/* gesture.c
 *
 * (c)12/3/1994 Stuart N. John
 *
 *
 * History:
 *
 *   12-3-1994 - Created module
 *
 */

#include <sys/types.h>
#include <stdio.h>

#include "glove.h"
#include "mode.h"
#include "client.h"
#include "posture.h"
#include "feature.h"
#include "gesture.h"


/* global variables */

int           client_id = -1;  /* identifier for client connection        */
Glove_t       *glove;          /* current client glove data sample        */
Glove_t       g;               /* local copy of client glove data sample  */
Glove_t       prevg;           /* previous glove data sample              */
Vector_t      ps;              /* current path segment vector             */
double        lps;             /* length of current path segment vector   */
Vector_t      nps;             /* current normalised path segment vector  */
Vector_t      prevnps;         /* previous normalised path segment vector */
Vector_t      ss;              /* path segment speed                      */
double        ls;              /* current linear speed                    */
Vector_t      ncp;             /* normalised cross product vector         */
double        ndp;             /* normalised dot product vector           */
postureData_t *pd;             /* current posture data                    */

char          d_tokens[4];     /* movement direction tokens */
char          shape_tokens[3]; /* basic shape tokens        */
char          poly_tokens[4];  /* polygonal shape tokens    */


/* prototypes */

static int calculate_features(void);
static int check_circling(void);
static int d_tokeniser(void);
static int shape_tokeniser(void);
static int rectangle(void);


/* gesture control functions */

int init_gestures(void)
{
  /* open a connection to the glove server */

  client_id = client_get_service("pgloved");
  if (client_id < 0)
  {
    fprintf(stderr, "gesture: sorry, can't use the glove\n");
    return -1;
  }

  /* set client connection mode parameters */

  client_set_datalist(client_id, PGDATA_FLEX | PGDATA_WRIST | PGDATA_XYZ);
  client_set_comms(client_id, PGCOMMS_TWOWAY);
  client_set_filter(client_id, PGFILTER_ON);

  client_get_datalist(client_id);
  client_get_comms(client_id);
  client_get_filter(client_id);

  /* calculate initial features */

  /* get an initial glove data sample and store it */

  glove = client_get_record(client_id);
  if (glove == NULL)
  {
    fprintf(stderr, "gesture: couldn't get glove data\n");
    quit_gestures();
    return -1;
  }

  memcpy(&g, glove, sizeof(Glove_t));

  /* zero normalised initial path segment vector */

  nps.x = 0.0;
  nps.y = 0.0;
  nps.z = 0.0;

  if (calculate_features() < 0)
  {
    fprintf(stderr, "gesture: failed to calculate features\n");
    return -1;
  }

  /* initialise the posture look-up table */

  create_posture_table();

  fprintf(stderr, "gesture: posture table created\n");

  /* return successfully to application program */

  return client_id;
}


int recognise_gesture(gestureData_t *gesture)
{
  char direction;

  if (client_id < 0)
  {
    fprintf(stderr, "gesture: please call init_gestures() first\n");
    return -1;
  }

  /* check different gestures, eg. rectangle */

  rectangle();

  if (!strcmp(poly_tokens, "REC"))
    gesture->id = 0;
  else
    gesture->id = -1;

  gesture->glove   = &g;
  gesture->posture = pd;

  return 0;
}


/*
 * Return after one glove sample for real time glove tracking
 *
 */
int realtime_tracking(gestureData_t *ges)
{
  if (client_id < 0)
  {
    fprintf(stderr, "gesture: please call init_gestures() first\n");
    return -1;
  }

  if (calculate_features() < 0)
  {
    fprintf(stderr, "gesture: failed to calculate features\n");
    return -1;
  }

  ges->id      = 0;
  ges->glove   = &g;
  ges->posture = pd;

  return 0;
}


static int calculate_features(void)
{
  /* get a new glove data sample */

  glove = client_get_record(client_id);
  if (glove == NULL)
  {
    fprintf(stderr, "gesture: couldn't get glove data\n");
    quit_gestures();
    return -1;
  }

  /* update the previous glove sample and copy the current data */

  memcpy(&prevg, &g, sizeof(Glove_t));
  memcpy(&g, glove, sizeof(Glove_t));

  /* calculate the path segment vector */

  path_segment(&g, &prevg, &ps);

  /* calculate the length of the path segment vector */

  lps = length_path_segment(&ps);

  /* update previous normalised path segment vector and calculate new one */

  prevnps = nps;

  if (lps == 0.0)
  {
    /* no vector change, so i guess we zero it ? */

    nps.x = 0.0;
    nps.y = 0.0;
    nps.z = 0.0;
  }
  else
    norm_path_segment(&ps, lps, &nps);

  /* calculate normalised cross product for perpendicular vectors */

  norm_cross_product(&nps, &prevnps, &ncp);

  ndp = norm_dot_product(&nps, &prevnps);

  /* calculate speeds */

  segment_speed(&ps, &g, &ss);
  ls = linear_speed(lps, &g);

  /* classify the hand posture */

  pd = hand_posture(&g);
}


/*
 * function to tidy up memory when finished with gestures
 *
 */
void quit_gestures(void)
{
  delete_posture_table();

  fprintf(stderr, "gesture: posture table deleted\n");

  /* close connection to glove server */

  client_quit(client_id);
}


/*
 * Circling - function to recognise X-Z, X-Y, Y-Z circling gestures
 *
 */
int check_circling(void)
{
  if (ncp.x > 0.6 || ncp.x < -0.6)      return CIRCLE_XZ;
  else if (ncp.y > 0.6 || ncp.y < -0.6) return CIRCLE_YZ;
  else if (ncp.z > 0.6 || ncp.z < -0.6) return CIRCLE_XY;
  else return -1;
}


/*
 * Translates the direction vectors into single character tokens
 *
 */
static int d_tokeniser(void)
{
  int index = 0;

  if (calculate_features() < 0)
  {
    fprintf(stderr, "gesture: failed to calculate features\n");
    return -1;
  }

  /* recognise right-left movements */

  if (nps.x > 0.0)
  {
    if (nps.x < 0.4)
      d_tokens[index] = 'r';
    else
      d_tokens[index] = 'R';
  }
  else if (nps.x < 0.0)
  {
    if (nps.x > -0.4)
      d_tokens[index] = 'l';
    else
      d_tokens[index] = 'L';
  }
  else
    d_tokens[index] = '-';

  ++index;

  /* recognise up-down movements */

  if (nps.y > 0.0)
  {
    if (nps.y < 0.4)
      d_tokens[index] = 'u';
    else
      d_tokens[index] = 'U';
  }
  else if (nps.y < -0.0)
  {
    if (nps.y > -0.4)
      d_tokens[index] = 'd';
    else
      d_tokens[index] = 'D';
  }
  else
    d_tokens[index] = '-';

  ++index;

  /* recognise back-forward movements */

  if (nps.z > 0.0)
  {
    if (nps.z < 0.4)
      d_tokens[index] = 'b';
    else
      d_tokens[index] = 'B';
  }
  else if (nps.z < -0.0)
  {
    if (nps.z > -0.4)
      d_tokens[index] = 'f';
    else
      d_tokens[index] = 'F';
  }
  else
    d_tokens[index] = '-';

  fprintf(stderr, "D= %s", d_tokens);

  return 0;
}


/*
 * Translates the direction tokens into basic shape components
 * (right angles, etc. )
 *
 */
static int shape_tokeniser(void)
{
  static char st[] = { '-', '-', '\0' };  /* initially invalid directions */

  strcpy(shape_tokens, "--");

  /* get new direction tokens */

  if (d_tokeniser() < 0)
  {
    fprintf(stderr, "gesture: failed to tokenise movement directions\n");
    return -1;
  }

  /* examine principal movements, ie. upper case direction tokens */

  if (islower(d_tokens[0]) && islower(d_tokens[1]))
  {
    fprintf(stderr, "  S= %s\r", shape_tokens);
    return 0;
  }
  else if (isupper(d_tokens[0]))  /* check first axis */
  {
    if (isupper(d_tokens[1]))  /* if second axis also true then error */
    {
      fprintf(stderr, "  S= %s\r", shape_tokens);
      return 0;
    }
    else  /* use the first axis token (if different to last) */
    {
      if (st[1] != d_tokens[0])
      {
        /* get rid of old direction token */

        st[0] = st[1];

        st[1] = d_tokens[0];
      }
    }
  }
  else
  {
    if (st[1] != d_tokens[1])
    {
      /* get rid of old direction token */

      st[0] = st[1];

      st[1] = d_tokens[1];
    }
  }

  /* try to match rectangle corners based on hand movements */
  
  if (!strcmp(st, "DR") || !strcmp(st, "LU"))
  {
    strcpy(shape_tokens, "BL");
    fprintf(stderr, "  S= %s\r", shape_tokens);
    return 0;
  }

  if (!strcmp(st, "RU") || !strcmp(st, "DL"))
  {
    strcpy(shape_tokens, "BR");
    fprintf(stderr, "  S= %s\r", shape_tokens);
    return 0;
  }

  if (!strcmp(st, "UL") || !strcmp(st, "RD"))
  {
    strcpy(shape_tokens, "TR");
    fprintf(stderr, "  S= %s\r", shape_tokens);
    return 0;
  }

  if (!strcmp(st, "LD") || !strcmp(st, "UR"))
  {
    strcpy(shape_tokens, "TL");
    fprintf(stderr, "  S= %s\r", shape_tokens);
    return 0;
  }

  fprintf(stderr, "  S= %s\r", shape_tokens);
}


/*
 * Translates shape component tokens into a polygonal shape
 * (ie. rectangle)
 *
 */
static int rectangle(void)
{
  static char rectangle[] = { '-', '-', '-', '-', '-', '-', '-', '-', '\0' };
  int         index = 2;
  int         i;

  /* attempt to get the four corners of a rectangle */

  do
  {
    if (shape_tokeniser() < 0)
    {
      fprintf(stderr, "gesture: failed to tokenise shape components\n");
      return -1;
    }
  } while (!strcmp(shape_tokens, "--"));

  rectangle[0] = shape_tokens[0];
  rectangle[1] = shape_tokens[1];

  i = 0;
  while (i < 3)
  {
    if (shape_tokeniser() < 0)
    {
      fprintf(stderr, "gesture: failed to tokenise shape components\n");
      return -1;
    }

    if (rectangle[index-2] == shape_tokens[0] &&
        rectangle[index-1] == shape_tokens[1])
    {
      continue;
    }
    else
    {
      if (strcmp(shape_tokens, "--"))
      {
        rectangle[index++] = shape_tokens[0];
        rectangle[index++] = shape_tokens[1];
        i++;
      }
    }
  }

  fprintf(stderr, "\nREC= %s\n", rectangle);

  if (!strcmp(rectangle, "TLTRBRBL") ||
      !strcmp(rectangle, "TRBRBLTL") ||
      !strcmp(rectangle, "BRBLTLTR") ||
      !strcmp(rectangle, "BLTLTRBR") ||
      !strcmp(rectangle, "TLBLBRTR") ||
      !strcmp(rectangle, "BLBRTRTL") ||
      !strcmp(rectangle, "BRTRTLBL") ||
      !strcmp(rectangle, "TRTLBLBR"))
  {
    strcpy(poly_tokens, "REC");
  }
  else
  {
    strcpy(poly_tokens, "---");
  }

  return 0;
}
