/* client.c
 *
 * (c)19/12/1993 Stuart N. John
 *
 *
 * History:
 *
 *   19-12-1993 - Created module
 *   29-12-1993 - Added TCP/IP networking
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <stdio.h>

#include "powerglove.h"
#include "glove.h"
#include "mode.h"
#include "client.h"
#include "server.h"
#include "message.h"


typedef struct server
{
  char          *server;  /* name of server                       */
  char          *host;    /* host name on which server is running */
  int           port;     /* server master socket                 */
  int           socket;   /* client connection socket             */
  Glove_t       data;     /* glove readings from this server      */
  int           ref;      /* number of connections to this server */
  struct server *next;    /* data for next different server       */
  struct server *prev;    /* data for previous different server   */
} serverData_t;


typedef struct cd
{
  int       id;     /* id number of this client connection      */
  Glove_t   *data;  /* glove readings from server               */
  Mode_t    mode;   /* mode control flags for server connection */
  struct cd *next;
  struct cd *prev;
} clientData_t;


/* global variables */

int new_client_identifier = 0;  /* new client connection identifier num  */
serverData_t *s_list  = NULL;   /* list of server connections            */
clientData_t *cd_list = NULL;   /* list of client connections            */


/* prototypes */
static int client_set_byte(Msg_id msg, clientData_t *client, int flags);
static int client_get_byte(Msg_id send_msg, Msg_id reply_msg, int id);

static int fill_glove_structure(Glove_t *glove, char *data);

static clientData_t *add_client(serverData_t *server);
static clientData_t *find_client(int id);
static int delete_client(clientData_t *client);

static serverData_t *add_server(char *server, char *host, int port, int socket);
static serverData_t *find_server(Glove_t *data);
static serverData_t *match_server(char *server, char *host);
static int delete_server(serverData_t *serv);


/* functions */

int client_get_service(char *server)
{
  clientData_t *client;    /* pointer to new client structure       */
  serverData_t *serv;      /* pointer to (new) server structure     */
  struct protoent *ppe;    /* pointer to protocol information entry */
  struct sockaddr_in sin;  /* Internet address for master server    */
  int port = PORT;         /* server connection port                */
  int s;                   /* socket descriptor                     */

  /* see if we're already connected to this server */

  serv = match_server(server, HOST);

  if (serv)
    fprintf(stderr, "client: server '%s' already in use\n", server);
  else
  {
    /* we need to open a new client-server connection */

    fprintf(stderr, "client: connecting to new server '%s'\n", server);

    /* open a new socket to the named server */

    bzero((char *) &sin, sizeof(sin));
    sin.sin_family      = AF_INET;
    sin.sin_addr.s_addr = inet_addr(HOST);
    sin.sin_port        = htons((u_short) port);

    /* map protocol name to protocol number */

    if ((ppe = getprotobyname("tcp")) == 0)
    {
      fprintf(stderr, "client: can't get tcp protocol entry\n");
      perror("client");
      return -1;
    }

    /* allocate a socket */

    s = socket(AF_INET, SOCK_STREAM, ppe->p_proto);
    if (s < 0)
    {
      fprintf(stderr, "client: can't open stream socket fd\n");
      perror("client");
      return -1;
    }

    /* attempt connection to the master server socket */

    if (connect(s, (struct sockaddr *) &sin, sizeof(sin)) < 0)
    {
      fprintf(stderr, "client: can't connect to server\n");
      perror("client");
      close(s);
      return -1;
    }

    if (fcntl(s, F_SETFL, O_NONBLOCK) < 0)
    {
      fprintf(stderr,"client: can't set no-blocking\n");
      perror("client");
    }
  }

  if (!serv)
  {
    serv = add_server(server, HOST, port, s);
    if (serv == NULL)
    {
      fprintf(stderr, "client: could not store server information\n");
      close(s);
      return -1;
    }
  }

  /* increment reference count to this server */
  ++serv->ref;

  /* store new client information */

  client = add_client(serv);
  if (client == NULL)
  {
    fprintf(stderr, "client: could not store client information\n");
    delete_server(serv);
    return -1;
  }

  return client->id;  /* return client identifier number */
}


Glove_t *client_get_record(int id)
{
  clientData_t *client;          /* client for this connection       */
  serverData_t *serv;            /* server for this connection       */
  static char  buffer[G_BYTES];  /* temp input buffer for glove data */
  int          msgid;            /* message id of server reply       */

  /* find client & its server based on numeric identifier */

  client = find_client(id);
  if (client == NULL)
  {
    fprintf(stderr, "client: invalid identifier, can't get client data record\n");
    return NULL;
  }

  serv = find_server(client->data);
  if (serv == NULL)
  {
    fprintf(stderr, "client: not connected to server, can't get glove data record\n");
    return NULL;
  }

  /* request latest glove data from the server if in polling mode */

  if (client->mode.comms == PGCOMMS_TWOWAY)
  {
    if (send_message(PG_GET_GLOVEDATA, serv->socket, NULL, NULL) < 0)
    {
      fprintf(stderr, "client: error sending request for glove data\n");
      return NULL;
    }
  }

  /* get server message containing new glove data */

  msgid = get_message(serv->socket, &(client->mode), buffer);

  if (msgid != PG_REPLY_GLOVEDATA)
  {
    fprintf(stderr, "client: error reading glove data from server\n");
    return NULL;
  }

  /* fill server glove structure from powerglove bytes */

  fill_glove_structure(&(serv->data), buffer);

  /* return the glove data for this client */

  return client->data;
}


int client_set_datalist(int id, int datalist)
{
  clientData_t *client;  /* client for this connection */

  /* find client based on numeric identifier */

  client = find_client(id);
  if (client == NULL)
  {
    fprintf(stderr, "client: invalid identifier, can't get client data record\n");
    return -1;
  }

  /* store new value for the datalist combination in the client */

  client->mode.datalist = (char) datalist;

  /* send this new value to the server */

  if (client_set_byte(PG_SET_DATALIST, client, datalist) == -1)
  {
    fprintf(stderr, "client: can't set datalist combination\n");
    return -1;
  }

  return 0;
}


int client_get_datalist(int id)
{
  if (client_get_byte(PG_GET_DATALIST, PG_REPLY_DATALIST, id) == -1)
  {
    fprintf(stderr, "client: can't get datalist combination\n");
    return -1;
  }

  return 0;
}


int client_set_comms(int id, int mode)
{
  clientData_t *client;  /* client for this connection */

  /* find client based on numeric identifier */

  client = find_client(id);
  if (client == NULL)
  {
    fprintf(stderr, "client: invalid identifier, can't get client data record\n");
    return -1;
  }

  /* store new value for the communication mode in the client */

  client->mode.comms = (char) mode;

  /* send this new value to the server */

  if (client_set_byte(PG_SET_COMMS, client, mode) == -1)
  {
    fprintf(stderr, "client: can't set communication mode\n");
    return -1;
  }

  return 0;
}


int client_get_comms(int id)
{
  if (client_get_byte(PG_GET_COMMS, PG_REPLY_COMMS, id) == -1)
  {
    fprintf(stderr, "client: can't get communication mode\n");
    return -1;
  }

  return 0;
}


int client_set_filter(int id, int mode)
{
  clientData_t *client;  /* client for this connection */

  /* find client based on numeric identifier */

  client = find_client(id);
  if (client == NULL)
  {
    fprintf(stderr, "client: invalid identifier, can't get client data record\n");
    return -1;
  }

  /* store new value for the filter mode in the client */

  client->mode.filter = (char) mode;

  /* send this new value to the server */

  if (client_set_byte(PG_SET_FILTER, client, mode) == -1)
  {
    fprintf(stderr, "client: can't set filter mode\n");
    return -1;
  }

  return 0;
}


int client_get_filter(int id)
{
  if (client_get_byte(PG_GET_FILTER, PG_REPLY_FILTER, id) == -1)
  {
    fprintf(stderr, "client: can't get filter mode\n");
    return -1;
  }

  return 0;
}


int client_quit(int id)
{
  clientData_t *client;  /* client for this connection */

  /* find client based on numeric identifier */

  client = find_client(id);
  if (client == NULL)
  {
    fprintf(stderr, "client: invalid identifier, client doesn't exist, can't terminate\n");
    return 0;
  }

  delete_client(client);

  fprintf(stderr, "client: client terminated\n");

  return 0;
}


/*
 * Send one byte of data to the server
 *
 */
static int client_set_byte(Msg_id msg, clientData_t *client, int flags)
{
  serverData_t *serv;  /* server for this connection */

  serv = find_server(client->data);
  if (serv == NULL)
  {
    fprintf(stderr, "client: not connected to server, can't set flags\n");
    return -1;
  }

  if (send_message(msg, serv->socket, &(client->mode), NULL) < 0)
  {
    fprintf(stderr, "client: error sending request message to server\n");
    return -1;
  }

  return 0;
}


/*
 * Get one byte of data from the server
 *
 */
static int client_get_byte(Msg_id send_msg, Msg_id reply_msg, int id)
{
  clientData_t *client;  /* client for this connection                      */
  serverData_t *serv;    /* server for this connection                      */
  int          msgid;    /* message id of server reply                      */
  int          try = 3;  /* try at most 3 times to get correct server reply */

  /* find client & its server based on numeric identifier */

  client = find_client(id);
  if (client == NULL)
  {
    fprintf(stderr, "client: invalid identifier, can't get client data record\n");
    return -1;
  }

  serv = find_server(client->data);
  if (serv == NULL)
  {
    fprintf(stderr, "client: not connected to server, can't get flags\n");
    return -1;
  }

  if (send_message(send_msg, serv->socket, &(client->mode), NULL) < 0)
  {
    fprintf(stderr, "client: error sending request message to server\n");
    return -1;
  }

  /* try at most 3 times to get correct server reply */

  do
  {
    msgid = get_message(serv->socket, &(client->mode), NULL);
  } while ((msgid != reply_msg) && try--);

  if (msgid != reply_msg)
  {
    fprintf(stderr, "client: error reading flags from server\n");
    return -1;
  }

  return 0;
}


/*
 * fill the abstract glove structure with latest glove data
 *
 */
static int fill_glove_structure(Glove_t *glove, char *data)
{
  /* fill glove data structure from powerglove bytes */

  setX(glove, data[G_X]);
  setY(glove, data[G_Y]);
  setZ(glove, data[G_Z]);
  setRotation(glove, data[G_W]);
  setFingers (glove, (u_char) data[G_F]);
  setKey(glove, data[G_K]);
  setTime(glove, data[G_T]);

#ifdef DEBUGDATA
  fprintf(stderr, "x:%4d ", getX(glove));
  fprintf(stderr, "y:%4d ", getY(glove));
  fprintf(stderr, "z:%4d ", getZ(glove));
  fprintf(stderr, "r:%2d ", getRotation(glove));
  fprintf(stderr, "k:%2x ", getKey(glove));
  fprintf(stderr, "t:%3d ", getTime(glove));
  fprintf(stderr, "   \r");
#endif

  return 0;
}


/*
 * Client list routines
 *
 */
static clientData_t *add_client(serverData_t *server)
{
  clientData_t *client;  /* pointer to new client structure */

  /* create a new client structure and put it in the
     client list so we can keep track of it */

  client = (clientData_t *) malloc(sizeof(clientData_t));
  if (client == NULL) return NULL;

  /* point to glove data in the server list */

  client->data = &(server->data);

  /* assign an identifier number for this client */

  client->id = new_client_identifier++;  /* from global variable */

  if (cd_list)  /* normal double linked list pointer adjustment */
  {
    client->next  = cd_list;
    client->prev  = cd_list->prev;
    client->next->prev = client;
    client->prev->next = client;
  }
  else
    client->next = client->prev = client;  /* this is the only client in the list */

  cd_list = client;

  /* get default mode settings from the server */

  if (client_get_datalist(client->id) < 0)
  {
    delete_client(client);
    return NULL;
  }

  if (client_get_comms(client->id) < 0)
  {
    delete_client(client);
    return NULL;
  }

  if (client_get_filter(client->id) < 0)
  {
    delete_client(client);
    return NULL;
  }

  return client;
}


static clientData_t *find_client(int id)
{
  clientData_t *client;  /* pointer to client structure in list */

  if (cd_list)  /* make sure the client list is not empty */
  {
    client = cd_list;
    do
    {
      if (client->id == id) return client;
      client = client->next;
    } while (client != cd_list);
  }

  fprintf(stderr, "client: can't find client id %d\n", id);
  return NULL;
}


static int delete_client(clientData_t *client)
{
  serverData_t *serv;

  /* find the server data this client is using, and attempt to delete it */

  serv = find_server(client->data);
  if (serv) delete_server(serv);

  if (cd_list == client)  /* see if we're deleting the head of the list */
  {
    if (client == client->next)  /* only this element in the list */
    {
      cd_list = NULL;
      free(client);

      return 0;
    }

    cd_list = client->next;  /* move the start of list to the next client */
  }

  client->prev->next = client->next;  /* un-hook pointers from other clients */
  client->next->prev = client->prev;

  free(client);

  return 0;
}


/*
 * Server list routines
 *
 */
static serverData_t *add_server(char *server, char *host, int port, int socket)
{
  serverData_t *serv;  /* pointer to (new) server structure     */

  /* create a new server structure and put it in the
     server list so we can keep track of it */

  serv = (serverData_t *) malloc(sizeof(serverData_t));
  if (serv == NULL) return NULL;

  /* no clients using this server yet */
  serv->ref = 0;

  serv->server = (char *) malloc(strlen(server) + 1);
  if (serv->server == NULL)
  {
    fprintf(stderr, "client: could not store server name\n");
    free(serv);
    return NULL;
  }

  strcpy(serv->server, server);  /* store server name */

  serv->host = (char *) malloc(strlen(host) + 1);
  if (serv->host == NULL)
  {
    fprintf(stderr, "client: could not store host name\n");
    free(serv->server);
    free(serv);

    return NULL;
  }

  strcpy(serv->host, host);  /* store host name */

  serv->port   = port;    /* store server port number */
  serv->socket = socket;  /* store socket number      */

  if (s_list)  /* normal double linked list pointer adjustment */
  {
    serv->next  = s_list;
    serv->prev  = s_list->prev;
    serv->next->prev = serv;
    serv->prev->next = serv;
  }
  else
    serv->next = serv->prev = serv;  /* this is the only client in the list */

  s_list = serv;

  return serv;
}


static serverData_t *find_server(Glove_t *data)
{
  serverData_t *serv;  /* pointer to server structure in list */

  if (s_list)  /* make sure the server list is not empty */
  {
    serv = s_list;
    do
    {
      if (&(serv->data) == data) return serv;
      serv = serv->next;
    } while (serv != s_list);
  }

  fprintf(stderr, "client: can't find server information\n");
  return NULL;
}


static serverData_t *match_server(char *server, char *host)
{
  serverData_t *serv;  /* pointer to server structure in list */

  if (s_list)  /* make sure the server list is not empty */
  {
    serv = s_list;
    do
    {
      if ((!strcmp(server, serv->server)) && (!strcmp(host, serv->host))) return serv;
      serv = serv->next;
    } while (serv != s_list);
  }

  fprintf(stderr, "client: server '%s' not in use\n", server);
  return NULL;
}


static int delete_server(serverData_t *serv)
{
  /* a client is to be deleted, decrement the reference count */

  --serv->ref;

  /* check reference count, if > 0 then there are other clients using it */

  if (serv->ref > 0) return -1;

  /* send PG_QUIT message to the server */

  send_message(PG_QUIT, serv->socket, NULL, NULL);

  /* shut down the socket */

  close(serv->socket);

  if (s_list == serv)  /* see if we're deleting the head of the list */
  {
    if (serv == serv->next)  /* only this element in the list */
    {
      s_list = NULL;

      free(serv->server);
      free(serv->host);
      free(serv);

      return 0;
    }

    s_list = serv->next;  /* move the start of list to the next server */
  }

  serv->prev->next = serv->next;  /* un-hook pointers from other servers */
  serv->next->prev = serv->prev;

  free(serv->server);
  free(serv->host);
  free(serv);

  return 0;
}
