/*
   Level Connector 1.0beta

   Reads RAW or ASCII pgm file from stdin or from
   file specified in command line and writes binary
   pgm file to stdout.

   The input is treated as height field and the
   heights of pixels marked as free are calculated
   so that different levels are connected smoothly.
   
   The heights are found by solving Poisson's 2nd
   order differential equation in the free areas
   using modified Gauss-Seidel algorithm.

   Gauss-Seidel iteration calculates approximations
   for the [free] pixels based on previous values.
   [Inithgt] is used as the first approximation. The
   iteration stops when [maxiter] iterations are made
   or maximum difference between iterations is less
   than [limres]. If max difference reported by [stat]
   is much greater than 1 it means that the approximation
   is far from the final value and [maxiter] should be
   increased. That is, if a better solution for the
   Poisson's equation is desired. 
   
   If the input file contains no fixed areas inside
   free areas then the approximations go towards
   flat surface and [limres] should be set to 0.0.
   The iteration is in this case controlled by
   [maxiter] only.

   [Force] pushes the free areas upwards. Values greater
   than ~2 give unpredictable resuls. Negative force
   pushes downwards.

   If a pgm is generated using
   > giftopnm foo.gif| ppmtopgm > foo.pgm
   it may not preserve the original levels and some
   guessing is required to find the right [free] value.

   > giftoppm foo.gif| ppmtopgm > foo.pgm
   works fine.

   Kari Kivisalo
   kkivisal@vipunen.hut.fi
   http://www.hut.fi/~kkivisal/

*/
   

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#define MAXNODES 100000
#define RAW 5
#define ASCII 2

unsigned char *pgm;
long pgmSize,PixN,Node,*Pos,Nodes=0;
int pgmWidth,pgmHeight,inithgt,maxhgt,freepixv;
int maxiter,stat,ItrStat=1,Itr,*s[MAXNODES],levels=0;
float *u,*h,limres,force,MaxD;
FILE *input;

int pm_keymatch(char *arg,char *opt,int minlen)
{
  for (; *arg && *arg == *opt; arg++, opt++)
    minlen--;
  return !*arg && minlen <= 0;
}


int Load_pgm(void)
{
  int pgmType,mgv,byte;
  char line[100],cbyte;

  if(fscanf(input,"P%i \n",&pgmType) != 1) return(1);

  if((pgmType != RAW) && (pgmType != ASCII)) return(1);

  fgets(line,100,input);

  while(line[0] == '#')
    fgets(line,100,input);
      
  if(sscanf(line," %i %i %i \n",&pgmWidth,&pgmHeight,&mgv) == 2)
    fscanf(input," %i \n",&mgv);

  if((pgmSize=pgmHeight*pgmWidth) == 0) return(1);
  pgm=(unsigned char *)malloc(pgmSize);

  switch(pgmType)
    {
    case RAW :
      fread(pgm,1,pgmSize,input);
      break;
    case ASCII :
      for(PixN=0;PixN<pgmSize;PixN++)
	{ fscanf(input,"%i",&byte);
	  pgm[PixN]=(unsigned char)byte;
        }
      break;
    }

  fclose(input);
  return(0);
}

int comp(long *a,long *b)
{
  if (*a < *b) return -1; 
  else if (*a == *b) return 0;
  else return 1;
}

void MakeEquations(void)
{
  int xo[4]={1,-1,0,0},yo[4]={0,0,1,-1};
  long c,*NodeP;
  int x,y,xt,yt,i,c2,PixV,k,v[256]={0};

  maxhgt=0;

  for(PixN=0;PixN<pgmSize;PixN++)
    { PixV=pgm[PixN];
      if(PixV == freepixv)
        Nodes++;
      else
        { v[PixV]=1;
	  if(PixV > maxhgt) maxhgt=PixV;
	}
    }

  for(i=0;i<256;i++)
    levels=levels + v[i];

  if(levels == 1) maxhgt=255;

  if (Nodes > MAXNODES) {perror("Too many free pixels. (increase MAXNODES)");exit(1);}
  if (Nodes == 0) {perror("Not any free pixels.");exit(1);}

  u=(float *)malloc(Nodes*sizeof(float));
  h=(float *)malloc(Nodes*sizeof(float));
  Pos=(long *)malloc(Nodes*sizeof(long));

  for(Node=0;Node<Nodes;Node++)
  { int j;
    s[Node]=(int *)malloc(5*sizeof(int));
    if (s[Node] == NULL) {perror("Out of memory.");exit(1);}
    for(j=0;j<5;j++)
      s[Node][j]=0;
    u[Node]=inithgt;
    h[Node]=0;
    Pos[Node]=0;
  }

  c=0;
  for(PixN=0;PixN<pgmSize;PixN++)
    { if (pgm[PixN] == freepixv)
	{ Pos[c]=PixN;
	  c++;
        }
    }

  c=0;
  for(y=0;y<pgmHeight;y++)
    { for(x=0;x<pgmWidth;x++)
	{ if (pgm[y*pgmWidth+x] == freepixv)
	    { k=0;c2=0;
	      for(i=0;i<4;i++)
		{ yt=y+yo[i];
		  xt=x+xo[i];
		  if (yt == -1) yt=pgmHeight-1;
		  if (yt == pgmHeight) yt=0;
		  if (xt == -1) xt=pgmWidth-1;
		  if (xt == pgmWidth) xt=0;
		  PixN=yt*pgmWidth+xt;
		  PixV=pgm[PixN];
		  if (PixV != freepixv) k=k+PixV;
		  else
		    { c2++;
	              NodeP=(long *)bsearch((void *)&PixN,(void *)Pos,Nodes,sizeof(long),comp);
		      if (NodeP == NULL)
			{ perror("Error in bsearch."); exit(1); }

		      s[c][c2]=(int)(NodeP-Pos);
		    }
		}
	      h[c]=force-k;
	      s[c][0]=c2;
	      c++;
	    }
	}
    }
}

void SolveEquations()
{
  int i,c;
  float maxdiff,sum;

  for(i=0;i<maxiter;i++)
    { maxdiff=0;
      for(Node=0;Node<Nodes;Node++)
	{ sum=-1.0*h[Node];
	  for(c=1;c<=s[Node][0];c++)
	    sum=sum+0.25*u[s[Node][c]];
          if(fabs(u[Node]-sum)>maxdiff) maxdiff=fabs(u[Node]-sum);
	  u[Node]=sum;
        }
      if(maxdiff<limres)
        {ItrStat=0;break;}
    }

  MaxD=maxdiff;
  Itr=i;
  free(h);
}

void PrintUsage(void)
{
  fprintf(stderr,"usage: levcon\n");
  fprintf(stderr,
"       [-free #]     Value of free pixels in input file.    default: 255\n");
  fprintf(stderr,
"       [-inithgt #]  Initial height for the iteration.      default: 127\n");
  fprintf(stderr,
"       [-maxiter #]  Maximum number of iterations.          default:  50\n");
  fprintf(stderr,
"       [-limres #.#] Maximum difference between iterations. default: 1.0\n");
  fprintf(stderr,
"       [-force #.#]  Upward directed force.                 default: 0.0\n");
  fprintf(stderr,
"       [-stat]       Print some info.\n");
  fprintf(stderr,
"       [pgmfile]\n");
  exit(1);
}

void GetOptions(int argc,char *argv[])
{
  int i,argn;

  inithgt=127;
  maxhgt=255;
  maxiter=50;
  limres=1;
  force=0;
  stat=0;
  freepixv=255;
  
  argn=1;
  i=1;

  while ((i < argc) && (argv[i][0] == '-') && (argv[i][1] != '\0'))
    {
        if (pm_keymatch(argv[i], "-inithgt", 2))
	  { i++;argn=argn+2;
            if ((i == argc) || (sscanf(argv[i], "%i", &inithgt)  != 1))
                PrintUsage();
	  }

        else if (pm_keymatch(argv[i], "-maxhgt", 4))
	  { i++;argn=argn+2;
            if ((i == argc) || (sscanf(argv[i], "%i", &maxhgt)  != 1))
                PrintUsage();
	  }
        
        else if (pm_keymatch(argv[i], "-maxiter", 4))
	  { i++;argn=argn+2;
            if ((i == argc) || (sscanf(argv[i], "%i", &maxiter)  != 1))
                PrintUsage();
	  }
        else if (pm_keymatch(argv[i], "-limres", 2))
	  { i++;argn=argn+2;
            if ((i == argc) || (sscanf(argv[i], "%f", &limres)  != 1))
                PrintUsage();
	  }
        else if (pm_keymatch(argv[i], "-force", 2))
	  { i++;argn=argn+2;
            if ((i == argc) || (sscanf(argv[i], "%f", &force)  != 1))
                PrintUsage();
          }
        else if (pm_keymatch(argv[i], "-free", 2))
	  { i++;argn=argn+2;
            if ((i == argc) || (sscanf(argv[i], "%i", &freepixv)  != 1))
                PrintUsage();
          }
        else if (pm_keymatch(argv[i], "-stat", 2))
	  { argn=argn+1;
            stat=1;
	  }
        else
	  PrintUsage();
	i++;
      }

  if (argn < argc)
    { input=fopen(argv[argn],"rb");
      argn=argn+1;
    }
  else
    input=stdin;

  if(argn != argc)
    PrintUsage();
 
  force=-force;
}

void Write_pgm(void)
{
  float scale,maxval;

  maxval=u[0];

  for(Node=1;Node<Nodes;Node++)
    if (fabs(u[Node])>maxval) maxval=fabs(u[Node]);

  scale=(float)maxhgt/maxval;

  for(Node=0;Node<Nodes;Node++)
    pgm[Pos[Node]]=(unsigned char)(fabs(u[Node]) * scale);

  printf("P5\n%i %i\n%i\n",pgmWidth,pgmHeight,255);

  fwrite(pgm,1,pgmSize,stdout);
}

void PrintStat()
{
   fprintf(stderr,"Image size:       %ix%i\n",pgmWidth,pgmHeight);
   fprintf(stderr,"Levels:           %i\n",levels);
   fprintf(stderr,"Free pixels:      %li\n",Nodes);
   fprintf(stderr,"Iterations:       %i\n",Itr);
   fprintf(stderr,"Max iterations:   %i\n",maxiter);
   fprintf(stderr,"Max difference:   %f\n",MaxD);
   fprintf(stderr,"Resolution limit: %f\n",limres);
   fprintf(stderr,"Initial height:   %i\n",inithgt);
   fprintf(stderr,"Max height:       %i\n",maxhgt);
   fprintf(stderr,"Force:            %f\n",-force);
}

/*---------------------------------------------------------------*/
int main(int argc,char *argv[])
{
  GetOptions(argc,argv);

  if (Load_pgm())
    { perror("Error in input stream.");
      exit(1);
    }

  MakeEquations();

  SolveEquations();

  if (stat)
    PrintStat();

  Write_pgm();

  return(0);
}
