////////////////////////////////////////////////////////////////////////////////
//  Implementation of Light sources.                                          //  
//  LAST EDIT: Fri Aug  5 08:55:01 1994 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include "usefligh.h"

const char *RTN_POINT_LIGHT = "PointLight";
const char *RTSN_PL = "PL";

RT_Color RT_PointLight::shade( const RT_Vector &p, const RT_Vector &n, const RT_Vector &, const RT_Surface &sf, const RT_Camera *camera) {
    if (!get_on()) return RT_Color( 0, 0, 0 );

    // lighting model:
    // c = tf * (ca + cd + cs)

    // tf - is the slowest transmission factor
    // 0 means that there is no transmission

    // #### ca - ambient light ####
    // ca = 0

    // #### cd - diffuse light #### 
    // cd = sd * lc * (n * l) * la;
    // l - light source position vector
    // la - attenuation vector
    // sd - diffuse surface parameters

    // #### cs - specular light #### 
    // cs = ss * lc * (e * p)^se * la
    // ss - specular surface parameter
    // se - specular exponent

    RT_Vector l = get_origin() - p; 
    double d = l.ABS();
    double la = exp( -0.1 * d );
    l = l.UNITIZE();

    RT_Ray ray;
    ray.pt = p;
    ray.dir = l;

    if (n.DOT( l ) <= 0) return RT_Color( 0, 0, 0 );

    double tf = 1; 
    RT_InterSectionList inter;
    RT_IntersectPrimitiveFunc func( ray, inter);
    camera->get_scene()->doWithElements( &func);

    if (func.getInterSect()) {
	if (rt_RefractedShadows) {
	    class LFunctoid: public RT_GeneralListFunctoid {
		double xtf, xd;
		void exec( RT_GeneralListEntry *e, void *) {
		    if (xtf) {
			RT_InterSection *i = (RT_InterSection*)e;
			if ( (i->getT() > rt_RayEps ) 
			    && ( i->getT()  < (xd - rt_RayEps))
			    && (i->getSurface()->tran < xtf)
			    ) xtf = i->getSurface()->tran;
		    }
		}
	      public:
		LFunctoid(double _xd): xd( _xd ), xtf( 1 ) {}
		double get() { return xtf; }
	    } func( d );
	    
	    inter.doWithElements( &func );
	    tf = func.get();
	} else {
	    double t = inter.first()->getT();
	    if ( (t > rt_RayEps ) && (t< (d - rt_RayEps))) tf = 0;
	}
    } 

    // if anything blocks the ray totally:
    if (!tf) return RT_Color(0, 0, 0);

    RT_Color c = get_color() * tf * la;
    RT_Color cs = sf.diff * n.DOT( l );

    double dot = (p.UNITIZE()).DOT( l );
    RT_Color cd( 0, 0, 0 );
    if (dot > 0) cd = sf.spec * pow( dot, sf.shin );
    return get_color() * tf * la * ( cs + cd );
}

int RT_PointLight::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String} {Creates a new point light called {ARG 1 Name}. The short name is ", RTSN_PL, ".}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	new RT_PointLight( argv[1] ); 
	RTM_classReturn;
    }
    return res; 
}

#ifdef RTD_RSY

//Note:
//A point light is a hard task for a radiosity method.
//You should use a light emitting (extended) surface instead.
void RT_PointLight::toRSY(RT_RSYScene* rsc) {
    float pdim = 0.01;
    RT_Color rt_emit = get_color();
    RT_Vector rt_pos = get_origin();
    
    RT_RS_Scene *rscene = rsc->get_rscene(); //the radiosity scene
    RT_RS_DiffVtx* rs_dvtx;
    RT_RS_DiffArea* rs_dar;
    RT_RS_3DVector rs_pos, rs_pn;
    RT_RS_DiffRGB rs_no_reflx, rs_demit;
    
    rs_pos.x = rt_pos.x; rs_pos.y = rt_pos.y; rs_pos.z = rt_pos.z;
    rs_demit.r = rt_emit.r; rs_demit.g = rt_emit.g; rs_demit.b = rt_emit.b;
    rs_demit = rs_demit * 100000.;
    
    //get a new mesh number
    long mesh_nr = rscene->Areas->get_new_mesh_nr();
    
    long p_cnt = rscene->Points->get_diffcnt();
    long ar_cnt = rscene->Areas->get_diffcnt();
    
    //approx. the point light using a small light emitting "cube-sphere":
    //the 8 corner points:
    //front:
    //bottom:
    //left:
    { RT_RS_3DVector tmpv(-pdim,-pdim,-pdim); rs_pn = tmpv; }
    rs_dvtx = new RT_RS_DiffVtx(rs_pos + rs_pn, rs_pn, rs_no_reflx, curved);
    rscene->Points->insert(rs_dvtx, &rs_demit);
    //right: 
    { RT_RS_3DVector tmpv(pdim,-pdim,-pdim); rs_pn = tmpv; }
    rs_dvtx = new RT_RS_DiffVtx(rs_pos + rs_pn, rs_pn, rs_no_reflx, curved);
    rscene->Points->insert(rs_dvtx, &rs_demit);
    //upper:
    //right:
    { RT_RS_3DVector tmpv(pdim,-pdim,pdim); rs_pn = tmpv; }
    rs_dvtx = new RT_RS_DiffVtx(rs_pos + rs_pn, rs_pn, rs_no_reflx, curved);
    rscene->Points->insert(rs_dvtx, &rs_demit);
    //left:
    { RT_RS_3DVector tmpv(-pdim,-pdim,pdim); rs_pn = tmpv; }
    rs_dvtx = new RT_RS_DiffVtx(rs_pos + rs_pn, rs_pn, rs_no_reflx, curved);
    rscene->Points->insert(rs_dvtx, &rs_demit);
    //back:
    //bottom:
    //left:
    { RT_RS_3DVector tmpv(-pdim,pdim,-pdim); rs_pn = tmpv; }
    rs_dvtx = new RT_RS_DiffVtx(rs_pos + rs_pn, rs_pn, rs_no_reflx, curved);
    rscene->Points->insert(rs_dvtx, &rs_demit);
    //right:
    { RT_RS_3DVector tmpv(pdim,pdim,-pdim); rs_pn = tmpv; }
    rs_dvtx = new RT_RS_DiffVtx(rs_pos + rs_pn, rs_pn, rs_no_reflx, curved);
    rscene->Points->insert(rs_dvtx, &rs_demit);
    //upper:
    //right:
    { RT_RS_3DVector tmpv(pdim,pdim,pdim); rs_pn = tmpv; }
    rs_dvtx = new RT_RS_DiffVtx(rs_pos + rs_pn, rs_pn, rs_no_reflx, curved);
    rscene->Points->insert(rs_dvtx, &rs_demit);
    //left:
    { RT_RS_3DVector tmpv(-pdim,pdim,pdim); rs_pn = tmpv; }
    rs_dvtx = new RT_RS_DiffVtx(rs_pos + rs_pn, rs_pn, rs_no_reflx, curved);
    rscene->Points->insert(rs_dvtx, &rs_demit);
    //the 6 areas:
    //front:
    rs_dar = new RT_RS_DiffArea(p_cnt, p_cnt + 1, p_cnt + 2, p_cnt + 3,
				ar_cnt + 5, ar_cnt + 1, ar_cnt + 4, ar_cnt + 3, mesh_nr);
    rscene->Areas->insert(rs_dar);
    //right:
    rs_dar = new RT_RS_DiffArea(p_cnt + 1, p_cnt + 5, p_cnt + 6, p_cnt + 2,
				ar_cnt + 5, ar_cnt + 2, ar_cnt + 4, ar_cnt, mesh_nr);
    rscene->Areas->insert(rs_dar);
    //back:
    rs_dar = new RT_RS_DiffArea(p_cnt + 5, p_cnt + 4, p_cnt + 7, p_cnt + 6,
				ar_cnt + 5, ar_cnt + 3, ar_cnt + 4, ar_cnt + 1, mesh_nr);
    rscene->Areas->insert(rs_dar);
    //left:
    rs_dar = new RT_RS_DiffArea(p_cnt + 4, p_cnt, p_cnt + 3, p_cnt + 7,
				ar_cnt + 5, ar_cnt, ar_cnt + 4, ar_cnt + 2, mesh_nr);
    rscene->Areas->insert(rs_dar);
    //top:
    rs_dar = new RT_RS_DiffArea(p_cnt + 3, p_cnt + 2, p_cnt + 6, p_cnt + 7,
				ar_cnt, ar_cnt + 1, ar_cnt + 2, ar_cnt + 3, mesh_nr);
    rscene->Areas->insert(rs_dar);
    //bottom:
    rs_dar = new RT_RS_DiffArea(p_cnt, p_cnt + 4, p_cnt + 5, p_cnt + 1,
				ar_cnt + 3, ar_cnt + 2, ar_cnt + 1, ar_cnt, mesh_nr);
    rscene->Areas->insert(rs_dar);
}

#endif

const char *RTN_AMBIENT_LIGHT = "AmbientLight";
const char *RTSN_AL = "AL";

int RT_AmbientLight::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String} {Creates a new ambient light called {ARG 1 Name}. The short name is ", RTSN_AL, ".}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	new RT_AmbientLight( argv[1] ); 
	RTM_classReturn;
    }
    return res; 
}

RT_Color RT_AmbientLight::shade( const RT_Vector &, const RT_Vector &, const RT_Vector &, const RT_Surface &sf, const RT_Camera *) {
    if (!get_on()) return RT_Color( 0, 0, 0 );
    return get_color() * sf.ambi;
}

