#include <yart.h>
#include <deltabl/deltabl.h>

// ##### A class for constraint variables with Tcl access #####

const char *RTN_VARIABLE = "Variable";
const char *RTN_CONSTANT = "Constant";

class RT_Variable: public RT_Object, public DB_Variable {
    DB_Boolean cv;
    static int sF, sG, asF;
    static double sV, asV;
    static RT_ParseEntry table[];

    RT_String setExpr; 
    // something like "sp -radius"
    // will be expanded to "sp -radius 0.3" for instance

    RT_String getExpr;
    // something like "sp -get_radius"

  public:
    RT_Variable(char *c, const char *_set, const char *_get,
		DB_Boolean _cv = FALSE): 
    RT_Object( c ), DB_Variable(c, _cv), setExpr( _set ), getExpr( _get ) { cv = _cv; }

    const char *get_class() const { return RTN_VARIABLE; }
    const char *get_description() const { return "A parameterizable variable that may be used by DeltaBlue constraint solver."; }
    void print(FILE *f) const {
	fprintf( f, "%s %s {%s} {%s}\n", cv ? RTN_CONSTANT : RTN_VARIABLE, RT_Object::get_name(), (char*)setExpr, (char*)getExpr );
    }
    int isA(const char *_c) const { return RT_Object::isA( _c ) || DB_Variable::isA( _c ); }

    static int classCMD(ClientData, Tcl_Interp *, int, char *[]); 
    int objectCMD(char *argv[]) { 
	int r = 0;
	RT_parseTable( argv, table );
	if (sF) { set( sV ); r++; }
	if (asF) { assign( asV ); r++; }
	if (sG) {
	    char tmp[20]; r++;
	    RT_double2string( get(), tmp );
	    RT_Object::result( tmp );
	}
	return r;
    }
    void assign(double d) {
	DB_Constraint *editC = new DB_EditC(this, DBS_REQUIRED );
	if (editC->satisfied()) {
	    set( d );
	    DB_List *plan = editC->extractPlanFromConstraint();
	    plan->executePlan();
	    delete plan;
	}
	delete editC;
    }
    // the following set&get methods may be oberloaded in derived classes
    virtual void set(double d) {
	static RT_String tmp( 100 );
	static char tmp2[10];
	sprintf( tmp2, "%lf", d );
	tmp = setExpr; tmp+= ' '; tmp += tmp2;
	RT_eval( (char*)tmp );
    }
    virtual double get() {
	double val; 
	if (!RT_string2double( RT_eval( getExpr ), val )) 
	    rt_Output->errorVar( "Bad get function ", RT_Object::get_name(), "!", 0 );
	resResult();
	return val;
    }
};

int RT_Variable::sF, RT_Variable::sG; double RT_Variable::sV;
int RT_Variable::asF; double RT_Variable::asV;

RT_ParseEntry RT_Variable::table[] = {
    { "-assign", RTP_DOUBLE, (char*)&asV, &asF, "Assign a new {ARG 1 value} for the variable. This will invoke a constraint solving process.", RTPS_DOUBLE },
    { "-set", RTP_DOUBLE, (char*)&sV, &sF, "Set a {ARG 1 value} for the variable.", RTPS_DOUBLE },
    { "-get", RTP_NONE, 0, &sG, "Get the value of the variable.", RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_Variable::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{ ", argv[0], " { String String String} {Creates a {ARG 1 variable} for constraint solver. There must be specified a {ARG 2 set} expression and a {ARG 3 get} expression.}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	if (argc != 4) {
	    Tcl_AppendResult( ip, "Bad number of arguments. Try \"", argv[0], " ?\". ", 0 );
	    return TCL_ERROR;
	}
	new RT_Variable( argv[1], argv[2], argv[3], strcmp( argv[0], RTN_CONSTANT ) ? FALSE : TRUE ); 
	RTM_classReturn;
    }
    return res; 
}

// ##### A class for constraints with Tcl access #####

const char *RTN_CONSTRAINT = "Constraint";

class RT_Constraint: public RT_Object, public DB_Constraint {
    RT_String body; 
    DB_Variable **list; 

    // may be overloaded in subclasses:
    void execute() {
	char tmp[10];
	sprintf( tmp, "%i", whichMethod );
	Tcl_VarEval( rt_Ip, "proc _local {} { set whichMethod ", tmp, "\n", (char*)body, "}; _local", 0 );
    }
  public:
    RT_Constraint(char *, DB_Variable **, const char *, DB_Strength, int); 
    const char *get_class() const { return RTN_CONSTRAINT; }
    const char *get_description() const { return "A parameterizable constraint for Tcl access."; }
    void print(FILE *) const {}
    int isA(const char *_c) const { return RT_Object::isA( _c ) || DB_Constraint::isA( _c ); }

    static int classCMD(ClientData, Tcl_Interp *, int, char *[]); 
    int objectCMD(char *[]) { return 0; }
};

int RT_Constraint::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{ ", argv[0], " { String Variables_List String String} {Creates a {ARG 1 constraint}. There must be specified a list of {ARG 2 variables}, a procedure {ARG 3 body} and a {ARG 4 strength}.}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	if (argc != 5) {
	    Tcl_AppendResult( ip, argv[0], ": need four arguments: <name> <variables> <body> <strength>", 0 );
	    return TCL_ERROR;
	}
	int nr;
	char **xargv;
	Tcl_SplitList(ip, argv[2], &nr, &xargv);
	if ( !nr ) {
	    free((char*)xargv);
	    Tcl_AppendResult( ip , argv[0],": need at least one variable", 0 );
	    return TCL_ERROR;
	}

	// check the list elements and put them into a list:
	// the list will cleared in the constraint class

	DB_Variable **list = new DB_Variable*[ nr ];
	for (int i = 0; i < nr; i++ ) {
	    DB_Variable *v = (RT_Variable*)RT_Object::getObject( xargv[i] );
	    if (!v) {
		rt_Output->errorVar( argv[0],": No such variable: ", xargv[i], "!" , 0 );
		delete list;
		free((char*)xargv);
		return TCL_ERROR;
	    }
	    list[i] = v;
	}
	free((char*)xargv);
	
	DB_Strength str;
	if ((str = DB_string2strength( argv[4])) == -1) {
	    Tcl_AppendResult( ip, argv[0], ": ", " bad strength: ", argv[4], "! Must be one of the following: ", 0 );
	    char **tmp = DB_STRENGTH_NAMES;
	    while (*tmp) Tcl_AppendResult( ip, *tmp++, " ", 0 );
	    return TCL_ERROR;
	}
	new RT_Constraint( argv[1], list, argv[3], str, nr ); 
	RTM_classReturn;
    }
    return res;
}

RT_Constraint::RT_Constraint(char *n, DB_Variable **_list, const char *_body, DB_Strength str, int nr)
: RT_Object( n ), DB_Constraint( nr, str ), body( _body ), list( _list ) {
    methodCount = nr;
    for (int i = 0; i < nr; i++ ) {
	variables[i] = list[i];
	methodOuts[i] = i;
    }
    add();
}

#define RTM_init() \
if ( RT_init() != TCL_OK) exit( 1 );\
DB_init();\
RTM_command( RTN_VARIABLE, RT_Variable::classCMD );\
RTM_command( RTN_CONSTANT, RT_Variable::classCMD );\
RTM_command( RTN_CONSTRAINT, RT_Constraint::classCMD );\

#include <rtsh.C>



