/*
 * DrawTreeNode.cc -- member routines for class DrawTreeNode
 * 
 * -----------------------------------------------------------------------------
 * Copyright 1993 Allan Brighton.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies.  Allan
 * Brighton make no representations about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied
 * warranty.
 * -----------------------------------------------------------------------------
 * 
 * The basic algorithm used below is taken from:
 * 
 *     IEEE Software: July 1990 "Drawing Dynamic Trees"
 * 
 * I also used the Xview extension "Slingshot" as a reference.
 */
 

#include <DrawTreeNode.h>


inline int max(int x, int y) { return (x > y) ? x : y; }

implementPtrList(PolylinePtrList, Polyline);

    
/*
 * constructor for new child node
 */
DrawTreeNode::DrawTreeNode(const String& tag, int x, int y, int w, int h, int border) 
: tag_(tag), parent_(0), child_(0), sibling_(0), width_(w), height_(h), border_(border), pos_(x, y)
{
}


/*
 * set the bounding box of a node
 */
void DrawTreeNode::box(int x1, int y1, int x2, int y2)
{
    pos_.x = x1;
    pos_.y = y1;
    width_ = x2 - x1;
    height_ = y2 - y1;
}


/*
 * clean up the mess...	
 */
DrawTreeNode::~DrawTreeNode()
{
    for(ListItr(PolylinePtrList) it(polylines_); it.more(); it.next())
       delete it.cur();
    polylines_.remove_all();
       
    if (child_) {
	delete  child_;
	child_ = 0;
    }
        
    if (sibling_) {
	delete  sibling_;
	sibling_ = 0;
    }
}


/*
 * Keep track of the allocated Polylines for later garbage collection
 */
Polyline* DrawTreeNode::newPolyline(int x, int y, Polyline* pl)
{
    Polyline* p = new Polyline(x, y, pl);
    polylines_.append(p);
    return p;
}


/*
 * find node with tag and return it or 0 if not found
 */
DrawTreeNode* DrawTreeNode::find(const String& tag)
{
    DrawTreeNode* result = 0;
    
    if (tag == tag_)
        return this;
        
    if (child_)
        if (result = child_->find(tag))
            return result;
        
    if (sibling_)
        return sibling_->find(tag);
        
    return result;
}


/*
 * Calculate the node positions and draw the tree 
 */
void DrawTreeNode::drawTree()
{
    // these will hold the bounding box of the entire tree
    int x1=pos_.x, y1=pos_.y, x2=pos_.x+width_, y2=pos_.y+height_;
    
    layout();
    setXY(x1, y1, x2, y2);
    setDrawArea(x1, y1, x2, y2);
    draw();
}


/*
 * remove and delete this nodes children
 */
void DrawTreeNode::prune()
{
    if (child_) {
        for (DrawTreeNode* p = child_; p; p = p->sibling_) {
            p->prune();
            p->removeNode();
            p->removeLine();
        }
        delete child_;
        child_ = 0;
    }
}


/*
 * remove and delete this node
 */
void DrawTreeNode::rmLink()
{
    prune();
    unLink ();
    removeNode();
    removeLine();
    delete this;
}



/*
 * unlink this subtree from the tree
 */
void DrawTreeNode::unLink ()
{
    if (parent_) {
        if (parent_->child_ == this) {
            parent_->child_ = sibling_;
        }
        else {
            for (DrawTreeNode* p = parent_->child_; p; p = p->sibling_) {
                if (p->sibling_ == this) {
                    p->sibling_ = sibling_;
                    break;
                }
            }
        }
    }
    drawLine(Point(), Point()); 	// make line invisible - length 0
    parent_ = 0;
    sibling_ = 0;
}


/*
 * Draw the tree 
 */
void DrawTreeNode::draw()
{
    // if the position has changed, redraw the node
    if (pos_ != prev_pos_)
        drawNode();
    
    // redraw the line from the parent to the node if one of the
    // nodes has been moved
    if (parent_ && parent_->parent_ && (pos_ != prev_pos_ || parent_->pos_ != parent_->prev_pos_)) {
        if (horizontal()) {
            drawLine(
                 Point(parent_->pos_.x + parent_->width_ + 1, 
                       parent_->pos_.y + parent_->height_ / 2),
                 Point(pos_.x - 1, 
                       pos_.y + height_ / 2)); 
        }
        else {
            drawLine(
                 Point(parent_->pos_.x + parent_->width_/2, 
                       parent_->pos_.y + parent_->height_ + 1),
                 Point(pos_.x + width_ / 2, 
                       pos_.y - 1)); 
        }
    }
    
    // do for each node ...    
    if (child_)
        child_->draw();
        
    if (sibling_)
        sibling_->draw();
}


/*
 * Set the absolute positions of the tree nodes from the relative
 * offsets set by layout().
 * 
 * The offset_ field is relative to the nodes predecessor 
 * (parent or sibling). From that we want to set the pos_ field.
 * 
 * Assume that this node already has the correct pos.
 *
 * x1, y1, x2, y2 are set to the bounding box of the entire tree
 */
void DrawTreeNode::setXY(int& x1, int& y1, int& x2, int& y2)
{
    if (child_) 
	child_->setPosFrom(this, x1, y1, x2, y2);
        
    if (sibling_) 
	sibling_->setPosFrom(this, x1, y1, x2, y2);
}


/*
 * Set this node's position from its relative offset and its predecessor's
 * position.
 *
 * x1, y1, x2, y2 are kept up to date as the bounding box of the tree
 */
void DrawTreeNode::setPosFrom(DrawTreeNode* pred, int& x1, int& y1, int& x2, int& y2)
{
    prev_pos_ = pos_;
    if (horizontal())
        pos_ = pred->pos_ + offset_;
    else {
        pos_.x = pred->pos_.x + offset_.y;
        pos_.y = pred->pos_.y + offset_.x;
    }
    
    // keep track of the outer bounds of the tree
    if (pos_.x < x1)
        x1 = pos_.x;
    if (pos_.y < y1)
        y1 = pos_.y;
    if ((pos_.x+width_) > x2)
        x2 = pos_.x+width_;
    if ((pos_.y+height_) > y2)
        y2 = pos_.y+height_;
        
    setXY(x1, y1, x2, y2);
}


/*
 * insert the node in the tree as a child of this node
 */
void DrawTreeNode::addLink(DrawTreeNode* node)
{
    node->parent_ = this;
    if (!child_)
        child_ = node;
    else {
	for (DrawTreeNode* p = child_; p->sibling_; p = p->sibling_);
	p->sibling_ = node;
    }
}

        
/*
 * calculate the offsets of the nodes in the tree
 */
void DrawTreeNode::layout()
{
    for (DrawTreeNode *p = child_; p; p = p->sibling_)
        p->layout();
    
    if (child_)
        attach_parent(join());
    else
        layout_leaf();
}


/*
 * Calculate the leaf contours (see the article mentioned at the top
 * of this file for more details)
 */
void DrawTreeNode::layout_leaf() 
{
    if (horizontal()) {
	contour_.upper.tail = newPolyline(width_ + 2 * border_, 0, 0);
	contour_.upper.head = contour_.upper.tail;
	contour_.lower.tail = newPolyline(0, -height_ - 2 * border_, 0);
	contour_.lower.head = newPolyline(width_ + 2 * border_, 0, contour_.lower.tail);
    }
    else {
	contour_.upper.tail = newPolyline(height_ + 2 * border_, 0, 0);
	contour_.upper.head = contour_.upper.tail;
	contour_.lower.tail = newPolyline(0, -width_ - 2 * border_, 0);
	contour_.lower.head = newPolyline(height_ + 2 * border_, 0, contour_.lower.tail);
    }
}
    

/*
 * See the article mentioned at the top of this file for details on
 * what this function does.
 */
void DrawTreeNode::attach_parent(int h) 
{
    int     x, y1, y2;
    x = border_ + parentDistance();
    
    if (horizontal()) {
	y2 = (h - height_) / 2 - border_;
	y1 = y2 + height_ + 2 * border_ - h;
	child_->offset_.x = x + width_;
	child_->offset_.y = y1;
	contour_.upper.head = newPolyline(width_, 0, newPolyline(x, y1, contour_.upper.head));
	contour_.lower.head = newPolyline(width_, 0, newPolyline(x, y2, contour_.lower.head));
    } else {
	y2 = (h - width_) / 2 - border_;
	y1 = y2 + width_ + 2 * border_ - h;
	child_->offset_.x = x + height_;
	child_->offset_.y = y1;
	contour_.upper.head = newPolyline(height_, 0, newPolyline(x, y1, contour_.upper.head));
	contour_.lower.head = newPolyline(height_, 0, newPolyline(x, y2, contour_.lower.head));
    }
}


/*
 * See the article mentioned at the top of this file for details on
 * what this function does.
 */
int DrawTreeNode::join() 
{
    DrawTreeNode* node;
    int d, h, sum;
    int horiz = horizontal();
    
    node = child_;
    contour_ = node->contour_;
    if (horiz) 
        sum = h = node->height_ + 2 * node->border_;
    else
        sum = h = node->width_ + 2 * node->border_;
    
    node = node->sibling_;
    while (node) {
	d = merge(contour_, node->contour_);
	node->offset_.y = d + h;
	node->offset_.x = 0;
	if (horiz)
	    h = node->height_ + 2 * node->border_;
	else
	    h = node->width_ + 2 * node->border_;
	sum += d + h;
	node = node->sibling_;
    }
    return sum;
}


/*
 * merge() walks through two contours and prevents overlap. The lower
 * contour of the upper node is compared against the upper contour of the
 * lower node.
 * See the article mentioned at the top of this file for details on
 * what this function does.
 */
int DrawTreeNode::merge(Polygon& p1, Polygon& p2)
{
    int     x, y, total, d;
    Polyline *lower, *upper, *b;

    x = y = total = 0;
    upper = p1.lower.head;
    lower = p2.upper.head;

    while (lower && upper) {	/* compute offset total */

	d = offset(x, y, lower->p.x, lower->p.y, upper->p.x, upper->p.y);
	y += d;
	total += d;

	if (x + lower->p.x <= upper->p.x) {
	    y += lower->p.y;
	    x += lower->p.x;
	    lower = lower->link;
	}
	else {
	    y -= upper->p.y;
	    x -= upper->p.x;
	    upper = upper->link;
	}
    }
    
    /* store result in c1 */
    if (lower) {
	b = bridge(p1.upper.tail, 0, 0, lower, x, y);
	p1.upper.tail = (b->link) ? p2.upper.tail : b;
	p1.lower.tail = p2.lower.tail;
    }
    else {
	b = bridge(p2.lower.tail, x, y, upper, 0, 0);
	if (!b->link)
	    p1.lower.tail = b;
    }
    p1.lower.head = p2.lower.head;

    return total;
}


/*
 * See the article mentioned at the top of this file for details on
 * what this function does.
 */
int DrawTreeNode::offset(int p1, int p2, int a1, int a2, int b1, int b2)
{
    int     d, s, t;

    if (b1 <= p1 || p1 + a1 <= 0)
	return 0;

    t = b1 * a2 - a1 * b2;
    if (t > 0) {
	if (p1 < 0) {
	    s = p1 * a2;
	    d = s / a1 - p2;
	}
	else if (p1 > 0) {
	    s = p1 * b2;
	    d = s / b1 - p2;
	}
	else
	    d = -p2;
    }
    else {
	if (b1 < p1 + a1) {
	    s = (b1 - p1) * a2;
	    d = b2 - (p2 + s / a1);
	}
	else if (b1 > p1 + a1) {
	    s = (a1 + p1) * b2;
	    d = s / b1 - (p2 + a2);
	}
	else
	    d = b2 - (p2 + a2);
    }
    return max(0, d);
}


/*
 * See the article mentioned at the top of this file for details on
 * what this function does.
 */
Polyline* DrawTreeNode::bridge(Polyline* line1, int x1, int y1, Polyline* line2, int x2, int y2)
{
    int dy, dx, s;
    Polyline *r;

    dx = x2 + line2->p.x - x1;
    if (line2->p.x == 0)
	dy = line2->p.y;
    else {
	s = dx * line2->p.y;
	dy = s / line2->p.x;
    }
    r = newPolyline(dx, dy, line2->link);
    line1->link = newPolyline(0, y2 + line2->p.y - dy - y1, r);
    return r;
}


/*
 * These routines belong to the algorithm and will be implemented if a
 * performance bottleneck is detected at this point.
 * 
 * It seems fast enough to just recalculate the positions for the whole
 * tree each time.  The bottleneck is more likely in displaying the nodes.
 */
void DrawTreeNode::branch(DrawTreeNode* /*child*/, DrawTreeNode* /*sibling*/)
{
}

void DrawTreeNode::unzip()
{
}

void DrawTreeNode::zip()
{
}

