/*-------------------------------------------------------------------------
 *
 * itemptr.h--
 *    POSTGRES disk item pointer definitions.
 *
 *
 * Copyright (c) 1994, Regents of the University of California
 *
 * itemptr.h,v 1.3 1995/04/09 20:27:31 andrew Exp
 *
 *-------------------------------------------------------------------------
 */
#ifndef	ITEMPTR_H
#define ITEMPTR_H

#include "c.h"
#include "storage/block.h"
#include "storage/off.h"
#include "storage/pos.h"
#include "storage/itemid.h"

typedef struct ItemPointerData {
    BlockIdData		ip_blkid;
    PositionIdData	ip_posid;
} ItemPointerData;

typedef ItemPointerData	*ItemPointer;

/* ----------------
 *	support macros
 * ----------------
 */

/*
 * ItemPointerIsValid --
 *	True iff the disk item pointer is not NULL.
 */
#define ItemPointerIsValid(pointer) \
    ((bool) (PointerIsValid(pointer) && ((pointer)->ip_posid != 0)))

/*
 * ItemPointerGetBlockNumber --
 *	Returns the block number of a disk item pointer.
 */
#define ItemPointerGetBlockNumber(pointer) \
    (AssertMacro(ItemPointerIsValid(pointer)) ? \
     BlockIdGetBlockNumber(&(pointer)->ip_blkid) : (BlockNumber) 0)

/*
 * ItemPointerGetOffsetNumber --
 *	Returns the offset number of a disk item pointer.
 */
#define ItemPointerGetOffsetNumber(pointer) \
    (AssertMacro(ItemPointerIsValid(pointer)) ? \
     PositionIdGetOffsetNumber(&((pointer)->ip_posid)) : \
     (OffsetNumber) 0)
/*
 * ItemPointerGetOffsetIndex --
 *    Returns the offset index of a disk item pointer.
 */
#define ItemPointerGetOffsetIndex(pointer) \
    (AssertMacro(ItemPointerIsValid(pointer)) ? \
     ItemPointerGetOffsetNumber(pointer) - 1 : \
     (OffsetIndex) 0)

/*
 * ItemPointerSet --
 *	Sets a disk item pointer to the specified block and offset.
 */
#define ItemPointerSet(pointer, blockNumber, offNum) \
    AssertMacro(PointerIsValid(pointer)); \
    BlockIdSet(&((pointer)->ip_blkid), blockNumber); \
    PositionIdSet(&((pointer)->ip_posid), offNum)

/*
 * ItemPointerCopy --
 *	Copies the contents of one disk item pointer to another.
 */
#define ItemPointerCopy(fromPointer, toPointer) \
    Assert(PointerIsValid(toPointer)); \
    Assert(PointerIsValid(fromPointer)); \
    *(toPointer) = *(fromPointer)

/*
 * ItemPointerSetInvalid --
 *	Sets a disk item pointer to be invalid.
 */
#define ItemPointerSetInvalid(pointer) \
    Assert(PointerIsValid(pointer)); \
    BlockIdSetInvalid(&((pointer)->ip_blkid)); \
    PositionIdSetInvalid(&((pointer)->ip_posid))

/* ----------------
 *	externs
 * ----------------
 */

extern bool ItemPointerEquals(ItemPointer pointer1, ItemPointer pointer2);

#endif	/* ITEMPTR_H */
