/* -------------------------------------------------------------------------- */
/* Surface.h */
/* Header file for surface project */
/* -------------------------------------------------------------------------- */

/* A few #includes we need for the prototypes... */
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>

/* -------------------------------------------------------------------------- */

/* ... a little something for those winter nights ... */
#define bool char

/* -------------------------------------------------------------------------- */

/* ... and a smattering of typedefs for that finishing touch. */
typedef double matrix[4][4];
typedef double vector[3];

typedef enum {
   ALL, CURRENT
} Itemsmode;

typedef enum {
   ZEROTH, FIRST
} Continuitymode;

typedef enum {
   NS, EW
} jointype;

typedef enum {
    N, S, E, W
} direction;

typedef struct {
   int x, y;
} connectpoint;

typedef struct {
   Position x, y;
   double z;
   int viewcode;
} screenpoint;

typedef struct {
   int width, height;
   double scale;
} windowdata;

typedef struct vertex {
   struct vertex *next, *previous;
   int occurrences;
   int number;
   screenpoint screen;
   double world[4];
} vertex;

typedef struct {
   vertex *vertexptr[3][3];
   vector move[3][3];
} joindata;

typedef struct patch {
   struct patch *next, *previous, *n, *s, *e, *w;
   vertex *data[4][4];
   matrix differences[3];
   struct {
      int x, y;
   }      connectpos;
   bool visited;
   int number;
} patch;

typedef struct {
   int xmax, xmin, ymax, ymin;
   double scale;
   Position xcentre, ycentre;
} connectdata;

typedef struct {
   struct {
      vector min, max;
   } bounds;
   connectdata connect;
   patch *firstpatch, *lastpatch, *currentpatch;
   vertex *firstvertex, *lastvertex;
   int numpatches, numvertices;
} objectdata;

/* -------------------------------------------------------------------------- */

/* Oh yes, I almost forgot... */
struct axis {
   double x1, x2, x3, y1, y2, y3, z1, z2, z3;
};

/* -------------------------------------------------------------------------- */

/* Program constants */
#define DEFAULTITEMS ALL	   /* Display all patches or current one only */
#define DEFAULTCONTINUITY FIRST    /* Maintain tangential continuity */

#define MAXNUMLINES 40	/* Max subdivisions for Bezier display */

#define SCALE 0.175

#define CP_DEF_X 50	/* Default positions and sizes of windows */
#define CP_DEF_Y 10
#define CP_DEF_H 400
#define CP_DEF_W 400
#define BS_DEF_X 500
#define BS_DEF_Y 10
#define BS_DEF_H 400
#define BS_DEF_W 400
#define CN_DEF_X 50
#define CN_DEF_Y 450
#define CN_DEF_H 400
#define CN_DEF_W 400
#define ST_DEF_X 500
#define ST_DEF_Y 750
#define ST_H 80
#define ST_W 400

#define STATUSWIDTH 64 	/* Width of Status display in characters */

#ifdef DOUBLEBUFFERING
  #define CP_MAX_H 800	/* If double-buffering, set max size for the */
  #define CP_MAX_W 800  /* double-buffered windows to be the size of */
  #define BS_MAX_H 800  /* the corresponding pixmaps.                */
  #define BS_MAX_W 800
#endif

#define ZSENSITIVITY 20.0       /* Sensitivity to movement along Z axis */

#define BOXGAP 2		/* Gap between boxes in CN display */
#define CHARX 3			/* Approx size of chars for CN display */
#define CHARY 5	
#define DOTSIZE 0.1		/* Size of points in CP display */ 
#define AXISPOSX 40		/* Axis position and size */
#define AXISPOSY 360
#define AXISSIZE 0.1
#define FRONTCLIP 1		/* Clip lines at this distance from camera */ 
#define CLIPMARGIN 2		/* Clip this far from edge of display */

#define DEFAULTTHETA (250.0*M_PI/180.0)	/* Default camera attributes */
#define DEFAULTPHI (215*M_PI/180.0)
#define DEFAULTR 7.5
#define DEFAULTVPDIST 5.5
#define DEFAULTNUMLINES 12

#define SELECTDIST 50	/* Distance from point click will still select it */

#define BIG 10000000000.0
#define BIGINT 100000

#define FLOATLENGTH 10

#ifdef DOUBLEBUFFERING	
  #define CONTROLDRAWABLE controlpixmap
  #define BEZIERDRAWABLE bezierpixmap 
#else 
  #define CONTROLDRAWABLE controlwindow
  #define BEZIERDRAWABLE bezierwindow
#endif

#ifdef LONGFLOAT
  #define FLOATTYPE double
  #define ERRORMARGIN 1E-10
  #define SIN(x) sin(x)
  #define COS(x) cos(x)
#else
  #define FLOATTYPE float
  #define ERRORMARGIN 1E-5
  #ifdef SPTRIG
    /* Some machines support sinf() and cosf() for single precision floats... */
    #define SIN(x) sinf(x)
    #define COS(x) cosf(x)
  #else
    #define SIN(x) sin(x)
    #define COS(x) cos(x)
  #endif
#endif

/* -------------------------------------------------------------------------- */

/* Macros */

#define drawopenpoint(coord) XDrawArc(controldisplay, CONTROLDRAWABLE,\
       gc, coord.x-dotscale/coord.z, coord.y-dotscale/coord.z,\
       2*dotscale/coord.z, 2*dotscale/coord.z, 0, 360*64)

#define drawfilledpoint(coord) XFillArc(controldisplay, CONTROLDRAWABLE,\
       gc, coord.x-dotscale/coord.z, coord.y-dotscale/coord.z,\
       2*dotscale/coord.z, 2*dotscale/coord.z, 0, 360*64)

#define calcviewcode(pt, window) ((pt.x<CLIPMARGIN) + \
          2*(pt.x>window->width-CLIPMARGIN) + 4*(pt.y<CLIPMARGIN) + \
          8*(pt.y>window->height-CLIPMARGIN) + 16*(pt.z<FRONTCLIP))

#define distsq(x1,y1,x2,y2) ((x1)-(x2))*((x1)-(x2)) + ((y1)-(y2))*((y1)-(y2))
#define min(x,y) ((x)<(y)?(x):(y))
#define max(x,y) ((x)<(y)?(y):(x))
#define difference(x,y) ((x>y)?(x-y):(y-x)) 
#define cube(a) ((a)*(a)*(a))
#define sqr(a) ((a)*(a))
#define rad(a) (a*M_PI/180)
#define deg(a) (a*180/M_PI)
#define scalarprod(a,b) (a[0]*b[0] + a[1]*b[1] + a[2]*b[2])
#define gradient(p1,p2) (double *) (p1.y-p2.y)/(p1.x-p2.x)

/* These macros used to do cunning things with resource strings in miscx.x */
#define stringify(x) #x
#define string(x) stringify(x)

/* -------------------------------------------------------------------------- */

/* Prototypes */

/* in file bezier.c */
void calcmatrices(int);
void calcbezier(objectdata *);
void displaybezier(objectdata *);
void drawbezierpatch(patch *);

/* -------------------------------------------------------------------------- */

/* in file clbacks.c */
void destroypopup(Widget, XtPointer, XtPointer);
void popupexitbox(Widget, XtPointer, XtPointer);
void theend(Widget, XtPointer, XtPointer);
void popupsetviewbox(Widget, XtPointer, XtPointer);
void xview(Widget, XtPointer, XtPointer);
void yview(Widget, XtPointer, XtPointer);
void zview(Widget, XtPointer, XtPointer);
void defaultvalues(Widget, XtPointer, XtPointer);
void increase(Widget, XtPointer, XtPointer);
void decrease(Widget, XtPointer, XtPointer);
void setvalue(Widget, XtPointer, XtPointer);
void popupdispitemsbox(Widget, XtPointer, XtPointer);
void allmode(Widget, XtPointer, XtPointer);
void currentmode(Widget, XtPointer, XtPointer);
void popupcontinuitybox(Widget, XtPointer, XtPointer);
void zerothmode(Widget, XtPointer, XtPointer);
void firstmode(Widget, XtPointer, XtPointer);
void popupclearbox(Widget, XtPointer, XtPointer);
void clear(Widget, XtPointer, XtPointer);
void popupsavebox(Widget, XtPointer, XtPointer);
void save(Widget, XtPointer, XtPointer);
void popuploadbox(Widget, XtPointer, XtPointer);
void load(Widget, XtPointer, XtPointer);
void popupdelpatchbox(Widget, XtPointer, XtPointer);
void delete(Widget, XtPointer, XtPointer);
void dontdelete(Widget, XtPointer, XtPointer);
void popupaddpatchbox(Widget, XtPointer, XtPointer);
void add(Widget, XtPointer, XtPointer);
void dontadd(Widget, XtPointer, XtPointer);
void controlclicked(Widget, XtPointer, XtPointer);
void connectivityclicked(Widget, XtPointer, XtPointer);
void bezierclicked(Widget, XtPointer, XtPointer);

/* -------------------------------------------------------------------------- */

/* in file connect.c */
void drawconnectivity(objectdata *);
void tracevisit(patch *, patch *);
void traceconnect(patch *);
patch ***formparent(objectdata *);
bool mendobject(objectdata *);
bool connectshared(objectdata *, patch ***);
bool joinpatches(objectdata *, jointype, patch *, patch *, patch ***);
bool tidyobject(objectdata *);
bool mergevertices(objectdata *, vertex **, vertex **);
bool createpatch(objectdata **, connectpoint*);
void connectpatches(objectdata **, jointype, patch *, patch *);
void fixcontinuity(int, int, vector, joindata *);
void formjoinstruct(patch *, int, int, joindata *);

/* -------------------------------------------------------------------------- */

/* in file control.c */
void calccontrol(objectdata *);
void drawcontrol(objectdata *);


/* -------------------------------------------------------------------------- */

/* in file filing.c */
bool saveobject(char *name);
bool loadobject(char *name);
objectdata *initobject(void);
void wipeobject(objectdata *);

/* -------------------------------------------------------------------------- */

/* in file miscx.c */
void setupwindows(int argc, char **argv);
void setupgc(void);
void updatestatus(int, char *);
void carriagereturn(Widget, XEvent *, String *, Cardinal *);

/* -------------------------------------------------------------------------- */

/* in file util.c */
void matrixmultiply(matrix, matrix, matrix);
FLOATTYPE magnitude(vector);
void vecprod(vector, vector, vector);
void normalise(vector);
bool nearestpoint(Position, Position, int *, int *);
void movepoint(vertex *, vector);
void error(char *);
void transpose(matrix, matrix);
void renumber(objectdata *);
void printstatus(char *);
void killvertex(objectdata *, vertex *);
void deletepatch(objectdata *, patch *);
vertex *createvertex(objectdata *);
void copyobject(objectdata *, objectdata *);
void findworldbounds(objectdata *);
void getconnectcoords(objectdata *);
void unvisit(objectdata *);
objectdata *createobject(int, int);
void getconnectivitybounds(objectdata *);

/* -------------------------------------------------------------------------- */

/* in file view.c */
void calctransform(objectdata *);
void transformpoint(vertex *, windowdata *);
void clipline(screenpoint *, screenpoint *, Display *, Drawable, GC, windowdata *);

/* -------------------------------------------------------------------------- */

/* in file debug.c */
#ifdef DEBUG
  void check(objectdata *);
#endif

