/* Dialog managing.
   Copyright (C) 1994 Miguel de Icaza.
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
#include <ncurses.h>
#include <stdio.h>
#include <stdarg.h>
#include <sys/types.h>
#include <string.h>
#include "global.h"
#include "dialog.h"
#include "input.h"
#include "color.h"
#include "util.h"
#include "win.h"
#include "mouse.h"
#include "main.h"

static char rcsid [] = "$Id: dialog.c,v 1.11 1994/10/13 04:27:26 miguel Exp $";
static char buffer [4096];

static Dialog *current_dialog = 0;
WINDOW *top_window;

static Refresh *refresh_list = 0;

void push_refresh (void (*new_refresh)())
{
    Refresh *new;

    new = xmalloc (sizeof (Refresh), "push_refresh");
    new->next = (struct Refresh *) refresh_list;
    new->refresh_fn = new_refresh;
    refresh_list = new;
}

void pop_refresh ()
{
    Refresh *old;
    
    if (!refresh_list)
	fprintf (stderr, "\n\n\nrefresh stack underflow!\n\n\n");
    else {
	old = refresh_list;
	refresh_list = refresh_list->next;
	free (old);
    }
}

void do_refresh ()
{
    if (!refresh_list)
	fprintf (stderr, "\n\nNo refresh handlers defined!\n\n");
    else 
	(*(refresh_list->refresh_fn))();
}

/* Poor man's window puts, it doesn't handle auto-wrap */
void my_wputs (WINDOW *w, int y, int x, char *text)
{
    char p;

    wmove (w, y, x);
    while ((p = *text++) != 0){
	if (p == '\n')
	    wmove (w, ++y, x);
	else 
	    waddch (w, p);
    }
}

void create_dialog (int cols, int rows, char *header, char *text, int error)
{
    int base = LINES/3;
    int xpos, ypos;
    Dialog *newd;
    
    if (rows+1 > LINES/3+2)
	base = LINES/3+2;

    newd = (Dialog *) xmalloc (sizeof (Dialog), "create_dialog");
    newd->backpointer = current_dialog;
    current_dialog = newd;

    xpos = COLS/2-(cols+5)/2;
    ypos = base-(rows+1)/2;
    newd->d = newwin (rows+4, cols+4, ypos, xpos);
    newd->f = derwin (newd->d, rows+2, cols+2, 1, 1);
    newd->t = derwin (newd->d, rows, cols, 2, 2);
    
    push_event (1, 1, COLS, LINES, (mouse_h) null_event, 0);
    push_frame (xpos, ypos, 0);
    
    leaveok (newd->d, TRUE);
    leaveok (newd->f, TRUE);
    leaveok (newd->t, TRUE);
    if ((error & D_ERROR)) {
	wattron (newd->d, ERROR_COLOR | A_BOLD);
	wclr (newd->d);
	wattron (newd->f, ERROR_COLOR | A_BOLD);
	wattron (newd->t, ERROR_COLOR | A_BOLD);
    } else {
	wattron (newd->d, REVERSE_COLOR);
	wclr (newd->d);
	wattron (newd->f, REVERSE_COLOR);
	wattron (newd->t, REVERSE_COLOR);
    }
    box (newd->f, ACS_VLINE, ACS_HLINE);
    mvwprintw (newd->f, 0, (cols-strlen (header)-2)/2, header);
    wclr (newd->t);
    my_wputs (newd->t, 0, 0, text); 
    touchwin (newd->t);
    touchwin (newd->f);
    touchwin (newd->d);
    
    wrefresh (newd->d);
    top_window = newd->d;
}

WINDOW *get_top_text ()
{
    return current_dialog->t;
}

int run_dialog ()
{
    int key;

    push_event (1, 1, COLS, LINES, click_may_break_loop, quit_event);
    key = mi_getch ();
    pop_event ();
    do_refresh ();
    return key;
}

void destroy_dialog ()
{
    Dialog *p;
    
    if (!current_dialog){
	fprintf (stderr, "Trying to destroy non-existant dialog\n");
	exit (1);
    }
    delwin (current_dialog->t);
    delwin (current_dialog->f);
    delwin (current_dialog->d);
    p = current_dialog;
    current_dialog = p->backpointer;
    if (current_dialog)
	top_window = current_dialog->d;
    else
	top_window = 0;
    free (p);
    pop_frame ();
    pop_event ();
}

int message (int error, char *header, char *text,  ...)
{
    va_list  args;
    int      len, lines;
    char     *buf;
    int      v;

    buf = buffer;

    strcpy (buffer, "\n");
    va_start (args, text);
    vsprintf (&buffer [1], text, args);
    strcat (buf, "\n");
    va_end (args);
    len = max (strlen (header), msglen (buf, &lines));
    create_dialog (len, lines, header, buf, error);
    if (!(error & D_INSERT)){
	v = run_dialog ();
	destroy_dialog ();
	return v;
    }
    return 0; 
}

int error (int status, int errnum, char *text,  ...)
{
    va_list  args;
    int      len, lines;
    char     *buf;
    int      v;
    char     *header = " Error ";
    
    buf = buffer;

    strcpy (buffer, "\n");
    va_start (args, text);
    vsprintf (&buffer [1], text, args);
    strcat (buf, "\n");
    
    len = max (strlen (header), msglen (buf, &lines));
    create_dialog (len, lines, header, buf, 1);
    v = run_dialog ();
    destroy_dialog ();
    va_end (args);
    return 0;
}

char *input_dialog (char *header, char *text, char *def_text)
{
    int    len, lines;
    char   *return_value;
    Input  *in;
    int    c;

    if (strlen (text) >= sizeof (buffer))
	text [sizeof (buffer)-1] = 0;

    strcpy (buffer, text);
    
    len = max (strlen (header), msglen (buffer, &lines));
    len = max (len, 60);
    create_dialog (len, lines+1, header, buffer, 0);
    in = create_input (1, lines, current_dialog->t, INPUT_COLOR, 58, def_text);
    return_value = in->buffer;
    while ((c = mi_getch ()) != '\n'){
	if (c == ESC_CHAR){
	    return_value = 0;
	    break;
	}
	handle_char (in, c);
    }
    destroy_dialog ();
    do_refresh ();
    destroy_input (in, IN_KEEP_BUFFER); 
    return (return_value);
}

char *input_expand_dialog (char *header, char *text, char *def_text)
{
    char *result;
    char *expanded;

    result = input_dialog (header, text, def_text);
    if (result){
	expanded = tilde_expand (result, no_gc);
	if (result != expanded){
	    free (result);
	    return expanded;
	}
    }
    return result;
}
