/*
 * gpn.c - support functions for gpm-Linux
 *
 * Copyright 1993   ajh@gec-mrc.co.uk (Andrew Haylett)
 * Copyright 1994   rubini@ipvvis.unipv.it (Alessandro Rubini)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 ********/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>        /* strerror(); ?!? memcpy() */
#include <signal.h>
#include <stdarg.h>        /* Log uses it */
#include <errno.h>
#include <unistd.h>        /* getopt() */
#include <sys/param.h>
#include <sys/time.h>      /* timeval */
#include <sys/wait.h>      /* wait() */

#ifdef	SIGTSTP		/* true if BSD system */
#include <sys/file.h>
#include <sys/ioctl.h>
#endif

#include "gpmInt.h"
#include "gpm.h"        /* this includes socket and internet headers */
#include "getopt.h"     /* uninfluent for linux, needed in sunos */

extern int	errno;


/*===================================================================*
 *     ==>    the logging utility
 *
 * This procedure is used to log info (mostly debugging ones)
 * the log file is quite large, actually, so be careful....
 *-------------------------------------------------------------------*/
#ifdef CONFIG_GPM_LOG      /* only if enabled */
int Log(char *fmt,...)
{
  va_list args;
  static time_t mytime;
  static char str_time[15];
  static FILE *f;
  static int pid;
  static char temp[80];

  if (!f)
    {
    pid=getpid();
    f=fopen(GPM_NODE_LOG,"a");
    if (!f) f=fopen("/dev/null","w");      /* not null, hopefully */
    setvbuf(f,NULL,_IONBF,0);
    }

  time(&mytime);
  strftime(str_time,15,"%H:%M:%S",localtime(&mytime));
  sprintf(temp,"%s - %i: %s\n",str_time,pid,fmt);

  va_start(args,fmt);           /* but the second argument is not used... */
  vfprintf(f,temp,args);
  return 1;
}
#endif


/*===================================================================*/
int _oops(char *f, int n, char *s, int err)
{
  LOG(("%s(%i): Exiting: \"%s: %s\"",f,n,s,strerror(err) ));
  DEBUG((stderr,"oops() invoked from %s(%i)\n",f,n));
  fprintf(stderr,"%s: %s: %s\n",prgname,s,strerror(err));
  exit(1);
}

/*===================================================================*/
static int loadlut(char *charset)
{
int i, c, fd;
static unsigned long long_array[5]={
  0x05050505,      /* ugly, but preserves alignment */
  0x0,0x0,0x0,0x0
};

#define inwordLut (long_array+1)
#define SET(c) (inwordLut[((c)>>5)&3] |= (1<<((c)&0x1F)))

   for (i=0; (c=charset[i]); i++)
     {
     if (c=='\\')    /* use the next, if any */
       {
       if (charset[i+1]=='\0') break;
       c=charset[++i];
       SET(c);
       }
    else if (c=='-' && i!=0 && charset[i+1]!='\0') /* range */
      for (c=charset[i-1]; c<charset[i+1]; c++) SET(c);
    else SET(c);
    }

#ifdef linux
  if ((fd=open("/dev/console", O_WRONLY)) < 0)
    oops("/dev/console");
  if ((fd<0) || (ioctl(fd, TIOCLINUX, &long_array)<0))
    {
    fprintf(stderr,"Is your kernel compiled with CONFIG_SELECTION on?\n");
    oops("loadlut");
    }
  close(fd);
#endif

  return 0;
}

/*===================================================================*/
static int usage(void)
{
 printf(GPM_NAME " " GPM_RELEASE ", " GPM_DATE "\n"
        "Usage: %s [options]\n",prgname);
 printf("  Valid options are (not all of them are implemented)\n"
 "    -a accel         sets the acceleration (default %d)\n"
 "    -b baud-rate     sets the baud rate (default %d)\n"
 "    -B sequence      allows changing the buttons (default '%s')\n"
 "    -d delta         sets the delta value (default %d)\n"
 "    -D               dirty operation (debug only)\n"
 "    -i interval      max time interval for multiple clicks (default %i)\n"
 "    -l charset       loads the inword() LUT (default '%s')\n"
 "    -L charset       load LUT and exit\n"
 "    -m mouse-device  sets mouse device\n"
 "    -r number        sets the responsiveness (default %i)\n"
 "    -s sample-rate   sets the sample rate (default %d)\n"
 "    -t mouse-type    sets mouse type (default '%s')\n"
 "                     Microsoft = `ms', Mouse Systems Corp = `msc',\n"
 "                     MM Series = `mm', Logitech = `logi', BusMouse = `bm',\n"
 "                     MSC 3-bytes = `sun', PS/2 = `ps2')\n"
 "    -T               test: read mouse, no clients\n"
 "    -v               print version info and exit\n",
        DEF_ACCEL, DEF_BAUD, DEF_SEQUENCE, DEF_DELTA, DEF_TIME, DEF_LUT,
	DEF_SCALE, DEF_SAMPLE, DEF_TYPE);
  return 1;
}

/*===================================================================*/
int cmdline(int argc, char **argv)
{
char options[]="a:b:B:d:Dhi:l:L:m:r:s:t:Tv23";
int opt, fd;
static struct {char *in; char *out;} seq[] = {
  {"123","01234567"},
  {"132","02134657"},
  {"213","01452367"},
  {"231","02461357"},
  {"312","04152637"},
  {"321","04261537"},
  {NULL,NULL}
};

  while ((opt = getopt(argc, argv, options)) != -1)
    {
    switch (opt)
      {
      case 'a':
          opt_accel = atoi(optarg);
          break;
      case 'b':
          opt_baud = atoi(optarg);
          break;
      case 'B':
	  opt_sequence = optarg;
	  break;
      case 'd':
          opt_delta = atoi(optarg);
          break;
      case 'D':
          opt_dirty++;
	  break;
      case 'h':
	  exit(usage());
      case 'i':
	  opt_time=atoi(optarg);
	  break;
      case 'l': /* load look-up-table */
          opt_lut=optarg;
          break;
      case 'L':
	  loadlut(optarg);
	  exit(0);
      case 'm':
          opt_dev = optarg;
          break;
      case 'r':
	  /* being called responsiveness, I must take the inverse */
	  opt_scale=atoi(optarg);
          if (!opt_scale) opt_scale=100; /* a maximum */
	  else opt_scale=100/opt_scale;
	  break;
      case 's':
          opt_sample = atoi(optarg);
          break;
      case 't':
          opt_type=optarg;
	  break;
      case 'T':
	  opt_test++;
	  break;
      case 'v':
	  printf(GPM_NAME " " GPM_RELEASE ", " GPM_DATE "\n"); exit(0);
      case '2':
          opt_three=-1;
          break;
      case '3':
	  opt_three=1;
	  break;
      default:
          fprintf(stderr,"%s: try \"%s -h\" to get help\n",
		  prgname, prgname);
	  exit(1);
      }
    }

  if (opt_accel<2) exit(usage());
  if (opt_delta<2) exit(usage());
  if (strlen(opt_sequence)!=3 || atoi(opt_sequence)<100) exit(usage());

  /* look for the type */
  for (m_type=mice; m_type->fun; m_type++)
    if (!strcmp(opt_type,m_type->name))
      break;
  if (!(m_type->fun)) /* not found */
    exit(usage());

  /* open the device */
  if (opt_dev)
    {
    if ((fd=open(opt_dev,O_RDWR))<0)
      oops(opt_dev); /* user failed */
    }
  else /* use "/dev/mouse", and the the type-specific name */
    {
    opt_dev = "/dev/mouse";
    if ((fd=open(opt_dev,O_RDWR))<0)
      {
      DEBUG((stderr,"%s: %s: %s\n",prgname,opt_dev,strerror(errno)));
      opt_dev = m_type->device;
      if ((fd=open(opt_dev,O_RDWR))<0)
        oops(m_type->device); /* default failed */
      }
    }

  /* init the device */
  if (m_type->init)
    (m_type->init)(fd, m_type->flags);

  loadlut(opt_lut);

  /* choose the sequence */
  for (opt=0; seq[opt].in && strcmp(seq[opt].in,opt_sequence); opt++)
    ;
  if (!seq[opt].in) exit(usage());
  opt_sequence=seq[opt].out;

#ifndef linux
  /* make the mouse non blocking... */
  if ((opt=fcntl(fd,F_GETFL,0))<0)  oops("fcntl()");
  if (fcntl(fd,F_SETFL,opt|O_NONBLOCK)<0) oops("fcntl()");
#endif


  return fd;
}
  
