/* Main program for the MouseLess Commander
   Copyright (C) 1994 Miguel de Icaza, Mauricio Plaza
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <ncurses.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <errno.h>
#include <pwd.h>
#include <unistd.h>
#include <fcntl.h>		/* For O_RDWR */
#include <signal.h>
#include "dir.h"
#include "color.h"
#include "panel.h"
#include "input.h"
#include "global.h"
#include "dialog.h"
#include "menu.h"
#include "file.h"
#include "main.h"
#include "util.h"
#include "win.h"
#include "user.h"

static char rcsid [] = "$Header: /usr/users/miguel/c/CVS/nc/main.c,v 1.5 1994/06/27 22:53:09 miguel Exp $";

/* The structures for the panels */
Panel left_panel, right_panel;

/* Pointers to the selected and unselected panel */
Panel *current_panel, *other_panel;

Input cmdline;

/* Usefull macros to avoid too much typing */
#define cpanel current_panel
#define opanel other_panel

/* Set when main loop should be terminated */
int   quit = 0;

/* If true, marking a files moves the cursor down */
int   mark_moves_down = 1;

/* If true, at startup the user-menu is invoked */
int   auto_menu = 0;

/* If true, after executing a command, wait for a keystroke */
int   pause_after_run = 1;

/* It true saves the setup when quitting */
int auto_save_setup = 0;

/* If true, next character is quoted */
int quote = 0;

/* If true use the internal viewer */
int use_internal_view = 1;

/* Have we shown the fast-reload warning in the past? */
int fast_reload_w = 0;

/* If true all typed chars go to the incremental search routine */
int searching = 0;

char *prompt = "$ ";

/* For slow terminals */
int force_slow = 0;

/* Controls screen clearing before an exec */
int clear_before_exec = 1;

/* WINDOW handles for the command line and the function keys */
WINDOW *cmdline_win;
WINDOW *fkeys;
WINDOW *clean_screen;

/* The currently selected file */
file_entry *selection;

/* The home directory */
char *home_dir;

/* The value of the other directory, only used when loading the setup */
char *other_dir = 0;

/* If true, then print on stdout the last directory we were at */
int print_last_wd = 0;

/* Ugly hack in order to distinguish between left and right panel in menubar */
int is_right;
#define MENU_PANEL  (is_right ? &right_panel : &left_panel)
#define OTHER_PANEL (is_right ? &left_panel : &right_panel)

/* Quick search status */
char search_buffer [256] = { 0 };
char cmd_buf [512];

extern char *sys_errlist [];

void save_setup_cmd ();

inline void unselect_item (Panel *panel)
{
    set_attr (cpanel, 0, selection->f.marked);
    repaint_file (cpanel, cpanel->selected);
    panel_refresh (cpanel);
}

#define SELECT_ITEM(panel) \
    selection = &current_panel->dir [current_panel->selected]; \
    set_attr (cpanel, 1, selection->f.marked);

/* This function sets the selection variable and redisplays the data */
inline void select_item (Panel *panel)
{
    SELECT_ITEM (panel);
    repaint_file (cpanel, cpanel->selected);
    panel_refresh (cpanel);
    display_mini_info (cpanel);
}

void init_panels ()
{
    init_panel (&left_panel, 0, 0, COLS/2, LINES-2);
    left_panel.active = 1;
    left_panel.count  = do_load_dir (&left_panel.dir, left_panel.sort_type);
    if (other_dir)
	chdir (other_dir);
    init_panel (&right_panel, COLS/2, 0, COLS, LINES-2);
    right_panel.count = do_load_dir (&right_panel.dir, right_panel.sort_type);
    if (other_dir)
	chdir (left_panel.cwd);

    current_panel = &left_panel;
    other_panel   = &right_panel;
    
    select_item (current_panel);/* Inits selection variable */
    display_mini_info (cpanel);	/* It needs selection defined */
    display_mini_info (opanel);	/* Same here */

    /* Now, update the screen */
    paint_panel (&left_panel);
    paint_panel (&right_panel);
}

void init_entry ()
{
    cmdline_win = newwin (1, COLS, LINES-2, 0);
    wprintw (cmdline_win, prompt);
    raw ();
    noecho ();
    keypad (stdscr, TRUE);
    wrefresh (cmdline_win);
}

void init_labels ()
{
    fkeys   = new_fkey ();
    define_label (fkeys, 1, "Help");
    define_label (fkeys, 2, "Menu");
    define_label (fkeys, 3, "View");
    define_label (fkeys, 4, "Edit");
    define_label (fkeys, 5, "Copy");
    define_label (fkeys, 6, "RenMov");
    define_label (fkeys, 7, "Mkdir");
    define_label (fkeys, 8, "Delete");
    define_label (fkeys, 9, "PullDn");
    define_label (fkeys, 10, "Quit");

    /* workaround ncurses 1.8.5 bug */
    wmove (fkeys, 0, 0);
    waddch (fkeys, '1');
    
    wrefresh (fkeys);
}

void try_to_select (Panel *panel, char *name)
{
    Xtry_to_select (panel, name);
    SELECT_ITEM (panel);
    display_mini_info (panel);
}

/* This routine reloads the directory in both panels. It tries to
** select current_file in current_panel and other_file in other_panel.
** If current_file == -1 then it automatically sets current_file and
** other_file to the currently selected files in the panels.
*/
void update_panels (int force_update, char *current_file, char *other_file)
{
    int free_pointers = 0;

    if (force_update == UP_RELOAD){
	bzero (&(cpanel->dir_stat), sizeof (cpanel->dir_stat));
	bzero (&(opanel->dir_stat), sizeof (opanel->dir_stat));
    }
    
    /* If current_file == -1 (an invalid pointer, then preserve selection */
    if (current_file == UP_KEEPSEL){
	free_pointers = 1;
	current_file = strdup (cpanel->dir [cpanel->selected].fname);
	other_file   = strdup (opanel->dir [opanel->selected].fname);
    }
    panel_reload (cpanel);
    try_to_select (cpanel, current_file);
    panel_reload (opanel);
    try_to_select (opanel, other_file);
    paint_dir (cpanel);
    paint_dir (opanel);
    if (free_pointers){
	free (current_file);
	free (other_file);
    }
    chdir (cpanel->cwd);
}

/* Sets up the terminal before executing a program */
void pre_exec ()
{
    if (clear_before_exec){
	wstandend (clean_screen);
	werase (clean_screen);
	touchwin (clean_screen);
	wrefresh (clean_screen);
    } else {
	printf ("\n\n");
    }
    keypad (stdscr, FALSE);
    reset_shell_mode ();
    endwin ();
}

/* Restores the terminal after an exec, see execute for extra post-exec code */
void post_exec ()
{
    reset_prog_mode ();
    flushinp ();
    cbreak ();
    keypad (stdscr, TRUE);
}

/* Save current stat of directories to avoid reloading the panels */
/* when no modifications have took place */
void save_cwds_stat ()
{
    if (fast_reload){
	stat (cpanel->cwd, &(cpanel->dir_stat));
	stat (opanel->cwd, &(opanel->dir_stat));
    }
}

void execute (char *command)
{
#ifdef OLD_SYSTEM_CODE
    char buf [4096];
#endif
    struct passwd *pw_data;

    pw_data = getpwuid (geteuid ());

#ifdef OLD_SYSTEM_CODE
    sprintf (buf, "%s -c \"%s\"", pw_data->pw_shell, command);
#endif
    
    save_cwds_stat ();
    pre_exec ();
#ifdef OLD_SYSTEM_CODE
    system (buf);
#else
    my_system (pw_data->pw_shell, command);
#endif
    post_exec ();

    if (pause_after_run){
	printf ("Press any key to continue...");
	fflush (stdout);
	getch ();
    }
    update_panels (UP_OPTIMIZE, UP_KEEPSEL, 0);
    refresh_screen ();
}

/* Changes the currently selected panel to the other panel */
void change_panel ()
{
    if (other_panel->view_type == view_info)
	return;

    if (other_panel->view_type & VIEW_DISABLED)
	return;
    
    current_panel->active = 0;
    show_dir (current_panel);
    unselect_item (current_panel);
    other_panel = current_panel;
    if (current_panel == &left_panel)
	current_panel = &right_panel;
    else
	current_panel = &left_panel;
    current_panel->active = 1;
    if (chdir (current_panel->cwd) != 0){
	message (1, " Error ", " Can't chdir to %s ", current_panel->cwd);
    }
    show_dir (current_panel);
    select_item (current_panel);
}

void cmd_quit ()
{
    if (query_dialog (" The MouseLess Commander  ",
		      " Do you really want to quit the Mouse-less commander? ",
		      0, 2, " Yes ", " No ") == 0)
	quit = 1;
}

/* Returns the number of items in the given panel */
int ITEMS (Panel *p)
{
    if (p->view_type == view_brief)
	return p->lines * 2;
    else
	return p->lines;
}

void move_down ()
{
    if (cpanel->selected+1 == cpanel->count)
	return;
    
    unselect_item (cpanel);
    cpanel->selected++;
    
    if (cpanel->selected - cpanel->top_file == ITEMS (cpanel)){
	/* Scroll window half screen */
	cpanel->top_file += ITEMS (cpanel)/2;
	paint_dir (cpanel);
	select_item (cpanel);
    } else 
	select_item (cpanel);
    panel_refresh (cpanel);
}

void move_up ()
{
    if (cpanel->selected == 0)
	return;
    
    unselect_item (cpanel);
    cpanel->selected--;
    if (cpanel->selected < cpanel->top_file){
	/* Scroll window half screen */
	cpanel->top_file -= ITEMS (cpanel)/2;
	if (cpanel->top_file < 0) cpanel->top_file = 0;
	paint_dir (cpanel);
    } else
	select_item (cpanel);
    panel_refresh (cpanel);
}

void move_home ()
{
    if (cpanel->selected == 0)
	return;
    unselect_item (cpanel);
    cpanel->top_file = 0;
    cpanel->selected = 0;
    paint_dir (cpanel);
    select_item (cpanel);
    panel_refresh (cpanel);
}

void move_end ()
{
    if (cpanel->selected == cpanel->count-1)
	return;
    unselect_item (cpanel);
    cpanel->selected = cpanel->count-1;
    cpanel->top_file = cpanel->selected - ITEMS (cpanel) + 1;
    if (cpanel->top_file < 0) cpanel->top_file = 0;
    paint_dir (cpanel);
    select_item (cpanel);
    panel_refresh (cpanel);
}

void prev_page ()
{
    unselect_item (cpanel);
    cpanel->selected -= ITEMS (cpanel);
    cpanel->top_file -= ITEMS (cpanel);
    if (cpanel->selected < 0) cpanel->selected = 0;
    if (cpanel->top_file < 0) cpanel->top_file = 0;
    paint_dir (cpanel);
    select_item (cpanel);
    panel_refresh (cpanel);
}

void next_page ()
{
    unselect_item (cpanel);
    cpanel->selected += ITEMS (cpanel);
    cpanel->top_file += ITEMS (cpanel);
    if (cpanel->selected >= cpanel->count)
	cpanel->selected = cpanel->count - 1;
    if (cpanel->top_file >= cpanel->count)
	cpanel->top_file = cpanel->count - 1;
    paint_dir (cpanel);
    select_item (cpanel);
    panel_refresh (cpanel);
}

void mark_file ()
{
    if (S_ISDIR (selection->buf.st_mode))
	return;
    
    selection->f.marked = selection->f.marked ? 0 : 1;
    set_attr  (cpanel, 1, selection->f.marked);
    repaint_file (cpanel, cpanel->selected);
    if (selection->f.marked){
	cpanel->marked++;
	cpanel->total += selection->buf.st_size;
	set_colors (cpanel);
	if (cpanel->marked == 1)
	    mvwprintw (cpanel->win_file, cpanel->lines+3, 1,
		       "%-*s", cpanel->cols, "", COLS/2-2);
    } else {
	cpanel->marked--;
	cpanel->total -= selection->buf.st_size;
    }
    if (mark_moves_down)
	move_down ();
    display_mini_info (cpanel);
    panel_refresh (cpanel);
}

/* This routine untouches the first line on both panels in order */
/* to avoid the refreshing the menu bar */
void untouch_bar ()
{
    touchwin (cpanel->win_file);
    touchwin (opanel->win_file);
    wtouchln (cpanel->win_file, 0, 1, 0);
    wtouchln (opanel->win_file, 0, 1, 0);
    panel_refresh (cpanel);
    panel_refresh (opanel);
}

void clr_scr ()
{
    wstandend (clean_screen);
    wclear (clean_screen);
    touchwin (clean_screen);
    wrefresh (clean_screen);
}

void repaint_screen (int clear)
{
    if (clear){
	clr_scr ();
    }
    touchwin (current_panel->win_file);
    panel_refresh (current_panel);
    touchwin (other_panel->win_file);
    panel_refresh (other_panel);
    
    touchwin (cmdline_win);
    touchwin (fkeys);
    
    wrefresh (cmdline_win);
    wrefresh (fkeys);
}

void repaint_screen_cmd ()
{
    repaint_screen (RP_CLEAR);
}

void only_refresh_screen ()
{
    panel_refresh (current_panel);
    panel_refresh (other_panel);
    wrefresh (cmdline_win);
    wrefresh (fkeys);
}

void refresh_screen ()
{
    touchwin (current_panel->win_file);
    touchwin (other_panel->win_file);
    touchwin (cmdline_win);
    touchwin (fkeys);
    only_refresh_screen ();
}

void help_cmd ()
{
    interactive_display (HELPFILE);
}

int do_cd (char *new_dir)
{
    int ret;

    if (!*new_dir)
	ret = chdir (home_dir);
    else
	ret = chdir (new_dir);
    if (ret == -1)
	return 0;
    clean_dir (&cpanel->dir, cpanel->count);
    cpanel->count = do_load_dir (&cpanel->dir, cpanel->sort_type);
    cpanel->top_file = 0;
    cpanel->selected = 0;
    cpanel->marked = 0;
    cpanel->total = 0;
    getwd (cpanel->cwd);
    return 1;
}

void enter ()
{
    char *old_dir;

    if (strlen (cmdline.buffer)){
	if (strncmp (cmdline.buffer, "cd ", 3) == 0 ||
	    strcmp (cmdline.buffer, "cd") == 0){
	    if (cmdline.buffer [2] == 0)
		cmdline.buffer [3] = 0;
	    if (!do_cd (&cmdline.buffer [3])){
		message (1, " Error ", " Can't chdir to '%s' ",
			 &cmdline.buffer [3]);
		return;
	    }
	    paint_panel (cpanel);
	    select_item (cpanel);
	    *cmdline.buffer = 0;
	    cmdline.pos = 0;
	    update_input (&cmdline);
	} else {
	    execute (cmdline.buffer);
	    cmdline.buffer [0] = 0;
	    cmdline.pos = 0;
	    update_input (&cmdline);
	}
	return;
    }
    if (S_ISDIR (selection->buf.st_mode) || link_isdir (selection)){
	if (!strcmp (selection->fname, ".."))
	    old_dir = strdup (cpanel->cwd);
	else
	    old_dir = 0;
	if (!do_cd (selection->fname))
	    return;
	try_to_select (cpanel, old_dir);
	free (old_dir);
	paint_panel (cpanel);
	select_item (cpanel);
	return;
    }
    if (is_exe (selection->buf.st_mode)){
	/* Need to append "./" to the command */
	execute (selection->fname);
	return;
    }
    regex_command (selection->fname);
}

void do_search (int c_code)
{
    int l, i;
    
    l = strlen (search_buffer);
    search_buffer [l] = c_code & 0x7f;
    search_buffer [l+1] = 0;
    l++;

    for (i = 0; i < cpanel->count; i++){
	if (strncmp (cpanel->dir [i].fname, search_buffer, l) == 0){
	    cpanel->selected = i;
	    cpanel->top_file = cpanel->selected - cpanel->lines/2;
	    if (cpanel->top_file < 0)
		cpanel->top_file = 0;

	    select_item (cpanel);
	    paint_panel (cpanel);
	    break;
	}
    }
}

void start_search ()
{
    searching = 1;
}

/* If the key pressed is not bound to any command, this function is called */
void default_key (int c_code)
{
    if (searching)
	do_search (c_code);
    else {
	handle_char (&cmdline, c_code);
    }
}

void view_cmd ()
{
    static char *viewer = 0;

    if (use_internal_view){
	view (selection->fname);
	refresh_fn = refresh_screen;
	repaint_screen (RP_NOCLEAR);
    } else {
	if (!viewer){
	    viewer = getenv ("PAGER");
	    if (!viewer)
		viewer = "view";
	}
	sprintf (cmd_buf, "%s %s", viewer, selection->fname);
	execute (cmd_buf);
    }
}

void do_edit (const char *what)
{
    static char *editor = 0;

    if (!editor){
	editor = getenv ("EDITOR");
	if (!editor)
	    editor = "vi";
    }
    sprintf (cmd_buf, "%s %s", editor, what);
    execute (cmd_buf);
}

void edit_cmd ()
{
    do_edit (selection->fname);
}

/* Returns currently selected file or the first marked file if there is one */
char *get_file (Panel *panel)
{
    int i;
    
    if (panel->marked){
	for (i = 0; i < panel->count; i++)
	    if (panel->dir [i].f.marked)
		return panel->dir [i].fname;
    } else
	return panel->dir [panel->selected].fname;
    fprintf (stderr, "Internal error\n");
    mi_getch ();
    return "";
}

/* Clears all files in the panel, used only when one file was marked */

void unmark_file (Panel *panel)
{
    int i;

    if (!panel->marked)
	return;
    for (i = 0; i < panel->count; i++)
	panel->dir [i].f.marked = 0;
    panel->marked = 0;
    panel->total = 0;
}

void copy_cmd ()
{
    char *dest;
    int  i, c, stat_r;
    struct stat s;

    if (cpanel->marked > 0)
	sprintf (cmd_buf, " Copy %d files to:", cpanel->marked);
    else {
	if (!S_ISREG (selection->buf.st_mode))
	    return;
	sprintf (cmd_buf," Copy \"%s\" file to:",name_trunc(selection->fname, 30));
    }
    dest = input_dialog (" Copy ", cmd_buf, other_panel->cwd);
    if (!dest)
	return;
    
    stat_r = stat (dest, &s);
    save_cwds_stat ();
    if (S_ISREG (s.st_mode) || stat_r != 0){
	if (cpanel->marked > 1){
	    message (1, " Error ", "Can't copy multiple files to one file");
	    free (dest);
	    return;
	}
	copy_file_file (get_file (cpanel), dest);
	unmark_file (cpanel);
    } else {
	if (!S_ISDIR (s.st_mode)){
	    message (1, " Copy ", "Unknown copy destination");
	    free (dest);
	    return;
	}
	/* Destination is a directory */
	if (!cpanel->marked)
	    copy_file_dir (selection->fname, dest);
	else {
	    nodelay (stdscr, TRUE);
	    for (i = 0; i < cpanel->count; i++){
		if (cpanel->dir [i].f.marked){
		    copy_file_dir (cpanel->dir [i].fname, dest);
		    cpanel->dir [i].f.marked = 0;
		    cpanel->marked--;
		    cpanel->total -= cpanel->dir [i].buf.st_size;
		}
		if ((c = getch ()) != ERR)
		    if (c == '\e' || c == 3)
			break;
	    }
	    nodelay (stdscr, FALSE);
	}
    }
    free (dest);
    update_panels (UP_OPTIMIZE, UP_KEEPSEL, 0);
    repaint_screen (RP_NOCLEAR);
}

void ren_cmd ()
{
    struct stat s;
    int i, c, stat_r;
    
    char *new_name = input_dialog (" Rename/Move ", "Enter rename mask:",
				   opanel->cwd);
    if (!new_name)
	return;
    if (strchr (new_name, '*') || strchr (new_name, '?')){
	/* Mask rename */
	message (0, " Missing feature ",
		 " Mask renaming not yet implemented, sorry ");
	free (new_name);
	return;
    }
    save_cwds_stat ();
    stat_r = stat (new_name, &s);
    if (S_ISREG (s.st_mode) || stat_r != 0){
	if (cpanel->marked <= 1){
	    move_file (get_file (cpanel), new_name);
	    unmark_file (cpanel);
	} else {
	    message (1," Error "," Can't rename multiple files to same name ");
	    free (new_name);
	    return;
	}
    } else if (!S_ISDIR (s.st_mode)){
	message (1, " Error ", " Destination is not a directory or a file ");
	free (new_name);
	return;
    } else {
	/* destination is a directory */
	if (cpanel->marked){
	    nodelay (stdscr, TRUE);
	    for (i = 0; i < cpanel->marked; i++){
		if (cpanel->dir [i].f.marked)
		    move_file_dir (cpanel->dir [i].fname, new_name);
		if ((c = getch ()) != ERR)
		    if (c == '\e' || c == 3)
			break;
	    }
	    nodelay (stdscr, FALSE);
	} else
	    move_file_dir (selection->fname, new_name);
    }
    free (new_name);
    update_panels (UP_OPTIMIZE, 0, 0);
    repaint_screen (RP_NOCLEAR);
}

void mkdir_cmd ()
{
    char *dir = input_dialog (" Mkdir ", "Enter directoy name:" , "");
    if (!dir)
	return;

    save_cwds_stat ();
    if (mkdir (dir, 0777) == 0){
	update_panels (UP_OPTIMIZE, UP_KEEPSEL, 0);
	unselect_item (cpanel);
	try_to_select (cpanel, dir);
	repaint_screen (RP_NOCLEAR);
	free (dir);
	return;
    }
    free (dir);
    message (1, " Error ", "  %s  ", sys_errlist [errno]);
}

void delete_cmd ()
{
    int  i;

    save_cwds_stat ();
    if (cpanel->marked > 0){
	sprintf (cmd_buf, " Do you really want to delete %d files? ",
		 cpanel->marked);
	if (query_dialog (" Delete ", cmd_buf, 3, 2, " Yes ", " No ") == 0){
	    for (i = 0; i < cpanel->count; i++){
		if (cpanel->dir [i].f.marked)
		    erase_file (cpanel->dir [i].fname);
	    }
	} else
	    return;
    } else {
	if (S_ISDIR (selection->buf.st_mode))
	    erase_dir (selection->fname);
	else
	    erase_file (selection->fname);
    }
    update_panels (UP_OPTIMIZE, UP_KEEPSEL, 0);
    repaint_screen (RP_NOCLEAR);
}

void find_cmd ()
{
    do_find ();
}

void ext_cmd ()
{
    char buffer [200];
    
    sprintf (buffer, "%s/.nc.ext", home_dir);
    do_edit (buffer);
}

void select_cmd ()
{
    char *reg_exp;
    int i;
    int c;

    reg_exp = input_dialog (" Select ","", "*");
    if (!reg_exp)
	return;
    for (i = 0; i < cpanel->count; i++){
	if (S_ISDIR (cpanel->dir [i].buf.st_mode))
	    continue;
	c = regexp_match (reg_exp, cpanel->dir [i].fname);
	if (c == -1){
	    message (1, " Error ", "  Malformed regular expression  ");
	    free (reg_exp);
	    return;
	}
	if (c){
	    if (!cpanel->dir [i].f.marked){
		cpanel->marked++;
		cpanel->total += cpanel->dir [i].buf.st_size;
	    }
	    cpanel->dir [i].f.marked = 1;
	}
    }
    paint_panel (cpanel);
    panel_refresh (cpanel);
    panel_refresh (cpanel);
    free (reg_exp);
}

void unselect_cmd ()
{
    char *reg_exp;
    int i;
    int c;

    reg_exp = input_dialog (" Unselect ","", "*");
    if (!reg_exp)
	return;
    for (i = 0; i < cpanel->count; i++){
	if (S_ISDIR (cpanel->dir [i].buf.st_mode))
	    continue;
	c = regexp_match (reg_exp, cpanel->dir [i].fname);
	if (c == -1){
	    message (1, " Error ", "  Malformed regular expression  ");
	    free (reg_exp);
	    return;
	}
	if (c){
	    if (cpanel->dir [i].f.marked){
		cpanel->marked--;
		cpanel->total -= cpanel->dir [i].buf.st_size;
	    }
	    cpanel->dir [i].f.marked = 0;
	}
    }
    paint_panel (cpanel);
    panel_refresh (cpanel);
    panel_refresh (cpanel);
    free (reg_exp);
}

/* Switches to full display panel, well, not so full */
void full_cmd ()
{
    Panel *p = MENU_PANEL;

    full_frame (p);
    p->view_type = view_full;
    left_panel.view_type &= (~VIEW_DISABLED);
    right_panel.view_type &= (~VIEW_DISABLED);
    paint_panel (p);
}

/* Switches the panel to brief directory listing */
void brief_cmd ()
{
    Panel *p = MENU_PANEL;

    brief_frame (p);
    p->view_type = view_brief;
    left_panel.view_type &= (~VIEW_DISABLED);
    right_panel.view_type &= (~VIEW_DISABLED);
    paint_panel (p);
}

void info_cmd ()
{
    
}

/* Switches the panel to long display (like ls -l) */
void long_cmd ()
{
    Panel *p = MENU_PANEL;

    long_frame (p); 
    p->view_type = view_long;
    if (p == &left_panel)
	right_panel.view_type |= VIEW_DISABLED;
    else
	left_panel.view_type  |= VIEW_DISABLED;
    paint_panel (p);
}

/* Panel sorting related routines */
void do_re_sort (Panel *panel)
{
    do_sort (&panel->dir, panel->sort_type, panel->count-1);
    paint_dir (panel);
    panel_refresh (panel);
}

void by_name_cmd ()
{
    Panel *panel = MENU_PANEL;
    panel->sort_type = (sortfn *) sort_name;
    do_re_sort (panel);
}

void by_ext_cmd ()
{
    Panel *panel = MENU_PANEL;
    panel->sort_type = (sortfn *) sort_ext;
    do_re_sort (panel);
}

void by_time_cmd ()
{
    Panel *panel = MENU_PANEL;
    panel->sort_type = (sortfn *) sort_time;
    do_re_sort (panel);
}

void by_size_cmd ()
{
    Panel *panel = MENU_PANEL;
    panel->sort_type = (sortfn *) sort_size;
    do_re_sort (panel);
}

void unsorted_cmd ()
{
    Panel *panel = MENU_PANEL;
    panel->sort_type = (sortfn *) unsorted;
    do_re_sort (panel);
}

void reread_cmd ()
{
    update_panels (UP_RELOAD, UP_KEEPSEL, 0);
    repaint_screen (RP_NOCLEAR);
}

void link_cmd ()
{
}

void menu_cmd ();

menu_entry PanelMenu [] = {
    "  Full",  'F', full_cmd,
    "  Brief", 'B', brief_cmd,
    "  Long",  'L', long_cmd, 
/*    "  Info",  'I', info_cmd, */
    "", ' ', 0,
    "  Name",      'N', by_name_cmd,
    "  Extension", 'E', by_ext_cmd,
    "  Time",      'T', by_time_cmd,
    "  Size",      'S', by_size_cmd,
    "  Unsorted",  'U', unsorted_cmd,
    "", ' ', 0,
    "  Reread",    'R', reread_cmd,
/*    "  Filter...", 'l', reread_cmd, */
};

menu_entry RightMenu [] = {
    "  Full",  'F', full_cmd,
    "  Brief", 'B', brief_cmd,
    "  Long",  'L', long_cmd, 
/*    "  Info",  'I', info_cmd, */
    "", ' ', 0,
    "  Name",      'N', by_name_cmd,
    "  Extension", 'E', by_ext_cmd,
    "  Time",      'T', by_time_cmd,
    "  Size",      'S', by_size_cmd,
    "  Unsorted",  'U', unsorted_cmd,
    "", ' ', 0,
    "  Reread",    'R', reread_cmd,
    /*
    "  Filter...", 'l', reread_cmd,
    */
};

menu_entry FileMenu [] = {
    "Help            F1", KEY_F(1), help_cmd,
    "User menu       F2", KEY_F(2), user_menu_cmd,
    "View            F3", KEY_F(3), view_cmd,
    "Edit            F4", KEY_F(4), edit_cmd,
    "Copy            F5", KEY_F(5), copy_cmd,
/*    "Link              ", ' ', link_cmd, */
    "Rename/Move     F6", KEY_F(6), ren_cmd,
    "Mkdir           F7", KEY_F(7), mkdir_cmd,
    "Delete          F8", KEY_F(8), delete_cmd,
    "", ' ', 0,
    "Select group     +", '+', select_cmd,
    "Unselect group   \\", '\\', unselect_cmd,
    "", ' ', 0,
    "Quit           F10", KEY_F(10), cmd_quit
};

menu_entry CmdMenu [] = {
    "Find file (test)", 'F', find_cmd,
    "Save setup",       'S', save_setup_cmd,
    "Extension file edit", 'E', ext_cmd, 
};

/* Must keep in sync with the constants in menu_cmd */
menu_entry OptMenu [] = {
    "  show Backup files", 'b', toggle_show_backup,
    "  show Hidden files", 'H', toggle_show_hidden,
    "  Verbose operation", 'V', toggle_verbose,
    "  mark moves Down",   'D', toggle_mark_move_down,
    "  Pause after run",   'P', toggle_pause_after_run,
    "  show mini-Status",  'S', toggle_show_mini_status,
    "  shEll patterns",    'E', toggle_easy_patterns,
    "  Auto save setup",   'A', toggle_auto_save,
    "  auto Menus",        'M', toggle_auto_menu,
    "  use Internal view", 'I', toggle_internal,
    "  miX all files",     'X', toggle_mix_all_files,
    "  Fast dir-reload",   'F', toggle_fast_reload,
/*    "  Align extensions",  'A', toggle_align_extensions, */
};

#define menu_entries(x) sizeof(x)/sizeof(menu_entry)

Menu *MenuBar [5];

void init_menu ()
{
    MenuBar [0] = create_menu (" Left ", PanelMenu, menu_entries (PanelMenu));
    MenuBar [1] = create_menu (" File ", FileMenu, menu_entries (FileMenu));
    MenuBar [2] = create_menu (" Command ", CmdMenu, menu_entries (CmdMenu));
    MenuBar [3] = create_menu (" Options ", OptMenu, menu_entries (OptMenu));
    MenuBar [4] = create_menu (" Right ", RightMenu, menu_entries (PanelMenu));
}

void check_options_menu (int index, int flag)
{
    *OptMenu [index].text = flag ? '*' : ' ';
}

void check_menu_panel (Panel *panel, menu_entry PanelMenu [])
{
    PanelMenu [0].text [0] = panel->view_type == view_full ? '*' : ' ';
    PanelMenu [1].text [0] = panel->view_type == view_brief ? '*' : ' ';
    PanelMenu [2].text [0] = panel->view_type == view_long ? '*' : ' ';
/*    PanelMenu [3].text [0] = panel->view_type == view_info ? '*' : ' '; */

    PanelMenu[4].text[0]= panel->sort_type == (sortfn *)sort_name ? '*' : ' ';
    PanelMenu[5].text[0]= panel->sort_type == (sortfn *) sort_ext ? '*' : ' ';
    PanelMenu[6].text[0]= panel->sort_type == (sortfn *) sort_time ? '*' : ' ';
    PanelMenu[7].text[0]= panel->sort_type == (sortfn *) sort_size ? '*' : ' ';
    PanelMenu[8].text[0]= panel->sort_type == (sortfn *) unsorted ? '*' : ' ';
}

void menu_cmd ()
{
    int panel;
    
    check_options_menu (0, show_backups);
    check_options_menu (1, show_dot_files);
    check_options_menu (2, verbose);
    check_options_menu (3, mark_moves_down);
    check_options_menu (4, pause_after_run);
    check_options_menu (5, show_mini_info);
    check_options_menu (6, easy_patterns);
    check_options_menu (7, auto_save_setup);
    check_options_menu (8, auto_menu);
    check_options_menu (9, use_internal_view);
    check_options_menu (10, mix_all_files);
    check_options_menu (11, fast_reload);
    
    check_menu_panel (&left_panel, PanelMenu);
    check_menu_panel (&right_panel, RightMenu);
    panel = current_panel == &right_panel ? 4 : 0;
    run_bar (stdscr, 0, 0, 80, 12, 5, panel, MenuBar, BAR_FILL, A_BOLD,
	     SELECTED_COLOR);
}

/* Flag toggling functions */

void toggle_fast_reload ()
{
    fast_reload = !fast_reload;
    if (fast_reload_w){
	message (0, " Information ",
		 " Using the fast reload option may not reflect the exact "
		 " directory contents. In this cases you'll need to do a  "
		 " manual reload of the directory. See the man page for   "
		 " the details.");
    }
}

void toggle_mix_all_files ()
{
    mix_all_files = !mix_all_files;
    update_panels (UP_RELOAD, UP_KEEPSEL, 0);
}

void toggle_internal ()
{
    use_internal_view = !use_internal_view;
}

void toggle_auto_menu ()
{
    auto_menu = !auto_menu;
}

void toggle_show_backup ()
{
    show_backups = !show_backups;
    update_panels (UP_RELOAD, UP_KEEPSEL, 0);
}

void toggle_show_hidden ()
{
    show_dot_files = !show_dot_files;
    update_panels (UP_RELOAD, UP_KEEPSEL, 0);
}

void toggle_verbose ()
{
    verbose = !verbose;
}

void toggle_auto_save ()
{
    auto_save_setup = !auto_save_setup;
}

void toggle_mark_move_down ()
{
    mark_moves_down = !mark_moves_down;
}

void toggle_pause_after_run ()
{
    pause_after_run = !pause_after_run;
}

void toggle_show_mini_status ()
{
    if (show_mini_info){
	left_panel.lines += 2;
	right_panel.lines +=2;
    } else {
	left_panel.lines -= 2;
	right_panel.lines -= 2;
    }
    show_mini_info = !show_mini_info;
    paint_panel (cpanel);
    paint_panel (opanel);
}

void toggle_easy_patterns ()
{
    easy_patterns = !easy_patterns;
}

void toggle_align_extensions ()
{
    align_extensions = !align_extensions;
}

void dump_marked ()
{
    int i;
    
    system ("clear");
    move (1, 1);
    for (i = 1; i < 8; i++){
	attron (COLOR_PAIR (i));
	printw ("Este es el color %d\n", i);
    }
    getch ();
    fprintf (stderr, "Hay %d archivos marcados\n", cpanel->marked);
    for (i = 0; i < cpanel->count; i++){
	if (cpanel->dir [i].f.marked)
	    fprintf (stderr, "%s\n", cpanel->dir [i].fname);
    }
    getch ();
    repaint_screen (RP_NOCLEAR);
}

/* Inserts the selected file name into the input line */
void copy_prog_name ()
{
    char *k = selection->fname;

    while (*k)
	handle_char (&cmdline, *k++);
    handle_char (&cmdline, ' ');
}

/* Inserts the selected file name in the other panel into the input line */
void copy_other_prog_name ()
{
    char *k = opanel->dir [opanel->selected].fname;

    if ((opanel->view_type==view_info) || (opanel->view_type & VIEW_DISABLED))
	return;
	
    while (*k)
	handle_char (&cmdline, *k++);
    
    handle_char (&cmdline, ' ');
}

void suspend_cmd ()
{
    save_cwds_stat ();
    pre_exec ();
    kill (getpid (), SIGTSTP);
    post_exec ();
    update_panels (UP_OPTIMIZE, UP_KEEPSEL, 0);
/*    repaint_screen (RP_CLEAR); */
}

void quote_next ()
{
    quote = 1;
}

struct {
    int   key_code;
    void  (*fn)(int);
} key_map [] = {
    KEY_F(10), 	cmd_quit,
    KEY_F(1),   help_cmd,
    KEY_F(2),   user_menu_cmd, 
    KEY_F(3),   view_cmd,
    KEY_F(4),   edit_cmd,
    KEY_F(5),   copy_cmd,
    KEY_F(6),   ren_cmd,
    KEY_F(7),   mkdir_cmd,
    KEY_F(8),   delete_cmd,
    KEY_F(9),   menu_cmd,
    KEY_F(12),  dump_marked,
    '+',        select_cmd,
    '\\',       unselect_cmd,
    '\t',  	change_panel,
    '\n',       enter,		/* Enter */
    KEY_DOWN,   move_down,
    KEY_UP, 	move_up,
    KEY_IC,     mark_file,
    KEY_HOME,	move_home,
    KEY_C1,     move_end,
    KEY_END,    move_end,
    KEY_A1,     move_home,
    KEY_NPAGE,  next_page,
    KEY_PPAGE,  prev_page,
    XCTRL('l'), repaint_screen_cmd, /* C-l */

    ALT('\n'),  copy_prog_name,
    ALT('\t'),  copy_other_prog_name,
    XCTRL('t'), mark_file,
    
    /* Emacs-like bindings */
    XCTRL('v'), next_page,      /* C-v like emacs */
    ALT('v'),   prev_page,	/* M-v like emacs */
    XCTRL('p'), move_up,	/* C-p like emacs */
    XCTRL('n'), move_down,	/* C-n like emacs */
    XCTRL('q'), quote_next,	/* C-q like emacs */
    XCTRL('s'), start_search,	/* C-s like emacs */
    ALT('s'),   start_search,	/* M-s not like emacs */
    
    /* Suspend */
    XCTRL('z'), suspend_cmd,
    
    /* Default key handling */
    0,          default_key,
};

void save_setup ()
{
    char profile [100];

    sprintf (profile, "%s/.nc.ini", home_dir);
    
    set_int (profile, "show_backups", show_backups);
    set_int (profile, "show_dot_files", show_dot_files);
    set_int (profile, "verbose", verbose);
    set_int (profile, "mark_moves_down", mark_moves_down);
    set_int (profile, "pause_after_run", pause_after_run);
    set_int (profile, "show_mini_info", show_mini_info);
    set_int (profile, "shell_patterns", easy_patterns);
    set_int (profile, "auto_save_setup", auto_save_setup);
    set_int (profile, "align_extensions", align_extensions);
    set_int (profile, "auto_menu", auto_menu);
    set_int (profile, "use_internal_view", use_internal_view);
    set_int (profile, "clear_before_exec", clear_before_exec);
    set_int (profile, "mix_all_files", mix_all_files);
    set_int (profile, "fast_reload", fast_reload);
    set_int (profile, "fast_reload_msg_shown", fast_reload_w);
    WritePrivateProfileString ("Dirs", "other_dir", opanel->cwd, profile);
    sync_profiles ();
}

void save_setup_cmd ()
{
    save_setup ();
    message (0, " Setup ", " Setup saved to ~/.nc.ini ");
}

void load_setup ()
{
    static char buffer [100];
    char   *profile;
    
    sprintf (buffer, "%s/.nc.ini", home_dir);
    
    if (exist_file (buffer)){
	profile = buffer;
    } else if (exist_file (PROFILE)){
	profile = PROFILE;
    } else
	return;
    
    show_backups    = get_int (profile, "show_backups", 0);
    show_dot_files  = get_int (profile, "show_dot_files", 1);
    verbose         = get_int (profile, "verbose", 1);
    mark_moves_down = get_int (profile, "mark_moves_down", 1);
    pause_after_run = get_int (profile, "pause_after_run", 1);
    show_mini_info  = get_int (profile, "show_mini_info", 1);
    easy_patterns   = get_int (profile, "shell_patterns", 1);
    auto_save_setup = get_int (profile, "auto_save_setup", 1);
    align_extensions= get_int (profile, "align_extensions", 1);
    auto_menu       = get_int (profile, "auto_menu", 0);
    fast_reload     = get_int (profile, "fast_reload", 0);
    fast_reload_w   = get_int (profile, "fast_reload_msg_shown", 0);
    mix_all_files   = get_int (profile, "mix_all_files", 0);
    use_internal_view=get_int (profile, "use_internal_view", 1);
    clear_before_exec=get_int (profile, "clear_before_exec", 1);
    
    if (!other_dir){
	GetPrivateProfileString ("Dirs", "other_dir", ".", buffer,
			     sizeof (buffer), profile);
	other_dir = buffer;
    }
}

void do_nc ()
{
    int key;
    int i;

    refresh_fn = refresh_screen;
    if (COLS < 70 || LINES < 12){
	fprintf (stderr, "Screen too small: you need at leas 70x12\n");
	return;
    }
    clean_screen = newwin (0, 0, 0, 0);
    home_dir = getenv ("HOME");
    home_dir = home_dir ? home_dir : "/";
    load_setup ();
    init_colors ();
    init_labels ();
    init_key ();
    init_entry ();
    init_panels ();
    init_menu ();
    create_input (&cmdline, strlen (prompt), 0, cmdline_win, 0, COLS-strlen (prompt), "");

    if (auto_menu)
	user_menu_cmd ();
    
    for (quit = 0; !quit;){
	key = mi_getch ();
	if (quote){
	    default_key (key);
	    quote = 0;
	    continue;
	}
	for (i = 0; key_map [i].key_code; i++){
	    if (key == key_map [i].key_code){
		searching = 0;
		(*key_map [i].fn)(key);
		break;
	    }
	}
	if (key_map [i].key_code)
	    continue;

	/* Default key handling */
	(*key_map [i].fn)(key);
	wrefresh (cmdline_win);
    }
    if (auto_save_setup)
	save_setup ();
    clr_scr ();
    keypad (cmdline_win, FALSE);
    keypad (stdscr, FALSE);
    reset_shell_mode ();
    noraw ();
}

main (int argc, char *argv [])
{
    extern char *optarg;
    extern int optind, opterr;
    char   *this_dir = 0;
    int    c;
    int    output1, output2;

    while ((c = getopt (argc, argv, "Vbs")) != -1){
	switch (c){
	case 'V':
	    printf ("MouseLess Commander %s\n", VERSION);
	    return 0;

	case 'b':
	    disable_colors = 1;
	    break;

case 'P':
	    print_last_wd = 1;
	    break;
	    
	case 's':
	    force_slow = 1;	/* Currently it does nothing */
	    break;

	default:
	    exit (1);
	}
    }

    /* sets the current dir and the other dir */
    for (; optind < argc; optind++){
	if (this_dir){
	    if (other_dir)
		break;
	    else
		other_dir = argv [optind];
	} else
	    this_dir = argv [optind];
    }
    if (this_dir)
	chdir (this_dir);
    if (print_last_wd){
	output1 = dup (1);
	close (1);
	output2 = open ("/dev/tty", O_RDWR);
    }
    save_stop_handler ();
    initscr ();
    do_nc ();
    endwin ();
    if (print_last_wd){
	close (output2);
	output2 = dup (output1);
	close (output1);
	if (isatty (1))
	    printf ("%s", cpanel->cwd);
    }
    return 0;
}
