/* Dialog managing.
   Copyright (C) 1994 Miguel de Icaza.
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
#include <ncurses.h>
#include <stdio.h>
#include <stdarg.h>
#include <sys/types.h>
#include "global.h"
#include "dialog.h"
#include "input.h"
#include "color.h"
#include "util.h"
#include "win.h"

static char rcsid [] = "$Header: /usr/users/miguel/c/CVS/nc/dialog.c,v 1.3 1994/06/27 22:52:51 miguel Exp $";
static char buffer [4096];

static Dialog *current_dialog = 0;
WINDOW *top_window;

void (*refresh_fn)();

void create_dialog (int cols, int rows, char *header, char *text, int error)
{
    int base = LINES/3;
    Dialog *newd;
    
    if (rows+1 > LINES/3+2)
	base = LINES/3+2;

    newd = (Dialog *) xmalloc (sizeof (Dialog), "create_dialog");
    newd->backpointer = current_dialog;
    current_dialog = newd;
    
    newd->d = newwin (rows+4, cols+4, base-(rows+1)/2, COLS/2-(cols+5)/2);
    newd->f = derwin (newd->d, rows+2, cols+2, 1, 1);
    newd->t = derwin (newd->d, rows, cols, 2, 2);
    leaveok (newd->d, TRUE);
    leaveok (newd->f, TRUE);
    leaveok (newd->t, TRUE);
    if ((error & D_ERROR)) {
	wattron (newd->d, ERROR_COLOR | A_BOLD);
	wclr (newd->d);
	wattron (newd->f, ERROR_COLOR | A_BOLD);
	wattron (newd->t, ERROR_COLOR | A_BOLD);
    } else {
	wattron (newd->d, A_REVERSE);
	wclr (newd->d);
	wattron (newd->f, A_REVERSE);
	wattron (newd->t, A_REVERSE);
    }
    box (newd->f, ACS_VLINE, ACS_HLINE);
    mvwprintw (newd->f, 0, (cols-strlen (header)-2)/2, header);
    wclr (newd->t); 
    mvwprintw (newd->t, 0, 0, text); 
    touchwin (newd->t);
    touchwin (newd->f);
    touchwin (newd->d);
    
    wrefresh (newd->d);
    top_window = newd->d;
}

WINDOW *get_top_text ()
{
    return current_dialog->t;
}

int run_dialog ()
{
    int key;
    
    key = getch ();
    refresh_screen ();
    return key;
}

void done_dialog ()
{
    Dialog *p;
    
    if (!current_dialog){
	fprintf (stderr, "Trying to destroy non-existant dialog\n");
	exit (1);
    }
    delwin (current_dialog->t);
    delwin (current_dialog->f);
    delwin (current_dialog->d);
    p = current_dialog;
    current_dialog = p->backpointer;
    if (current_dialog)
	top_window = current_dialog->d;
    else
	top_window = 0;
    free (p);
}

int message (int error, char *header, char *text,  ...)
{
    va_list  args;
    int      len, lines;
    char     *buf;
    int      v;

    buf = buffer;

    strcpy (buffer, "\n");
    va_start (args, text);
    vsprintf (&buffer [1], text, args);
    strcat (buf, "\n");
    va_end (args);
    len = max (strlen (header), msglen (buf, &lines));
    create_dialog (len, lines, header, buf, error);
    if (!(error & D_INSERT)){
	v = run_dialog ();
	done_dialog ();
	return v;
    }
    return 0; 
}

int error (int status, int errnum, char *text,  ...)
{
    va_list  args;
    int      len, lines;
    char     *buf;
    int      v;
    char     *header = " Error ";
    
    buf = buffer;

    strcpy (buffer, "\n");
    va_start (args, text);
    vsprintf (&buffer [1], text, args);
    strcat (buf, "\n");
    
    len = max (strlen (header), msglen (buf, &lines));
    create_dialog (len, lines, header, buf, 1);
    v = run_dialog ();
    done_dialog ();
    va_end (args);
    return 0;
}

char *input_dialog (char *header, char *text, char *def_text)
{
    int    len, lines;
    char   *return_value;
    Input  in;
    int    c;

    strcpy (buffer, text);
    
    len = max (strlen (header), msglen (buffer, &lines));
    len = max (len, 60);
    create_dialog (len, lines+1, header, buffer, 0);
    create_input (&in, 1, lines, current_dialog->t, INPUT_COLOR,
		  58, def_text);

    return_value = in.buffer;
    while ((c = mi_getch ()) != '\n'){
	if (c == '\e'){
	    return_value = 0;
	    free (in.buffer);
	    break;
	}
	handle_char (&in, c);
    }
    done_dialog ();
    (*refresh_fn)();
    return (return_value);
}

