/*
 * Copyright 1996-1999 Hans Reiser
 */
#include "fsck.h"
#include <strings.h>

/*  on pass2 we take leaves which could not be inserted into tree
    during pass1 and insert each item separately. It is possible that
    items of different objects with the same key can be found. We
    treat that in the following way: we put it into tree with new key
    and link it into root directory with name made of dir,oid. When
    coming item is a directory - we delete object from the tree, put
    it back with different key, link it to root directory and insert
    directory as it is */

/* remaping rules: we have an item (it is taken from "non-insertable"
   leaf). It has original key yet. We check to see if object with this
   key is remapped. Object can be only remapped if it is not a piece
   of directory */



/* in list of this structures we keep list of "remapped" objects. Only
   directory can not be remapped. */
struct remapped {
    unsigned long old_dir_id;
    unsigned long old_objectid;
    unsigned long new_objectid;
    mode_t mode;
    struct remapped * next;
};

/* this list is created on pass2, and it is used after semantic pass
   to link remapped files */
struct remapped * remapped_list;


/* return objectid the object has to be remapped with */
__u32 objectid_for_remapping (struct key * key, mode_t mode)
{
    struct remapped * cur;

    cur = remapped_list;

    while (cur) {
	if (cur->old_dir_id == key->k_dir_id && cur->old_objectid == key->k_objectid &&
	    (S_IFMT & cur->mode) == (S_IFMT & mode))
	    /* object is remapped already */
	    return cur->new_objectid;
	cur = cur->next;
    }

    cur = getmem (sizeof (struct remapped));
    cur->old_dir_id = key->k_dir_id;
    cur->old_objectid = key->k_objectid;
    cur->new_objectid = get_unused_objectid (fs);
    cur->mode = mode;
    cur->next = remapped_list;
    remapped_list = cur;
    fsck_log ("remap_object: (%lu %lu) is remapped to (%lu, %lu). look for it in /\n",
	      key->k_dir_id, key->k_objectid, key->k_dir_id, cur->new_objectid);
    return cur->new_objectid;
}




/* this item is in tree. All unformatted pointer are correct. Do not
   check them */
static void save_item_2 (struct si ** head, struct item_head * ih, 
			 char * item, __u32 blocknr)
{
    struct si * si, * cur;

    si = getmem (sizeof (*si));
    si->si_dnm_data = getmem (ih_item_len(ih));
    /*si->si_blocknr = blocknr;*/
    memcpy (&(si->si_ih), ih, IH_SIZE);
    memcpy (si->si_dnm_data, item, ih_item_len(ih));

    if (*head == 0)
	*head = si;
    else {
	cur = *head;
	while (cur->si_next)
	    cur = cur->si_next;
	cur->si_next = si;
    }
    return;
}


struct si * save_and_delete_file_item (struct si * si, struct path * path)
{
    struct buffer_head * bh = PATH_PLAST_BUFFER (path);
    struct item_head * ih = PATH_PITEM_HEAD (path);

    save_item_2 (&si, ih, B_I_PITEM (bh, ih), bh->b_blocknr);

    /* delete item temporary - do not free unformatted nodes */
    reiserfsck_delete_item (path, 1/*temporary*/);
    return si;
}


/* this is called by rewrite_object to insert items which were in the
   tree already but were deleted. unformatted node pointed do not get
   freed/allocated. 'do_remap' says to insert_item_separately
   whether it should remap items */
struct si * remove_saved_item (struct si * si);
static void put_saved_items_into_tree_2 (struct si * si, int do_remap)
{
    while (si) {
	insert_item_separately (&(si->si_ih), si->si_dnm_data,
				1/*was in tree*/, do_remap);
	si = remove_saved_item (si);
    }
}


/* delete all items (but directory ones) with the same first two
   components and insert them back remapped */
void rewrite_object (struct item_head * ih, int do_remap)
{
    struct key key;
    struct key * rkey;
    struct path path;
    struct item_head * path_ih;
    struct si * si;

    /* starting with the leftmost one - look for all items of file,
       store them and delete */
    copy_short_key (&key, &ih->ih_key);
    set_type_and_offset (KEY_FORMAT_1, &key, SD_OFFSET, TYPE_STAT_DATA);

    si = 0;
    while (1) {
	usearch_by_key (fs, &key, &path);
	if (get_item_pos (&path) == B_NR_ITEMS (get_bh (&path))) {
	    rkey = uget_rkey (&path);
	    if (rkey && !not_of_one_file (&key, rkey)) {
		/* file continues in the right neighbor */
		copy_key (&key, rkey);
		pathrelse (&path);
		continue;
	    }
	    /* there is no more items of file */
	    pathrelse (&path);
	    break;
	}
	path_ih = get_ih (&path);
	if (not_of_one_file (&key, &(path_ih->ih_key))) {
	    pathrelse (&path);
	    break;
	}

	/* ok, item found, but make sure that it is not a directory one */
	if (is_direntry_ih (path_ih)) {
	    copy_key (&key, &(path_ih->ih_key));
	    set_offset (KEY_FORMAT_1, &key, get_offset (&key) + 1);
	    pathrelse (&path);
	    continue;
	}
	si = save_and_delete_file_item (si, &path);
    }
	    
    /* put all items back into tree */
    put_saved_items_into_tree_2 (si, do_remap);
}






#define st_mode(sd) le16_to_cpu((sd)->sd_mode)
#define st_mtime(sd) le32_to_cpu((sd)->sd_mtime)


/*  sigh, ugly but overwrite_new_stat_data and overwrite_old_stat_data are so
    similar */
#define remap_stat_data(s) \
{\
    fsck_log (": \'%M\' found in tree, "\
	      "new is \'%M\'\n", st_mode (in_tree_sd),\
	      st_mode (new_sd));\
    \
    if (S_ISDIR (st_mode (new_sd))) {\
	/* new item is a directory stat data. delete and insert\
	   remapped items of another object which are in tree now */\
	pathrelse (path);\
\
	rewrite_object (new_ih, 1/*do remap*/);\
\
	mark_item_unreachable (new_ih);\
\
	new_sd->sd_nlink = 0;\
	usearch_by_key (fs, &new_ih->ih_key, path);\
	reiserfsck_insert_item (path, new_ih, new_item);\
	return;\
    }\
\
    /* directory is in tree. remap new stat data (assigning new objectid to\
       it) */\
    new_ih->ih_key.k_objectid = cpu_to_le32 (objectid_for_remapping (&new_ih->ih_key,\
								     st_mode (new_sd)));\
    mark_item_unreachable (new_ih);\
\
    new_sd->sd_nlink = 0;\
    pathrelse (path);\
    usearch_by_key (fs, &new_ih->ih_key, path);\
    reiserfsck_insert_item (path, new_ih, new_item);\
}


static void overwrite_new_stat_data (struct item_head * new_ih,
				     void * new_item, struct path * path)
{
    struct stat_data * new_sd, * in_tree_sd;

    in_tree_sd = (struct stat_data *)get_item (path);
    new_sd = (struct stat_data *)new_item;

    if ((S_IFMT & st_mode (in_tree_sd)) != (S_IFMT & st_mode (new_sd))) {
	/* stat datas of different objects with the same key found */
	remap_stat_data ("overwrite_new_stat_data");
	return;

    } else {
	/* old and new stat data are of one type of file. Overwrite if
           new is newer */
	if (st_mtime (new_sd) > st_mtime (in_tree_sd)) {
	    /* new sd is newer than the found one */
	    memcpy (in_tree_sd, new_sd, SD_SIZE);
	    in_tree_sd->sd_nlink = 0;
	    mark_buffer_dirty (get_bh (path));
	}
	pathrelse (path);
    }
}

static void overwrite_old_stat_data (struct item_head * new_ih,
				     void * new_item, struct path * path)
{
    struct stat_data_v1 * new_sd, * in_tree_sd;

    in_tree_sd = (struct stat_data_v1 *)get_item (path);
    new_sd = (struct stat_data_v1 *)new_item;

    if ((S_IFMT & st_mode (in_tree_sd)) != (S_IFMT & st_mode (new_sd))) {
	/* stat datas of different objects with the same key found */
	remap_stat_data ("overwrite_old_stat_data");
	return;

    } else {
	/* old and new stat data are of one type of file. Overwrite if
           new is newer */
	if (st_mtime (new_sd) > st_mtime (in_tree_sd)) {
	    /* new sd is newer than the found one */
	    memcpy (in_tree_sd, new_sd, SD_V1_SIZE);
	    in_tree_sd->sd_nlink = 0;
	    mark_buffer_dirty (get_bh (path));
	}
	pathrelse (path);
    }
}


/* insert sd item if it does not exist, overwrite it otherwise */
static void put_sd_into_tree (struct item_head * new_ih, char * new_item)
{
    struct path path;
    int key_format = ih_key_format (new_ih);

    if (usearch_by_key (fs, &(new_ih->ih_key), &path) == ITEM_FOUND) {

        if (ih_key_format (get_ih(&path)) != key_format) {
	    /* found two stat datas of one object and one of them is old,
               another is new */
	    if (key_format == KEY_FORMAT_1) {
		/* new sd is am old one */
		struct stat_data_v1 * new_sd;
		struct stat_data * in_tree_sd;

		new_sd = (struct stat_data_v1 *)new_item;
		in_tree_sd = (struct stat_data *)get_item (&path);

		if (!S_ISDIR (st_mode (new_sd))) {
		    /* new sd is not a directory - remap it */
		    new_ih->ih_key.k_objectid = cpu_to_le32 (objectid_for_remapping (&new_ih->ih_key,
										     st_mode (new_sd)));
		    /* need to find new place for insertion */
		    pathrelse (&path);
		    usearch_by_key (fs, &new_ih->ih_key, &path);
		    mark_item_unreachable (new_ih);
		    new_sd->sd_nlink = 0;
		    reiserfsck_insert_item (&path, new_ih, new_item);
		    return;
		} else {
		    /* new sd is of directory one */
		    if (!S_ISDIR (st_mode (in_tree_sd))) {
			/* what is in tree - is not a directory - so we can
                           remap it */
			pathrelse (&path);
			rewrite_object (new_ih, 1/*do remap*/);

			usearch_by_key (fs, &new_ih->ih_key, &path);
			mark_item_unreachable (new_ih);
			new_sd->sd_nlink = 0;
			reiserfsck_insert_item (&path, new_ih, new_item);
			return;
		    } else {
			fsck_log ("put_sd_into_tree: found old and new sd-s of one directory %k"
				  " - old format stat data ignored\n", &new_ih->ih_key);
			pathrelse (&path);
			return;
		    }
		}
	    } else {
		/* the stat data in the tree is an old one */
		struct stat_data * new_sd;
		struct stat_data_v1 * in_tree_sd;

		new_sd = (struct stat_data *)new_item;
		in_tree_sd = (struct stat_data_v1 *)get_item (&path);
		if (S_ISDIR (st_mode (new_sd))) {
		    if (S_ISDIR (st_mode (in_tree_sd))) {
			/* delete old stat data.. */
			reiserfsck_delete_item (&path, 0/*not temporary*/);
		    } else {
			/* remap old object, and insert new */
			pathrelse (&path);

			rewrite_object (new_ih, 1/*do remap*/);
		    }
		} else {
		    /* new sd is not a directory so we can insert it remapped */
		    new_ih->ih_key.k_objectid = cpu_to_le32 (objectid_for_remapping (&new_ih->ih_key,
										     st_mode (new_sd)));
		    pathrelse (&path);
		}

		/* we took care of old stat datt and can now insert a new one */
		usearch_by_key (fs, &new_ih->ih_key, &path);
		mark_item_unreachable (new_ih);
		new_sd->sd_nlink = 0;
		reiserfsck_insert_item (&path, new_ih, new_item);
	    }
	    /* we have done with sd-s of different versions */
	    return;
	}


	/* overwrite stat data in the tree */
	if (!stat_data_v1 (new_ih))	{
	    overwrite_new_stat_data (new_ih, new_item, &path);
	} else {
	    overwrite_old_stat_data (new_ih, new_item, &path);
	}
    } else {
	__u32 objectid;

	/* item not found, insert a new one */
	if (stat_data_v1 (new_ih)) {
	    assert (new_ih->ih_item_len == cpu_to_le16 (SD_V1_SIZE));
	    ((struct stat_data_v1 *)new_item)->sd_nlink = 0;
	} else {
	    assert (new_ih->ih_item_len == cpu_to_le16 (SD_SIZE));
	    ((struct stat_data *)new_item)->sd_nlink = 0;
	}
	mark_item_unreachable (new_ih);
	reiserfsck_insert_item (&path, new_ih, new_item);

	objectid = le32_to_cpu (new_ih->ih_key.k_objectid);

	if (mark_objectid_really_used (proper_id_map (fs), objectid)) {
	    stat_shared_objectid_found (fs);
	    mark_objectid_really_used (shared_id_map (fs), objectid);
	}	    
    }
}


static int reiserfsck_find_entry (struct key * key, struct reiserfs_de_head * deh,
				  struct path * path)
{
    struct key entry_key;

    copy_key (&entry_key, key);
    set_offset (KEY_FORMAT_1, &entry_key, deh->deh_offset);
    set_type (KEY_FORMAT_1, &entry_key, TYPE_DIRENTRY);

    return usearch_by_entry_key (fs, &entry_key, path);
}


/* this tries to put each item entry to the tree, if there is no items
   of the directory, insert item containing 1 entry */
static void put_directory_item_into_tree (struct item_head * comingih, char * item)
{
    /*  struct item_head * ih;*/
    struct reiserfs_de_head * deh;
    int i, retval;
    struct path path;
    int size;
    char * buf, * entry;
    int remapped = 0;

    deh = (struct reiserfs_de_head *)item;

    for (i = 0; i < ih_entry_count (comingih); i ++, deh ++) {

	if (!is_properly_hashed (fs, name_in_entry (deh, i),
				 name_length (comingih, deh, i),
				 le32_to_cpu (deh->deh_offset))) {
	    die ("put_directory_item_into_tree: should be hashed properly ()");
	}

	entry = item + deh->deh_location;

	retval = reiserfsck_find_entry (&(comingih->ih_key), deh, &path);
	switch (retval) {
	case POSITION_FOUND:
	    pathrelse (&path);
	    break;

	case POSITION_NOT_FOUND:
	    /* paste_into_item accepts entry to paste as buffer, beginning
	       with entry header and body, that follows it */
	    buf = getmem (size = entry_length (comingih, deh, i) + DEH_SIZE);
	    memcpy (buf, deh, DEH_SIZE);
	    ((struct reiserfs_de_head *)buf)->deh_location = 0;
	    memcpy (buf + DEH_SIZE, entry, size - DEH_SIZE);

	    reiserfsck_paste_into_item (&path, buf, size);

	    freemem (buf);
	    break;

	case DIRECTORY_NOT_FOUND:
	{
	    struct item_head tmpih;

	    // insert item containing on entry into the tree
	    buf = getmem (size = entry_length(comingih, deh, i) + DEH_SIZE);
	    memcpy (buf, deh, DEH_SIZE);
	    ((struct reiserfs_de_head *)buf)->deh_location = cpu_to_le16 (DEH_SIZE);
	    memcpy (buf + DEH_SIZE, entry, size - DEH_SIZE);

	    set_key_format (&tmpih, KEY_FORMAT_1);
	    copy_key (&(tmpih.ih_key), &(comingih->ih_key));
	    tmpih.ih_item_len = cpu_to_le16 (size);
	    tmpih.u.ih_entry_count = cpu_to_le16 (1);
	    mark_item_unreachable (&tmpih);
      
	    reiserfsck_insert_item (&path, &tmpih, buf);

	    freemem (buf);
	    break;
	}
	case REGULAR_FILE_FOUND:
	    if (remapped)
		fsck_log ("put_directory_item_into_tree: hmm, remapping (%lu %lu) failed, trying again",
			  comingih->ih_key.k_dir_id, comingih->ih_key.k_objectid);
	    pathrelse (&path);
	    rewrite_object (comingih, 1/*do remap*/);
	    deh --;
	    i --;
	    break;
	}
    }
}


static void link_one (struct remapped * file)
{
    char entry [80];
    char * name;
    struct item_head ih;
    struct reiserfs_de_head * deh;
    int len;

    deh = (struct reiserfs_de_head *)entry;
    name = entry + DEH_SIZE;
    sprintf (name, "%lu,%lu", file->old_dir_id, file->new_objectid);
    /* to do not care about 3.5 and 3.6 formats - append name with '_'
       up to 8 byte boundary */
    len = strlen (name);
    memset (name + len, '_', ROUND_UP (len) - len);
    name[ROUND_UP (len)] = 0;
    deh->deh_dir_id = cpu_to_le32 (file->old_dir_id);
    deh->deh_objectid = cpu_to_le32 (file->new_objectid);
    /* FIXME: gen counter is 0 */

    deh->deh_offset = GET_HASH_VALUE (cpu_to_le32 (reiserfs_hash (fs) (name, strlen (name))));

    set_bit (DEH_Visible, &(deh[0].deh_state));
    deh->deh_location = cpu_to_le16 (DEH_SIZE);

    ih.ih_key = root_dir_key;
    ih.ih_item_len = DEH_SIZE + strlen (name);
    ih.u.ih_entry_count = 1;

    put_directory_item_into_tree (&ih, entry);
}


static void link_remapped_files (void)
{
    struct remapped * cur;
    int count;

    count = 0;
    cur = remapped_list;
    while (cur) {
	link_one (cur);
	cur = cur->next;
	count ++;
    }

    fsck_log ("There were %d remapped files. %s\n", count, 
	      count ? "Look for then in \"/\"" : "");
}


/* if remapped - item will be put with newly allocated objectid */
void insert_item_separately (struct item_head * ih,
			     char * item, int was_in_tree,
			     int remapped)
{
    if (remapped)
	/* */
	ih->ih_key.k_objectid = objectid_for_remapping (&ih->ih_key, S_IFREG);

    // FIXME: refuse transparently bad items
    if (ih->ih_key.k_dir_id == ih->ih_key.k_objectid)
	return;
    if (is_stat_data_ih (ih)) {
	put_sd_into_tree (ih, item);
    } else if (is_direntry_ih (ih)) {
	put_directory_item_into_tree (ih, item);
    } else {
	reiserfsck_file_write (ih, item, was_in_tree);
    }
}


/* 1 if object was remapped, 0 otherwise. When it was remapped - set
   ih->ih_key.k_objectid to correct one if necessary */
int is_remapped (struct item_head * ih)
{
    struct remapped * cur;

    cur = remapped_list;

    while (cur) {
	if (cur->old_dir_id == ih->ih_key.k_dir_id &&
	    (cur->old_objectid == ih->ih_key.k_objectid ||
	     cur->new_objectid == ih->ih_key.k_objectid)) {
	    /* object is remapped already */
	    ih->ih_key.k_objectid = cur->new_objectid;
	    return 1;
	}
	cur = cur->next;
    }

    return 0;
/*    return ih->ih_key.k_objectid;*/
}


static int is_stat_data_of_directory (struct buffer_head * bh, struct item_head * ih)
{
    struct stat_data * sd; /* sd_mode is first16 bit of both new and old stat
                              data */

    sd = (struct stat_data *)B_I_PITEM (bh, ih);
    return S_ISDIR (st_mode (sd));
}


static void put_items (struct buffer_head * bh)
{
    int i;
    struct item_head * ih;

    ih = B_N_PITEM_HEAD (bh, 0);
    for (i = 0; i < B_NR_ITEMS (bh); i ++, ih ++) {
	if (i && bad_pair (fs, bh, i)) {
	    /* skip item if it is in wrong order */
	    continue;
	}
	/*	mark_objectid_as_used (fs, le32_to_cpu (ih->ih_key.k_objectid));*/
	
	/* if this file was remapped already - then put coming item
           with the key the object was assigned with */
	if (is_direntry_ih (ih) || is_stat_data_of_directory (bh, ih)) {
	    /* directory can not be remapped */
	    insert_item_separately (ih, B_I_PITEM (bh, ih), 0/*was in tree*/, 0/*do not remap*/);
	    continue;
	}

	if (is_remapped (ih))
	    ;
	insert_item_separately (ih, B_I_PITEM (bh, ih), 0/*was in tree*/, 0/*do not remap*/);
    }
}


/* this is actually pass 2.   
   uninsertable blocks are marked by 0s in g_uninsertable_leaf_bitmap
   during the pass 1. They must be not in the tree */
extern reiserfs_bitmap_t uninsertable_leaf_bitmap;
void pass_2_take_bad_blocks_put_into_tree (void)
{
    struct buffer_head * bh;
    unsigned long j;
    unsigned long bb_counter = 0;
    int what_node;

    fsck_progress ("%lu leaves could not be inserted during pass 1\nPass 2 - ",
		   how_many_uninsertables_were_there ());

    fsck_log ("####### Pass 2 #######\n");

    j = 0;
    while (reiserfs_bitmap_find_zero_bit (uninsertable_leaf_bitmap, &j) == 0) {
	bh = bread (fs->s_dev, j, fs->s_blocksize);
	if (bh == 0) {
	    fsck_log ("pass_2_take_bad_blocks_put_into_tree: "
		      "unable to read %lu block on device 0x%x\n",
		      j, fs->s_dev);
	    goto next;
	}
	
	if (is_block_used (bh->b_blocknr)) {
	    fsck_log ("pass_2_take_bad_blocks_put_into_tree: "
		      "block %d can not be in tree\n", bh->b_blocknr);
	    goto next;
	}
	/* this must be leaf */
	what_node = who_is_this (bh->b_data, fs->s_blocksize);
	if (what_node != THE_LEAF) { // || B_IS_KEYS_LEVEL(bh)) {
	    fsck_log ("take_bad_blocks_put_into_tree: buffer (%b %z) must contain leaf\n", bh, bh);
	    goto next;
	}

	fsck_log ("block %lu is being inserted\n", bh->b_blocknr);
	put_items (bh);
	
	print_how_far (&bb_counter, stats(fs)->uninsertable_leaves);
	
    next:
	brelse (bh);
	j ++;
    }


    if (bb_counter != stats(fs)->uninsertable_leaves)
	die ("take_bad_blocks_put_into_tree: found bad block %d, must be %d", 
	     bb_counter, stats(fs)->uninsertable_leaves);

    /* link all files remapped into root directory */
    link_remapped_files ();

    fsck_progress ("\n");
}




















































