/*
 * Copyright 1996-2001 Hans Reiser
 */
#include "fsck.h"
#include <stdlib.h>


#define NR_TO_READ 8

static unsigned long tmp_zeroed;

/* pass 0 scans the partition (used part). It creates two maps which will be
   used on the pass 1. These are a map of nodes looking like leaves and a map
   of "bad" unformatted nodes. */

/* leaves */
reiserfs_bitmap_t leaves_bitmap;
#define pass0_is_leaf(block) __is_marked (leaves, block)
#define pass0_mark_leaf(block) __mark (leaves, block)

/* nodes which are referred to from only one indirect item */
reiserfs_bitmap_t good_unfm_bitmap;
#define pass0_is_good_unfm(block) __is_marked (good_unfm, block)
#define pass0_mark_good_unfm(block) __mark (good_unfm, block)
#define pass0_unmark_good_unfm(block) __unmark (good_unfm, block)

/* nodes which are referred to from more than one indirect item */
reiserfs_bitmap_t bad_unfm_bitmap;
#define pass0_is_bad_unfm(block) __is_marked (bad_unfm, block)
#define pass0_mark_bad_unfm(block) __mark (bad_unfm, block)
#define pass0_unmark_bad_unfm(block) __unmark (bad_unfm, block)



static void make_aux_bitmaps (struct super_block * s)
{
    leaves_bitmap = reiserfs_create_bitmap (SB_BLOCK_COUNT (s));

    good_unfm_bitmap = reiserfs_create_bitmap (SB_BLOCK_COUNT (s));

    bad_unfm_bitmap = reiserfs_create_bitmap (SB_BLOCK_COUNT (s));

}


/* register block some indirect item points to */
static void register_unfm (unsigned long block)
{
    if (!pass0_is_good_unfm (block) && !pass0_is_bad_unfm (block)) {
	/* this block was not pointed by other indirect items yet */
	pass0_mark_good_unfm (block);
	return;
    }

    if (pass0_is_good_unfm (block)) {
	/* block was pointed once already, unmark it in bitmap of good
           unformatted nodes and mark in bitmap of bad pointers */
	pass0_unmark_good_unfm (block);
	pass0_mark_bad_unfm (block);
	return;
    }

    assert (pass0_is_bad_unfm (block));
}





/* 'upper' item is correct if 'upper + 2' exists and its key is greater than
   key of 'upper' */
static int upper_correct (struct buffer_head * bh, struct item_head * upper,
			  int upper_item_num)
{
    if (upper_item_num + 2 < B_NR_ITEMS (bh)) {
	if (comp_keys (&upper->ih_key, &(upper + 2)->ih_key) != -1)
	    /* item-num's item is out of order of order */
	    return 0;
	return 1;
    }
    
    /* there is no item above the "bad pair" */
    return 2;
}


/* 'lower' item is correct if 'lower - 2' exists and its key is smaller than
   key of 'lower' */
static int lower_correct (struct buffer_head * bh, struct item_head * lower,
			  int lower_item_num)
{
    if (lower_item_num - 2 >= 0) {
	if (comp_keys (&(lower - 2)->ih_key, &lower->ih_key) != -1)
	    return 0;
	return 1;
    }
    return 2;
}


/* return 1 if something was changed */
static int correct_key_format (struct item_head * ih)
{
    int dirty = 0;

    if (is_stat_data_ih (ih)) {
	/* for stat data we have no way to check whether key format in item
	   head matches to the key format found from the key directly */
	if (ih_item_len (ih) == SD_V1_SIZE) {
	    if (ih_key_format (ih) != KEY_FORMAT_1) {
		fsck_log ("correct_key_format: ih_key_format of (%H) is set to format 1\n",
			  ih);
		set_key_format (ih, KEY_FORMAT_1);
		return 1;
	}
	    return 0;
	}
	if (ih_item_len (ih) == SD_SIZE) {
	    if (ih_key_format (ih) != KEY_FORMAT_2) {
		fsck_log ("correct_key_format: ih_key_format of (%H) is set to format 2\n",
			  ih);
		set_key_format (ih, KEY_FORMAT_2);
		return 1;
	    }
	    return 0;
	}
	
	die ("stat data of wrong length");
    }
    
    if (key_format (&ih->ih_key) != ih_key_format (ih)) {
	fsck_log ("correct_key_format: ih_key_format of (%H) is set to format found in the key\n",
		  ih);
	set_key_format (ih, key_format (&ih->ih_key));
	dirty = 1;
    }
    
    if (type_unknown (&ih->ih_key)) {
	/* FIXME: */
	set_type (key_format (&ih->ih_key), &ih->ih_key, TYPE_DIRECT);
	dirty = 1;	
    }
    
    return dirty;
}


static int prob_name (reiserfs_filsys_t fs,
		      char ** name, int max_len, __u32 deh_offset)
{
    int start; /* */
    int len;

    for (start = 0; start < max_len; start ++) {
	for (len = 0; len < max_len - start; len ++) {
	    if (is_properly_hashed (fs, *name + start, len + 1, deh_offset)) {
		*name = *name + start;
		return len + 1;
	}
	}
    }
    return 0;
}


/* check directory item and try to recover something */
static int verify_directory_item (reiserfs_filsys_t fs, struct buffer_head * bh,
				  int item_num)
{
    struct item_head * ih;
    char * item;
    struct reiserfs_de_head * deh;
    char * entries, * end, * name;
    int total_entry_len, name_len;
    int i;
    int bad_entries; /* how many bad neighboring entries */
    int dirty;
    int entry_count;

    ih = B_N_PITEM_HEAD (bh, item_num);
    item = B_I_PITEM (bh,ih);
    deh = (struct reiserfs_de_head *)item;

    dirty = 0;
    entry_count = ih_entry_count (ih);

    /* check first */
    for (i = 0; i < ih_entry_count (ih); i ++) {
	name = name_in_entry (deh + i, i);
	name_len = name_length (ih, deh + i, i);
	if (!is_properly_hashed (fs, name, name_len, deh_offset (deh + i))) {
	    dirty ++;
		    }
		}
		
    if (!dirty)
	return 0;

    fsck_log ("pass0: block %lu: %d-th item (%H) has %d bad entries..",
	      bh->b_blocknr, item_num, ih, dirty);

    if (get_offset (&ih->ih_key) == DOT_OFFSET) {
	/* first item of directory - make sure that "." and ".." are in place */
	if (deh_offset (deh) != DOT_OFFSET || name_in_entry (deh, 0)[0] != '.') {
	    deh->deh_offset = cpu_to_le32 (DOT_OFFSET);
	    name_in_entry (deh, 0)[0] = '.';
		}
	if (deh_offset (deh + 1) != DOT_DOT_OFFSET ||
	    name_in_entry (deh + 1, 1)[0] != '.' || name_in_entry (deh + 1, 1)[1] != '.') {
	    (deh + 1)->deh_offset = cpu_to_le32 (DOT_DOT_OFFSET);
	    name_in_entry (deh + 1, 1)[0] = '.';
	    name_in_entry (deh + 1, 1)[1] = '.';
	    }
	}

    end = item + ih_item_len (ih);
    deh += ih_entry_count (ih);
    entries = (char *)deh;
    total_entry_len = ih_item_len (ih) - DEH_SIZE * ih_entry_count (ih);
    i = ih_entry_count (ih);

    bad_entries = 0;
    do {
	i --;
	deh --;
	name_len = prob_name (fs, &entries, total_entry_len, deh_offset (deh));
	if (!name_len) {
	    if (!bad_entries) {
		deh->deh_location = cpu_to_le16 (entries - item);
	    } else {
		deh->deh_location = cpu_to_le16 (deh_location (deh + 1) + 1);
	}
	    bad_entries ++;

	    /*fsck_log ("verify_directory_item: entry %d: in string \'%s\' there is no substring matching hash %ld\n",
	      i, bad_name (entries, total_entry_len), masked_offset);*/
		mark_de_bad (deh);
	    continue;
    }
	bad_entries = 0;
	/*fsck_log ("verify_directory_item: entry %d: found \"%s\" name matching hash %ld\n",
	  i, bad_name (entries, name_len), masked_offset);*/

	/* 'entries' points now to the name which match given offset - so, set deh_location */
	deh->deh_location = cpu_to_le16 (entries - item);
	deh->deh_state = 0;
	mark_de_visible (deh);

	entries += name_len;
	total_entry_len = end - entries;
	/* skip padding zeros */
	while (!*entries) {
	    entries ++;
	    total_entry_len --;
	}
	/* 'entries' points now at the place where next (previous) entry
           should start */
    } while ((char *)deh != item);


    /* fixme: this will not work if all entries are to be deleted */
    for (i = 0; i < ih_entry_count (ih); i ++, deh ++) {
	deh = (struct reiserfs_de_head *)B_I_PITEM (bh, ih) + i;
	if (de_bad (deh)) {
	    cut_entry (fs, bh, item_num, i);
	    i --;
	}
/*
	    fsck_log ("verify_directory_item: %d-th entry is to be deleted: "
		      "\"%s\" does not match to hash %lu\n", 
		      i, bad_name (name_in_entry (deh, i), name_length (ih, deh, i)),
		      deh_offset (deh));
*/
    }

    fsck_log ("%d entries were deleted\n", entry_count - ih_entry_count (ih));
	mark_buffer_dirty (bh);

    return 0;
}


/* do this on pass 0 with every leaf marked used */

/* FIXME: we can improve fixing of broken keys: we can ssfe direct items which
   go after stat data and have broken keys */
static void pass0_correct_leaf (reiserfs_filsys_t fs,
				struct buffer_head * bh)
{
    int i, j;
    struct item_head * ih;
    __u32 * ind_item;
    unsigned long unfm_ptr;
    int dirty = 0;

 start_again:

    ih = B_N_PITEM_HEAD (bh, 0);
    for (i = 0; i < B_NR_ITEMS (bh); i ++, ih ++) {
	if (ih->ih_key.k_dir_id == 0 || ih->ih_key.k_objectid == 0) {
	    /* sometimes stat datas get k_objectid==0 or k_dir_id==0 */
	    if (i == (B_NR_ITEMS (bh) - 1)) {
		/* */
		if (i == 0) {
		    fsck_log ("pass0: item %k is alone in the block %lu\n",
			      &ih->ih_key, bh->b_blocknr);
		    return;
		}
		/* delete last item */
		delete_item (fs, bh, i - 1);
		return;
	    }
	    /* there is next item: if it is not stat data - take its k_dir_id
               and k_objectid. if key order will be still wrong - the changed
               item will be deleted */
	    if (!is_stat_data_ih (ih + 1)) {
		ih->ih_key.k_dir_id = (ih + 1)->ih_key.k_dir_id;
		ih->ih_key.k_objectid = (ih + 1)->ih_key.k_objectid;
		set_offset (KEY_FORMAT_1, &ih->ih_key, 0);
		set_type (KEY_FORMAT_1, &ih->ih_key, TYPE_STAT_DATA);
		fsck_log ("pass0: block %lu: item %d had zero k_dir_id or k_objectid - fixed to %k\n",
			  bh->b_blocknr, i, &ih->ih_key);
		dirty = 1;
	    } else if (i == 0) {
		delete_item (fs, bh, i);
		goto start_again;
	    }
	}

	/* this recovers corruptions like the below: 
	   1774 1732 0 0
	   116262638 1732 1 3
	   1774 1736 0 0 */
	if (i && is_stat_data_ih (ih - 1) && !is_stat_data_ih (ih)) {
	    if (ih->ih_key.k_objectid != (ih - 1)->ih_key.k_objectid ||
		ih->ih_key.k_dir_id != (ih - 1)->ih_key.k_dir_id ||
		get_offset (&ih->ih_key) != 1) {
		fsck_log ("pass0: block %lu: %d-th item (%H) fixed to ", 
			  bh->b_blocknr, i, ih);
		ih->ih_key.k_dir_id = (ih - 1)->ih_key.k_dir_id;
		ih->ih_key.k_objectid = (ih - 1)->ih_key.k_objectid;
		set_offset (key_format (&(ih->ih_key)), &(ih->ih_key), 1);
		if (type_unknown (&ih->ih_key))
		    /* FIXME: we could look at the stat data and at the item
                       itself */
		    set_type (key_format (&(ih->ih_key)), &(ih->ih_key), TYPE_DIRECT);
		fsck_log ("%H\n", ih);
		dirty = 1;
	    }
	}

	/* FIXME: corruptions like:
	   56702 66802 1 2
	   56702 65536 0 0
	   56702 66803 1 2
	   do not get recovered (both last items will be deleted) */
	/* delete item if it is not in correct order of object items */
	if (i && not_of_one_file (&ih->ih_key, &(ih - 1)->ih_key) &&
	    !is_stat_data_ih (ih)) {
	    fsck_log ("pass0: block %lu: %H follows non stat item %H - deleted\n",
		      bh->b_blocknr, ih, ih - 1);
	    delete_item (fs, bh, i);
	    goto start_again;
	}

	if (i &&  comp_keys (&(ih - 1)->ih_key, &ih->ih_key) != -1) {
	    /* previous item has key not smaller than the key of currect item */
	    if (is_stat_data_ih (ih - 1) && !is_stat_data_ih (ih)) {
		/* fix key of stat data such as if it was stat data of that item */
		fsck_log ("pass0: block %lu: %d-th item %k is out of order, made a stat data of %d-th (%k)\n",
			  bh->b_blocknr, i - 1, &(ih - 1)->ih_key, i, &ih->ih_key);
		(ih - 1)->ih_key.k_dir_id = ih->ih_key.k_dir_id;
		(ih - 1)->ih_key.k_objectid = ih->ih_key.k_objectid;
		set_offset (KEY_FORMAT_1, &(ih - 1)->ih_key, 0);
		set_type (KEY_FORMAT_1, &(ih - 1)->ih_key, TYPE_STAT_DATA);
		dirty = 1;
	    } else {
		/* ok, we have to delete one of these two - decide which one */
		int retval;

		/* something will be deleted */
		dirty = 1;
		retval = upper_correct (bh, ih - 1, i - 1);
		switch (retval) {
		case 0:
		    /* delete upper item */
		    fsck_log ("pass0: block %lu: %d-th (upper) item (%k) is out of order, deleted\n",
			      bh->b_blocknr, i - 1, &(ih - 1)->ih_key);
		    delete_item (fs, bh, i - 1);
		    goto start_again;

		case 1:
		    /* delete lower item */
		    fsck_log ("pass0: block %lu: %d-th (lower) item (%k) is out of order, deleted\n",
			      bh->b_blocknr, i, &ih->ih_key);
		    delete_item (fs, bh, i);
		    goto start_again;

		default:
		    /* upper item was the first item of a node */
		}

		retval = lower_correct (bh, ih, i);
		switch (retval) {
		case 0:
		    /* delete lower item */
		    fsck_log ("pass0: block %lu: %d-th (lower) item (%k) is out of order, deleted\n",
			      bh->b_blocknr, i, &ih->ih_key);
		    delete_item (fs, bh, i);
		    goto start_again;

		case 1:
		    /* delete upper item */
		    fsck_log ("pass0: block %lu: %d-th (upper) item (%k) is out of order, deleted\n",
			      bh->b_blocknr, i - 1, &(ih - 1)->ih_key);
		    delete_item (fs, bh, i - 1);
		    goto start_again;

		default:
		    /* there wer only two items in a node, so we could not
                       decide what to delete, go and ask user */
		}
		fsck_log ("pass0: which of these items looks better (other will be deleted)?\n"
			  "%h\n%h\n", ih - 1, ih);
		if (fsck_user_confirmed (fs, "1 or 2?", "1\n", 1))
		    delete_item (fs, bh, i - 1);
		else
		    delete_item (fs, bh, i);
		goto start_again;
	    }
	}

	dirty += correct_key_format (ih);

	if (is_stat_data_ih (ih)) {
	    ;/*correct_stat_data (fs, bh, i);*/
	}

	if (is_direntry_ih (ih)) {
	    verify_directory_item (fs, bh, i);
	    continue;
	}

	if (!is_indirect_ih (ih))
	    continue;
	
	ind_item = (__u32 *)B_I_PITEM (bh, ih);
	for (j = 0; j < I_UNFM_NUM (ih); j ++) {
	    unfm_ptr = le32_to_cpu (ind_item [j]);
	    if (!unfm_ptr)
		continue;
	    
	    if (fsck_mode (fs) == FSCK_ZERO_FILES) {
		/* FIXME: this is temporary mode of fsck */
		ind_item [j] = 0;
		reiserfs_bitmap_clear_bit (fsck_new_bitmap(fs), unfm_ptr);
		tmp_zeroed ++;
		dirty = 1;
		continue;
	    }

	    if (not_data_block (fs, unfm_ptr) || /* journal area or bitmap or super block */
		unfm_ptr >= SB_BLOCK_COUNT (fs)) {/* garbage in pointer */

		stat_wrong_pointer_found (fs);
		fsck_log ("pass0: %d-th pointer (%lu) in item %k (leaf block %lu) is wrong\n",
			  j, unfm_ptr, &ih->ih_key, bh->b_blocknr);
		ind_item [j] = 0;
		dirty = 1;
		continue;
	    }
	    if (!was_block_used (unfm_ptr)) {
	      /* this will get to a pool of allocable blocks */
	      ind_item [j] = 0;
	      dirty = 1;
	      stat_wrong_pointer_found (fs);
	      continue;
	    }

	    /* mark block in bitmaps of unformatted nodes */
	    register_unfm (unfm_ptr);
	}
    }

    if (B_NR_ITEMS (bh) < 1)
	die ("pass0: block %lu got all items deleted\n",
	     bh->b_blocknr);

    pass0_mark_leaf (bh->b_blocknr);

    if (dirty) {
	stat_pass0_leaf_corected (fs);
	mark_buffer_dirty (bh);
    }
}


static int is_bad_sd (struct item_head * ih, char * item)
{
    struct stat_data * sd = (struct stat_data *)item;

    if (ih->ih_key.u.k_offset_v1.k_offset || ih->ih_key.u.k_offset_v1.k_uniqueness)
	return 1;

    if (ih_item_len (ih) == SD_V1_SIZE) {
	/* looks like old stat data */
	if (ih_key_format (ih) != KEY_FORMAT_1)
	    fsck_log ("item %H has wrong format\n", ih);
	return 0;
    }

    if (!S_ISDIR (sd->sd_mode) && !S_ISREG(sd->sd_mode) &&
	!S_ISCHR (sd->sd_mode) && !S_ISBLK(sd->sd_mode) &&
	!S_ISLNK (sd->sd_mode) && !S_ISFIFO(sd->sd_mode) &&
	!S_ISSOCK(sd->sd_mode)) {
	fsck_log ("file %k unexpected mode encountered 0%o\n", &ih->ih_key, sd->sd_mode);
    }
    return 0;
}


static int bad_hash (char * name, int namelen, __u32 deh_offset)
{
    return !is_properly_hashed (fs, name, namelen, deh_offset);

}

int is_bad_directory (struct item_head * ih, char * item, int dev, int blocksize)
{
    int i;
    char * name;
    int namelen, entrylen;
    struct reiserfs_de_head * deh = (struct reiserfs_de_head *)item;
    __u32 prev_offset = 0;
    __u16 prev_location = ih_item_len (ih);
    int min_entry_size = 1;/* we have no way to understand whether the
                              filesystem were created in 3.6 format or
                              converted to it. So, we assume that minimal name
                              length is 1 */

    if (ih_item_len (ih) / (DEH_SIZE + min_entry_size) < ih_entry_count (ih))
	/* entry count is too big */
	return 1;

    for (i = 0; i < ih_entry_count (ih); i ++, deh ++) {
	entrylen = entry_length(ih, deh, i);
	if (entrylen > REISERFS_MAX_NAME_LEN (blocksize)) {
	    return 1;
	}
	if (deh_offset (deh) <= prev_offset) {
	    return 1;
	}
	prev_offset = deh_offset (deh);

	if (deh_location(deh) + entrylen != prev_location) {
	    return 1;
	}
	prev_location = deh_location (deh);

	namelen = name_length (ih, deh, i);
	name = name_in_entry (deh, i);
	if (bad_hash (name, namelen, deh_offset (deh))) {
	    return 1;
	}
    }
    return 0;
}


/* change incorrect block adresses by 0. Do not consider such item as incorrect */
static int is_bad_indirect (struct item_head * ih, char * item, int dev, int blocksize)
{
    int i;
    int bad = 0;
    int blocks;

    if (ih_item_len(ih) % UNFM_P_SIZE) {
	fsck_log ("is_bad_indirect: indirect item of %h of invalid length\n", ih);
	return 1;
    }

    blocks = SB_BLOCK_COUNT (fs);
  
    for (i = 0; i < I_UNFM_NUM (ih); i ++) {
	__u32 * ind = (__u32 *)item;

	if (le32_to_cpu (ind[i]) >= blocks) {
	    bad ++;
	    fsck_log ("is_bad_indirect: %d-th pointer of item %H looks bad (%lu)\n",
		      i, ih, le32_to_cpu (ind [i]));
	    continue;
	}
    }
    return bad;
}


/* this is used by pass1.c:save_item and check.c:is_leaf_bad */
int is_bad_item (struct buffer_head * bh, struct item_head * ih, char * item)
{
    int blocksize, dev;

    blocksize = bh->b_size;
    dev = bh->b_dev;

    // FIXME: refuse transparently bad items
    if (ih->ih_key.k_dir_id == ih->ih_key.k_objectid)
	return 1;
    if (!ih->ih_key.k_dir_id || !ih->ih_key.k_objectid)
	return 1;

    if (is_stat_data_ih(ih))
	return is_bad_sd (ih, item);

    if (is_direntry_ih (ih))
	return is_bad_directory (ih, item, dev, blocksize);

    if (is_indirect_ih (ih))
	return is_bad_indirect (ih, item, dev, blocksize);

    if (is_direct_ih (ih))
	return 0;

    return 1;
}

#if 0
/* 1 if i-th and (i-1)-th items can not be neighbors */
int bad_pair (struct super_block * s, struct buffer_head * bh, int i)
{
    struct item_head * ih;

    ih = B_N_PITEM_HEAD (bh, i);

    if (comp_keys (&((ih - 1)->ih_key), &ih->ih_key) != -1)
	return 1;

    if (is_stat_data_ih (ih))
	/* left item must be of another object */
	if (comp_short_keys (&((ih - 1)->ih_key), &ih->ih_key) != -1)
	    return 1;

    if (is_direct_ih(ih)) {
	/* left item must be indirect or stat data item of the same
	   file */
	if (not_of_one_file (&((ih - 1)->ih_key), &ih->ih_key))
	    return 1;

	if (!((is_stat_data_ih (ih - 1) && get_offset (&ih->ih_key) == 1) ||
	      (is_indirect_ih (ih - 1) &&	
	       get_offset (&(ih - 1)->ih_key) + get_bytes_number (ih-1, bh->b_size) ==
	       get_offset (&ih->ih_key))))
	    return 1;
	
    }

    if (is_indirect_ih (ih) || is_direntry_ih (ih)) {
	/* left item must be stat data of the same object */
	if (not_of_one_file (&((ih - 1)->ih_key), &ih->ih_key))
	    return 1;
	
	if (!is_stat_data_ih (ih - 1))
	    return 1;
    }

    return 0;
}
#endif

int is_leaf_bad (struct buffer_head * bh)
{
    int i;
    struct item_head * ih;
    int bad = 0;

    assert (is_leaf_node (bh));

    for (i = 0, ih = B_N_PITEM_HEAD (bh,  0); i < B_NR_ITEMS (bh); i ++, ih ++) {
	if (is_bad_item (bh, ih, B_I_PITEM (bh, ih))) {
	    fsck_log ("is_leaf_bad: block %lu: %d-th item (%H) is bad\n",
		      bh->b_blocknr, i, ih);
	    bad = 1;
	    continue;
	}

	if (i && bad_pair (fs, bh, i)) {
	    fsck_log ("is_leaf_bad: block %luL %d-th item (%H) and "
		      "the next one (%H) are in wrong order\n",
		     bh->b_blocknr, i - 1, ih - 1, ih);
	    bad = 1;
	}
    }

    return bad;
}


static int at_least_one_used (reiserfs_filsys_t fs, unsigned long first, int count)
{
    int i;

    for (i = 0; i < count; i ++)
	if (SB_BLOCK_COUNT (fs) > first + i &&
	    was_block_used (first + i))
	    return 1;
    return 0;
}


static void go_through (struct super_block * s)
{
    struct buffer_head * bbh, * bh;
    int nr_to_read = NR_TO_READ;
    int i, j;
    char * data;
    int what_node;
    unsigned long handled_blocks = 0;
    /*FILE * block_list;*/

    /* for debug only */
    /*block_list = fopen ("pass0-leaves", "w");*/

    for (i = 0; i < SB_BLOCK_COUNT (s); i += nr_to_read) {
	if (at_least_one_used (s, i, nr_to_read)) {
	    /* at least one of nr_to_read blocks is to be checked */
	    bbh = bread (s->s_dev, i / nr_to_read, s->s_blocksize * nr_to_read);
	    if (bbh) {
		for (j = 0; j < nr_to_read; j ++) {
		    if (!was_block_used (i + j)) {
			stat_free_block_found (s);
			continue;
		    }
		    if (not_data_block (s, i + j)) {
			stat_not_data_block_found (s);
			continue;
		    }

		    stat_used_block_found (s);

		    print_how_far (&handled_blocks, g_blocks_to_read);
		    data = bbh->b_data + j * s->s_blocksize;

		    what_node = who_is_this (data, s->s_blocksize);
		    if ( what_node != THE_LEAF )
			continue;
		    
		    /* fprintf (block_list, "leaf %d\n", i + j);*/

		    /* the node looks like a leaf, but it still can be
		       not perfect */
		    bh = make_buffer (s->s_dev, i + j, s->s_blocksize, data);

		    pass0_correct_leaf (s, bh);
		    brelse (bh);
		}
		bforget (bbh);
	    } else {
		/* bread failed */
		if (nr_to_read != 1) {
		    /* we tryied to read bunch of blocks. Try to read them by one */
		    i -= nr_to_read;
		    nr_to_read = 1;
		}
	    }
	}

	if (nr_to_read == 1 && ((i + 1) % NR_TO_READ) == 0) {
	    /* we have read NR_TO_READ blocks one at time, switch back to
               reading NR_TO_READ blocks at time */
	    i -= (NR_TO_READ - 1);
	    nr_to_read = NR_TO_READ;
	}
    }

    /*    fclose (block_list);*/
}


/* this makes a map of blocks which can be allocated which fskc will continue: */
static void check_bitmaps (struct super_block * s)
{
    int i;
 
    stats (s)->all_blocks = SB_BLOCK_COUNT (s);

    /* find how many leaves are not pointed by any indirect items */
    for (i = 0; i < SB_BLOCK_COUNT (s); i ++) {
	if (not_data_block (s, i))
	    continue;

	if (!was_block_used (i)) {
	    /* marked free in the on-disk bitmap */
	    make_allocable (i);
	    continue;
	}
	  
	if (pass0_is_leaf (i)) {
	    /* marked in the bitmap of leaves */
	    stat_leaf_found (s);
	    if (pass0_is_good_unfm (i) || pass0_is_bad_unfm (i))
		stat_pointed_leaf_found (s);
	}

	if (pass0_is_good_unfm (i) && pass0_is_bad_unfm (i))
	    die ("check_bitmap: bad and good unformatted");

	if (pass0_is_good_unfm (i)) {
	    stat_pointed_found (s);
	    stat_pointed_once_found (s);
	    continue;
	}
	if (pass0_is_bad_unfm (i)) {
	    stat_pointed_found (s);
	    stat_pointed_more_than_once_found (s);
	    continue;
	}

	/* blocks which marked used but are not leaves and are not
	   pointed (internals in short) get into bitmap of allocable
	   blocks */
	if (!(pass0_is_leaf (i))) {
	    make_allocable (i);
	}
    }
}


static int should_skip_pass0 (reiserfs_filsys_t fs)
{
    return 0;

#if 0
    struct stat st;
    unsigned long size;
    int answer;
    
    size = ((SB_BLOCK_COUNT (fs) + 7) / 8) * 4;
    if (stat (".pass0.bmps", &st) != 0 || st.st_size != size)
	return 0;
    
    answer = fsck_user_confirmed (fs, "There seems to be correct results "
				  "of pass 0 - should I skip it (Yes)?",
				  "Yes\n", 0);
    return answer;
#endif
	
}

static void save_pass0_result (struct super_block * s)
{
    return;

#if 0
    FILE * fp;
    unsigned long size;
    
    size = ((SB_BLOCK_COUNT (fs) + 7) / 8);
 
    fp = fopen (".pass0.bmps", "w");
    if (!fp)
	return;

    if (fwrite (leaves_bitmap->bm_map, size, 1, fp) != 1)
	return;
    if (fwrite (good_unfm_bitmap->bm_map, size, 1, fp) != 1)
	return;
    if (fwrite (bad_unfm_bitmap->bm_map, size, 1, fp) != 1)
	return;
    if (fwrite (fsck_allocable_bitmap (s)->bm_map, size, 1, fp) != 1)
	return;
    fclose (fp);
#endif
}


static void load_pass0_result (struct super_block * s)
{
    reiserfs_panic ("This does not work");

#if 0
    FILE * fp;
    unsigned long size;
    
    size = ((SB_BLOCK_COUNT (fs) + 7) / 8);

    fp = fopen (".pass0.bmps", "r");
    if (!fp)
	die ("load_pass0_result: could not avoid pass0");
    
    if (fread (leaves_bitmap->bm_map, size, 1, fp) != 1 ||
	fread (good_unfm_bitmap->bm_map, size, 1, fp) != 1 ||
	fread (bad_unfm_bitmap->bm_map, size, 1, fp) != 1 ||
	fread (fsck_allocable_bitmap (s)->bm_map, size, 1, fp) != 1)
	die ("load_pass0_result: could not avoid pass0");

    fclose (fp);  
#endif
}


int are_there_used_leaves (unsigned long from, int count)
{
    int i;

    for (i = 0; i < count; i ++)
	if (pass0_is_leaf (from + i))
	    return 1;
    return 0;
}


int is_used_leaf (unsigned long block)
{
    return pass0_is_leaf (block);
}


int how_many_leaves_were_there (void)
{
    return stats (fs)->leaves;
}

/* these are used to correct uformatted node pointers */
int is_bad_unformatted (unsigned long block)
{
    return pass0_is_bad_unfm (block);
}

/* this is for check only. With this we make sure that all pointers we
   put into tree on pass 1 do not point to leaves (FIXME), do not
   point to journal, bitmap, etc, do not point out of fs boundary and
   are marked used in on-disk bitmap */
int still_bad_unfm_ptr_1 (unsigned long block)
{
    if (!block)
	return 0;
    if (pass0_is_leaf (block))
	return 1;
    if (pass0_is_bad_unfm (block) && !is_bad_unfm_in_tree_once (block))
	return 1;
    if (not_data_block (fs, block))
	return 1;
    if (!was_block_used (block))
	return 1;
    if (block >= stats (fs)->all_blocks)
	return 1;
    return 0;
    
}


/* pointers to data block which get into tree are checked with this */
int still_bad_unfm_ptr_2 (unsigned long block)
{
    if (!block)
	return 0;
    if (is_block_used (block))
	return 1;
    if (block >= stats (fs)->all_blocks)
	return 1;
    return 0;
}


/* these are used to allocate blocks for tree building */
int are_there_allocable_blocks (int amout_needed)
{
    if (reiserfs_bitmap_zeros (fsck_allocable_bitmap (fs)) < amout_needed) {
	int zeros = 0, i;
	
	fsck_progress ("Hmm, there are not enough allocable blocks, checking bitmap...");fflush (stdout);
	for (i = 0; i < fsck_allocable_bitmap (fs)->bm_bit_size; i ++)
	    if (!reiserfs_bitmap_test_bit (fsck_allocable_bitmap (fs), i))
		zeros ++;
	fsck_progress ("there are %d zeros, btw\n", zeros);
	return 0;
    }
    return 1;
}


unsigned long alloc_block (void)
{
    unsigned long block = 0; /* FIXME: start point could be used */

    if (reiserfs_bitmap_find_zero_bit (fsck_allocable_bitmap (fs), &block)) {
	die ("alloc_block: allocable blocks counter is wrong");
	return 0;
    }
    reiserfs_bitmap_set_bit (fsck_allocable_bitmap (fs), block);
    return block;
}


void make_allocable (unsigned long block)
{
    reiserfs_bitmap_clear_bit (fsck_allocable_bitmap (fs), block);
}


unsigned long how_many_uninsertables_were_there (void)
{
    return stats (fs)->uninsertable_leaves;
}


unsigned long how_many_items_were_saved (void)
{
    return stats (fs)->saved_on_pass1;
}


int pass_0 (reiserfs_filsys_t fs)
{
    int i;

    if (fsck_mode (fs) == FSCK_ZERO_FILES) {
	reiserfs_bitmap_fill (fsck_new_bitmap (fs));
	fsck_progress ("Zeroing existing files - all blocks marked used\n");
    }

    make_aux_bitmaps (fs);

    i = should_skip_pass0 (fs);
    if (i) {
	load_pass0_result (fs);
    } else {
	fsck_progress ("Pass 0 - ");fflush (stdout);
	go_through (fs);
	fsck_progress (" - done\n");
	save_pass0_result (fs);
    }

    /* find blocks which can be allocated */
    check_bitmaps (fs);

    if (fsck_mode (fs) == FSCK_ZERO_FILES) {
	/* flush bitmaps and exit */
	fsck_progress ("Zeroing existing files - zeroed %lu blocks\n", tmp_zeroed);
	reiserfs_flush_bitmap (fsck_new_bitmap (fs), fs);
	reiserfs_close (fs);
	exit (0);
    }


    stage_report (0, fs);
    return 0;
}


#if 0

/* node looks like a leaf (block head and ih_item_length & ih_location
   of item_head array are correct. This function recovers (tries to) node's key
   table and directory items. */
static void recover_leaf (reiserfs_filsys_t fs, struct buffer_head * bh)
{
    int i;
    struct item_head * ih;

    /* mark transparently corrupted items - bad */
    ih = B_N_PITEM_HEAD (bh, 0);
    for (i = 0; i < node_item_number (bh); i ++, ih ++) {
	if (type_unknown (&ih->ih_key) ||
	    ih->ih_key.k_dir_id == 0 ||
	    ih->ih_key.k_objectid) {
	    mark_ih_bad (ih);
	    continue;
	}
    }
}

#endif

