/* 
 * Copyright 2000 by Hans Reiser, licensing governed by reiserfs/README
 */
  
/*
 * 2000/10/26 - Initial version.
 */

#include <assert.h>
#include "includes.h"


/* create clean bitmap */
reiserfs_bitmap_t reiserfs_create_bitmap (unsigned int bit_count)
{
    reiserfs_bitmap_t bm;

    bm = getmem (sizeof (*bm));
    if (!bm)
	return 0;
    bm->bm_bit_size = bit_count;
    bm->bm_byte_size = (bit_count + 7) / 8;
    bm->bm_set_bits = 0;
    bm->bm_map = getmem (bm->bm_byte_size);
    if (!bm->bm_map) {
	freemem (bm);
	return 0;
    }
    return bm;
}

/* Expand existing bitmap.  Return non-zero if can't. FIXME: it is
   assumed that bit_count is new number of blocks to be addressed */
int reiserfs_expand_bitmap (reiserfs_bitmap_t bm, unsigned int bit_count)
{
    unsigned int byte_count = ((bit_count + 7) / 8);
    char * new_map;

    new_map = expandmem (bm->bm_map, bm->bm_byte_size,
			 byte_count - bm->bm_byte_size);

    if (!new_map) {
	return 1;
    }
    
    bm->bm_map = new_map;
    bm->bm_byte_size = byte_count;
    bm->bm_bit_size = bit_count;
    return 0;
}

/* bitmap destructor */
void reiserfs_delete_bitmap (reiserfs_bitmap_t bm)
{
    freemem(bm->bm_map);
    bm->bm_map = NULL;		/* to not reuse bitmap handle */
    bm->bm_bit_size = 0;
    bm->bm_byte_size = 0;
    freemem(bm);
}


void reiserfs_bitmap_copy (reiserfs_bitmap_t to, reiserfs_bitmap_t from)
{
    if (to->bm_byte_size != from->bm_byte_size)
	die ("reiserfs_bitmap_copy: bitmap sizes mismatch");
    memcpy (to->bm_map, from->bm_map, from->bm_byte_size);
    to->bm_bit_size = from->bm_bit_size;
    to->bm_set_bits = from->bm_set_bits;
}


int reiserfs_bitmap_compare (reiserfs_bitmap_t bm1, reiserfs_bitmap_t bm2)
{
    if (bm1->bm_byte_size != bm2->bm_byte_size)
	die ("reiserfs_bitmap_compare: bitmap sizes mismatch");
    return memcmp (bm1->bm_map, bm2->bm_map, bm1->bm_byte_size);
}


void reiserfs_bitmap_set_bit (reiserfs_bitmap_t bm, unsigned int bit_number)
{
    assert(bit_number <= bm->bm_bit_size);
    set_bit(bit_number, bm->bm_map);
    bm->bm_set_bits ++;
}

void reiserfs_bitmap_clear_bit (reiserfs_bitmap_t bm, unsigned int bit_number)
{
    assert(bit_number <= bm->bm_bit_size);
    clear_bit(bit_number, bm->bm_map);
    bm->bm_set_bits --;
}


int reiserfs_bitmap_test_bit (reiserfs_bitmap_t bm, unsigned int bit_number)
{
    assert(bit_number <= bm->bm_bit_size);
    return test_bit(bit_number, bm->bm_map);
}


int reiserfs_bitmap_zeros (reiserfs_bitmap_t bm)
{
    return bm->bm_bit_size - bm->bm_set_bits;
}


int reiserfs_bitmap_ones (reiserfs_bitmap_t bm)
{
    return bm->bm_set_bits;
}


int reiserfs_bitmap_find_zero_bit (reiserfs_bitmap_t bm, unsigned long * start)
{
    unsigned int  bit_nr = *start;
    assert(*start < bm->bm_bit_size);

    bit_nr = find_next_zero_bit(bm->bm_map, bm->bm_bit_size, *start);

    if (bit_nr >= bm->bm_bit_size) { /* search failed */	
	return 1;
    }

    *start = bit_nr;
    return 0;
}

/* copy reiserfs filesystem bitmap into memory bitmap */
int reiserfs_fetch_disk_bitmap (reiserfs_bitmap_t bm, reiserfs_filsys_t fs)
{
    int i;
    int bytes;
    char * p;

    bytes = fs->s_blocksize;
    p = bm->bm_map;
    for (i = 0; i < SB_BMAP_NR (fs); i ++) {
	if (i == (SB_BMAP_NR (fs) - 1))
	    bytes = bm->bm_byte_size % fs->s_blocksize;

	memcpy (p, SB_AP_BITMAP (fs)[i]->b_data, bytes);
	p += bytes;
    }
    bm->bm_set_bits = 0;
    /* FIXME: optimize that */
    for (i = 0; i < bm->bm_bit_size; i ++)
	if (reiserfs_bitmap_test_bit (bm, i))
	    bm->bm_set_bits ++;

    /* unused part of last bitmap block is filled with 0s */
    for (i = SB_BLOCK_COUNT (fs) % (fs->s_blocksize * 8); i < fs->s_blocksize * 8; i ++)
	if (!test_bit (i, SB_AP_BITMAP (fs)[SB_BMAP_NR (fs) - 1]->b_data)) {
	    reiserfs_warning ("fetch_bitmap: on-disk bitmap is not filled properly\n");
	    break;
	}

    return 0;
}

/* copy bitmap to buffers which hold on-disk bitmap */
int reiserfs_flush_bitmap (reiserfs_bitmap_t bm, reiserfs_filsys_t fs)
{
    int i;
    int bytes;
    char * p;

    bytes = fs->s_blocksize;
    p = bm->bm_map;
    for (i = 0; i < SB_BMAP_NR (fs); i ++) {
	if (i == (SB_BMAP_NR (fs) - 1))
	    bytes = bm->bm_byte_size % fs->s_blocksize;

	memcpy (SB_AP_BITMAP (fs)[i]->b_data, p, bytes);
	mark_buffer_dirty (SB_AP_BITMAP (fs)[i]);
	
	p += bytes;
    }
    /* unused part of last bitmap block is filled with 0s */
    for (i = SB_BLOCK_COUNT (fs) % (fs->s_blocksize * 8); i < fs->s_blocksize * 8; i ++)
	set_bit (i, SB_AP_BITMAP (fs)[SB_BMAP_NR (fs) - 1]->b_data);
}


void reiserfs_bitmap_zero (reiserfs_bitmap_t bm)
{
    memset (bm->bm_map, 0, bm->bm_byte_size);
    bm->bm_set_bits = 0;
}


void reiserfs_bitmap_fill (reiserfs_bitmap_t bm)
{
    memset (bm->bm_map, 0xff, bm->bm_byte_size);
    bm->bm_set_bits = bm->bm_bit_size;
}


