/*
 *  Copyright 2000 by Hans Reiser, licensing governed by reiserfs/README
 */

#ifndef REISERFS_LIB_H
#define REISERFS_LIB_H

typedef struct super_block * reiserfs_filsys_t;

#include "reiserfs_fs.h"

struct _bitmap {
    unsigned long bm_byte_size;
    unsigned long bm_bit_size;
    char * bm_map;
    unsigned long bm_set_bits;
};

typedef struct _bitmap * reiserfs_bitmap_t;

struct super_block {
    int s_version;		/* on-disk format version */
    reiserfs_bitmap_t bitmap;	/* copy of reiserfs on-disk bitmap */
    
    int s_dev; /* descriptor of opened block device file */
    int s_blocksize;
    struct buffer_head ** s_ap_bitmap; /* array of buffers containing bitmap
                                          blocks */
    struct buffer_head * s_sbh;  /* buffer containing super block */
    struct reiserfs_super_block * s_rs; /* pointer to its b_data */
    int s_dirt;
    hashf_t s_hash_function; /* pointer to function which is used to sort
				names in directory. It is set by reiserfs_open
				if it is set in the super block, otherwise it
				is set by first is_properly_hashed */
    char * file_name;	/* file name of underlying device */
    int s_flags;
    void * s_vp;
    int (*block_allocator) (reiserfs_filsys_t fs, 
			    unsigned long * free_blocknrs,
			    unsigned long start, int amount_needed);
    int (*block_deallocator) (reiserfs_filsys_t fs, unsigned long block);
};


/* reiserfslib.c */

reiserfs_filsys_t reiserfs_open (char * filename, int flags, int * error, void * vp);
int no_reiserfs_found (reiserfs_filsys_t fs);
void reiserfs_reopen (reiserfs_filsys_t fs, int flags);
void reiserfs_flush (reiserfs_filsys_t fs);
void reiserfs_free (reiserfs_filsys_t fs);
void reiserfs_close (reiserfs_filsys_t fs);
int reiserfs_new_blocknrs (reiserfs_filsys_t fs, 
			   unsigned long * free_blocknrs, unsigned long start,
			   int amount_needed);
int reiserfs_free_block (reiserfs_filsys_t fs, unsigned long block);


/* bitmap.c */

reiserfs_bitmap_t reiserfs_create_bitmap (unsigned int bit_count);
int reiserfs_expand_bitmap (reiserfs_bitmap_t bm, unsigned int bit_count);
void reiserfs_delete_bitmap (reiserfs_bitmap_t bm);
void reiserfs_bitmap_copy (reiserfs_bitmap_t to, reiserfs_bitmap_t from);
int reiserfs_bitmap_compare (reiserfs_bitmap_t bm1, reiserfs_bitmap_t bm2);
void reiserfs_bitmap_set_bit (reiserfs_bitmap_t bm, unsigned int bit_number);
void reiserfs_bitmap_clear_bit (reiserfs_bitmap_t bm, unsigned int bit_number);

int reiserfs_bitmap_test_bit (reiserfs_bitmap_t bm, unsigned int bit_number);
int reiserfs_bitmap_find_zero_bit (reiserfs_bitmap_t bm, unsigned long * start);
int reiserfs_fetch_disk_bitmap (reiserfs_bitmap_t bm, reiserfs_filsys_t fs);
void reiserfs_bitmap_zero (reiserfs_bitmap_t bm);
void reiserfs_bitmap_fill (reiserfs_bitmap_t bm);
int reiserfs_bitmap_ones (reiserfs_bitmap_t bm);
int reiserfs_bitmap_zeros (reiserfs_bitmap_t bm);


/* node_formats.c */

#define THE_LEAF 1
#define THE_INTERNAL 2
#define THE_SUPER 3
#define THE_JDESC 4
#define THE_UNKNOWN 5

/* node_format.c */
int who_is_this (char * buf, int blocksize);
int journal_size (struct super_block * s);
int not_data_block (struct super_block * s, unsigned long block);
int not_journalable (reiserfs_filsys_t fs, unsigned long block);
int block_of_bitmap (reiserfs_filsys_t fs, unsigned long block);
int block_of_journal (reiserfs_filsys_t fs, unsigned long block);



typedef int (*check_unfm_func_t) (reiserfs_filsys_t fs, __u32);
int is_it_bad_item (reiserfs_filsys_t, struct item_head *, char *,
		    check_unfm_func_t, int bad_dir);

#define hash_func_is_unknown(fs) ((fs)->s_hash_function == 0)
#define reiserfs_hash(fs) ((fs)->s_hash_function)
int hash_code (reiserfs_filsys_t);

/* prints.c */
#define reiserfs_panic(fmt, list...) \
{\
	reiserfs_warning2 (stderr, fmt, ## list);\
	exit(4);\
}

#endif /* REISERFS_LIB_H */

