/*
 * Copyright 1996-1999 Hans Reiser
 */
#include "fsck.h"
#include <time.h>
/*#include <math.h>*/


struct path_key
{
    struct short_key
    {
        __u32 k_dir_id;
        __u32 k_objectid;
    } key;
    struct path_key * next, * prev;
};

struct path_key * head_key = NULL;
struct path_key * tail_key = NULL;

void check_path_key(struct key * key)
{
    struct path_key * cur = head_key;

    while(cur != NULL)
    {
        if (!comp_short_keys(&cur->key, key))
            die("\nsemantic check: loop found %k", key);
        cur = cur->next;
    }
}

void add_path_key(struct key * key)
{
    check_path_key(key);

    if (tail_key == NULL)
    {
        tail_key = getmem(sizeof(struct path_key));
        head_key = tail_key;
        tail_key->prev = NULL;
    }else{
        tail_key->next = getmem(sizeof(struct path_key));
        tail_key->next->prev = tail_key;
        tail_key = tail_key->next;
    }
    copy_short_key (&tail_key->key, key);
    tail_key->next = NULL;
}

void del_path_key()
{
    if (tail_key == NULL)
        die("wrong path_key structure");

    if (tail_key->prev == NULL)
    {
        freemem(tail_key);
        tail_key = head_key = NULL;
    }else{
        tail_key = tail_key->prev;
        freemem(tail_key->next);
        tail_key->next = NULL;
    }
}

static void print (char * dir_name, __u32 len)
{
    int i;

    printf("/");
    for (i = 0; i<len; i++, dir_name++)
        printf ("%c", *dir_name);
    fflush (stdout);
}

static void erase_print (__u32 len)
{
    int i;

    for (i = 0; i<=len; i++)
        printf("\b");
    for (i = 0; i<=len; i++)
        printf(" ");
    for (i = 0; i<=len; i++)
        printf("\b");
    fflush (stdout);
}


/* *size is "real" file size, sd_size - size from stat data */
static int wrong_st_size (struct key * key, int blocksize, __u64 * size, __u64 sd_size)
{
    if (sd_size >= *size) {
	/* size in stat data can be bigger than size calculated by items */
	*size = sd_size;
	return 0;
    }

    if (!(*size % blocksize)) {
	/* last item is indirect */
	if (((sd_size & ~(blocksize - 1)) == (*size - blocksize)) && sd_size % blocksize) {
	    /* sd_size is somewhere in the last file block */
	    *size = sd_size;
	    return 0;
	}
    } else {
	/* last item is a direct one */
	if (!(*size % 8)) {
	    if (((sd_size & ~7) == (*size - 8)) && sd_size % 8) {
		/* sd_size */
		*size = sd_size;
		return 0;
	    }
	}
    }

    fsck_log ("file %k has wrong sd_size %Ld, has to be %Ld\n",
	      key, sd_size, *size);
    stat_fixed_size (fs);
    return 1;
}


/* sd_blocks is 32 bit only */
static int wrong_st_blocks (struct key * key, __u32 blocks, __u32 sd_blocks)
{
    if (blocks == sd_blocks)
	return 0;

    fsck_log ("file %k has wrong sd_blocks %d, has to be %d\n",
	      key, sd_blocks, blocks);
    return 1;
}


/* key is a key of last file item */
static int wrong_first_direct_byte (struct key * key, int blocksize, 
				    __u32 * first_direct_byte,
				    __u32 sd_first_direct_byte, __u32 size)
{
    if (!size || is_indirect_key (key)) {
	/* there is no direct item */
	*first_direct_byte = NO_BYTES_IN_DIRECT_ITEM;
	if (sd_first_direct_byte != NO_BYTES_IN_DIRECT_ITEM) {
	    return 1;
	}
	return 0;
    }

    /* there is direct item */
    *first_direct_byte = (get_offset (key) & ~(blocksize - 1)) + 1;
    if (*first_direct_byte != sd_first_direct_byte) {
	fsck_log ("file %k has wrong first direct byte %d, has to be %d\n",
		  key, sd_first_direct_byte, *first_direct_byte);
	return 1;
    }
    return 0;
}


/* path is path to stat data */
static void check_regular_file (struct path * path, void * sd)
{
    int mark_passed_items;
    struct key key, sd_key;
    __u64 size, min_size, saved_size;
    __u32 blocks, saved_blocks;
    struct buffer_head * bh = PATH_PLAST_BUFFER (path);/* contains stat data */
    struct item_head * ih = PATH_PITEM_HEAD (path);/* stat data item */
    int fix_sd;
    int symlnk = 0;


    if (ih_key_format (ih) == KEY_FORMAT_2)
    {
	struct stat_data * sd_v2 = sd;
	int fix_sd;
	
	if (sd_v2->sd_nlink == 0) {
	    /*    print_how_far (&stat_datas, get_event (STAT_DATA_ITEMS));*/
	    if (S_ISREG (sd_v2->sd_mode))
		stat_regular_file_found (fs);
	    else if (S_ISLNK (sd_v2->sd_mode))
	    {
		symlnk = 1;
		stat_symlink_found (fs);
	    }
	    else
		stat_other_found (fs);

	    saved_size = le64_to_cpu (sd_v2->sd_size);
	    saved_blocks = le32_to_cpu (sd_v2->sd_blocks);
	
	    if (fsck_mode (fs) != FSCK_CHECK)
	    {
                sd_v2->sd_nlink = cpu_to_le32 (1);
	        mark_item_reachable (ih, bh);
	        /*mark_objectid_as_used (&g_sb, ih->ih_key.k_objectid);*/
            }else{
                if (!is_objectid_used (fs, ih->ih_key.k_objectid))
                     reiserfs_panic (fs, "check_regular_file: unused objectid found %k\n", &ih->ih_key);
            }

	    /* ih's key is stat data key */
	    copy_key (&key, &(ih->ih_key));
	    copy_key (&sd_key, &key);

	    pathrelse (path);

	    if (are_file_items_correct (&key, KEY_FORMAT_2, &size, &min_size, &blocks,
	                mark_passed_items = ((fsck_mode (fs) == FSCK_CHECK)?0:1), symlnk, saved_size) != 1) {
		/* unpassed items will be deleted in pass 4 as they left unaccessed */
		stat_broken_file_found (fs);
	    }
	    /* are_file_items_correct could perform indirect_to_direct, bh could be changed */

	    fix_sd = 0;
	    fix_sd += wrong_st_size (&sd_key, fs->s_blocksize, &size, saved_size);
	    fix_sd += wrong_st_blocks (&sd_key, blocks, saved_blocks);
	    if (fix_sd && fsck_mode (fs) == FSCK_REBUILD) {
		/* find stat data and correct it */
		if (usearch_by_key (fs, &sd_key, path) != ITEM_FOUND)
		    die ("check_regular_file: stat data not found");

		sd_v2 = (struct stat_data *)B_I_PITEM (get_bh (path), get_ih (path));
		sd_v2->sd_size = cpu_to_le64 (size);
		sd_v2->sd_blocks = cpu_to_le32 (blocks);
		mark_buffer_dirty (get_bh (path));
		pathrelse (path);
	    }
	} else {
	    /* one more link found. FIXME: we do not check number of links in
               check mode */
	    if (!is_item_reachable (ih))
		die ("check_regular_file: new stat data item must be accessed already");
	    if (fsck_mode (fs) == FSCK_REBUILD) {
	      sd_v2->sd_nlink = cpu_to_le32 (le32_to_cpu (sd_v2->sd_nlink) + 1);
	      mark_buffer_dirty (bh);
	    }
	}
    } else {
	struct stat_data_v1 * sd_v1 = sd;
	__u32 first_direct_byte, sd_first_direct_byte;

	if (sd_v1->sd_nlink == 0) {
	    /*    print_how_far (&stat_datas, get_event (STAT_DATA_ITEMS));*/
	    if ((sd_v1->sd_mode & S_IFMT) == S_IFREG)
		stat_regular_file_found(fs);
	    else if ((sd_v1->sd_mode & S_IFMT) == S_IFLNK)
	    {
		stat_symlink_found(fs);
		symlnk = 1;
	    }
	    else
		stat_other_found(fs);

	    /* save fields which will be checked */
	    saved_size = le32_to_cpu (sd_v1->sd_size);
	    saved_blocks = le32_to_cpu (sd_v1->u.sd_blocks);
	    sd_first_direct_byte = le32_to_cpu (sd_v1->sd_first_direct_byte);

		
	    if (fsck_mode (fs) != FSCK_CHECK)
	    {
	        sd_v1->sd_nlink = cpu_to_le16 (1);
	        mark_item_reachable (ih, bh);
	        /*mark_objectid_as_used (&g_sb, ih->ih_key.k_objectid);*/
            }else{
                if (!is_objectid_used (fs, ih->ih_key.k_objectid))
                     reiserfs_panic(fs, "check_regular_file: unused objectid found %k\n", &ih->ih_key);
            }

	    /* ih's key is stat data key */
	    copy_key (&key, &(ih->ih_key));
	    copy_key (&sd_key, &key);

	    pathrelse (path);
	    if (are_file_items_correct (&key, ih_key_format (ih), &size, &min_size, &blocks,
	                        mark_passed_items = ((fsck_mode (fs) == FSCK_CHECK)?0:1), symlnk, sd_v1->sd_size) != 1) {
		/* unpassed items will be deleted in pass 4 as they left unaccessed */
		stat_broken_file_found(fs);
	    }
	    /* are_file_items_correct could perform indirect_to_direct, bh could be changed */

	    fix_sd = 0;
	    fix_sd += wrong_st_size (&sd_key, fs->s_blocksize, &size, saved_size);
	    fix_sd += wrong_st_blocks (&sd_key, blocks, saved_blocks);
	    fix_sd += wrong_first_direct_byte (&key, fs->s_blocksize,
					       &first_direct_byte, sd_first_direct_byte, size);
	    if (fix_sd && fsck_mode (fs) == FSCK_REBUILD) {
		if (usearch_by_key (fs, &sd_key, path) != ITEM_FOUND)
		    die ("check_regular_file: stat data not found");

		sd_v1 = (struct stat_data_v1 *)B_I_PITEM (get_bh (path), get_ih (path));
		sd_v1->sd_size = cpu_to_le32 (size);
		sd_v1->u.sd_blocks = cpu_to_le32 (blocks);
		sd_v1->sd_first_direct_byte = cpu_to_le32 (first_direct_byte);
		mark_buffer_dirty (get_bh (path));
	    }

	} else {
	    /* one more link found. FIXME: we do not check number of links in
               check mode */
	    if (!is_item_reachable (ih))
		die ("check_regular_file: old stat data item must be accessed already");
	    if (fsck_mode (fs) == FSCK_REBUILD) {
	      sd_v1->sd_nlink = cpu_to_le16 (le16_to_cpu (sd_v1->sd_nlink) + 1);
	      mark_buffer_dirty (bh);
	    }
	}
    }
}


static int is_rootdir_key (struct key * key)
{
    if (comp_keys (key, &root_directory_key))
	return 0;
    return 1;
}

static int is_rootdir_entry_key (struct key * key)
{
    if (comp_short_keys (key, &root_directory_key))
	return 0;
    return 1;
}


/* when root direcotry can not be found */
static void create_root_directory (struct path * path)
{
    struct item_head ih;
    void * sd;
    struct stat_data sd_v2;

    sd = &sd_v2;

    /* insert stat data item */
    copy_key (&ih.ih_key, &root_directory_key);
    set_free_space(&ih,MAX_US_INT);
    mark_item_unreachable (&ih);

    if (SB_VERSION(fs) == REISERFS_VERSION_2)
    {
	ih.ih_item_len = cpu_to_le16 (SD_SIZE);
	set_key_format (&ih, KEY_FORMAT_2);

	sd_v2.sd_mode = cpu_to_le16 (S_IFDIR + 0755);
	sd_v2.sd_nlink = 0;
	sd_v2.sd_uid = 0;
	sd_v2.sd_gid = 0;
	sd_v2.sd_size = cpu_to_le64 (EMPTY_DIR_SIZE);
	sd_v2.sd_atime = sd_v2.sd_ctime = sd_v2.sd_mtime = cpu_to_le32 (time (NULL));
	sd_v2.sd_blocks = 0;
	sd_v2.u.sd_rdev = 0;
    }
    else
    {
	struct stat_data_v1 * sd_v1;

	sd_v1 = sd;
	ih.ih_item_len = cpu_to_le16 (SD_V1_SIZE);
	set_key_format (&ih, KEY_FORMAT_1);

	sd_v1->sd_mode = cpu_to_le16 (S_IFDIR + 0755);
	sd_v1->sd_nlink = 0;
	sd_v1->sd_uid = 0;
	sd_v1->sd_gid = 0;
	sd_v1->sd_size = cpu_to_le32 (EMPTY_DIR_SIZE_V1);
	sd_v1->sd_atime = sd_v1->sd_ctime = sd_v1->sd_mtime = cpu_to_le32 (time (NULL));
	sd_v1->u.sd_blocks = 0;
	sd_v1->u.sd_rdev = 0;
	sd_v1->sd_first_direct_byte = MAX_UL_INT;

    }
    reiserfsck_insert_item (path, &ih, sd);
}


static void paste_dot_and_dot_dot (struct path * path)
{
    char dir[EMPTY_DIR_SIZE];
    struct reiserfs_de_head * deh;
    struct key key;
  
    copy_key (&key, &root_directory_key);

    deh = (struct reiserfs_de_head *)dir;
    deh[0].deh_offset = DOT_OFFSET;
    deh[0].deh_dir_id = root_directory_key.k_dir_id;
    deh[0].deh_objectid = root_directory_key.k_objectid;
    deh[0].deh_state = 0;
    set_bit (DEH_Visible, &(deh[0].deh_state));
    dir[DEH_SIZE] = '.';

    if (SB_VERSION(fs) == REISERFS_VERSION_2)
    {
      reiserfsck_paste_into_item (path, dir, DEH_SIZE + ROUND_UP (strlen (".")));
    }else{
      reiserfsck_paste_into_item (path, dir, DEH_SIZE + 1);
    }

    set_type_and_offset (KEY_FORMAT_1, &key, DOT_DOT_OFFSET, TYPE_DIRENTRY);
    //set_le_key_k_offset(ITEM_VERSION_1, &key, DOT_DOT_OFFSET);
    //set_le_key_k_type(ITEM_VERSION_1, &key, TYPE_DIRENTRY);

    if (usearch_by_entry_key (fs, &key, path) == POSITION_FOUND) {
	fsck_log ("paste_dot_and_dot_dot: \"..\" found\n");
	pathrelse (path);
	return;
    }
    deh[0].deh_offset = DOT_DOT_OFFSET;
    deh[0].deh_dir_id = 0;
    deh[0].deh_objectid = root_directory_key.k_dir_id;
    deh[0].deh_state = 0;
    set_bit (DEH_Visible, &(deh[0].deh_state));
    dir[DEH_SIZE] = '.';
    dir[DEH_SIZE + 1] = '.';

    if (SB_VERSION(fs) == REISERFS_VERSION_2)
    {
      reiserfsck_paste_into_item (path, dir, DEH_SIZE + ROUND_UP (strlen ("..")) );
    }else{
      reiserfsck_paste_into_item (path, dir, DEH_SIZE + 2);
    }
}


static void insert_dot_and_dot_dot (struct path * path)
{
    struct item_head ih;
    char dir[EMPTY_DIR_SIZE];
    struct reiserfs_de_head * deh;

    copy_key (&ih.ih_key, &root_directory_key);

    set_key_format (&ih, KEY_FORMAT_1);
    set_type_and_offset (KEY_FORMAT_1, &ih.ih_key, DOT_OFFSET, TYPE_DIRENTRY);
    
    deh = (struct reiserfs_de_head *)dir;

    if (SB_VERSION(fs) == REISERFS_VERSION_2)
    {
	ih.ih_item_len = cpu_to_le16 (EMPTY_DIR_SIZE);
	deh[0].deh_location = cpu_to_le16 (ih_item_len (&ih) - ROUND_UP (strlen (".")));
	deh[1].deh_location = cpu_to_le16 (deh_location (&deh[0]) - ROUND_UP (strlen ("..")));
    }
    else
    {
	ih.ih_item_len = cpu_to_le16 (EMPTY_DIR_SIZE_V1);
	deh[0].deh_location = cpu_to_le16 (ih_item_len (&ih) - 1/*strlen (".")*/);
	deh[1].deh_location = cpu_to_le16 (deh_location (&deh[0]) - 2/*strlen ("..")*/);
    }

    set_entry_count (&ih, 2);
    mark_item_unreachable (&ih);

    deh[0].deh_offset = cpu_to_le32 (DOT_OFFSET);
    deh[0].deh_dir_id = cpu_to_le32 (root_directory_key.k_dir_id);
    deh[0].deh_objectid = cpu_to_le32 (root_directory_key.k_objectid);
    deh[0].deh_state = 0;
    set_bit (DEH_Visible, &(deh[0].deh_state));

    deh[1].deh_offset = cpu_to_le32 (DOT_DOT_OFFSET);
    deh[1].deh_dir_id = 0;
    deh[1].deh_objectid = cpu_to_le32 (root_directory_key.k_dir_id);
    deh[1].deh_state = 0;
    set_bit (DEH_Visible, &(deh[1].deh_state));
    dir[DEH_SIZE * 2] = '.';
    dir[DEH_SIZE * 2 + 1] = '.';
    dir[DEH_SIZE * 2 + 2] = '.';

    reiserfsck_insert_item (path, &ih, dir);
}


/* returns buffer, containing found directory item.*/
char * get_next_directory_item (struct path * path, struct key * key,
				struct key * parent, struct item_head * ih)
{
    char * dir_item;
    struct key * rdkey;
    struct buffer_head * bh;
    struct reiserfs_de_head * deh;
    int i;
    int retval;
    int length;

    if ((retval = usearch_by_entry_key (fs, key, path)) != POSITION_FOUND) {
	if (get_offset (key) != DOT_OFFSET)
	    die ("get_next_directory_item: entry not found");

	/* first directory item not found */
	if (is_rootdir_entry_key (key)) {
	    /* add "." and ".." to the root directory */
            if (fsck_mode (fs) == FSCK_CHECK)
                reiserfs_panic(fs, "get_next_directory_item: root has no \".\" entry %k\n", key);

	    if (retval == POSITION_NOT_FOUND)
		paste_dot_and_dot_dot (path);
	    else if (retval == DIRECTORY_NOT_FOUND)
		insert_dot_and_dot_dot (path);
	    else
		die ("get_next_directory_item: invalid return value");
	    usearch_by_entry_key (fs, key, path);
	} else {
	    /* it is ok for directories but the root one that "." is not found */
	    pathrelse (path);
	    return 0;
	}
    }
    /* leaf containing directory item */
    bh = PATH_PLAST_BUFFER (path);

    memcpy (ih, PATH_PITEM_HEAD (path), IH_SIZE);

    /* make sure, that ".." exists as well */
    if (get_offset (key) == DOT_OFFSET) {
	if (ih_entry_count (ih) < 2) {
	    pathrelse (path);
	    return 0;
	}
	deh = B_I_DEH (bh, ih) + 1;
	if (name_length (ih, deh, 1) != 2 ||
	    name_in_entry (deh, 1)[0] != '.' || name_in_entry (deh, 1)[1] != '.') {
	    fsck_log ("get_next_directory_item: \"..\" not found in %H\n", ih);
	    pathrelse (path);
	    return 0;
	}
    }

    deh = B_I_DEH (bh, ih);

    /* mark hidden entries as visible, reset ".." correctly */
    for (i = 0; i < ih_entry_count (ih); i ++, deh ++) {
	int namelen;
	char * name;

	name = name_in_entry (deh, i);
	namelen = name_length (ih, deh, i);
	if (de_hidden (deh))
	{
            if (fsck_mode (fs) == FSCK_CHECK)
                reiserfs_panic(fs, "get_next_directory_item: item %k has hidden entry %d\n", key, i);
	
	    fsck_log ("\nget_next_directory_item: %k: hidden entry \'%s\'\n",
		      key, bad_name (name, namelen));

	    mark_de_visible (deh);
	    mark_buffer_dirty (bh);
	}
	
	if (deh->deh_offset == DOT_OFFSET)
	{
	    if (comp_short_keys (&(deh->deh_dir_id), key) &&
		deh->deh_objectid != REISERFS_ROOT_PARENT_OBJECTID)
            {
		fsck_log ("get_next_directory_item: wrong \".\" found %k\n", key);
                if (fsck_mode (fs) == FSCK_REBUILD) {
		    deh->deh_dir_id = key->k_dir_id;
		    deh->deh_objectid = key->k_objectid;
		    mark_buffer_dirty (bh);
		}
	    }
	}
	
	if (deh->deh_offset == DOT_DOT_OFFSET)
	{
	    /* set ".." so that it points to the correct parent directory */
	    if (comp_short_keys (&(deh->deh_dir_id), parent) &&
		deh->deh_objectid != REISERFS_ROOT_PARENT_OBJECTID)
            {
		fsck_log ("\nget_next_directory_item: %k: \"..\" pointed to [%lu %lu], "
			  "fixed to [%lu %lu]\n",
			  key, deh->deh_dir_id, deh->deh_objectid,
			  parent->k_dir_id, parent->k_objectid);
		if (fsck_mode (fs) == FSCK_REBUILD) {
		    deh->deh_dir_id = parent->k_dir_id;
		    deh->deh_objectid = parent->k_objectid;
		    mark_buffer_dirty (bh);
		}
	    }
	}
    }

    /* copy directory item to the temporary buffer */
    dir_item = getmem (ih_item_len (ih)); 
    memcpy (dir_item, B_I_PITEM (bh, ih), ih_item_len (ih));

    /* unmark entries marked DEH_Lost_Found */
    deh = B_I_DEH (bh, ih);
    for (i = 0; i < ih_entry_count (ih); i ++, deh ++) {
	if (de_lost_found (deh)) {
	    unmark_de_lost_found (deh);
	    mark_buffer_dirty (bh);
	}
    }    


    /* next item key */
    if (PATH_LAST_POSITION (path) == (B_NR_ITEMS (PATH_PLAST_BUFFER (path)) - 1) &&
	(rdkey = uget_rkey (path)))
	copy_key (key, rdkey);
    else {
	key->k_dir_id = 0;
	key->k_objectid = 0;
    }

    if (fsck_mode (fs) != FSCK_CHECK)
        mark_item_reachable (PATH_PITEM_HEAD (path), PATH_PLAST_BUFFER (path));
    return dir_item;
}


// get key of an object pointed by direntry and the key of the entry itself
static void get_object_key (struct reiserfs_de_head * deh, struct key * key, 
			    struct key * entry_key, struct item_head * ih)
{
    key->k_dir_id = deh->deh_dir_id;
    key->k_objectid = deh->deh_objectid;
    key->u.k_offset_v1.k_offset = SD_OFFSET;
    key->u.k_offset_v1.k_uniqueness = V1_SD_UNIQUENESS;

    entry_key->k_dir_id = ih->ih_key.k_dir_id;
    entry_key->k_objectid = ih->ih_key.k_objectid;
    entry_key->u.k_offset_v1.k_offset = deh->deh_offset;
    entry_key->u.k_offset_v1.k_uniqueness = DIRENTRY_UNIQUENESS;
}


static void reiserfsck_cut_entry (struct key * key)
{
    INITIALIZE_PATH (path);
    struct item_head * ih;

    if (usearch_by_entry_key (fs, key, &path) != POSITION_FOUND || get_offset (key) == DOT_OFFSET)
	die ("reiserfsck_cut_entry: entry not found");

    ih = get_ih (&path);
    if (ih_entry_count (ih) == 1)
	reiserfsck_delete_item (&path, 0);
    else {
	struct reiserfs_de_head * deh = B_I_DEH (get_bh (&path), ih) + path.pos_in_item;

	reiserfsck_cut_from_item (&path, -(DEH_SIZE + entry_length (ih, deh, path.pos_in_item)));
    }
}



/* check recursively the semantic tree. Returns 0 if entry points to
   good object, and -1 or -2 if this entry must be deleted (stat data
   not found or directory does have any items).  Hard links are not
   allowed, but if directory rename has been interrupted by the system
   crash, it is possible, that fsck will find two entries (not "..") 
   pointing to the same directory. In this case fsck keeps only the
   first one. */
#define OK 0
#define STAT_DATA_NOT_FOUND -1
#define DIRECTORY_HAS_NO_ITEMS -2

static unsigned long stat_datas = 0;


// FIXME: hash can be detected wrong when two hash functions give the
// same value on the first name which gets here
static void detect_check_and_set_hash (struct super_block * s, char * name, int namelen,
				       __u32 deh_hash_value)
{
    int hash_code;
    
    if ((namelen == 1 && !strncmp (name, ".", 1)) ||
	(namelen == 2 && !strncmp (name, "..", 2)))
	return;

    if (deh_hash_value == GET_HASH_VALUE(keyed_hash (name, namelen))){
	hash_code = TEA_HASH;
    } else if (deh_hash_value == GET_HASH_VALUE(yura_hash (name, namelen))) {
	hash_code = YURA_HASH;
    } else if (deh_hash_value == GET_HASH_VALUE(r5_hash (name, namelen))) {
	hash_code = R5_HASH;
    } else {
	hash_code = TEA_HASH; // to avoid compiler's warning
	die ("check_semantic_tree: unknown hash is used");
    }

    if (!s->s_hash_function) {
	// set hash function as it is not set yet
//	if (opt_fsck_mode == FSCK_SEMANTIC)
//	        die("detect_check_and_set_hash: hash is not set in superblock");
	
	s->s_hash_function = code2function (hash_code);
	set_hash (s->s_rs, hash_code);
    } else {
	// check whether other hashes were used
	if (hash_code != rs_hash (s->s_rs))
	    printf ("check_semantic_tree: \"%s\" hash is used. Should be \"%s\" only\n", hash_name (hash_code), 
		    hash_name (rs_hash (s->s_rs)));
    }
}

/* for directories st_blocks is number of 512 byte units which fit into dir size round up to blocksize */
#define dir_size2st_blocks(s,size) \
(((size + ((s)->s_blocksize - 1)) / (s)->s_blocksize) * ((s)->s_blocksize / 512))


/* this can be called to scan either whole filesystem tree or lost+found
   only. In later case - it has to not skip reading of a directory if its
   sd_nlink is not 0 already and proceed only new names created by lost+found pass (they are marked "DEH_Found") */
int check_semantic_tree (struct key * key, struct key * parent, int is_dot_dot, int lost_found)
{
    struct path path;
    void * sd;
    int version;
    __u32 nlink;
    __u32 entry_len = 0;

    if (!KEY_IS_STAT_DATA_KEY (key))
	die ("check_semantic_tree: key must be key of a stat data");

    /* look for stat data of an object */
    if (usearch_by_key (fs, key, &path) == ITEM_NOT_FOUND) {
	if (fsck_mode (fs) == FSCK_CHECK) {
	    pathrelse (&path);
	    return STAT_DATA_NOT_FOUND;
	}

	if (is_rootdir_key (key)) {
	    /* stat data of the root directory not found. Make it */
	    create_root_directory (&path);
	    usearch_by_key (fs, key, &path);
	} else {
	    pathrelse (&path);
	    return STAT_DATA_NOT_FOUND;
	}
    }

    /* stat data has been found */
    version = ih_key_format (get_ih (&path));
    sd = get_item(&path);

    if (version == KEY_FORMAT_2)
    {
	struct stat_data * sd_v2 = sd;

	if ( (sd_v2->sd_nlink == 0) && (fsck_mode (fs) != FSCK_CHECK))
	    print_how_far (&stat_datas, 0/*get_event (STAT_DATA_ITEMS)*/);
	
        if ((sd_v2->sd_nlink == 0) && (fsck_mode (fs) == FSCK_CHECK))
            fsck_log ("check_semantic_tree: sd_nlink is 0 of %k\n", &(get_ih(&path)->ih_key));
	
	if ( !S_ISDIR (sd_v2->sd_mode) )
	{
	    /* object is not a directory (regular, symlink, device file) */
	    check_regular_file (&path, sd);
	    pathrelse (&path);
	    return OK;
	}
	
#if 0
	/* stat data of a directory found. Check if it is /lost+found. We skip
           it during semantic pass and will go through it on "looking for lost
           files" */
	if (lost_found && parent->k_objectid == REISERFS_ROOT_OBJECTID) {
	    if (fsck_stage (fs) == STAGE_SEMANTIC) {
		pathrelse (&path);
		return OK;
	    }
	}
#endif

	if (fsck_mode (fs) != FSCK_CHECK)
	{
	    if (!lost_found) {
		/* we found one more link to a directory, adjust sd_nlink */
		nlink = le32_to_cpu (sd_v2->sd_nlink) + 1;
		sd_v2->sd_nlink = cpu_to_le32 (nlink +
					       ((key->k_objectid == REISERFS_ROOT_OBJECTID && nlink == 1) ? 1 : 0));
	    } else {
		/* we are going to scan /lost+found for new names only */
		nlink = 1;
	    }
	}
    }
    else
    {
	struct stat_data_v1 * sd_v1 = sd;
	
	if ((sd_v1->sd_nlink == 0) && (fsck_mode (fs) != FSCK_CHECK))
	    print_how_far (&stat_datas, 0/*get_event (STAT_DATA_ITEMS)*/);

        if ((sd_v1->sd_nlink == 0) && (fsck_mode (fs) == FSCK_CHECK))
            reiserfs_panic(fs, "check_semantic_tree: sd_nlink is 0 %k\n", &(get_ih(&path)->ih_key));
	
	if (!S_ISDIR (sd_v1->sd_mode))
	{
	    /* object is not a directory (regular, symlink, device file) */
	    check_regular_file (&path, sd);
	    pathrelse (&path);
	    return OK;
	}
#if 0
	/* stat data of a directory found. Check if it is /lost+found. We skip
           it during semantic pass and will go through it on "looking for lost
           files" */
	if (lost_found && parent->k_objectid == REISERFS_ROOT_OBJECTID) {
	    if (fsck_stage (fs) == STAGE_SEMANTIC) {
		pathrelse (&path);
		return OK;
	    }
	}
#endif
	if (fsck_mode (fs) != FSCK_CHECK)
	{
	    if (!lost_found) {
		/* we found one more link to a directory, adjust sd_nlink */
		nlink = le16_to_cpu (sd_v1->sd_nlink) + 1;
		sd_v1->sd_nlink = cpu_to_le16 (nlink +
					       ((key->k_objectid == REISERFS_ROOT_OBJECTID && nlink == 1) ? 1 : 0));
	    } else {
		/* we are going to scan /lost+found for new names only */
		nlink = 1;
	    }
        }
    }

    if (fsck_mode (fs) != FSCK_CHECK)
        mark_buffer_dirty (PATH_PLAST_BUFFER (&path));

    /* object is directory */
    if (nlink == 1 || fsck_mode (fs) == FSCK_CHECK || lost_found) {
	char * dir_item;
	struct item_head ih;
	struct key item_key, entry_key, object_key;
	__u64 dir_size = 0;
        __u32 blocks;

	/*print_how_far (&stat_datas, get_event (STAT_DATA_ITEMS));*/

	stat_directory_found (fs);
	copy_key (&item_key, key);
	item_key.u.k_offset_v1.k_offset = DOT_OFFSET;
	item_key.u.k_offset_v1.k_uniqueness = DIRENTRY_UNIQUENESS;
	pathrelse (&path);
	while ((dir_item = get_next_directory_item (&path, &item_key, parent, &ih)) != 0) {
	    /* dir_item is copy of the item in separately allocated memory */
	    int i;
	    int retval;
	    struct reiserfs_de_head * deh = (struct reiserfs_de_head *)dir_item + path.pos_in_item;

/*&&&&&&&&&&&&&&&*/
	    if (dir_size == 0) {
		if (deh->deh_offset != DOT_OFFSET || (deh + 1)->deh_offset != DOT_DOT_OFFSET)
		    die ("check_semantic_tree: Directory without \".\" or \"..\"");
	    }
/*&&&&&&&&&&&&&&&*/

	    for (i = path.pos_in_item; i < ih_entry_count (&ih); i ++, deh ++) {
		char * name;
		int namelen;

		name = name_in_entry (deh, i);
		namelen = name_length (&ih, deh, i);
		detect_check_and_set_hash (fs, name, namelen,
					   GET_HASH_VALUE(deh_offset (deh)));

		get_object_key (deh, &object_key, &entry_key, &ih);
		if (fsck_mode (fs) != FSCK_CHECK) {
		    if (!lost_found || de_lost_found (deh)) {
			/* check entry if we are not in lost+found pass or
                           this name was added on lost+found pass */
			retval = check_semantic_tree (&object_key, key,
						      (deh->deh_offset == DOT_OFFSET ||
						       deh->deh_offset == DOT_DOT_OFFSET) ? 1 : 0,
						      0);
		    } else {
			/* this is old entry in /lost+found directory */
			retval = OK;
		    }
		} else {
		    if (deh->deh_offset != DOT_OFFSET && deh->deh_offset != DOT_DOT_OFFSET)
		    {
		        add_path_key(&object_key);
	                print(name, namelen);
                        retval = check_semantic_tree (&object_key, key, 0, 0/* do not skip "lost+found" */);
                        erase_print(namelen);
		        del_path_key();
                    }else
                        retval = OK;
		}

		if (retval != OK)
		{
		    if (fsck_mode (fs) == FSCK_CHECK)
                        fsck_log ("\ncheck_semantic_tree: name \"%s\" in directorty %k points to nowhere\n",
				  bad_name (name, namelen), &ih.ih_key);
		    else {
			if (get_offset (&entry_key) == DOT_DOT_OFFSET && object_key.k_objectid == REISERFS_ROOT_PARENT_OBJECTID) {
			    /* ".." of root directory can not be found */
			    if (retval != STAT_DATA_NOT_FOUND)
				die ("check_semantic_tree: stat data of parent directory of root directory found");
			    dir_size += DEH_SIZE + ((version == KEY_FORMAT_2) ? ROUND_UP (strlen ("..")) : strlen (".."));
			    continue;
			}
			stat_entry_deleted(fs);
			reiserfsck_cut_entry (&entry_key);
		    }
		} else {
		    /* OK */
                    dir_size += DEH_SIZE + entry_length (&ih, deh, i);
		}
	    }

	    freemem (dir_item);

	    if (not_of_one_file (&item_key, key)) {
		pathrelse (&path);
		break;
	    }
	    pathrelse (&path);
	}

	if (dir_size == 0)
	    return DIRECTORY_HAS_NO_ITEMS;

	if (usearch_by_key (fs, key, &path) != ITEM_FOUND)
	    die ("check_semantic_tree: stat data not found");


	if (fsck_mode (fs) == FSCK_CHECK)
	{
	    if (!is_objectid_used(fs, PATH_PITEM_HEAD (&path)->ih_key.k_objectid))
		fsck_log ("check_semantic_tree: objectid %d is unused\n", PATH_PITEM_HEAD (&path)->ih_key.k_objectid);
	}else
            /*mark_objectid_as_used (&g_sb, PATH_PITEM_HEAD (&path)->ih_key.k_objectid)*/;

        sd = get_item(&path);
        /*blocks = ceill((double)dir_size / 512);*/
	blocks = dir_size2st_blocks (fs, dir_size);

        if (version == KEY_FORMAT_2)
        {
	    struct stat_data * sd_v2 = sd;

	    if (dir_size != le64_to_cpu (sd_v2->sd_size))
	    {
	        if (fsck_mode (fs) == FSCK_CHECK)
	            fsck_log ("\ncheck_semantic_tree: new dir %k has st_size mismatch (calculated %Ld, in stat data %Ld)\n",
			      &(get_ih(&path)->ih_key), dir_size, le64_to_cpu (sd_v2->sd_size));
	
		/*		stat_fixed_size(&g_sb);*/
		sd_v2->sd_size = cpu_to_le64 (dir_size);
	    }
	    if (blocks != le32_to_cpu (sd_v2->sd_blocks))
	    {
	        if (fsck_mode (fs) == FSCK_CHECK)
	            fsck_log ("\ncheck_semantic_tree: new file %k has st_block mismatch "
			      "(found %d, in stat data %d)\n", &(get_ih(&path)->ih_key),
			      blocks, le32_to_cpu (sd_v2->sd_blocks));
	
	        sd_v2->sd_blocks = cpu_to_le32(blocks);
	    }
        }
	else
	{
	    struct stat_data_v1 * sd_v1 = sd;

	    if (dir_size != le32_to_cpu (sd_v1->sd_size))
	    {
	        if (fsck_mode (fs) == FSCK_CHECK)
	            fsck_log ("\ncheck_semantic_tree: old dir %k has st_size mismatch "
			      "(calculated %Ld, in stat data %d)\n",
			      &(get_ih(&path)->ih_key), dir_size, le32_to_cpu (sd_v1->sd_size));
		else
		    /*		stat_fixed_size(&g_sb);*/
		    sd_v1->sd_size = cpu_to_le32 (dir_size);
	    }
	    if (blocks != le32_to_cpu (sd_v1->u.sd_blocks))
	    {
	        if (fsck_mode (fs) == FSCK_CHECK)
	            fsck_log ("\ncheck_semantic_tree: old file %k has st_block mismatch "
			      "(found %d, in stat data %d)\n", &(get_ih(&path)->ih_key),
			      blocks, le32_to_cpu (sd_v1->u.sd_blocks));
		else
		    sd_v1->u.sd_blocks = cpu_to_le32(blocks);	
	    }
        }

	/* stat data of a directory is accessed */
	if (fsck_mode (fs) != FSCK_CHECK)
	    mark_item_reachable (PATH_PITEM_HEAD (&path), PATH_PLAST_BUFFER (&path));
    } else {
	/* we have accessed directory stat data not for the first time. we
	   can come here only from "." or "..". Other names must be removed
	   to avoid creation of hard links */
	if (fsck_mode (fs) == FSCK_CHECK)
            die ("check_semantic_tree: can not get here");
	
	if (!is_dot_dot) {
	    if (version == KEY_FORMAT_2)
	    {
		struct stat_data * sd_v2 = sd;

		nlink = le32_to_cpu (sd_v2->sd_nlink);
		sd_v2->sd_nlink = cpu_to_le32 (nlink - 1);
	    }
	    else
	    {
		struct stat_data_v1 * sd_v1 = sd;

		nlink = le16_to_cpu (sd_v1->sd_nlink);
		sd_v1->sd_nlink = cpu_to_le16 (nlink - 1);
	    }

	    fsck_log ("\ncheck_semantic_tree: more than one name "
		      "(neither \".\" nor \"..\") of a directory. Removed\n");
	    pathrelse (&path);
	    return STAT_DATA_NOT_FOUND;
	}
    }
    pathrelse (&path);


    return OK;
}

struct key root_directory_key = {REISERFS_ROOT_PARENT_OBJECTID, REISERFS_ROOT_OBJECTID, {{0, 0},}};
struct key parent_root_directory_key = {0, REISERFS_ROOT_PARENT_OBJECTID, {{0, 0},}};
struct key lost_found_dir_key = {REISERFS_ROOT_OBJECTID, 0, {{0, 0}, }};

static void zero_nlink (int old, void * sd)
{
    if (old) {
	struct stat_data_v1 * sd_v1 = sd;

	sd_v1->sd_nlink = 0;
    } else {
	struct stat_data * sd_v2 = sd;

	sd_v2->sd_nlink = 0;
    }
}

static int not_a_directory (void * sd)
{
    /* mode is at the same place and of the same size in both stat
       datas (v1 and v2) */
    struct stat_data_v1 * sd_v1 = sd;

    return !(S_ISDIR (le16_to_cpu (sd_v1->sd_mode)));
}


/* mkreiserfs should have created this */
static __u32 make_lost_found_dir (struct key * lost_found_key)
{
    int retval;
    INITIALIZE_PATH (path);
    struct stat_data sd;
    struct item_head ih;
    char empty [EMPTY_DIR_SIZE];
    __u32 lost_found; // objectid of directory "/lost+found"
    int key_format;


    if (SB_VERSION(fs) == REISERFS_VERSION_2)
	key_format = KEY_FORMAT_2;
    else 
	key_format = KEY_FORMAT_1;

    /* look for "lost+found" in the root directory */
    lost_found_key->k_objectid = find_entry (&root_directory_key, "lost+found");
    if (lost_found_key->k_objectid) {
	retval = usearch_by_key (fs, lost_found_key, &path);
	if (retval != ITEM_FOUND || not_a_directory (get_item (&path))) {
	    fsck_progress ("make_lost_found_dir: \"lost+found\" points to nothing or is not a directory\n");
	    lost_found_key->k_objectid = 0;
	}
	pathrelse (&path);
	return lost_found_key->k_objectid;
    }

    // key of the stat data
    lost_found_key->k_objectid = get_unused_objectid (fs);
    if (!lost_found_key->k_objectid)
	return 0;
    
    /* compose item to insert: ih and stat data body */
    make_dir_stat_data (key_format, lost_found_key->k_dir_id, 
			lost_found_key->k_objectid, &ih, (char *)&sd);
    zero_nlink (key_format == KEY_FORMAT_1, &sd);
    
    retval = usearch_by_key (fs, lost_found_key, &path);
    if (retval != ITEM_NOT_FOUND)
	return 0;
    
    mark_item_unreachable (&ih);
    reiserfsck_insert_item (&path, &ih, (char *)&sd);
    
    /* compose empty dir item and its item_head */
    set_offset (KEY_FORMAT_1, &ih.ih_key, DOT_OFFSET);
    set_type (KEY_FORMAT_1, &ih.ih_key, TYPE_DIRENTRY);
    ih.u.ih_entry_count = cpu_to_le16 (2);

    if (key_format == KEY_FORMAT_1) {
	ih.ih_item_len = cpu_to_le16 (EMPTY_DIR_SIZE_V1);
	make_empty_dir_item_v1 (empty, lost_found_key->k_dir_id, lost_found_key->k_objectid,
				REISERFS_ROOT_PARENT_OBJECTID, REISERFS_ROOT_OBJECTID);
    } else {
	ih.ih_item_len = cpu_to_le16 (EMPTY_DIR_SIZE);
	make_empty_dir_item (empty, lost_found_key->k_dir_id, lost_found_key->k_objectid,
			     REISERFS_ROOT_PARENT_OBJECTID, REISERFS_ROOT_OBJECTID);
    }

    // look for place to insert it
    set_offset (KEY_FORMAT_1, lost_found_key, DOT_OFFSET);
    set_type (KEY_FORMAT_1, lost_found_key, TYPE_DIRENTRY);
    retval = usearch_by_key (fs, lost_found_key, &path);
    if (retval != ITEM_NOT_FOUND)
	die ("make_lost_found_dir: can not create empty dir body of \'lost+found\'");

    mark_item_unreachable (&ih);
    reiserfsck_insert_item (&path, &ih, empty);

    // add entry to the root directory
    add_entry (&root_directory_key, "lost+found", lost_found_key, 0/*not lost_found name*/);

#if 0
    /* update root directory */
    if (usearch_by_key (fs, &root_key, &path) != ITEM_FOUND)
	die ("make_lost_found_dir: can not find root directory");
    if (ih_key_format (get_ih (&path)) == KEY_FORMAT_1) {
	struct stat_data_v1 * root_sd = get_item (&path);

	root_sd->sd_size = cpu_to_le32 (le32_to_cpu (root_sd->sd_size) + DEH_SIZE + strlen ("lost+found"));
	root_sd->sd_nlink = cpu_to_le16 (le16_to_cpu (root_sd->sd_nlink) + 1);
    } else {
	struct stat_data * root_sd = get_item (&path);

	root_sd->sd_size = cpu_to_le64 (le64_to_cpu (root_sd->sd_size) + DEH_SIZE + strlen ("lost+found"));
	root_sd->sd_nlink = cpu_to_le32 (le32_to_cpu (root_sd->sd_nlink) + 1);
    }
    // FIXME: sd_blocks left unchanged
    mark_buffer_dirty (get_bh (&path));
    pathrelse (&path);
#endif

    set_offset (KEY_FORMAT_1, lost_found_key, SD_OFFSET);
    set_type (KEY_FORMAT_1, lost_found_key, TYPE_STAT_DATA);

    return lost_found_key->k_objectid;
}


void pass_3_semantic (void)
{

//    init_objectid_list();

    fsck_progress ("Pass 3 (semantic) - ");
    fsck_log ("####### Pass 3 #########\n");

    /* create lost+found directory (if it is not there) */
    lost_found_dir_key.k_objectid = make_lost_found_dir (&lost_found_dir_key);

    if (check_semantic_tree (&root_directory_key, &parent_root_directory_key, 0, 0/*not lost+found*/) != OK)
        die ("check_semantic_tree: bad root found");

    stage_report (3, fs);

//    free_objectid_maps();
}


/* called when --check is given */
void semantic_check (void)
{
    fsck_progress ("Checking Semantic tree...\n");

//    init_objectid_list();

    if (check_semantic_tree (&root_directory_key, &parent_root_directory_key, 0, 0/*not lost+found yet*/) != OK)
        die ("check_semantic_tree: bad root found");

//    free_objectid_maps();

    fsck_progress ("Ok\n");

}
