/*
 * Copyright 1996-2000  Hans Reiser
 */
#include "fsck.h"
#include <getopt.h>
#include <sys/mount.h>

#include "../version.h"

//
// by default journal will be replayed via mount -o replay-only
// --replay-whole-journal
// --no-replay-journal - journal replaying will be skipped
// --scan-whole-partition - will scan whole parititon looking for the leaves
//


#define print_usage_and_exit() die ("Usage: %s [options] "\
" device\n"\
"\n"\
"Options:\n"\
"\t--check                  consistency checking (default)\n"\
"\t--rebuild-sb             super block checking and rebuilding if needed\n"\
"\t--rebuild-tree           force fsck to rebuild filesystem from scratch\n"\
"\t                         (takes a long time)\n"\
"\t--interactive, -i	    make fsck to stop after every stage\n"\
"\t-l|--logfile logfile     make fsck to complain to specifed file\n"\
"\t-a                       makes fsck to do nothing\n"\
"\t-p do nothing, exist for compatibility with fsck(8)\n"\
"\t-r\n", argv[0]);



/* fsck is called with one non-optional argument - file name of device
   containing reiserfs. This function parses other options, sets flags
   based on parsing and returns non-optional argument */
static char * parse_options (struct fsck_data * data, int argc, char * argv [])
{
    int c;
    static int mode = FSCK_CHECK;

    while (1) {
	static struct option options[] = {
	    /* options */
	    {"check", no_argument, &mode, FSCK_CHECK},
	    {"rebuild-sb", no_argument, &mode, FSCK_SB},
	    {"rebuild-tree", no_argument, &mode, FSCK_REBUILD},
/*
	    {"fast-rebuild", no_argument, &opt_fsck_mode, FSCK_FAST_REBUILD},
	    {"correct-bitmap", no_argument, &opt_fsck_mode, FSCK_CORRECT_BITMAP},
*/
	    /* options */
	    {"logfile", required_argument, 0, 'l'},
	    {"interactive", no_argument, 0, 'i'},

	    /* special option: will mark free blocks used, zero all
               unformatted node pointers and mark them free */
	    {"zero-files", no_argument, &mode, FSCK_ZERO_FILES},
	    {0, 0, 0, 0}
	};
	int option_index;
      
	c = getopt_long (argc, argv, "aprl:i", options, &option_index);
	if (c == -1)
	    break;

	switch (c) {
	case 0:
	    /* long option specifying fsck mode is found */
	    break;

	case 'i':
	    data->options |= OPT_INTERACTIVE;
	    break;

	case 'l':
	    data->log = fopen (optarg, "w");
	    if (!data->log)
		fprintf (stderr, "reiserfsck: could not open \'%s\'", optarg);

	    break;

	case 'p': /* these do nothing */
	case 'r':
	    break;

	case 'a':
	    mode = DO_NOTHING;
	    break;

	default:
	    print_usage_and_exit();
	}
    }

  if (optind != argc - 1)
    /* only one non-option argument is permitted */
    print_usage_and_exit();
  
  data->mode = mode;
  if (!data->log)
      data->log = stderr;

  return argv[optind];
}


reiserfs_filsys_t fs;



static void reset_super_block (struct super_block * s)
{
    __u32 * oids;
    struct reiserfs_super_block * rs;
    int super_block_size;

    rs = (struct reiserfs_super_block *)(SB_BUFFER_WITH_SB (s)->b_data);

    set_free_blocks (rs, SB_BLOCK_COUNT (s));
    set_root_block (rs, ~0);
    set_tree_height (rs, ~0);
    set_state (rs, REISERFS_ERROR_FS);


    if (is_reiser2fs_magic_string (rs)) {
	set_version (rs, REISERFS_VERSION_2);
	super_block_size = SB_SIZE;
    }
    if (is_reiserfs_magic_string (rs)) {
	set_version (rs, REISERFS_VERSION_1);
	super_block_size = SB_SIZE_V1;
    }

    /* can be not set yet. If so, hash function will be set when first dir
       entry will be found */
    s->s_hash_function = hashes[rs_hash (rs)];

    /* objectid map is not touched */

    mark_buffer_dirty (SB_BUFFER_WITH_SB (s));
    /* make file system invalid unless fsck done () */
    bwrite (SB_BUFFER_WITH_SB (s));

    
}

#if 0
static void update_super_block (void)
{
    struct reiserfs_super_block * rs;

    rs = (struct reiserfs_super_block *)(SB_BUFFER_WITH_SB (fs)->b_data);

    set_state (rs, REISERFS_VALID_FS);

    set_fsck_state (, 0);

/*    reset_journal (fs);*/

    mark_buffer_dirty (SB_BUFFER_WITH_SB (fs));
}
#endif

/*char ** g_disk_bitmap;
  char ** g_new_bitmap;*/
reiserfs_bitmap_t uninsertable_leaf_bitmap;
char ** g_formatted;
char ** g_unformatted;

int g_blocks_to_read;


/* on-disk bitmap is read, fetch it. create new bitmap, mark used
   blocks which are always used (skipped, super block, journal
   area, bitmaps), create other auxiliary bitmaps */
static void init_bitmaps (reiserfs_filsys_t fs)
{
    unsigned long i, j;

    fsck_disk_bitmap (fs) = reiserfs_create_bitmap (SB_BLOCK_COUNT (fs));
    reiserfs_fetch_disk_bitmap (fsck_disk_bitmap (fs), fs);


    /* find how many blocks will be read in pass 0 (used blocks not of not
       data area) */
    g_blocks_to_read = 0;
    for (i = 0; i < SB_BLOCK_COUNT (fs); i ++) {
	if (not_data_block (fs, i))
	    continue;
	if (reiserfs_bitmap_test_bit (fsck_disk_bitmap (fs), i))
	    g_blocks_to_read ++;
    }

    fsck_progress ("%lu blocks marked used in on disk bitmap\n",
		   g_blocks_to_read);

    fsck_new_bitmap (fs) = reiserfs_create_bitmap (SB_BLOCK_COUNT (fs));

    /* mark_block_used skips 0, ste the bit explicitly */
    reiserfs_bitmap_set_bit (fsck_new_bitmap (fs), 0);

    /* mark other skipped blocks and super block used */
    for (i = 1; i <= SB_BUFFER_WITH_SB (fs)->b_blocknr; i ++)
	mark_block_used (i);

    /* mark bitmap blocks as used */
    for (i = 0; i < SB_BMAP_NR (fs); i ++)
	mark_block_used (SB_AP_BITMAP (fs)[i]->b_blocknr);

    /* mark journal area as used */
    for (i = 0; i < JOURNAL_BLOCK_COUNT + 1; i ++)
	mark_block_used (i + SB_JOURNAL_BLOCK (fs));


    uninsertable_leaf_bitmap = reiserfs_create_bitmap (SB_BLOCK_COUNT (fs));
    reiserfs_bitmap_fill (uninsertable_leaf_bitmap);
    
    fsck_allocable_bitmap (fs) = reiserfs_create_bitmap (SB_BLOCK_COUNT (fs));
    reiserfs_bitmap_fill (fsck_allocable_bitmap (fs));


#if 0
    /* allocate space for bitmap of uninsertable leaves */
    g_uninsertable_leaf_bitmap = getmem (sizeof (char *) * SB_BMAP_NR (fs));
    for (i = 0; i < SB_BMAP_NR (fs); i ++) {
	g_uninsertable_leaf_bitmap[i] = getmem (fs->s_blocksize);
	memset (g_uninsertable_leaf_bitmap[i], 0xff, fs->s_blocksize);
    }

    /* bitmap of formatted nodes */
    g_formatted = getmem (sizeof (char *) * SB_BMAP_NR (s));
    for (i = 0; i < SB_BMAP_NR (s); i ++) {
	g_formatted[i] = getmem (s->s_blocksize);
	memset (g_formatted[i], 0, s->s_blocksize);
    }
    /* bitmap of unformatted nodes */
    g_unformatted = getmem (sizeof (char *) * SB_BMAP_NR (s));
    for (i = 0; i < SB_BMAP_NR (s); i ++) {
	g_unformatted[i] = getmem (s->s_blocksize);
	memset (g_unformatted[i], 0, s->s_blocksize);
    }
#endif
}

#if 0
static void release_aux_bitmaps (struct super_block * s)
{
    int i;

    for (i = 0; i < SB_BMAP_NR (s); i ++) {
	if (g_disk_bitmap)
	    /* when !g_disk_bitmap - g_new_bitmap[i] is mapped into
               SB_AP_BITMAP(s)[i]->b_data */
	    freemem (g_new_bitmap[i]);
	/* g_disk_bitmap[i] points to corresponding cautious bitmap's b_data */
	if (g_uninsertable_leaf_bitmap)
	    freemem (g_uninsertable_leaf_bitmap[i]);
    }
    if (g_disk_bitmap)
	freemem (g_disk_bitmap);
    freemem (g_new_bitmap);
    if (g_uninsertable_leaf_bitmap)
	freemem (g_uninsertable_leaf_bitmap);
}

static void release_super_block (void)
{
    bwrite (SB_BUFFER_WITH_SB (fs));
    freemem (SB_AP_BITMAP (fs));
    brelse (SB_BUFFER_WITH_SB (fs));

    freemem (g_old_rs);
}
#endif




#define REBUILD_WARNING \
"\nThis is an experimental version of reiserfsck, MAKE A BACKUP FIRST!\n\
Don't run this program unless something is broken. \n\
Some types of random FS damage can be recovered\n\
from by this program, which basically throws away the internal nodes\n\
of the tree and then reconstructs them.  This program is for use only\n\
by the desperate, and is of only beta quality.  Email\n\
reiserfs@devlinux.com with bug reports. \nWill rebuild the filesystem tree\n"

/* 
   warning #2
   you seem to be running this automatically.  you are almost
   certainly doing it by mistake as a result of some script that
   doesn't know what it does.  doing nothing, rerun without -p if you
   really intend to do this.  */

void warn_what_will_be_done (struct fsck_data * data)
{
    char * answer = 0;
    size_t n = 0;
    __u32 dir_id, objectid;

    /* warn about fsck mode */
    switch (data->mode) {
    case FSCK_CHECK:
	fsck_progress ("Will read-only check consistency of the partition\n");
	break;

    case FSCK_SB:
	fsck_progress ("Will check SB and rebuild if it is needed\n");
	break;

    case FSCK_REBUILD:
	fsck_progress (REBUILD_WARNING);
	break;
    case FSCK_ZERO_FILES:
	fsck_progress ("Will zero existing files and mark free blocks as used\n");
    }

    fsck_progress ("Will put log info to %s\n", (data->log != stderr) ?
		   "file you specified" : "stderr");
    if (data->options & OPT_INTERACTIVE)
	fsck_progress ("Will stop after every stage and ask for confirmation before continuing\n");

    if (!user_confirmed ("Do you want to run this program?[N/Yes] (note need to type Yes):", "Yes\n"))
	exit (0);
}


#if 0
void end_fsck (void)
{
    int i;

    update_super_block ();
    update_bitmap (fs);
    release_aux_bitmaps (fs);
    for (i = 0; i < SB_BMAP_NR (fs); i ++) {
	brelse (SB_AP_BITMAP (fs)[i]);
    }
    release_super_block ();

    fsck_progress ("Syncing.."); fflush (stdout);
    
    flush_buffers ();
    sync ();
    
    fsck_progress ("done\n"); fflush (stdout);

    fsck_progress ("Checking mem..");
	
    free_overwritten_unfms ();
    check_and_free_buffer_mem ();

    fsck_progress ("Ok\n");
    exit (0);
}
#endif



static void start_rebuild (reiserfs_filsys_t fs)
{
    reset_super_block (fs);
    init_bitmaps (fs);

    proper_id_map (fs) = init_id_map ();
    shared_id_map (fs) = init_id_map ();
}


/* called before semantic pass starts */
static void end_rebuilding (reiserfs_filsys_t fs)
{
    reiserfs_flush_bitmap (fsck_new_bitmap (fs), fs);
    flush_objectid_map (proper_id_map (fs), fs);
    set_fsck_state (fs->s_rs, TREE_IS_BUILT);
    set_free_blocks (fs->s_rs, reiserfs_bitmap_zeros (fsck_new_bitmap (fs)));
    mark_buffer_dirty (SB_BUFFER_WITH_SB (fs));
    
    /* write all dirty blocks */
    fsck_progress ("Syncing.."); fflush (stdout);
    reiserfs_flush (fs);
    fsck_progress ("done\n"); fflush (stdout);

    /* release what will not be needed */
    reiserfs_delete_bitmap (fsck_disk_bitmap (fs));
    reiserfs_delete_bitmap (fsck_allocable_bitmap (fs));

    /* FIXME: could be not a bitmap */
    reiserfs_delete_bitmap (uninsertable_leaf_bitmap);

    if (fsck_user_confirmed (fs, "Tree building completed. "
			     "You can stop now and restart from this point later "
			     "(this is probably not what you need). Do you want to stop? ",
			     "Yes\n", 0/*default*/)) {
	reiserfs_close (fs);
        exit (4);
    }
}


static int skip_rebuilding (reiserfs_filsys_t fs)
{
    if (fsck_state (fs->s_rs) == TREE_IS_BUILT) {
	if (fsck_user_confirmed (fs, "S+ tree of filesystem looks built. Skip rebuilding? ", "Yes\n", 0/*default*/)) {
	    
	    fsck_new_bitmap (fs) = reiserfs_create_bitmap (SB_BLOCK_COUNT (fs));
	    reiserfs_fetch_disk_bitmap (fsck_new_bitmap (fs), fs);

	    proper_id_map (fs) = init_id_map ();
	    fetch_objectid_map (proper_id_map (fs), fs);
	    return 1;
	}
    }
    return 0;
}


static void start_continuing (reiserfs_filsys_t fs)
{
    fsck_allocable_bitmap (fs) = reiserfs_create_bitmap (SB_BLOCK_COUNT (fs));
    reiserfs_bitmap_copy (fsck_allocable_bitmap (fs), fsck_new_bitmap (fs));
}


static void the_end (reiserfs_filsys_t fs)
{
    reiserfs_flush_bitmap (fsck_new_bitmap (fs), fs);
    flush_objectid_map (proper_id_map (fs), fs);
    set_fsck_state (fs->s_rs, 0);
    set_free_blocks (fs->s_rs, reiserfs_bitmap_zeros (fsck_new_bitmap (fs)));
    set_state (fs->s_rs, REISERFS_VALID_FS);
    mark_buffer_dirty (SB_BUFFER_WITH_SB (fs));

    /* write all dirty blocks */
    fsck_progress ("Syncing.."); fflush (stdout);
    reiserfs_flush (fs);
    fsck_progress ("done\n"); fflush (stdout);

    reiserfs_delete_bitmap (fsck_new_bitmap (fs));
}


static void rebuild_tree (reiserfs_filsys_t fs)
{
    if (is_mounted (fs->file_name)) {
	reiserfs_warning ("rebuild_tree: can not rebuild tree of mounted filesystem\n");
	return;
    }

    reiserfs_reopen (fs, O_RDWR);

    /* rebuild starts with journal replaying */
    reiserfs_replay_journal (fs);

    fsck_progress ("Rebuilding..\n");
	    
    if (!skip_rebuilding (fs)) {
	start_rebuild (fs);

	pass_0 (fs);
    
	/* passes 1 and 2. building of the tree */
	pass_1_pass_2_build_the_tree ();

	end_rebuilding (fs);
    }

    /* re-building of filesystem tree is now separated of sematic pass of the
       fsck */
    start_continuing (fs);

    /* 3. semantic pass */
    pass_3_semantic ();
    
    /* if --lost+found is set - link unaccessed directories to lost+found
       directory */
    look_for_lost (fs);
    
    /* 4. look for unaccessed items in the leaves */
    pass_4_check_unaccessed_items ();

    
    the_end (fs);


    free_id_map (&proper_id_map(fs));
    if (shared_id_map(fs))
	free_id_map (&shared_id_map(fs));


    /* update super block, bitmap, flush buffers */
    /*    end_fsck (fs);*/
}


static void zero_files (reiserfs_filsys_t fs)
{
    init_bitmaps (fs);
    reiserfs_reopen (fs, O_RDWR);
    pass_0 (fs);
}


/* check umounted or read-only mounted filesystems only */
static void check_fs (reiserfs_filsys_t fs)
{
    int i;

    if (!is_mounted (fs->file_name)) {
	/* filesystem is not mounted, replay journal before checking */
	reiserfs_reopen (fs, O_RDWR);

	reiserfs_replay_journal (fs);

	reiserfs_reopen (fs, O_RDONLY);
    } else {
	/* filesystem seems mounted. we do not check filesystems mounted with
           r/w permissions */
	if (!is_mounted_read_only (fs->file_name)) {
	    fsck_progress ("Device %s is mounted w/ write permissions, can not check it\n",
			   fs->file_name);
	    reiserfs_close (fs);
	    exit (0);
	}
	fsck_progress ("Filesystem seems mounted read-only. Skipping journal replay..\n");
    }

    fsck_disk_bitmap (fs) = reiserfs_create_bitmap (SB_BLOCK_COUNT (fs));
    reiserfs_fetch_disk_bitmap (fsck_disk_bitmap (fs), fs);

    check_fs_tree (fs);

    semantic_check ();

    reiserfs_delete_bitmap (fsck_disk_bitmap (fs));
    reiserfs_close (fs);
}


#include <sys/resource.h>

int main (int argc, char * argv [])
{
    char * file_name;
    struct rlimit rlim = {0xffffffff, 0xffffffff};
    struct fsck_data * data;

    data = getmem (sizeof (struct fsck_data));

#if 0
    /* this is only needed (and works) when running under 2.4 on regural files */
    if (setrlimit (RLIMIT_FSIZE, &rlim) == -1) {
	perror ("setrlimit failed");
	return 0;
    }
#endif 

    print_banner ("reiserfsck");

    file_name = parse_options (data, argc, argv);

    warn_what_will_be_done (data); /* and ask confirmation Yes */


    fs = reiserfs_open (file_name, O_RDONLY, 0, data);
    if (!fs)
	die ("reiserfsck: could not open filesystem on \"%s\"", file_name);

    if (fsck_mode (fs) == FSCK_SB) {
	reiserfs_reopen (fs, O_RDWR);
	rebuild_sb (fs);
	reiserfs_close (fs);
	return 0;
    }

    if (no_reiserfs_found (fs)) {
	fsck_progress ("reiserfsck: --rebuild-sb may restore reiserfs super block\n");
	reiserfs_close (fs);
	return 0;
    }


    fs->block_allocator = reiserfsck_reiserfs_new_blocknrs;
    fs->block_deallocator = reiserfsck_reiserfs_free_block;


    if (fsck_mode (fs) == DO_NOTHING) {
	/* this should work when fsck is called by fsck -a */
	fsck_progress ("%s: clean, %d/%d %ldk blocks\n", file_name,
		       SB_BLOCK_COUNT (fs) - SB_FREE_BLOCKS(fs),
		       SB_BLOCK_COUNT (fs), fs->s_blocksize / 1024);
	brelse (SB_BUFFER_WITH_SB (fs));
	return 0;
    }

    if (fsck_mode (fs) == FSCK_CHECK) {
	check_fs (fs);
	return 0;
    }

#ifdef FAST_REBUILD_READY /* and tested */
    if (opt_fsck_mode == FSCK_FAST_REBUILD) {
	__u32 root_block = SB_ROOT_BLOCK(fs);
	reopen_read_write (file_name);
	printf ("Replaying log..");
	reiserfs_replay_journal (fs);
	printf ("done\n");
	if (opt_fsck == 1)
	    printf ("ReiserFS : checking %s\n",file_name);
	else
	    printf ("Rebuilding..\n");

	
	reset_super_block (fs);
	SB_DISK_SUPER_BLOCK(fs)->s_root_block = cpu_to_le32 (root_block);
	init_bitmaps (fs);

	/* 1,2. building of the tree */
	recover_internal_tree(fs);

	/* 3. semantic pass */
	pass3_semantic ();

	/* if --lost+found is set - link unaccessed directories to
           lost+found directory */
       look_for_lost (fs);

	/* 4. look for unaccessed items in the leaves */
	pass4_check_unaccessed_items ();
	
	end_fsck ();
    }
#endif /* FAST REBUILD READY */


    if (fsck_mode (fs) == FSCK_ZERO_FILES)
	zero_files (fs);

    if (fsck_mode (fs) != FSCK_REBUILD)
	return 0;


    /* the --rebuild-tree is here */
    rebuild_tree (fs);
    return 0;

}
