/*
 *  Copyright 2000 by Hans Reiser, licensing governed by reiserfs/README
 */

#include "includes.h"


/* read super block and bitmaps. fixme: only 4k blocks, pre-journaled format
   is refused */
reiserfs_filsys_t reiserfs_open (char * filename, int flags, int *error, void * vp)
{
    reiserfs_filsys_t fs;
    struct buffer_head * bh;
    struct reiserfs_super_block * rs;
    int fd, i;
    unsigned long block;
    
    fd = open (filename, flags | O_LARGEFILE);
    if (fd == -1) {
	if (error)
	    *error = errno;
	return 0;
    }

    fs = getmem (sizeof (*fs));
    fs->s_dev = fd;
    asprintf (&fs->file_name, "%s", filename);

    /* reiserfs super block is either in 16-th or in 2-nd 4k block of the
       device */
    for (i = 16; i > 0; i -= 14) {
	bh = bread (fd, i, 4096);
	if (!bh) {
	    reiserfs_warning ("reiserfs_open: bread failed reading block %d\n", i);
	} else {
	    rs = (struct reiserfs_super_block *)bh->b_data;
	    
	    if (is_reiser2fs_magic_string (rs) || is_reiserfs_magic_string (rs))
		goto found;

	    /* reiserfs signature is not found at the i-th 4k block */
	    brelse (bh);
	}
    }

    reiserfs_warning ("reiserfs_open: neither new nor old reiserfs format "
		      "found on %s\n", filename);
    if (error)
	*error = 0;
    return fs;

 found:

    /* fixme: we could make some check to make sure that super block looks
       correctly */
    fs->s_version = is_reiser2fs_magic_string (rs) ? REISERFS_VERSION_2 :
	REISERFS_VERSION_1;
    fs->s_blocksize = rs_blocksize (rs);
    fs->s_hash_function = hashes [rs_hash (rs)];
    SB_BUFFER_WITH_SB (fs) = bh;
    fs->s_rs = rs;
    fs->s_flags = flags; /* O_RDONLY or O_RDWR */
    fs->s_vp = vp;

    /* read bitmaps */
    SB_AP_BITMAP (fs) = getmem (sizeof (void *) * rs_bmap_nr (rs));
    for (i = 0, block = bh->b_blocknr + 1; i < rs_bmap_nr (rs); i ++) {
	SB_AP_BITMAP (fs)[i] = bread (fd, block, fs->s_blocksize);
	if (!SB_AP_BITMAP (fs)[i]) {
	    reiserfs_warning ("reiserfs_open: bread failed reading bitmap #%d (%lu)\n", i, block);
	    SB_AP_BITMAP (fs)[i] = getblk (fd, block, fs->s_blocksize);
	    memset (SB_AP_BITMAP (fs)[i]->b_data, 0xff, fs->s_blocksize);
	    set_bit (BH_Uptodate, &SB_AP_BITMAP (fs)[i]->b_state);
	}
	block = (bh->b_blocknr == 16 ? ((i + 1) * fs->s_blocksize * 8) : (block + 1));
    }

    return fs;

}


int no_reiserfs_found (reiserfs_filsys_t fs)
{
    return (fs->s_blocksize == 0) ? 1 : 0;
}

void reiserfs_reopen (reiserfs_filsys_t fs, int flag)
{
    close (fs->s_dev);
    fs->s_dev = open (fs->file_name, flag | O_LARGEFILE);
    if (fs->s_dev == -1)
	die ("reiserfs_reopen: could not reopen device");
}


/* flush all changes made on a filesystem */
void reiserfs_flush (reiserfs_filsys_t fs)
{
    flush_buffers ();
}


/* free all memory involved into manipulating with filesystem */
void reiserfs_free (reiserfs_filsys_t fs)
{
    int i;

    /* release bitmaps */
    for (i = 0; i < SB_BMAP_NR (fs); i ++)
	brelse (SB_AP_BITMAP (fs) [i]);
    freemem (SB_AP_BITMAP (fs));

    /* release super block and memory used by filesystem handler */
    brelse (SB_BUFFER_WITH_SB (fs));

    free_buffers ();

    free (fs->file_name);
    freemem (fs);
}


void reiserfs_close (reiserfs_filsys_t fs)
{
    reiserfs_flush (fs);
    reiserfs_free (fs);
}


int reiserfs_new_blocknrs (reiserfs_filsys_t fs, 
			   unsigned long * free_blocknrs, unsigned long start, int amount_needed)
{
    if (fs->block_allocator)
	return fs->block_allocator (fs, free_blocknrs, start, amount_needed);
    die ("block allocator is not defined\n");
    return 0;
}


int reiserfs_free_block (reiserfs_filsys_t fs, unsigned long block)
{
    if (fs->block_deallocator)
	return fs->block_deallocator (fs, block);
    die ("block allocator is not defined\n");
    return 0;
}



static inline int _bin_search (void * key, void * base, int num, int width, __u32 *ppos)
{
    __u32 rbound, lbound, j;

    lbound = 0;
    rbound = num - 1;
    for (j = (rbound + lbound) / 2; lbound <= rbound; j = (rbound + lbound) / 2) {
	switch (comp_keys ((void *)((char *)base + j * width), key)) {
	case -1:/* second is greater */
	    lbound = j + 1;
	    continue;

	case 1: /* first is greater */
	    if (j == 0) {
                *ppos = lbound;
                return ITEM_NOT_FOUND;
	    }
	    rbound = j - 1;
	    continue;

	case 0:
	    *ppos = j;
	    return ITEM_FOUND;
	}
    }

    *ppos = lbound;
    return ITEM_NOT_FOUND;
}


static int _search_by_key (reiserfs_filsys_t fs, struct key * key, struct path * path)
{
    struct buffer_head * bh;
    unsigned long block = SB_ROOT_BLOCK (fs);
    struct path_element * curr;
    int retval;
    
    path->path_length = ILLEGAL_PATH_ELEMENT_OFFSET;
    while (1) {
	curr = PATH_OFFSET_PELEMENT (path, ++ path->path_length);
	bh = curr->pe_buffer = bread (fs->s_dev, block, fs->s_blocksize);
        if (bh == 0) {
	    path->path_length --;
	    pathrelse (path);
	    return ITEM_NOT_FOUND;
	}
	retval = _bin_search (key, B_N_PKEY (bh, 0), B_NR_ITEMS (bh),
			      is_leaf_node (bh) ? IH_SIZE : KEY_SIZE, &(curr->pe_position));
	if (retval == ITEM_FOUND) {
	    /* key found, return if this is leaf level */
	    if (is_leaf_node (bh)) {
		path->pos_in_item = 0;
		return ITEM_FOUND;
	    }
	    curr->pe_position ++;
	} else {
	    /* key not found in the node */
	    if (is_leaf_node (bh))
		return ITEM_NOT_FOUND;
	}
	block = B_N_CHILD_NUM (bh, curr->pe_position);
    }
    printf ("search_by_key: you can not get here\n");
    return ITEM_NOT_FOUND;
}


/* NOTE: this only should be used to look for keys who exists */
static int _search_by_entry_key (reiserfs_filsys_t fs, struct key * key, struct path * path)
{
    struct item_head * ih;
    struct reiserfs_de_head * deh;
    int i;
    
    if (_search_by_key (fs, key, path) == ITEM_FOUND) {
        path->pos_in_item = 0;
        return POSITION_FOUND;
    }
    if (PATH_LAST_POSITION (path) == 0) {
	pathrelse (path);
	return POSITION_NOT_FOUND;
    }

    /* take previous item */
    PATH_LAST_POSITION (path) --;
    ih = PATH_PITEM_HEAD (path);
    if (not_of_one_file (&(ih->ih_key), key) || !is_direntry_ih(ih)) {
	pathrelse (path);
	return POSITION_NOT_FOUND;
    }

    /* previous item is part of desired directory */
    deh = B_I_DEH (get_bh (path), ih);
    for (i = 0; i < ih_entry_count (ih); ih ++, deh ++, i ++) {
	if (deh->deh_offset == key->u.k_offset_v1.k_offset) {
	    path->pos_in_item = i;
	    return POSITION_FOUND;
	}
    }
    return POSITION_NOT_FOUND;
}

static void _init_tb_struct (struct tree_balance * tb, reiserfs_filsys_t fs,
			     struct path * path, int size)
{
    memset (tb, '\0', sizeof(struct tree_balance));
    tb->tb_sb = fs;
    tb->tb_path = path;

    PATH_OFFSET_PBUFFER(path, ILLEGAL_PATH_ELEMENT_OFFSET) = NULL;
    PATH_OFFSET_POSITION(path, ILLEGAL_PATH_ELEMENT_OFFSET) = 0;
    tb->insert_size[0] = size;
}

void mark_block_formatted (struct buffer_head * bh)
{

}
void unmark_block_formatted (struct buffer_head * bh)
{
}

int reiserfs_remove_entry (reiserfs_filsys_t fs, struct key * key)
{
    struct path path;
    struct tree_balance tb;
    struct item_head * ih;
    struct reiserfs_de_head * deh;

    if (_search_by_entry_key (fs, key, &path) != POSITION_FOUND) {
	pathrelse (&path);
	return 1;
    }

    ih = get_ih (&path);
    if (ih_entry_count (ih) == 1) {
	_init_tb_struct (&tb, fs, &path, -(IH_SIZE + ih_item_len (ih)));
	if (fix_nodes (/*tb.transaction_handle,*/ M_DELETE, &tb, 0) != CARRY_ON) {
	    unfix_nodes (/*tb.transaction_handle,*/ &tb);
	    return 1;
	}
	do_balance (/*tb.transaction_handle,*/ &tb, 0, 0, M_DELETE, 0);
	return 0;
    }

    deh = B_I_DEH (get_bh (&path), ih) + path.pos_in_item;
    _init_tb_struct (&tb, fs, &path, -(DEH_SIZE + entry_length (ih, deh, path.pos_in_item)));
    if (fix_nodes (/*tb.transaction_handle,*/ M_CUT, &tb, 0) != CARRY_ON) {
	unfix_nodes (/*tb.transaction_handle,*/ &tb);
	return 1;
    }
    do_balance (/*tb.transaction_handle,*/ &tb, 0, 0, M_CUT, 0);
    return 0;
}


