/*
 *  Copyright 2000 by Hans Reiser, licensing governed by reiserfs/README
 */


#include "includes.h"



/* this only checks that the node looks like a correct leaf. Item
   internals are not checked */
static int is_correct_leaf (char * buf, int blocksize)
{
    struct block_head * blkh;
    struct item_head * ih;
    int used_space;
    int prev_location;
    int i;
    int nr;

    blkh = (struct block_head *)buf;
    if (!is_leaf_block_head (buf))
	return 0;

    nr = le16_to_cpu (blkh->blk_nr_item);
    if (nr < 1 || nr > ((blocksize - BLKH_SIZE) / (IH_SIZE + MIN_ITEM_LEN)))
	/* item number is too big or too small */
	return 0;

    ih = (struct item_head *)(buf + BLKH_SIZE) + nr - 1;
    used_space = BLKH_SIZE + IH_SIZE * nr + (blocksize - ih_location (ih));
    if (used_space != blocksize - le16_to_cpu (blkh->blk_free_space))
	/* free space does not match to calculated amount of use space */
	return 0;

    // FIXME: it is_leaf will hit performance too much - we may have
    // return 1 here

    /* check tables of item heads */
    ih = (struct item_head *)(buf + BLKH_SIZE);
    prev_location = blocksize;
    for (i = 0; i < nr; i ++, ih ++) {
	if (ih_location (ih) >= blocksize || ih_location (ih) < IH_SIZE * nr)
	    return 0;
	if (ih_item_len (ih) < 1 || ih_item_len (ih) > MAX_ITEM_LEN (blocksize))
	    return 0;
	if (prev_location - ih_location (ih) != ih_item_len (ih))
	    return 0;
	prev_location = ih_location (ih);
    }

    // one may imagine much more checks
    return 1;
}


/* returns 1 if buf looks like an internal node, 0 otherwise */
static int is_correct_internal (char * buf, int blocksize)
{
    struct block_head * blkh;
    int nr;
    int used_space;

    blkh = (struct block_head *)buf;

    if (!is_internal_block_head (buf))
	return 0;
    
    nr = le16_to_cpu (blkh->blk_nr_item);
    if (nr > (blocksize - BLKH_SIZE - DC_SIZE) / (KEY_SIZE + DC_SIZE))
	/* for internal which is not root we might check min number of keys */
	return 0;

    used_space = BLKH_SIZE + KEY_SIZE * nr + DC_SIZE * (nr + 1);
    if (used_space != blocksize - le16_to_cpu (blkh->blk_free_space))
	return 0;

    // one may imagine much more checks
    return 1;
}


// make sure that bh contains formatted node of reiserfs tree of
// 'level'-th level
int is_tree_node (struct buffer_head * bh, int level)
{
    if (B_LEVEL (bh) != level)
	return 0;
    if (is_leaf_node (bh))
	return is_correct_leaf (bh->b_data, bh->b_size);

    return is_correct_internal (bh->b_data, bh->b_size);
}


static int is_desc_block (struct reiserfs_journal_desc * desc)
{
    if (!memcmp(desc->j_magic, JOURNAL_DESC_MAGIC, 8) &&
	le32_to_cpu (desc->j_len) > 0)
	return 1;
    return 0;
}



/* returns code of reiserfs metadata block (leaf, internal, super
   block, journal descriptor), unformatted */
int who_is_this (char * buf, int blocksize)
{
    struct reiserfs_journal_desc * desc;

    if (is_correct_leaf (buf, blocksize))
	/* block head and item head array seem matching (node level, free
           space, item number, item locations and length) */
	return THE_LEAF;

    if (is_correct_internal (buf, blocksize))
	return THE_INTERNAL;

    /* super block? */
    if (is_reiser2fs_magic_string ((void *)buf) || 
	is_reiserfs_magic_string ((void *)buf) ||
	is_prejournaled_reiserfs ((void *)buf))
	return THE_SUPER;

    /* journal descriptor block? */
    if (is_desc_block ((void *)buf))
	return THE_JDESC;

    /* contents of buf does not look like reiserfs metadata. Bitmaps
       are possible here */
    return THE_UNKNOWN;
}


static int block_of_journal (reiserfs_filsys_t fs, unsigned long block)
{
    if (block >= SB_JOURNAL_BLOCK (fs) && 
	block <= SB_JOURNAL_BLOCK (fs) + JOURNAL_BLOCK_COUNT)
	return 1;

    return 0;
}


int block_of_bitmap (reiserfs_filsys_t fs, unsigned long block)
{
    int i;

    for (i = 0; i < SB_BMAP_NR (fs); i ++)
	if (block == SB_AP_BITMAP (fs)[i]->b_blocknr)
	    return 1;

    return 0;
}


/* check whether 'block' can be pointed to by an indirect item */
int not_data_block (reiserfs_filsys_t fs, unsigned long block)
{
    if (block <= fs->s_sbh->b_blocknr)
	/* either super block or a block from skipped area at the
           beginning of filesystem */
	return 1;

    if (block_of_journal (fs, block))
	/* block of journal area */
	return 1;

    if (block_of_bitmap (fs, block))
	/* it is one of bitmap blocks */
	return 1;
    
    return 0;
}


/* check whether 'block' can be logged */
int not_journalable (reiserfs_filsys_t fs, unsigned long block)
{
    if (block < fs->s_sbh->b_blocknr)
	return 1;

    if (block_of_journal (fs, block))
	return 1;

    if (block >= SB_BLOCK_COUNT (fs))
	return 1;

    return 0;
}


// in reiserfs version 0 (undistributed bitmap)
// FIXME: what if number of bitmaps is 15?
int get_journal_old_start_must (struct reiserfs_super_block * rs)
{
    return 3 + rs_bmap_nr (rs);
}


// in reiserfs version 1 (distributed bitmap) journal starts at 18-th
//
int get_journal_start_must (int blocksize)
{
    return (REISERFS_DISK_OFFSET_IN_BYTES / blocksize) + 2;
}

int get_bmap_num (struct super_block * s)
{
    return ((is_prejournaled_reiserfs (s->s_rs)) ?
	    (((struct reiserfs_super_block_v0 *)s->s_rs)->s_bmap_nr) :
	    SB_BMAP_NR (s));
}

int get_block_count (struct super_block * s)
{
    return ((is_prejournaled_reiserfs (s->s_rs)) ?
	    (((struct reiserfs_super_block_v0 *)s->s_rs)->s_block_count) :
	    SB_BLOCK_COUNT (s));
}

int get_root_block (struct super_block * s)
{
    return  ((is_prejournaled_reiserfs (s->s_rs)) ?
	     (((struct reiserfs_super_block_v0 *)s->s_rs)->s_root_block) :
	     SB_ROOT_BLOCK (s));
}


static int new_format (struct super_block * s)
{
    return (SB_JOURNAL_BLOCK (s) == get_journal_start_must (s->s_blocksize));
}


#if 0
int uread_bitmaps (struct super_block * s)
{
    int i, bmp ;
    int error = 0;
    //struct reiserfs_super_block * rs = SB_DISK_SUPER_BLOCK(s);

    SB_AP_BITMAP (s) = getmem (sizeof (struct buffer_head *) * get_bmap_num (s));
    if (!SB_AP_BITMAP (s))
	die ("read_bitmaps: malloc failed\n");

    bmp = SB_BUFFER_WITH_SB (s)->b_blocknr + 1;
    if (is_prejournaled_reiserfs (s->s_rs))
	bmp += get_bmap_num (s);

    for (i = 0; i < get_bmap_num (s); i ++) {
	SB_AP_BITMAP (s)[i] = bread (s->s_dev, bmp, s->s_blocksize);
	if (!SB_AP_BITMAP (s)[i]) {
	    printf ("read_bitmaps: bread #%d bitmap failed (%d is a bad block)\n", i, bmp);
	    SB_AP_BITMAP (s)[i] = getblk (s->s_dev, bmp, s->s_blocksize);
	    memset (SB_AP_BITMAP (s)[i]->b_data, 0xff, s->s_blocksize);
	    set_bit(BH_Uptodate, &SB_AP_BITMAP (s)[i]->b_state);
	    error = IO_ERROR;
	}
	if (new_format (s) && !is_prejournaled_reiserfs (s->s_rs))
	    // FIXME: this new format is not good at all
	    bmp = (i + 1) * (s->s_blocksize * 8);
	else
	    // bitmaps are not distributed
	    bmp ++;
    }
    return error;
}
#endif

int journal_size (struct super_block * s)
{
    return JOURNAL_BLOCK_COUNT;
}



int check_item_f (reiserfs_filsys_t fs, struct item_head * ih, char * item);


/* make sure that key format written in item_head matches to key format
   defined looking at the key */
static int is_key_correct (struct item_head * ih)
{
    if (is_stat_data_ih (ih)) {
	/* stat data key looks identical in both formats */
	if (ih_item_len (ih) == SD_SIZE && ih_key_format (ih) == KEY_FORMAT_2)
	    return 1;
	if (ih_item_len (ih) == SD_V1_SIZE && ih_key_format (ih) == KEY_FORMAT_1)
	    return 1;
	return 0;
    }
    if (ih_key_format (ih) == key_format (&ih->ih_key))
	return 1;
    return 0;
}


/* check stat data item length, ih_free_space, mode */
static int is_bad_sd (reiserfs_filsys_t fs, struct item_head * ih, char * item)
{
    __u16 mode;

    if (ih_entry_count (ih) != 0xffff)
	return 1;

    if (ih_key_format (ih) == KEY_FORMAT_1) {
	struct stat_data_v1 * sd = (struct stat_data_v1 *)item;

	if (ih_item_len (ih) != SD_V1_SIZE)
	    /* old stat data must be 32 bytes long */
	    return 1;
	mode = le16_to_cpu (sd->sd_mode);
    }

    if (ih_key_format (ih) == KEY_FORMAT_2) {
	struct stat_data * sd = (struct stat_data *)item;

	if (ih_item_len (ih) != SD_SIZE)
	    /* new stat data must be 44 bytes long */
	    return 1;
	mode = le16_to_cpu (sd->sd_mode);
    }
    
    if (!S_ISDIR (mode) && !S_ISREG (mode) && !S_ISCHR (mode) && 
	!S_ISBLK (mode) && !S_ISLNK (mode) && !S_ISFIFO (mode) &&
	!S_ISSOCK (mode))
	return 1;

    return 0;
}


/* symlinks created by 3.6.x have direct items with ih_free_space == 0 */
static int is_bad_direct (reiserfs_filsys_t fs, struct item_head * ih, char * item)
{
    if (ih_entry_count (ih) != 0xffff && ih_entry_count (ih) != 0)
	return 1;
    return 0;
}


/* check item length, ih_free_space for pure 3.5 format, unformatted node
   pointers */
static int is_bad_indirect (reiserfs_filsys_t fs, struct item_head * ih, char * item,
			    check_unfm_func_t check_unfm_func)
{
    int i;
    __u32 * ind = (__u32 *)item;

    if (ih_item_len (ih) % UNFM_P_SIZE)
	return 1;

    for (i = 0; i < I_UNFM_NUM (ih); i ++) {
	if (!ind [i])
	    continue;
	if (check_unfm_func && check_unfm_func (fs, ind [i]))
	    return 1;
    }

    if (fs->s_version == REISERFS_VERSION_1) {
	/* check ih_free_space for 3.5 format only */
	if (ih_free_space (ih) > fs->s_blocksize - 1)
	    return 1;
    }
    
    return 0;
}

/* names in reiserfs directory do not end with 0. So, to print them as string
   we have to copy name to a buffer and append 0 */
static char bad_name [4096];

static char * name_from_entry (char * name, int namelen)
{
    memcpy (bad_name, name, namelen);
    bad_name [namelen] = 0;
    return bad_name;
}


hashf_t hashes[] = {0, keyed_hash, yura_hash, r5_hash};

#define good_name(hashfn,name,namelen,deh_offset) \
(GET_HASH_VALUE ((hashfn) (name, namelen)) == GET_HASH_VALUE (deh_offset))


int is_properly_hashed (reiserfs_filsys_t fs,
			char * name, int namelen, __u32 offset)
{
    int i;

    if (namelen == 1 && name[0] == '.') {
	if (offset == DOT_OFFSET)
	    return 1;
	return 0;
    }

    if (namelen == 2 && name[0] == '.' && name[1] == '.') {
	if (offset == DOT_DOT_OFFSET)
	    return 1;
	return 0;
    }

    if (hash_func_is_unknown (fs)) {
	/* try to find what hash function the name is sorted with */
	for (i = 1; i < sizeof (hashes)/ sizeof (hashes[0]); i ++) {
	    if (good_name (hashes [i], name, namelen, offset)) {
		if (!hash_func_is_unknown (fs)) {
		    /* two or more hash functions give the same value for this
                       name */
		    printf ("Detecting hash code: could not detect hash with name \"%s\"\n",
			    name_from_entry (name, namelen));
		    reiserfs_hash(fs) = 0;
		    return 1;
		}
		reiserfs_hash(fs) = hashes [i];
	    }
	}
    }

    if (good_name (reiserfs_hash(fs), name, namelen, offset))
	return 1;

    printf ("namelen %d, name \"%s\", offset %lu, hash %lu\n",
	    namelen, name_from_entry (name, namelen), GET_HASH_VALUE (offset),
	    GET_HASH_VALUE (reiserfs_hash(fs) (name, namelen)));
    /* we could also check whether more than one hash function match on the
       name */
#if 0
    for (i = 1; i < sizeof (hashes) / sizeof (hashes [0]); i ++) {
	if (i == g_real_hash)
	    continue;
	if (good_name (hashes[i], name, namelen, deh_offset)) {
	    die ("bad_hash: at least two hashes got screwed up with this name: \"%s\"",
		 bad_name (name, namelen));
	}
    }
#endif
    return 0;
}


int hash_code (reiserfs_filsys_t fs)
{
    int i;
    
    for (i = 1; i < sizeof (hashes) / sizeof (hashes [0]); i ++)
	if (reiserfs_hash(fs) == hashes [i])
	    return i;
    reiserfs_warning ("hash_code: hash function is not defined\nUsing tea hash\n");
    return TEA_HASH;
}


/* the only corruption which is not considered fatal - is hash mismatching. If
   bad_dir is set - directory item having such names is considered bad */
static int is_bad_directory (reiserfs_filsys_t fs, struct item_head * ih, char * item,
			     int bad_dir)
{
    int i;
    int namelen;
    struct reiserfs_de_head * deh = (struct reiserfs_de_head *)item;
    __u32 prev_offset = 0;
    __u16 prev_location = ih_item_len (ih);
    
    for (i = 0; i < ih_entry_count (ih); i ++, deh ++) {
	if (deh_location (deh) >= prev_location)
	    return 1;
	prev_location = deh_location (deh);
	    
	namelen = name_length (ih, deh, i);
	if (namelen > REISERFS_MAX_NAME_LEN (fs->s_blocksize)) {
	    return 1;
	}
	if (deh_offset (deh) <= prev_offset)
	    return 1;
	prev_offset = deh_offset (deh);
	
	/* check hash value */
	if (!is_properly_hashed (fs, item + prev_location, namelen, prev_offset)) {
	    if (bad_dir)
		/* make is_bad_leaf to not insert whole leaf. Node will be
		   marked not-insertable and put into tree item by item in
		   pass 2 */
		return 1;
	}
    }

    return 0;
}


/* with this one is allowed to */
int is_it_bad_item (reiserfs_filsys_t fs, struct item_head * ih, char * item,
		    
		    check_unfm_func_t check_unfm, int bad_dir)
{
    if (!is_key_correct (ih))
	return 1;

    if (is_stat_data_ih (ih))
	return is_bad_sd (fs, ih, item);

    if (is_direntry_ih (ih))
	return is_bad_directory (fs, ih, item, bad_dir);

    if (is_indirect_ih (ih))
	return is_bad_indirect (fs, ih, item, check_unfm);

    if (is_direct_ih (ih))
	return is_bad_direct (fs, ih, item);

    return 1;
}


