/*
 * Copyright 1996-2001 Hans Reiser
 */
#include "fsck.h"
#include <stdlib.h>


#define NR_TO_READ 8


/* leaves */
reiserfs_bitmap_t leaves_bitmap;
#define pass0_is_leaf(block) __is_marked (leaves, block)
#define pass0_mark_leaf(block) __mark (leaves, block)

/* nodes which are referred to from only one indirect item */
reiserfs_bitmap_t good_unfm_bitmap;
#define pass0_is_good_unfm(block) __is_marked (good_unfm, block)
#define pass0_mark_good_unfm(block) __mark (good_unfm, block)
#define pass0_unmark_good_unfm(block) __unmark (good_unfm, block)

/* nodes which are referred to from more than one indirect item */
reiserfs_bitmap_t bad_unfm_bitmap;
#define pass0_is_bad_unfm(block) __is_marked (bad_unfm, block)
#define pass0_mark_bad_unfm(block) __mark (bad_unfm, block)
#define pass0_unmark_bad_unfm(block) __unmark (bad_unfm, block)





static void make_aux_bitmaps (struct super_block * s)
{
    leaves_bitmap = reiserfs_create_bitmap (SB_BLOCK_COUNT (s));

    good_unfm_bitmap = reiserfs_create_bitmap (SB_BLOCK_COUNT (s));

    bad_unfm_bitmap = reiserfs_create_bitmap (SB_BLOCK_COUNT (s));

}


/* register block some indirect item points to */
static void register_unfm (unsigned long block)
{
    if (!pass0_is_good_unfm (block) && !pass0_is_bad_unfm (block)) {
	/* this block was not pointed by other indirect items yet */
	pass0_mark_good_unfm (block);
	return;
    }

    if (pass0_is_good_unfm (block)) {
	/* block was pointed once already, unmark it in bitmap of good
           unformatted nodes and mark in bitmap of bad pointers */
	pass0_unmark_good_unfm (block);
	pass0_mark_bad_unfm (block);
	return;
    }

    assert (pass0_is_bad_unfm (block));
}



/* do this on pass 0 with every leaf marked used */
static void handle_leaf (struct super_block * s, struct buffer_head * bh)
{
    int i, j;
    struct item_head * ih;
    __u32 * ind_item;
    unsigned long unfm_ptr;
    int dirty = 0;


    ih = B_N_PITEM_HEAD (bh, 0);
    for (i = 0; i < B_NR_ITEMS (bh); i ++, ih ++) {
	/* FIXME: should we do something with other items? */
	if (!is_indirect_ih (ih))
	    continue;
	
	ind_item = (__u32 *)B_I_PITEM (bh, ih);
	for (j = 0; j < I_UNFM_NUM (ih); j ++) {
	    unfm_ptr = le32_to_cpu (ind_item [j]);
	    if (!unfm_ptr)
		continue;
	    if (not_data_block (s, unfm_ptr) || /* journal area or bitmap or super block */
		unfm_ptr >= SB_BLOCK_COUNT (s)) {/* garbage in pointer */

		stat_wrong_pointer_found (s);
		fsck_log ("pass0: %d-th pointer (%lu) in itme %k (leaf block %lu) is wrong\n",
			  j, unfm_ptr, &ih->ih_key, bh->b_blocknr);
		continue;
	    }
	    if (!was_block_used (unfm_ptr)) {
	      /* this will get to a pool of allocable blocks */
	      ind_item [j] = 0;
	      dirty = 1;
	      stat_wrong_pointer_found (s);
	      continue;
	    }

	    /* mark block in bitmaps of unformatted nodes */
	    register_unfm (unfm_ptr);
	}
    }

    pass0_mark_leaf (bh->b_blocknr);

    if (dirty) {
	stat_pass0_leaf_corected (s);
	mark_buffer_dirty (bh);
    }
}

static int at_least_one_used (struct super_block * s, unsigned long first, int count)
{
    int i;

    for (i = 0; i < count; i ++)
	if (was_block_used (first + i))
	    return 1;
    return 0;
}

static void go_through (struct super_block * s)
{
    struct buffer_head * bbh, * bh;
    int nr_to_read = NR_TO_READ;
    int i, j;
    char * data;
    int what_node;
    unsigned long handled_blocks = 0;
    /*    FILE * block_list;*/

    /* for debug only */
    /*block_list = fopen ("pass0-leaves", "w");*/

    for (i = 0; i < SB_BLOCK_COUNT (s); i += nr_to_read) {
	if (at_least_one_used (s, i, nr_to_read)) {
	    /* at least one of nr_to_read blocks is to be checked */
	    bbh = bread (s->s_dev, i / nr_to_read, s->s_blocksize * nr_to_read);
	    if (bbh) {
		for (j = 0; j < nr_to_read; j ++) {
		    if (!was_block_used (i + j)) {
			stat_free_block_found (s);
			continue;
		    }
		    if (not_data_block (s, i + j)) {
			stat_not_data_block_found (s);
			continue;
		    }

		    stat_used_block_found (s);

		    print_how_far (&handled_blocks, g_blocks_to_read);
		    data = bbh->b_data + j * s->s_blocksize;

		    what_node = who_is_this (data, s->s_blocksize);
		    if ( what_node != THE_LEAF )
			continue;
		    
		    /* fprintf (block_list, "leaf %d\n", i + j);*/

		    /* the node looks like a leaf, but it still can be
		       not perfect */
		    bh = make_buffer (s->s_dev, i + j, s->s_blocksize, data);
		    handle_leaf (s, bh);
		    brelse (bh);
		}
		bforget (bbh);
	    } else {
		/* bread failed */
		if (nr_to_read != 1) {
		    /* we tryied to read bunch of blocks. Try to read them by one */
		    i -= nr_to_read;
		    nr_to_read = 1;
		}
	    }
	}

	if (nr_to_read == 1 && ((i + 1) % NR_TO_READ) == 0) {
	    /* we have read NR_TO_READ blocks one at time, switch back to
               reading NR_TO_READ blocks at time */
	    i -= (NR_TO_READ - 1);
	    nr_to_read = NR_TO_READ;
	}
    }
    if (i != SB_BLOCK_COUNT (s))
	die ("go_through: block counting failed");

    /*    fclose (block_list);*/
}


static void check_bitmaps (struct super_block * s)
{
    int i;
 
    stats (s)->all_blocks = SB_BLOCK_COUNT (s);

    /* find how many leaves are not pointed by any indirect items */
    for (i = 0; i < SB_BLOCK_COUNT (s); i ++) {
	if (not_data_block (s, i))
	    continue;

	if (!was_block_used (i)) {
	    /* marked free in the on-disk bitmap */
	    make_allocable (i);
	    continue;
	}
	  
	if (pass0_is_leaf (i)) {
	    /* marked in the bitmap of leaves */
	    stat_leaf_found (s);
	    if (pass0_is_good_unfm (i) || pass0_is_bad_unfm (i))
		stat_pointed_leaf_found (s);
	}

	if (pass0_is_good_unfm (i) && pass0_is_bad_unfm (i))
	    die ("check_bitmap: bad and good unformatted");

	if (pass0_is_good_unfm (i)) {
	    stat_pointed_found (s);
	    stat_pointed_once_found (s);
	    continue;
	}
	if (pass0_is_bad_unfm (i)) {
	    stat_pointed_found (s);
	    stat_pointed_more_than_once_found (s);
	    continue;
	}

	/* blocks which marked used but are not leaves and are not
	   pointed (internals in short) get into bitmap of allocable
	   blocks */
	if (!(pass0_is_leaf (i))) {
	    make_allocable (i);
	}
    }
}


static int should_skip_pass0 (struct super_block * s)
{
    struct stat st;
    unsigned long size;
    int answer;
    
    size = ((SB_BLOCK_COUNT (fs) + 7) / 8) * 4;
    if (stat (".pass0.bmps", &st) != 0 || st.st_size != size)
	return 0;
    
    answer = user_confirmed ("There seems to be correct results of pass 0 - should I skip it (Yes)?",
			     "Yes\n");
    return answer;
	
}

static void save_pass0_result (struct super_block * s)
{
    return;
#if 0
    FILE * fp;
    unsigned long size;
    
    size = ((SB_BLOCK_COUNT (fs) + 7) / 8);
 
    fp = fopen (".pass0.bmps", "w");
    if (!fp)
	return;

    if (fwrite (leaves_bitmap->bm_map, size, 1, fp) != 1)
	return;
    if (fwrite (good_unfm_bitmap->bm_map, size, 1, fp) != 1)
	return;
    if (fwrite (bad_unfm_bitmap->bm_map, size, 1, fp) != 1)
	return;
    if (fwrite (fsck_allocable_bitmap (s)->bm_map, size, 1, fp) != 1)
	return;
    fclose (fp);
#endif
}


static void load_pass0_result (struct super_block * s)
{
    die ("does not work now");
#if 0
    FILE * fp;
    unsigned long size;
    
    size = ((SB_BLOCK_COUNT (fs) + 7) / 8);

    fp = fopen (".pass0.bmps", "r");
    if (!fp)
	die ("load_pass0_result: could not avoid pass0");
    
    if (fread (leaves_bitmap, size, 1, fp) != 1 ||
	fread (good_unfm_bitmap, size, 1, fp) != 1 ||
	fread (bad_unfm_bitmap, size, 1, fp) != 1 ||
	fread (allocable_bitmap, size, 1, fp) != 1)
	die ("load_pass0_result: could not avoid pass0");

    fclose (fp);  
#endif  
}


int pass_0 (struct super_block * s)
{
    int i;

    make_aux_bitmaps (s);

    i = should_skip_pass0 (s);
    if (i) {
	load_pass0_result (s);
    } else {
	fsck_progress ("Pass 0 - ");fflush (stdout);
	go_through (s);
	fsck_progress (" - done\n");
	save_pass0_result (s);
    }

    /* find out node those were internals */
    check_bitmaps (s);

    stage_report (0, s);
}


int are_there_used_leaves (unsigned long from, int count)
{
    int i;

    for (i = 0; i < count; i ++)
	if (pass0_is_leaf (from + i))
	    return 1;
    return 0;
}


int is_used_leaf (unsigned long block)
{
    return pass0_is_leaf (block);
}


int how_many_leaves_were_there (void)
{
    return stats (fs)->leaves;
}

/* these are used to correct uformatted node pointers */
int is_bad_unformatted (unsigned long block)
{
    return pass0_is_bad_unfm (block);
}

/* this is for check only. With this we make sure that all pointers we
   put into tree on pass 1 do not point to leaves (FIXME), do not
   point to journal, bitmap, etc, do not point out of fs boundary and
   are marked used in on-disk bitmap */
int still_bad_unfm_ptr_1 (unsigned long block)
{
    if (!block)
	return 0;
    if (pass0_is_leaf (block))
	return 1;
    if (pass0_is_bad_unfm (block) && !is_bad_unfm_in_tree_once (block))
	return 1;
    if (not_data_block (fs, block))
	return 1;
    if (!was_block_used (block))
	return 1;
    if (block >= stats (fs)->all_blocks)
	return 1;
    return 0;
    
}


/* pointers to data block which get into tree are checked with this */
int still_bad_unfm_ptr_2 (unsigned long block)
{
    if (!block)
	return 0;
    if (is_block_used (block))
	return 1;
    if (block >= stats (fs)->all_blocks)
	return 1;
    return 0;
}

#if 0
int is_good_unformatted (unsigned long block)
{
    return _is_good_unformatted(block);
}

void mark_good_unformatted (unsigned long block)
{
    _mark_good_unformatted(block);
}
#endif

/* these are used to allocate blocks for tree building */
int are_there_allocable_blocks (int amout_needed)
{
    if (reiserfs_bitmap_zeros (fsck_allocable_bitmap (fs)) < amout_needed) {
	int zeros = 0, i;
	
	fsck_progress ("Hmm, there are not enough allocable blocks, checking bitmap...");fflush (stdout);
	for (i = 0; i < fsck_allocable_bitmap (fs)->bm_bit_size; i ++)
	    if (!reiserfs_bitmap_test_bit (fsck_allocable_bitmap (fs), i))
		zeros ++;
	fsck_progress ("there are %d zeros, btw\n", zeros);
	return 0;
    }
    return 1;
}


unsigned long alloc_block (void)
{
    unsigned long block = 0; /* FIXME: start point could be used */

    if (reiserfs_bitmap_find_zero_bit (fsck_allocable_bitmap (fs), &block)) {
	die ("alloc_block: allocable blocks counter is wrong");
	return 0;
    }
    reiserfs_bitmap_set_bit (fsck_allocable_bitmap (fs), block);
    return block;
}


void make_allocable (unsigned long block)
{
    reiserfs_bitmap_clear_bit (fsck_allocable_bitmap (fs), block);
}


unsigned long how_many_uninsertables_were_there (void)
{
    return stats (fs)->uninsertable_leaves;
}


unsigned long how_many_items_were_saved (void)
{
    return stats (fs)->saved_on_pass1;
}
