/*
 * Copyright 2000 by Hans Reiser, licensing governed by reiserfs/README
 */

#include "pack.h"


static reiserfs_bitmap_t bitmap;
int packed_leaves, bad_leaves, full_blocks;

/* these are to calculate compression */
unsigned long sent; /* how many bytes sent to stdout */
unsigned long had_to_be_sent; /* how many bytes were to be sent */


static void get_bitmap (struct super_block * s)
{
    int i, bmap_num;

    bitmap = reiserfs_create_bitmap (SB_BLOCK_COUNT (s));
    if (!bitmap)
	die ("get_bitmap: could not create bitmap");
    reiserfs_fetch_disk_bitmap (bitmap, s);
}


static int is_block_used (struct super_block * s, __u32 block)
{
    return reiserfs_bitmap_test_bit (bitmap, block);
}

static void mark_block_free (struct super_block * s, __u32 block)
{
    reiserfs_bitmap_clear_bit (bitmap, block);
}



static void pack_key (struct packed_item * pi, struct item_head * ih)
{
    if (pi->mask & DIR_ID) {
	fwrite32 (&ih->ih_key.k_dir_id);
	sent += sizeof (__u32);
    }

    if (pi->mask & OBJECT_ID) {
	fwrite32 (&ih->ih_key.k_objectid);
	sent += sizeof (__u32);
    }

    if (pi->mask & OFFSET_BITS_64) {
	__u64 offset;

	offset = get_offset (&ih->ih_key);
	fwrite64 (&offset);
	sent += sizeof (__u64);
    }

    if (pi->mask & OFFSET_BITS_32) {
	__u32 offset;

	offset = get_offset (&ih->ih_key);
	fwrite32 (&offset);
	sent += sizeof (__u32);
    }
}


static void pack_direct (struct packed_item * pi, struct buffer_head * bh, 
			  struct item_head * ih)
{
    pi->mask |= DIRECT_ITEM;

    /* send packed item header to stdout */
    fwrite (pi, sizeof (*pi), 1, stdout);
    sent += sizeof (*pi);
    
    /* send key components which are to be sent */
    pack_key (pi, ih);
}


/* if there is at least one extent longer than 2 - it is worth packing */
static int should_pack_indirect (__u32 * ind_item, int unfm_num)
{
    int i, len;

    for (i = 1, len = 1; i < unfm_num; i ++) {
	if (!ind_item [i] && !ind_item [i - 1] || /* hole continues */
	    ind_item [i] == ind_item [i - 1] + 1) { /* subsequent blocks */
	    len ++;
	    if (len > 2)
		return 1;
	} else {
	    /* sequence of blocks or hole broke */
	    len = 1;
	}
    }
    return 0;
}


/* indirect item can be either packed using "extents" (when it is
   worth doing) or be stored as is. Size of item in packed form is not
   stored. Unpacking will stop when full item length is reached */
static int pack_indirect (struct packed_item * pi, struct buffer_head * bh, 
			  struct item_head * ih)
{
    int i;
    __u32 * ind_item;
    __u16 len;

    pi->mask |= INDIRECT_ITEM;
    if (ih_entry_count (ih))
	pi->mask |= ENTRY_COUNT;
    
    ind_item = (__u32 *)B_I_PITEM (bh, ih);
    if (!should_pack_indirect (ind_item, I_UNFM_NUM (ih)))
	pi->mask |= WHOLE_INDIRECT;
    
    /* send packed item header to stdout */
    fwrite (pi, sizeof (*pi), 1, stdout);
    sent += sizeof (*pi);

    /* send key components which are to be sent */
    pack_key (pi, ih);

    if (pi->mask & ENTRY_COUNT) {
	__u16 ih_free_space;

	ih_free_space = ih_entry_count (ih);
	fwrite16 (&ih_free_space);
	sent += sizeof (__u16);
    }

    if (pi->mask & WHOLE_INDIRECT) {
	fwrite (ind_item, ih_item_len (ih), 1, stdout);
	sent += ih_item_len (ih);
	return;
    }

    fwrite32 (&ind_item [0]);
    sent += sizeof (__u32);
    for (i = 1, len = 1; i < I_UNFM_NUM (ih); i ++) {
	if (!ind_item [i] && !ind_item [i - 1] || /* hole continues */
	    ind_item [i] == ind_item[ i - 1] + 1) { /* subsequent blocks */
	    len ++;
	} else {
	    fwrite16 (&len);
	    fwrite32 (&ind_item[i]);
	    sent += (sizeof (__u32) + sizeof (__u16));
	    len = 1;
	}
    }
    fwrite16 (&len);
    sent += sizeof (__u16);

    return 0;
}


/* directory item is packed:
   entry count - 16 bits
   for each entry
   	mask (8 bits) - it shows whether there are any of (deh_dir_id, gen counter, deh_state)
	entry length 16 bits
	entry itself
	deh_objectid - 32 bits
		maybe deh_dir_id (32 bits)
		maybe gencounter (16)
		maybe deh_state (16)
*/
static int pack_direntry (reiserfs_filsys_t fs, struct packed_item * pi,
			  struct buffer_head * bh,
			  struct item_head * ih)
{
    int i;
    struct reiserfs_de_head * deh;
    struct packed_dir_entry pe;
    __u16 entry_count, gen_counter;
    __u8 mask_init;

    pi->mask |= (DIRENTRY_ITEM | ENTRY_COUNT);

    /* send packed item header to stdout */
    fwrite (pi, sizeof (*pi), 1, stdout);
    sent += sizeof (*pi);

    /* send key components which are to be sent */
    pack_key (pi, ih);

    /* entry count is sent unconditionally */
    entry_count = ih_entry_count (ih);
    fwrite16 (&entry_count);

    deh = B_I_DEH (bh, ih);
    for (i = 0; i < entry_count; i ++, deh ++) {
	pe.entrylen = entry_length (ih, deh, i);
	pe.mask = 0;
	if (deh_dir_id (deh) != le32_to_cpu (ih->ih_key.k_objectid))
	    /* entry points to name of another directory, store deh_dir_id */
	    pe.mask |= HAS_DIR_ID;

	gen_counter = GET_GENERATION_NUMBER (deh_offset (deh));
	if (gen_counter != 0)
	    /* store generation counter if it is != 0 */
	    pe.mask |= HAS_GEN_COUNTER;

	if (le16_to_cpu (deh->deh_state) != 4)
	    /* something unusual in deh_state. Store it */
	    pe.mask |= HAS_STATE;

	fwrite8 (&pe.mask);
	fwrite16 (&pe.entrylen);
	fwrite (name_in_entry (deh, i), pe.entrylen, 1, stdout);
	fwrite32 (&(deh->deh_objectid));
	sent += (sizeof (__u8) + sizeof (__u16) + pe.entrylen + sizeof (__u32));
	
	if (pe.mask & HAS_DIR_ID) {
	    fwrite32 (&deh->deh_dir_id);
	    sent += sizeof (__u32);
	}

	if (pe.mask & HAS_GEN_COUNTER) {
	    fwrite16 (&gen_counter);
	    sent += sizeof (__u16);
	}

	if (pe.mask & HAS_STATE) {
	    fwrite16 (&deh->deh_state);
	    sent += sizeof (__u16);
	}
    }
}


static void pack_stat_data (struct packed_item * pi, struct buffer_head * bh,
			    struct item_head * ih)
{
    pi->mask |= STAT_DATA_ITEM;

    if (stat_data_v1 (ih)) {
	/* for old stat data: we take
	   mode - 16 bits
	   nlink - 16 bits
	   size - 32 bits
	   blocks/rdev - 32 bits
	   maybe first_direct byte 32 bits
	*/
	struct stat_data_v1 * sd_v1;

	sd_v1 = (struct stat_data_v1 *)B_I_PITEM (bh, ih);
	if (sd_v1->sd_first_direct_byte != 0xffffffff)
	    pi->mask |= WITH_SD_FIRST_DIRECT_BYTE;

	/* we are done with packed_item send packed it to stdout */
	fwrite (pi, sizeof (*pi), 1, stdout);
	sent += sizeof (*pi);
	
	/* send key components which are to be sent */
	pack_key (pi, ih);
	
	fwrite16 (&sd_v1->sd_mode);
	fwrite16 (&sd_v1->sd_nlink);
	fwrite32 (&sd_v1->sd_size);
	fwrite32 (&sd_v1->u.sd_blocks);
	sent += (sizeof (__u16) * 2 + sizeof (__u32) * 2);
	if (pi->mask & WITH_SD_FIRST_DIRECT_BYTE) {
	    fwrite32 (&sd_v1->sd_first_direct_byte);
	    sent += sizeof (__u32);
	}
    } else {
	/* for new stat data
	   mode - 16 bits
	   nlink in either 16 or 32 bits
	   size in either 32 or 64 bits
	   blocks - 32 bits
	*/
	struct stat_data * sd;
	__u16 nlink16;
	__u32 nlink32, size32;
	__u64 size64;

	sd = (struct stat_data *)B_I_PITEM (bh, ih);
	if (sd->sd_nlink > 0xffff) {
	    pi->mask |= NLINK_BITS_32;
	    nlink32 = sd->sd_nlink;
	} else {
	    nlink16 = sd->sd_nlink;
	}
	if (sd->sd_size > 0xffffffff) {
	    pi->mask |= SIZE_BITS_64;
	    size64 = sd->sd_size;
	} else {
	    size32 = sd->sd_size;
	}

	/* we are done with packed_item send packed it to stdout */
	fwrite (pi, sizeof (*pi), 1, stdout);
	sent += sizeof (*pi);

	/* send key components which are to be sent */
	pack_key (pi, ih);

	fwrite16 (&sd->sd_mode);
	sent += sizeof (__u16);
	if (pi->mask & NLINK_BITS_32) {
	    fwrite32 (&nlink32);
	    sent += sizeof (__u32);
	} else {
	    fwrite16 (&nlink16);	
	    sent += sizeof (__u16);
	}

	if (pi->mask & SIZE_BITS_64) {
	    fwrite64 (&size64);
	    sent += sizeof (__u64);
	} else {
	    fwrite32 (&size32);
	    sent += sizeof (__u32);
	}
    
	fwrite32 (&sd->sd_blocks);
	sent += sizeof (__u32);
    }
}


static void pack_full_block (struct buffer_head * bh)
{
    __u16 magic;
    __u32 block;

    magic = FULL_BLOCK_START_MAGIC;
    fwrite16 (&magic);

    block = bh->b_blocknr;
    fwrite32 (&block);
    
    fwrite (bh->b_data, 4096, 1, stdout);
    sent += 4096;
    had_to_be_sent += 4096;

    full_blocks ++;
    
}


/* unformatted node pointer is considered bad when it points either to blocks
   of journal, bitmap blocks, super block or is transparently out of range of
   disk block numbers */
static int check_unfm_ptr (reiserfs_filsys_t fs, __u32 block)
{
    if (block >= SB_BLOCK_COUNT (fs))
        return 1;

    if (not_data_block (fs, block))
        return 1;

    return 0;
}


/* we only pack leaves which do not have any corruptions */
static int can_pack (reiserfs_filsys_t fs, struct buffer_head * bh)
{
    int i;
    struct item_head * ih;
    int bad_dir = 1; /* if name is not hashed properly - it is a bad directory */

    ih = B_N_PITEM_HEAD (bh, 0);
    for (i = 0; i < node_item_number (bh); i ++, ih ++) {
	if (is_it_bad_item (fs, ih, B_I_PITEM (bh, ih), check_unfm_ptr, 1/*bad dir*/))
	    return 0;
    }
    return 1;
}

/* pack leaf only if all its items are correct: keys are correct,
   direntries are hashed properly and hash function is defined,
   indirect items are correct, stat data ?, */
static void pack_leaf (reiserfs_filsys_t fs, struct buffer_head * bh)
{
    int i;
    char * buf, * start;
    struct item_head * ih;
    struct packed_item pi;
    __u16 v16;
    __u32 v32;

    if (!can_pack (fs, bh)) {
	/* if something looks suspicious in this leaf - pack whole block */
	pack_full_block (bh);
	fprintf (stderr, "leaf %lu is bad\n", bh->b_blocknr);
	bad_leaves ++;
	return;
    }

    /* start magic in low 8 bits, hash code in high 8 bits */
    v16 = (LEAF_START_MAGIC | (hash_code (fs) << 8));
    fwrite16 (&v16);
    
    /* block number */
    fwrite32 (&bh->b_blocknr);

    /* item number */
    v16 = node_item_number (bh);
    fwrite16 (&v16);

    ih = B_N_PITEM_HEAD (bh, 0);

    for (i = 0; i < node_item_number (bh); i ++, ih ++) {
#if 0
	v32 = ITEM_START_MAGIC;
	fwrite32 (&v32);
#endif

	pi.mask = 0;
	pi.item_len = ih_item_len (ih);

	// format
	if (ih_key_format (ih) == KEY_FORMAT_2)
	    pi.mask |= NEW_FORMAT;

	// k_dir_id
	if (!i || (i && ih->ih_key.k_dir_id != (ih - 1)->ih_key.k_dir_id)) {
	    /* if item is first in the leaf or if previous item has different
               k_dir_id - store it */
	    pi.mask |= DIR_ID;
	}
	// k_object_id
	if (!i || (i && ih->ih_key.k_objectid != (ih - 1)->ih_key.k_objectid)) {
	    /* if item is first in the leaf or if previous item has different
               k_objectid - store it */
	    pi.mask |= OBJECT_ID;
	}

	/* store offset if it is != 0 in 32 or 64 bits */
	if (get_offset (&ih->ih_key)) {
	    if (get_offset (&ih->ih_key) > 0xffffffffULL)
		pi.mask |= OFFSET_BITS_64;
	    else
		pi.mask |= OFFSET_BITS_32;
	}

	if (is_direct_ih (ih)) {
	    pack_direct (&pi, bh, ih);
	} else if (is_indirect_ih (ih))
	    pack_indirect (&pi, bh, ih);
	else if (is_direntry_ih (ih))
	    pack_direntry (fs, &pi, bh, ih);
	else if (is_stat_data_ih (ih))
	    pack_stat_data (&pi, bh, ih);
	else
	    die ("pack_leaf: unknown item found");
#if 0
	v32 = ITEM_END_MAGIC;
	fwrite32 (&v32);
#endif
    }

    v16 = LEAF_END_MAGIC;
    fwrite16 (&v16);

    packed_leaves ++;
    had_to_be_sent += 4096;

    return;
}


extern int opt_pack_all;

static int get_total_block_number (struct super_block * s)
{
    int i, j;
    int retval = 0;
    int bmap_num;
    
    retval = 0;
    
    if (opt_pack_all)
	retval = get_block_count (s);
    else {
	bmap_num = get_bmap_num (s);
	for (i = 0; i < bmap_num; i ++) {
	    for (j = 0; j < s->s_blocksize * 8; j ++)
		if (i * s->s_blocksize * 8 + j < get_block_count (s) &&
		    test_bit (j, SB_AP_BITMAP (s)[i]->b_data))
		    retval ++;
	}
    }
    return retval;
}


static void pack_super_block (struct super_block * s)
{
    pack_full_block (SB_BUFFER_WITH_SB (s));
    mark_block_free (s, SB_BUFFER_WITH_SB (s)->b_blocknr);
}


static void pack_bitmap (struct super_block * s)
{
    int i;
    int bmap_num;

    bmap_num = get_bmap_num (s);
    for (i = 0; i < bmap_num; i ++) {
	pack_full_block (SB_AP_BITMAP (s)[i]);
	mark_block_free (s, SB_AP_BITMAP (s)[i]->b_blocknr);
    }
}


static void pack_journal (reiserfs_filsys_t fs)
{
    int i;
    struct buffer_head * bh;

    if (is_prejournaled_reiserfs (fs->s_rs))
	return;
    
    fprintf (stderr, "Journal ..");
    for (i = SB_JOURNAL_BLOCK (fs);
	 i < SB_JOURNAL_BLOCK (fs) + journal_size (fs) + 1; i ++) {
	bh = bread (fs->s_dev, i, fs->s_blocksize);
	if (!bh)
	    continue;
	if (who_is_this (bh->b_data, bh->b_size) == THE_LEAF)
	    pack_leaf (fs, bh);
	else
	    pack_full_block (bh);
	mark_block_free (fs, bh->b_blocknr);
	brelse (bh);
    }
    fprintf (stderr, "done\n");
}


#define BLOCKS_PER_READ 8


// read blocks marked as used, pack leaves, mark leaves and internals as free
// which allows to have bitmap of unformatted nodes after this pass
static void pack_leaves (reiserfs_filsys_t fs)
{
    int i, j;
    struct buffer_head * bh;
    struct buffer_head tmp;
    unsigned long total_block_number, done;
    int block_count;

    // how many blocks have to be read. Needed to show progress
    total_block_number = get_total_block_number (fs);
    done = 0;

    tmp.b_size = fs->s_blocksize;
    block_count = get_block_count (fs);

    /* skip journal */
    i = SB_JOURNAL_BLOCK (fs) + journal_size (fs) + 1;
    for (; i < block_count / BLOCKS_PER_READ * BLOCKS_PER_READ; i += BLOCKS_PER_READ) {
	for (j = 0; j < BLOCKS_PER_READ; j ++)
	    if (is_block_used (fs, i + j))
		break;
	if (j == BLOCKS_PER_READ)
	    // all blocks are free
	    continue;

	bh = bread (fs->s_dev, i / BLOCKS_PER_READ, fs->s_blocksize * BLOCKS_PER_READ);
	if (!bh)
	    continue;

	for (j = 0; j < BLOCKS_PER_READ; j ++) {
	    if (!is_block_used (fs, i + j))
		continue;

	    tmp.b_blocknr = bh->b_blocknr * BLOCKS_PER_READ + j;
	    tmp.b_data = bh->b_data + j * tmp.b_size;
	    if (who_is_this (tmp.b_data, tmp.b_size) == THE_LEAF) {
		pack_leaf (fs, &tmp);
		/* free bit in bitmap of unformatted nodes */
		mark_block_free (fs, tmp.b_blocknr);
	    }
	    if (who_is_this (tmp.b_data, tmp.b_size) == THE_INTERNAL)
		/* free bit in bitmap of unformatted nodes */
		mark_block_free (fs, tmp.b_blocknr);

	    print_how_far (&done, total_block_number);
	}

	brelse (bh);
    }


    for (; i < block_count; i ++) {
	if (!is_block_used (fs, i))
	    continue;
	bh = bread (fs->s_dev, i, fs->s_blocksize);
	if (!bh)
	    continue;
	if (who_is_this (bh->b_data, bh->b_size) == THE_LEAF) {
	    pack_leaf (fs, &tmp);
	    /* free bit in bitmap of unformatted nodes */
	    mark_block_free (fs, tmp.b_blocknr);
	}
	if (who_is_this (tmp.b_data, tmp.b_size) == THE_INTERNAL)
	    /* free bit in bitmap of unformatted nodes */
	    mark_block_free (fs, tmp.b_blocknr);

	print_how_far (&done, total_block_number);

	brelse (bh);
    }
}


// bitmap is now bitmap of unformatted nodes
static void pack_unformatted_bitmap (struct super_block * s)
{
    __u16 magic;
    int i;
    __u16 v16;
    __u32 v32;

    magic = UNFORMATTED_BITMAP_START_MAGIC;
    fwrite16 (&magic);

    v16 = get_bmap_num (s);
    v32 = get_block_count (s);
    fwrite16 (&v16);
    fwrite32 (&v32);

    for (i = 0; i < v16; i ++)
	fwrite (SB_AP_BITMAP (s)[i]->b_data, s->s_blocksize, 1, stdout);
}


// write reiserfs partition metadata to stdout in the following format:
//
// 32 bit reiserfs magic
// 16 bit blocksize
// blocks. Super block, bitmaps, journal blocks, leaves (packed)
// bitmaps of unformatted nodes (start magic, bitmap1, ..., end magic)
// 16 bit end magic
//
void pack_partition (struct super_block * s)
{
    __u32 magic32;
    __u16 blocksize;
    __u16 magic16;

    // get pointers to bitmaps
    get_bitmap (s);

    magic32 = REISERFS_SUPER_MAGIC;
    fwrite32 (&magic32);

    blocksize = s->s_blocksize;
    fwrite16 (&blocksize);

    // 1. 
    pack_super_block (s);

    // 2.
    pack_bitmap (s);

    // 3.
    pack_journal (s);

    // 4.
    pack_leaves (s);

    // 5.
    pack_unformatted_bitmap (s);
    
    magic16 = END_MAGIC;
    fwrite16 (&magic16);

    fprintf (stderr, "Packed\n\tleaves %d\n\tfull block %d\n\tcorrupted leaves %d\n",
	     packed_leaves, full_blocks, bad_leaves);
    fprintf (stderr, "data packed with ratio %.2f\n", (double)sent / had_to_be_sent);

    reiserfs_delete_bitmap (bitmap);
}


void pack_one_block (reiserfs_filsys_t fs, unsigned long block)
{
    __u32 magic32;
    __u16 magic16;
    struct buffer_head * bh;

    // reiserfs magic
    magic32 = REISERFS_SUPER_MAGIC;
    fwrite32 (&magic32);

    // blocksize
    fwrite16 (&fs->s_blocksize);
    
    bh = bread (fs->s_dev, block, fs->s_blocksize);

    if (who_is_this (bh->b_data, bh->b_size) == THE_LEAF)
	pack_leaf (fs, bh);
    else
	pack_full_block (bh);

    brelse (bh);

    // end magic
    magic16 = END_MAGIC;
    fwrite16 (&magic16);

    fprintf (stderr, "Packed\n\tleaves %d\n\tfull block %d\n\tcorrupted leaves %d\n",
	     packed_leaves, full_blocks, bad_leaves);
}


#if 0
//
// this test program has two modes: 'pack file blocknr'
// and 'unpack file'
// in the first mode blocknr-th 4k block of the 'file' will be packed out to stdout
// the the second mode standart input will be converted to the reiserfs leaf on 'file'
//
static int do_unpack (char * file)
{
    char * buf;
    int fd;

    fd = open (file, O_RDONLY);
    if (fd == -1) {
	perror ("open failed");
	return 0;
    }
    
    buf = malloc (4096);
    if (!buf) {
	perror ("malloc failed");
	return 0;
    }

    fread (buf, 4096, 1, stdin);
    if (!feof (stdin)) {
	printf ("fread returned not eof\n");
	return 0;
    }

    unpack_leaf (buf, fd);

    free (buf);
    close (fd);
    return 0;
}

static int do_pack (char * file, int block)
{
    int fd;
    struct buffer_head * bh;
    char * buf;
    int len;

    fprintf (stderr, "dumping block %d of the \"%s\"\n", block, file);

    fd = open (file, O_RDONLY);
    if (fd == -1) {
	perror ("open failed");
	return 0;
    }
    
    bh = bread (fd, block, 4096);
    if (!bh) {
	fprintf (stderr, "bread failed\n");
	return 0;
    }

    if (who_is_this (bh->b_data, bh->b_size) != THE_LEAF) {
	fprintf (stderr, "block %d is not a leaf\n", block);
	return 0;
    }

    len = pack_leaf (bh, buf);
    fwrite (buf, len, 1, stdout);

    free (buf);
    close (fd);
    return 0;
}


int main (int argc, char ** argv)
{
    if (argc == 3 && !strcmp (argv[1], "unpack"))
	return do_unpack (argv[2]);

    if (argc == 4 && !strcmp (argv[1], "pack"))
	return do_pack (argv[2], atoi (argv[3]));

    fprintf (stderr, "Usage: \n\t%s pack filename block\nor\n"
	     "\t%s unpack filename\n", argv[0], argv[0]);
    return 0;
}

#endif
