/*
 * Copyright 2000 by Hans Reiser, licensing governed by reiserfs/README
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <asm/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/vfs.h>


#include "io.h"
#include "misc.h"
#include "reiserfs_lib.h"
#include "../version.h"



#define print_usage_and_exit() die ("Usage: %s [-b block-to-print][-idc] device\n\
-i Causes to print all items of a leaf\n\
-d                 content of directory items\n\
-c                 content of direct items\n\
-m                 bitmap blocks\n", argv[0]);



struct reiserfs_fsstat {
    int nr_internals;
    int nr_leaves;
    int nr_files;
    int nr_directories;
    int nr_unformatted;
} g_stat_info;


/*
 *  modes
 */
#define DO_DUMP 1  /* not a real dump, just dumping contents of tree nodes */
#define DO_PACK 2  /* helps debugging fsck */
#define DO_CORRUPT 3 /* used to make filesystem corruption and then test fsck */
#define DO_SCAN 4

int mode = DO_DUMP;

/*
 *  options
 */
int opt_print_regular_file_content = 0;/* -c */
int opt_print_details = 0;	/* -d */
int opt_print_leaf_items = 0;		/* -i */
int opt_print_objectid_map = 0;	/* -o */
int opt_print_block_map = 0;		/* -m */
/* when you want print one block specify -b # */
int opt_block_to_print = -1;
/* when you want to corrupt block specify -C # */
int opt_block_to_corrupt = -1;
int opt_pack = 0;	/* -P will produce output that should be |gzip -c > whatever.gz */
			/* -p will calculate number of bytes needed to transfer the partition */
int opt_print_journal;
int opt_pack_all = 0;


int print_mode (void)
{
    int mode = 0;

    if (opt_print_leaf_items == 1)
	mode |= PRINT_LEAF_ITEMS;
    if (opt_print_details == 1)
	mode |= (PRINT_LEAF_ITEMS | PRINT_ITEM_DETAILS);
    if (opt_print_regular_file_content == 1)
	mode |= (PRINT_LEAF_ITEMS | PRINT_DIRECT_ITEMS);
    return mode;
}


static void print_disk_tree (reiserfs_filsys_t fs, int block_nr)
{
    struct buffer_head * bh;

    bh = bread (fs->s_dev, block_nr, fs->s_blocksize);
    if (is_internal_node (bh)) {
	int i;
	struct disk_child * dc;

	g_stat_info.nr_internals ++;
	print_block (bh, print_mode (), -1, -1);
      
	dc = B_N_CHILD (bh, 0);
	for (i = 0; i <= B_NR_ITEMS (bh); i ++, dc ++)
	    print_disk_tree (fs, dc->dc_block_number);
      
    } else if (is_leaf_node (bh)) {
	g_stat_info.nr_leaves ++;
	print_block (bh, print_mode (), -1, -1);
    } else {
	print_block (bh, print_mode (), -1, -1);
	die ("print_disk_tree: bad block type");
    }
    brelse (bh);
}



static void print_one_block (reiserfs_filsys_t fs, int block)
{
    struct buffer_head * bh;
    
    if (test_bit (block % (fs->s_blocksize * 8), 
		  SB_AP_BITMAP (fs)[block / (fs->s_blocksize * 8)]->b_data))
	fprintf (stderr, "%d is used in true bitmap\n", block);
    else
	fprintf (stderr, "%d is free in true bitmap\n", block);
    
    bh = bread (fs->s_dev, block, fs->s_blocksize);
    if (!bh) {
	printf ("print_one_block: bread fialed\n");
	return;
    }

    if (opt_pack) {
	pack_one_block (fs, bh);
	brelse (bh);
	return;
    }

    if (who_is_this (bh->b_data, fs->s_blocksize) != THE_UNKNOWN)
	print_block (bh, PRINT_LEAF_ITEMS | PRINT_ITEM_DETAILS | 
		     (opt_print_regular_file_content == 1 ? PRINT_DIRECT_ITEMS : 0), -1, -1);
    else
	printf ("Looks like unformatted\n");
    brelse (bh);
    return;
}


static void corrupt_clobber_hash (char * name, struct item_head * ih, 
				  struct reiserfs_de_head * deh)
{
    printf ("\tCorrupting deh_offset of entry \"%s\" of [%u %u]\n", name,
	    ih->ih_key.k_dir_id, ih->ih_key.k_objectid);
    deh->deh_offset = 700;
}


static void corrupt_delete_entry (reiserfs_filsys_t fs,
				  char * name,
				  struct item_head * ih,
				  struct reiserfs_de_head * deh)
{
    struct key key;

    printf ("\tDeleting entry \"%s\" of [%u %u] - ", name,
	    ih->ih_key.k_dir_id, ih->ih_key.k_objectid);
	    
    key.k_dir_id = ih->ih_key.k_dir_id;
    key.k_objectid = ih->ih_key.k_objectid;
    key.u.k_offset_v1.k_offset = deh->deh_offset;
    key.u.k_offset_v1.k_uniqueness = 500;
    if (reiserfs_remove_entry (fs, &key))
	printf ("failed\n");
    else
	printf ("ok\n");
}


/* this reads list of desired corruptions from stdin and perform the
   corruptions. Format of that list:

   C name objectid - cut entry "name" form directory objectid
   H name objectid - clobber hash of entry "name" of directory objectid
*/
static void do_corrupt_one_block (reiserfs_filsys_t fs, int block)
{
    struct buffer_head * bh;
    int i, j;
    struct item_head * ih;
    char * line = 0;
    int n = 0;
    char code, name [100];
    __u32 objectid;
    

    if (test_bit (block % (fs->s_blocksize * 8), 
		  SB_AP_BITMAP (fs)[block / (fs->s_blocksize * 8)]->b_data))
	fprintf (stderr, "%d is used in true bitmap\n", block);
    else
	fprintf (stderr, "%d is free in true bitmap\n", block);
    
    bh = bread (fs->s_dev, block, fs->s_blocksize);
    if (!bh) {
	printf ("corrupt_one_block: bread fialed\n");
	return;
    }

    if (who_is_this (bh->b_data, fs->s_blocksize) != THE_LEAF) {
	printf ("Can not corrupt not a leaf node\n");
	brelse (bh);
	return;
    }

    printf ("Corrupting block %u..\n", bh->b_blocknr);

    while (getline (&line, &n, stdin) != -1) {
	if (line [0] == '#' || line [0] == '\n')
	    continue;
	if (sscanf (line, "%c %s %u\n", &code, name, &objectid) != 3) {
	    printf ("Wrong format\n");
	    continue;
	}

	ih = B_N_PITEM_HEAD (bh, 0);
	for (i = 0; i < node_item_number (bh); i ++, ih ++) {
	    struct reiserfs_de_head * deh;

	    if (ih->ih_key.k_objectid != objectid || !is_direntry_ih (ih))
		continue;

	    deh = B_I_DEH (bh, ih);

	    for (j = 0; j < ih_entry_count (ih); j ++, deh ++) {
		/* look for proper entry */
		if (name_length (ih, deh, j) != strlen (name) ||
		    strncmp (name, name_in_entry (deh, j), strlen (name)))
		    continue;

		/* ok, required entry found, make a corruption */
		switch (code) {
		case 'C': /* cut entry */
		    corrupt_delete_entry (fs, name, ih, deh);
		    if (!B_IS_IN_TREE (bh)) {
			printf ("NOTE: block is deleted from the tree\n");
			exit (0);
		    }
		    goto cont;
		    break;
		case 'H': /* clobber hash */
		    corrupt_clobber_hash (name, ih, deh);
		    goto cont;
		    break;
		default:
		    printf ("Unknown command found\n");
		}
		mark_buffer_dirty (bh);
	    }
	}
    cont:
    }
    free (line);
    printf ("Done\n");
    brelse (bh);
    return;
}


static char * parse_options (int argc, char * argv [])
{
    int c;
    char * tmp;
  
    while ((c = getopt (argc, argv, "b:C:icdmoMpPaAjs")) != EOF) {
	switch (c) {
	case 'b':	/* print a single node */
	    opt_block_to_print = strtol (optarg, &tmp, 0);
	    if (*tmp)
		die ("parse_options: bad block size");
	    break;
	case 'C':
	    mode = DO_CORRUPT;
	    opt_block_to_corrupt = strtol (optarg, &tmp, 0);
	    if (*tmp)
		die ("parse_options: bad block size");
	    break;
	    
	case 'p':
	case 'P':
	    mode = DO_PACK;
	    break;

	case 'i':	/* print items of a leaf */
	    opt_print_leaf_items = 1; break;

	case 'd':	/* print directories */
	    opt_print_details = 1; break;

	case 'c':	/* print contents of a regular file */
	    opt_print_regular_file_content = 1; break;

	case 'o':	/* print a objectid map */
	    opt_print_objectid_map = 1; break;

	case 'm':	/* print a block map */
	    opt_print_block_map = 1;  break;
	case 'M':	/* print a block map with details */
	    opt_print_block_map = 2;  break;
	case 'j':
	    opt_print_journal = 1; break; /* print journal */
	    
	case 's':
	    mode = DO_SCAN; break; /* read the device and print what reiserfs blocks were found */
	}
    }
    if (optind != argc - 1)
	/* only one non-option argument is permitted */
	print_usage_and_exit();
  
    return argv[optind];
}



/* print all valid transactions and found dec blocks */
static void print_journal (struct super_block * s)
{
    struct buffer_head * d_bh, * c_bh;
    struct reiserfs_journal_desc * desc ;
    struct reiserfs_journal_commit *commit ;
    int end_journal;
    int start_journal;
    int i, j;
    int first_desc_block = 0;
    int wrapped = 0;
    int valid_transactions = 0;

    start_journal = SB_JOURNAL_BLOCK (s);
    end_journal = start_journal + JOURNAL_BLOCK_COUNT;
    printf ("Start scanning from %d\n", start_journal);

    d_bh = 0;
    desc = 0;
    for (i = start_journal; i < end_journal; i ++) {
	d_bh = bread (s->s_dev, i, s->s_blocksize);
	if (who_is_this (d_bh->b_data, d_bh->b_size) == THE_JDESC) {
	    int commit_block;

	    if (first_desc_block == 0)
		/* store where first desc block found */
		first_desc_block = i;

	    print_block (d_bh); /* reiserfs_journal_desc structure will be printed */
	    desc = (struct reiserfs_journal_desc *)(d_bh->b_data);

	    commit_block = d_bh->b_blocknr + desc->j_len + 1;
	    if (commit_block >= end_journal) {
		printf ("-- wrapped?");
		wrapped = 1;
		break;
	    }

	    c_bh = bread (s->s_dev, commit_block, s->s_blocksize);
	    commit = bh_commit (c_bh);
	    if (does_desc_match_commit (desc, commit)) {
		printf ("commit block %d (trans_id %ld, j_len %ld) does not match\n", commit_block,
			commit->j_trans_id, commit->j_len);
		brelse (c_bh) ;
		brelse (d_bh);
		continue;
	    }

	    valid_transactions ++;
	    printf ("(commit block %d) - logged blocks (", commit_block);
	    for (j = 0; j < desc->j_len; j ++) {
		unsigned long block;

		if (j < JOURNAL_TRANS_HALF)
		    block = le32_to_cpu (desc->j_realblock[j]);
		else
		    block = le32_to_cpu (commit->j_realblock[i - JOURNAL_TRANS_HALF]);
			
		if (not_journalable (s, block))
		    printf (" xxxx");
		else {
		    printf (" %ld", desc->j_realblock[j]);
		    if (block_of_bitmap (s, desc->j_realblock[j]))
			printf ("(bmp)");
		}
	    }
	    printf (")\n");
	    i += desc->j_len + 1;
	    brelse (c_bh);
	}
	brelse (d_bh);
    }
    
    if (wrapped) {
	c_bh = bread (s->s_dev, first_desc_block - 1, s->s_blocksize);
	commit = bh_commit (c_bh);
	if (does_desc_match_commit (desc, commit)) {
	    printf ("No! commit block %d (trans_id %ld, j_len %ld) does not match\n",
		    first_desc_block - 1, commit->j_trans_id, commit->j_len);
	} else {
	    printf ("Yes! (commit block %d) - logged blocks (\n", first_desc_block - 1);
	    for (j = 0; j < desc->j_len; j ++) {
		unsigned long block;

		if (j < JOURNAL_TRANS_HALF)
		    block = le32_to_cpu (desc->j_realblock[j]);
		else
		    block = le32_to_cpu (commit->j_realblock[i - JOURNAL_TRANS_HALF]);
			
		if (not_journalable (s, block))
		    printf (" xxxx");
		else {
		    printf (" %ld", desc->j_realblock[j]);
		    if (block_of_bitmap (s, desc->j_realblock[j]))
			printf ("(bmp)");
		}
	    }
	    printf ("\n");
	}
	brelse (c_bh) ;
	brelse (d_bh);
    }

    printf ("%d valid transactions found\n", valid_transactions);

    {
	struct buffer_head * bh;
	struct reiserfs_journal_header * j_head;

	bh = bread (s->s_dev, SB_JOURNAL_BLOCK (s) + rs_journal_size (s->s_rs),
		    s->s_blocksize);
	j_head = (struct reiserfs_journal_header *)(bh->b_data);

	printf ("#######################\nJournal header:\n"
		"j_last_flush_trans_id %ld\n"
		"j_first_unflushed_offset %ld\n"
		"j_mount_id %ld\n", j_head->j_last_flush_trans_id, j_head->j_first_unflushed_offset,
		j_head->j_mount_id);
	brelse (bh);
    }
}


static void do_pack (reiserfs_filsys_t fs)
{
    if (opt_block_to_print != -1)
	pack_one_block (fs, opt_block_to_print);
    else
	pack_partition (fs);
	
}

/* FIXME: statistics does not work */
static void do_dump_tree (reiserfs_filsys_t fs)
{
    if (opt_block_to_print != -1) {
	print_one_block (fs, opt_block_to_print);
	return;
    }

    print_block (SB_BUFFER_WITH_SB (fs));
    
    if (opt_print_journal)
	print_journal (fs);
    
    if (opt_print_objectid_map == 1)
	print_objectid_map (fs);
    
    if (opt_print_block_map)
	print_bmap (fs, opt_print_block_map == 1 ? 1 : 0);
    
    if (opt_print_regular_file_content || opt_print_details ||
	opt_print_leaf_items) {
	print_disk_tree (fs, get_root_block (fs));
	
	/* print the statistic */
	printf ("File system uses %d internal + %d leaves + %d unformatted nodes = %d blocks\n",
		g_stat_info.nr_internals, g_stat_info.nr_leaves, g_stat_info.nr_unformatted, 
		g_stat_info.nr_internals + g_stat_info.nr_leaves + g_stat_info.nr_unformatted);
    }
}


static void do_scan (reiserfs_filsys_t fs)
{
    unsigned long i;
    struct buffer_head * bh;
    int type;

    for (i = 0; i < SB_BLOCK_COUNT (fs); i ++) {
	bh = bread (fs->s_dev, i, fs->s_blocksize);
	if (!bh) {
	    printf ("could not read block %lu\n", i);
	    continue;
	}
	type = who_is_this (bh->b_data, bh->b_size);
	switch (type) {
	case THE_JDESC:
	    printf ("block %lu is journal descriptor\n", i);
	    break;
	case THE_SUPER:
	    printf ("block %lu is reiserfs super block\n", i);
	    break;
	case THE_INTERNAL:
	    printf ("block %lu is reiserfs internal node\n", i);
	    break;
	case THE_LEAF:
	    printf ("block %lu is reiserfs leaf node\n", i);
	    break;
	}
	brelse (bh);
    }
}


/* FIXME: need to open reiserfs filesystem first */
int main (int argc, char * argv[])
{
    char * file_name;
    reiserfs_filsys_t fs;
    int error;

    printf ("%d %d\n", sizeof (struct reiserfs_super_block_v1), 
	    sizeof (struct reiserfs_super_block));
    print_banner ("debugreiserfs");
 
    file_name = parse_options (argc, argv);

    fs = reiserfs_open (file_name, O_RDWR, &error, 0);
    if (!fs) {
	fprintf (stderr, "\n\ndumpreiserfs: can not open reiserfs on \"%s\": %s\n\n",
		 file_name, error ? strerror (error) : "there is no one");
	return 0;
    }

    switch (mode) {
    case DO_PACK:
	do_pack (fs);
	break;

    case DO_CORRUPT:
	do_corrupt_one_block (fs, opt_block_to_corrupt);
	break;

    case DO_DUMP:
	do_dump_tree (fs);
	break;

    case DO_SCAN:
	do_scan (fs);
	break;
    }

    reiserfs_close (fs);
    return 0;
}






