/*
 * Copyright 2000 by Hans Reiser, licensing governed by reiserfs/README
 */

#include "debugreiserfs.h"


reiserfs_filsys_t fs;


#define print_usage_and_exit() die ("Usage: %s [options] \
 device\n\n\
Options:\n\
  -d\tprint block details
  -m\tprint bitmap blocks\n\
  -j\tprint journal\n\
  -o\tprint objectid map\n\n\
  -p\tsend filesystem metadata to stdout\n\
  -s\tscan device either for specific key or for any metadata\n\
  -n\tscan device for specific name in reiserfs directories\n\
  -S\tgo through whole device when running -p, -s or -n\n\
  -b bitmap_file\trunning -p, -s or -n read blocks marked in this bitmap only\n\
  -t\tstat the device\n\
  -B block\tblock to print\n\n\
  -q\tno speed info (for -p, -s and -n)\n\n", argv[0]);



#if 1
struct reiserfs_fsstat {
    int nr_internals;
    int nr_leaves;
    int nr_files;
    int nr_directories;
    int nr_unformatted;
} g_stat_info;
#endif



static void print_disk_tree (reiserfs_filsys_t fs, int block_nr)
{
    struct buffer_head * bh;

    bh = bread (fs->s_dev, block_nr, fs->s_blocksize);
    if (!bh) {
	die ("Could not read block %d\n", block_nr);
    }
    if (is_internal_node (bh)) {
	int i;
	struct disk_child * dc;

	g_stat_info.nr_internals ++;
	print_block (stdout, fs, bh, debug_mode (fs) & PRINT_DETAILS, -1, -1);
      
	dc = B_N_CHILD (bh, 0);
	for (i = 0; i <= B_NR_ITEMS (bh); i ++, dc ++)
	    print_disk_tree (fs, dc->dc_block_number);
      
    } else if (is_leaf_node (bh)) {
	g_stat_info.nr_leaves ++;
	print_block (stdout, fs, bh, debug_mode (fs) & PRINT_DETAILS, -1, -1);
    } else {
	print_block (stdout, fs, bh, debug_mode (fs) & PRINT_DETAILS, -1, -1);
	die ("print_disk_tree: bad block type");
    }
    brelse (bh);
}



void pack_one_block (reiserfs_filsys_t fs, unsigned long block);
static void print_one_block (reiserfs_filsys_t fs, int block)
{
    struct buffer_head * bh;
    
    if (test_bit (block % (fs->s_blocksize * 8), 
		  SB_AP_BITMAP (fs)[block / (fs->s_blocksize * 8)]->b_data))
	fprintf (stderr, "%d is used in true bitmap\n", block);
    else
	fprintf (stderr, "%d is free in true bitmap\n", block);
    
    bh = bread (fs->s_dev, block, fs->s_blocksize);
    if (!bh) {
	printf ("print_one_block: bread fialed\n");
	return;
    }

    if (debug_mode (fs) == DO_PACK) {
	pack_one_block (fs, bh->b_blocknr);
	brelse (bh);
	return;
    }

    if (who_is_this (bh->b_data, fs->s_blocksize) != THE_UNKNOWN)
	print_block (stdout, fs, bh, PRINT_DETAILS, -1, -1);
    else
	printf ("Looks like unformatted\n");
    brelse (bh);
    return;
}




/* debugreiserfs -p or -P compresses reiserfs meta data: super block, journal,
   bitmap blocks and blocks looking like leaves. It may save "bitmap" of
   blocks they packed in the file of special format. Reiserfsck can then load
   "bitmap" saved in that file and build the tree of blocks marked used in
   that "bitmap" */
char * where_to_save;


static char * parse_options (struct debugreiserfs_data * data, int argc, char * argv [])
{
    int c;
    char * tmp;
  
    data->scan_area = USED_BLOCKS;
    data->mode = DO_DUMP;

    while ((c = getopt (argc, argv, "a:b:CSB:psn:Nrdomjqt")) != EOF) {
	switch (c) {
	case 'a': /* -r will read this, -n and -N will write to it */
	    asprintf (&data->map_file, "%s", optarg);
	    break;

	case 'b':
	    /* will load bitmap from a file and read only blocks
               marked in it. This is for -p and -s */
	    asprintf (&data->input_bitmap, "%s", optarg);
	    data->scan_area = EXTERN_BITMAP;
	    break;

	case 'S':
	    /* have debugreiserfs -p or -s to read all the device */
	    data->scan_area = ALL_BLOCKS;
	    break;

	case 'B':	/* print a single node */
	    data->block = strtol (optarg, &tmp, 0);
	    if (*tmp)
		die ("parse_options: bad block number");
	    break;

	case 'C':
	    data->mode = DO_CORRUPT;
	    /*
	    data->block = strtol (optarg, &tmp, 0);
	    if (*tmp)
		die ("parse_options: bad block number");
	    */
	    break;
	    
	case 'p':
	    data->mode = DO_PACK;
	    break;

	case 't':
	    data->mode = DO_STAT;
	    break;

	case 's':
	    /* read the device and print what reiserfs blocks were found */
	    data->mode = DO_SCAN;
	    break;

	case 'n': /* scan for names matching a specified pattern */
	    data->mode = DO_SCAN_FOR_NAME;
	    asprintf (&data->pattern, "%s", optarg);
	    break;

	case 'N': /* search name in the tree */
	    data->mode = DO_LOOK_FOR_NAME;
	    break;

	case 'r':
	    data->mode = DO_RECOVER;
	    break;

	case 'd':
	    /*  print leaf details */
	    data->options |= PRINT_DETAILS;
	    break;

	case 'o':
	    /* print objectid map */
	    data->options |= PRINT_OBJECTID_MAP;
	    break;

	case 'm':	/* print a block map */
	case 'M':	/* print a block map with details */
	    data->options |= PRINT_BITMAP;
	    break;

	case 'j':
	    data->options |= PRINT_JOURNAL;
	    break;

	case 'q':
	    /* this makes packing to not show speed info during -p or -P */
	    data->options |= BE_QUIET;
	    break;
	}
    }

    if (optind != argc - 1)
	/* only one non-option argument is permitted */
	print_usage_and_exit();
  
    asprintf (&data->device_name, "%s", argv[optind]);
    return argv[optind];
}


void pack_partition (reiserfs_filsys_t fs);

static void do_pack (reiserfs_filsys_t fs)
{
    if (certain_block (fs))
	pack_one_block (fs, certain_block (fs));
    else
	pack_partition (fs);
	
}

/* FIXME: statistics does not work */
static void do_dump_tree (reiserfs_filsys_t fs)
{
    if (certain_block (fs)) {
	print_one_block (fs, certain_block (fs));
	return;
    }

    print_block (stdout, fs, SB_BUFFER_WITH_SB (fs));
    
    if (data (fs)->options & PRINT_JOURNAL)
	print_journal (fs);
    
    if (data (fs)->options & PRINT_OBJECTID_MAP)
	print_objectid_map (stdout, fs);
    
    if (data (fs)->options & PRINT_BITMAP)
	print_bmap (stdout, fs, 1/*opt_print_block_map == 1 ? 1 : 0*/);
    
    if (data (fs)->options & PRINT_DETAILS) {
	print_disk_tree (fs, SB_ROOT_BLOCK (fs));
	
	/* print the statistic */
	printf ("File system uses %d internal + %d leaves + %d unformatted nodes = %d blocks\n",
		g_stat_info.nr_internals, g_stat_info.nr_leaves, g_stat_info.nr_unformatted, 
		g_stat_info.nr_internals + g_stat_info.nr_leaves + g_stat_info.nr_unformatted);
    }
}




static void init_bitmap (reiserfs_filsys_t fs)
{
    FILE * fp;
    unsigned long block_count;

    block_count = SB_BLOCK_COUNT (fs);

    switch (scan_area (fs)) {
    case ALL_BLOCKS:
	debug_bitmap (fs) = reiserfs_create_bitmap (block_count);
	reiserfs_bitmap_fill (debug_bitmap (fs));
	reiserfs_warning (stderr, "Whole device (%d blocks) is to be scanned\n", 
			  reiserfs_bitmap_ones (debug_bitmap (fs)));	
	break;
	
    case USED_BLOCKS:
	reiserfs_warning (stderr, "Loading on-disk bitmap .. ");
	debug_bitmap (fs) = reiserfs_create_bitmap (block_count);
	reiserfs_fetch_disk_bitmap (debug_bitmap (fs), fs);
	reiserfs_warning (stderr, "%d bits set - done\n",
			  reiserfs_bitmap_ones (debug_bitmap (fs)));
	break;
	
    case EXTERN_BITMAP:
	fp = fopen (input_bitmap_file_name(fs), "r");
	if (!fp)
	    reiserfs_panic ("init_bitmap: could not load bitmap: %m\n");
	
	debug_bitmap (fs) = reiserfs_bitmap_load (fp);
	if (!debug_bitmap (fs))
	    reiserfs_panic ("could not load fitmap from \"%s\"", 
			    input_bitmap_file_name(fs));
	reiserfs_warning (stderr, "%d blocks marked in the given bitmap\n",
			  reiserfs_bitmap_ones (debug_bitmap (fs)));
	fclose (fp);
	break;

    default:
	reiserfs_panic ("No area to scan specified");
    }
}


/* FIXME: need to open reiserfs filesystem first */
int main (int argc, char * argv[])
{
    char * file_name;
    int error;
    struct debugreiserfs_data * data;

    print_banner ("debugreiserfs");
 
    data = getmem (sizeof (struct debugreiserfs_data));
    file_name = parse_options (data, argc, argv);

    fs = reiserfs_open (file_name, O_RDONLY, &error, data);
    if (!fs) {
	fprintf (stderr, "\n\ndumpreiserfs: can not open reiserfs on \"%s\": %s\n\n",
		 file_name, error ? strerror (error) : "there is no one");
	return 0;
    }

    init_bitmap (fs);

    switch (debug_mode (fs)) {
    case DO_STAT:
	do_stat (fs);
	break;

    case DO_PACK:
	do_pack (fs);
	break;

    case DO_CORRUPT:
	reiserfs_reopen (fs, O_RDWR);
	do_corrupt_one_block (fs);
	break;

    case DO_DUMP:
	do_dump_tree (fs);
	break;

    case DO_SCAN:
    case DO_SCAN_FOR_NAME:
    case DO_LOOK_FOR_NAME:
	do_scan (fs);
	break;

    case DO_RECOVER:
	do_recover (fs);
	break;

    case DO_TEST:
	/*do_test (fs);*/
	break;
    }

    reiserfs_close (fs);
    return 0;
}
