/*
 * Copyright 1996-2000  Hans Reiser
 */
#include "fsck.h"
#include <getopt.h>
#include <sys/resource.h>

#include "../version.h"


reiserfs_filsys_t fs;


#define print_usage_and_exit() {\
fsck_progress ("Usage: %s [options] "\
" device\n"\
"\n"\
"Options:\n\n"\
"  --check\t\tconsistency checking (default)\n"\
"  --rebuild-sb\t\tsuper block checking and rebuilding if needed\n"\
"  --rebuild-tree\tforce fsck to rebuild filesystem from scratch\n"\
"  \t\t\t(takes a long time)\n"\
"  -i | --interactive\tmake fsck to stop after every stage\n"\
"  -q | --quiet\t\tno speed info\n"\
"  -l | --logfile logfile\n"\
"  \t\t\tmake fsck to complain to specifed file\n"\
"  -n | --nolog\t\tmake fsck to not complain\n"\
"  -b | --scan-marked-in-bitmap file\n"\
"  \t\t\tbuild tree of blocks marked in the bitmapfile\n"\
"  -c | --create-bitmap-file\n"\
"  \t\t\tsave bitmap of found leaves\n"\
"  -d | --create-passes-dump\n"\
"  \t\t\tdump all information needed for the next pass in temp file\n"\
"  -S | --scan-whole-partition\n"\
"  -x | --fix-fixable\tfix corruptions which can be fixed w/o --rebuild-tree\n"\
"  -o | --fix-non-critical\n"\
"  \t\t\tfix strange modes, file sizes to real size and\n"\
"  \t\t\trelocate files using busy objectids\n"\
"  -h | --hash hashname\n"\
"  -g | --background\n"\
"  -V and -a\t\tprints version and exits\n"\
"  -p and -r\t\tignored\n", argv[0]);\
  exit (16);\
}



/* fsck is called with one non-optional argument - file name of device
   containing reiserfs. This function parses other options, sets flags
   based on parsing and returns non-optional argument */
static char * parse_options (struct fsck_data * data, int argc, char * argv [])
{
    int c;
    static int mode = FSCK_CHECK;

    data->u.rebuild.scan_area = USED_BLOCKS;
    while (1) {
	static struct option options[] = {
	    /* modes */
	    {"check", no_argument, &mode, FSCK_CHECK},
	    {"rebuild-sb", no_argument, &mode, FSCK_SB},
	    {"rebuild-tree", no_argument, &mode, FSCK_REBUILD},
/*
	    {"fast-rebuild", no_argument, &opt_fsck_mode, FSCK_FAST_REBUILD},
*/

	    /* options */
	    {"logfile", required_argument, 0, 'l'},
	    {"interactive", no_argument, 0, 'i'},
	    {"fix-fixable", no_argument, 0, 'x'},
	    {"fix-non-critical", no_argument, 0, 'o'},
	    {"quiet", no_argument, 0, 'q'},
	    {"nolog", no_argument, 0, 'n'},
	    
	    /* if file exists ad reiserfs can be load of it - only
               blocks marked used in that bitmap will be read */
	    {"scan-marked-in-bitmap", required_argument, 0, 'b'},

	    /* */
	    {"create-leaf-bitmap", required_argument, 0, 'c'},

	    {"create-passes-dump", required_argument, 0, 'd'},
	
	    /* all blocks will be read */
	    {"scan-whole-partition", no_argument, 0, 'S'},
	    /* useful for -S */
	    {"hash", required_argument, 0, 'h'},
	    
	    /* start reiserfsck in background and exit */
	    {"background", no_argument, 0, 'g'},

	    {0, 0, 0, 0}
	};
	int option_index;
      
	c = getopt_long (argc, argv, "iql:nb:Sc:xod:h:garpVt:y",
			 options, &option_index);
	if (c == -1)
	    break;
	
	switch (c) {
	case 0:
	    /* long option specifying fsck mode is found */
	    break;

	case 'i': /* --interactive */
	    data->options |= OPT_INTERACTIVE;
	    break;

	case 'q': /* --quiet */
	    data->options |= OPT_QUIET;
	    break;

	case 'l': /* --logfile */
	    asprintf (&data->log_file_name, "%s", optarg);
	    data->log = fopen (optarg, "w");
	    if (!data->log)
		fprintf (stderr, "reiserfsck: could not open \'%s\': %m", optarg);
	    break;

	case 'n': /* --nolog */
	    data->options |= OPT_SILENT;
	    break;

	case 'b': /* --scan-marked-in-bitmap */
	    /* will try to load a bitmap from a file and read only
               blocks marked in it. That bitmap could be created by
               previous run of reiserfsck with -c */
	    asprintf (&data->u.rebuild.bitmap_file_name, "%s", optarg);
	    data->u.rebuild.scan_area = EXTERN_BITMAP;
	    break;

	case 'S': /* --scan-whole-partition */
	    data->u.rebuild.scan_area = ALL_BLOCKS;
	    break;

	case 'c': /* --create-leaf-bitmap */
	    asprintf (&data->u.rebuild.new_bitmap_file_name, "%s", optarg);
	    data->options |= OPT_SAVE_EXTERN_BITMAP;
	    break;
	    
	case 'd': /* --create-passes-dump */
	    asprintf (&data->u.rebuild.passes_dump_file_name, "%s", optarg);
	    data->options |= OPT_SAVE_PASSES_DUMP;
	    break;
	
	case 'x': /* --fix-fixable */
	    data->options |= OPT_FIX_FIXABLE;
	    break;

	case 'o': /* --fix-non-critical */
	    data->options |= OPT_FIX_NON_CRITICAL;
	    break;

	case 'h': /* --hash: suppose that this hash was used on a filesystem */
	    asprintf (&data->u.rebuild.defined_hash, "%s", optarg);
	    if (name2func (data->u.rebuild.defined_hash) == 0)
		reiserfs_panic ("reiserfsck: unknown hash is defined: %s",
				data->u.rebuild.defined_hash);
	    data->options |= OPT_HASH_DEFINED;
	    break;

	case 'g': /* --background */
	    data->options |= OPT_BACKGROUND;
	    break;

	case 'r':
	case 'p':
	    break;
	    
	case 'V':
	case 'a':
	    mode = DO_NOTHING;
	    break;
#if 0
	case 'u': /* --undo */
	    
	    /* not ready */
	    data->options |= OPT_DONT_WRITE;
	    break;
#endif

	case 't':
	    mode = DO_TEST;
	    data->u.rebuild.test = atoi (optarg);
	    break;


	default:
	    print_usage_and_exit();
	}
    }

    if (optind != argc - 1 && mode != DO_NOTHING)
	/* only one non-option argument is permitted */
	print_usage_and_exit();

    data->mode = mode;
    if (!data->log)
	data->log = stdout;
    
    return argv[optind];
}


#define REBUILD_WARNING \
"\nThis is an experimental version of reiserfsck, MAKE A BACKUP FIRST!\n\
Don't run this program unless something is broken. \n\
Some types of random FS damage can be recovered\n\
from by this program, which basically throws away the internal nodes\n\
of the tree and then reconstructs them.  This program is for use only\n\
by the desperate, and is of only beta quality.  Email\n\
reiserfs@devlinux.com with bug reports. \nWill rebuild the filesystem tree\n"

/* 
   warning #2
   you seem to be running this automatically.  you are almost
   certainly doing it by mistake as a result of some script that
   doesn't know what it does.  doing nothing, rerun without -p if you
   really intend to do this.  */

void warn_what_will_be_done (struct fsck_data * data)
{
    FILE * warn_to;

    warn_to = (data->progress ?: stderr);

    reiserfs_warning (warn_to, "\n");

    /* warn about fsck mode */
    switch (data->mode) {
    case FSCK_CHECK:
	reiserfs_warning (warn_to,
			  "Will read-only check consistency of the partition\n");
	if (data->options & OPT_FIX_FIXABLE)
	    reiserfs_warning (fsck_progress_file(fs),
			      "\tWill fix what can be fixed w/o --rebuild-tree\n");
	break;

    case FSCK_SB:
	reiserfs_warning (warn_to,
			  "Will check SB and rebuild if it is needed\n");
	break;

    case FSCK_REBUILD:
	reiserfs_warning (warn_to, REBUILD_WARNING);
	if (data->options & OPT_SAVE_PASSES_DUMP) {
	    reiserfs_warning (warn_to,
			      "\tWill run only 1 step of rebuilding, write state file '%s' and exit\n",
			      data->u.rebuild.passes_dump_file_name);
	} else if (data->options & OPT_INTERACTIVE)
	    reiserfs_warning (warn_to,
			      "\tWill stop after every stage and ask for "
			      "confirmation before continuing\n");
	
	if (data->options & OPT_SAVE_EXTERN_BITMAP)
	    /* will die */
	    fsck_progress ("Will save list of found leaves in '%s'\n",
			   data->u.rebuild.new_bitmap_file_name);

	if (data->u.rebuild.bitmap_file_name)
	    reiserfs_warning (warn_to,
			      "\tWill try to load bitmap of leaves from file '%s'\n",
			      data->u.rebuild.bitmap_file_name);

	if (data->options & OPT_FIX_NON_CRITICAL)
	    reiserfs_warning (warn_to,
			      "\tWill fix following non-critical things:\n"
			      "\t\tunknown file modes will be set to regular files\n"
			      "\t\tfile sizes will be set to real file size\n"
			      "\t\tfiles sharing busy inode number will be relocated\n");

	if (data->options & OPT_HASH_DEFINED)
	    reiserfs_warning (warn_to,
			      "\tSuppose \"%s\" hash is in use\n",
			      data->u.rebuild.defined_hash);
	break;
    }

    reiserfs_warning (warn_to,
		      "Will put log info to '%s'\n", (data->log == stdout) ?
		      "stdout" : 
		      (data->log_file_name ?: "fsck.run"));

    if (!user_confirmed (warn_to, 
			 "Do you want to run this program?[N/Yes] (note need to type Yes):", "Yes\n"))
	exit (0);
}



static void reset_super_block (reiserfs_filsys_t fs)
{
    set_free_blocks (fs->s_rs, SB_BLOCK_COUNT (fs));
    set_root_block (fs->s_rs, ~0);
    set_tree_height (fs->s_rs, ~0);

    /* make file system invalid unless fsck done () */
    set_state (fs->s_rs, REISERFS_ERROR_FS);

    if (is_reiser2fs_magic_string (fs->s_rs)) {
	set_version (fs->s_rs, REISERFS_VERSION_2);
    }
    if (is_reiserfs_magic_string (fs->s_rs)) {
	set_version (fs->s_rs, REISERFS_VERSION_1);
    }

    if (fsck_hash_defined (fs)) {
	fs->s_hash_function = name2func (fsck_data (fs)->u.rebuild.defined_hash);
	set_hash (fs->s_rs, func2code (fs->s_hash_function));
    }

    /* objectid map is not touched */

    mark_buffer_dirty (fs->s_sbh);
    bwrite (fs->s_sbh);
}


#define START_FROM_THE_BEGINNING 1
#define START_FROM_PASS_1 2
#define START_FROM_PASS_2 3
#define START_FROM_SEMANTIC 4

/* this decides where to start from  */
static int where_to_start_from (reiserfs_filsys_t fs)
{
    int ret;
    FILE * fp = 0;
    int last_run_state;
    
    last_run_state = fsck_state (fs->s_rs);
    if (last_run_state == 0 || !fsck_restart (fs))
	/**/
	return START_FROM_THE_BEGINNING;
    
    /* ok, we asked to restart - we can only do that if there is file saved
       during previous runs */
    fp = open_file (state_dump_file (fs), "r");
    if (fp == 0) {
	set_fsck_state (fs->s_rs, 0);
	return START_FROM_THE_BEGINNING;
    }

    /* check start and end magics of dump file */
    ret = is_stage_magic_correct (fp);
    
    if (ret <= 0 || ret != last_run_state)
	return START_FROM_THE_BEGINNING;


    switch (last_run_state) {
    case PASS_0_DONE:
	/* skip pass 0 */
	if (!fsck_user_confirmed (fs, "Pass 0 seems done. Start from pass 1?(Yes)",
				  "Yes\n", 1))
	    fsck_exit ("Run without -d then\n");
	
	load_pass_0_result (fp, fs);
	fclose (fp);
	return START_FROM_PASS_1;
	
    case PASS_1_DONE:
	/* skip pass 1 */
	if (!fsck_user_confirmed (fs, "Passes 0 and 1 seems done. Start from pass 2?(Yes)",
				  "Yes\n", 1))
	    fsck_exit ("Run without -d then\n");
	
	load_pass_1_result (fp, fs);
	fclose (fp);
	return START_FROM_PASS_2;
	
    case TREE_IS_BUILT:
	if (!fsck_user_confirmed (fs, "S+ tree of filesystem looks built. Skip rebuilding?(Yes)",
				  "Yes\n", 1))
	    fsck_exit ("Run without -d then\n");
	
	load_pass_2_result (fs);
	fclose (fp);
	return START_FROM_SEMANTIC;
    }
    
    return START_FROM_THE_BEGINNING;
}



static void the_end (reiserfs_filsys_t fs)
{
    /* put bitmap and objectid map on place */
    reiserfs_flush_bitmap (fsck_new_bitmap (fs), fs);
    flush_objectid_map (proper_id_map (fs), fs);

    /* update super block */
    set_fsck_state (fs->s_rs, 0);
    set_free_blocks (fs->s_rs, reiserfs_bitmap_zeros (fsck_new_bitmap (fs)));
    set_state (fs->s_rs, REISERFS_VALID_FS);
    mark_buffer_dirty (SB_BUFFER_WITH_SB (fs));

    /* write all dirty blocks */
    fsck_progress ("Syncing..");
    reiserfs_flush (fs);
    sync ();
    fsck_progress ("done\n");
}


static void rebuild_tree (reiserfs_filsys_t fs)
{
    time_t t;

    if (is_mounted (fs->file_name)) {
	fsck_progress ("rebuild_tree: can not rebuild tree of mounted filesystem\n");
	return;
    }

    reiserfs_reopen (fs, O_RDWR);

    /* FIXME: for regular file take care of of file size */

    /* rebuild starts with journal replaying */
    reiserfs_replay_journal (fs);

    time (&t);
    fsck_progress ("###########\n"
		   "reiserfsck --rebuild-tree started at %s"
		   "###########\n", ctime (&t));

    switch (where_to_start_from (fs)) {
    case START_FROM_THE_BEGINNING:
	reset_super_block (fs);
	pass_0 (fs);

    case START_FROM_PASS_1:
	pass_1 (fs);
	
    case START_FROM_PASS_2:
	pass_2 (fs);

    case START_FROM_SEMANTIC:
	pass_3_semantic (fs);

	/* if --lost+found is set - link unaccessed directories to lost+found
	   directory */
	pass_3a_look_for_lost (fs);
	
	/* 4. look for unaccessed items in the leaves */
	pass_4_check_unaccessed_items ();
	
	the_end (fs);
    }

    time (&t);
    fsck_progress ("###########\n"
		   "reiserfsck finished at %s"
		   "###########\n", ctime (&t));
}


/* check umounted or read-only mounted filesystems only */
static int check_fs (reiserfs_filsys_t fs)
{
    int retval;
    time_t t;

    time (&t);
    fsck_progress ("###########\n"
		   "reiserfsck --check started at %s"
		   "###########\n", ctime (&t));

    if (!is_mounted (fs->file_name)) {
	/* filesystem is not mounted, replay journal before checking */
	reiserfs_reopen (fs, O_RDWR);

	reiserfs_replay_journal (fs);

	reiserfs_reopen (fs, O_RDONLY);
    } else {
	/* filesystem seems mounted. we do not check filesystems mounted with
           r/w permissions */
	if (!is_mounted_read_only (fs->file_name)) {
	    fsck_progress ("Device %s is mounted w/ write permissions, can not check it\n",
			   fs->file_name);
	    reiserfs_close (fs);
	    exit (0);
	}
	fsck_progress ("Filesystem seems mounted read-only. Skipping journal replay..\n");

	if (fsck_fix_fixable (fs)) {
	    fsck_progress ("--fix-fixable ignored\n");
	    fsck_data(fs)->options &= ~OPT_FIX_FIXABLE;
	}
    }


    if (fsck_fix_fixable (fs))
	reiserfs_reopen (fs, O_RDWR);

    check_fs_tree (fs);

    semantic_check ();


    if (fsck_data (fs)->u.check.fatal_corruptions) {
	fsck_progress ("There were found %d corruptions which can be fixed only during --rebuild-tree\n",
		       fsck_data (fs)->u.check.fatal_corruptions);
	retval = 2;
    } else if (fsck_data (fs)->u.check.fixable_corruptions) {
	if (!fsck_fix_fixable (fs))
	    fsck_progress ("There were found %d corruptions which can be fixed with --fix-fixable\n",
			   fsck_data (fs)->u.check.fixable_corruptions);
	retval = 1;
    } else {
	fsck_progress ("No corruptions found\n");
	retval = 0;
    }

    time (&t);
    fsck_progress ("###########\n"
		   "reiserfsck finished at %s"
		   "###########\n", ctime (&t));

    return retval;
}


static void fast_rebuild (reiserfs_filsys_t fs)
{
#ifdef FAST_REBUILD_READY /* and tested */
    if (opt_fsck_mode == FSCK_FAST_REBUILD) {
	__u32 root_block = SB_ROOT_BLOCK(fs);
	reopen_read_write (file_name);
	printf ("Replaying log..");
	reiserfs_replay_journal (fs);
	printf ("done\n");
	if (opt_fsck == 1)
	    printf ("ReiserFS : checking %s\n",file_name);
	else
	    printf ("Rebuilding..\n");

	
	reset_super_block (fs);
	SB_DISK_SUPER_BLOCK(fs)->s_root_block = cpu_to_le32 (root_block);
	init_bitmaps (fs);

	/* 1,2. building of the tree */
	recover_internal_tree(fs);

	/* 3. semantic pass */
	pass3_semantic ();

	/* if --lost+found is set - link unaccessed directories to
           lost+found directory */
       look_for_lost (fs);

	/* 4. look for unaccessed items in the leaves */
	pass4_check_unaccessed_items ();
	
	end_fsck ();
    }
#endif /* FAST REBUILD READY */

    reiserfs_panic ("Fast rebuild is not ready");

}



int main (int argc, char * argv [])
{
    char * file_name;
    struct fsck_data * data;
    struct rlimit rlim = {0xffffffff, 0xffffffff};
    int retval;

    print_banner ("reiserfsck");


    /* this is only needed (and works) when running under 2.4 on regural files */
    if (setrlimit (RLIMIT_FSIZE, &rlim) == -1) {
	reiserfs_warning (stderr, "could not setrlimit: %m\n");
    }

    data = getmem (sizeof (struct fsck_data));

    file_name = parse_options (data, argc, argv);

    if (data->mode == DO_NOTHING) {
	freemem (data);
	return 0;
    }

    if (data->options & OPT_BACKGROUND) {
	/* running in background reiserfsck appends progress information into
           'fsck.run'. Logs get there if log file was not specified*/
	data->options |= OPT_QUIET;
	data->progress = fopen ("fsck.run", "a+");
	if (!data->progress)
	    reiserfs_panic ("reiserfsck: could not open \"fsck.run\"");

	if (data->log == stdout)
	    /* no log file specifed - redirect log into 'fsck.run' */
	    data->log = data->progress;

	retval = fork ();
	if (retval == -1)
	    reiserfs_panic ("reiserfsck: fork failed: %m");
	if (retval != 0) {
	    return 0;
	}
	reiserfs_warning (stderr, "\nreiserfsck is running in background as [%d],\n"
			  "make sure that it get enough confirmation from stdin\n\n",
			  getpid ());
    }


    warn_what_will_be_done (data); /* and ask confirmation Yes */
    fs = reiserfs_open (file_name, O_RDONLY, 0, data);
    if (!fs)
	reiserfs_panic ("reiserfsck: could not open filesystem on \"%s\"", file_name);

    if (no_reiserfs_found (fs) && fsck_mode (fs) != FSCK_SB) {
	fsck_progress ("reiserfsck: --rebuild-sb may restore reiserfs super block\n");
	reiserfs_close (fs);
	return 0;
    }

    retval = 0;
    switch (fsck_mode (fs)) {
    case FSCK_SB:
	rebuild_sb (fs);
	break;

    case FSCK_CHECK:
	retval = check_fs (fs);
	break;

    case FSCK_REBUILD:
    case DO_TEST:
	rebuild_tree (fs);
	break;

    case FSCK_FAST_REBUILD:
	fast_rebuild (fs);
	break;
    }

    reiserfs_close (fs);
    return retval;
}

