/*
 * Copyright 2000 by Hans Reiser, licensing governed by reiserfs/README
 */

#include "debugreiserfs.h"


FILE * log;

static void look_for_key (struct buffer_head * bh, struct key * key)
{
    int i, j;
    struct item_head * ih;
    struct reiserfs_de_head * deh;


    ih = B_N_PITEM_HEAD (bh, 0);
    for (i = 0; i < B_NR_ITEMS (bh); i ++, ih ++) {
	if ((ih->ih_key.k_dir_id == key->k_dir_id || (int)key->k_dir_id == -1) &&
	    ih->ih_key.k_objectid == key->k_objectid) {
	    reiserfs_warning (log, "%d-th item of block %lu is item of file %K: %H\n",
			      i, bh->b_blocknr, key, ih);
	}
	if (!is_direntry_ih (ih))
	    continue;
	deh = B_I_DEH (bh, ih);
	for (j = 0; j < ih_entry_count (ih); j ++, deh ++) {
	    if ((deh->deh_dir_id == key->k_dir_id || (int)key->k_dir_id == -1) &&
		deh->deh_objectid == key->k_objectid) {
		reiserfs_warning (log, "dir item %d (%H) of block %lu has "
				  "entry (%d-th) %.*s pointing to %K\n",
				  i, ih, bh->b_blocknr, j,
				  name_length (ih, deh, j), name_in_entry (deh, j), key);
	    }
	}	
    }
    return;
}


static void look_for_name (struct buffer_head * bh, char * name)
{
    int i, j;
    struct item_head * ih;
    struct reiserfs_de_head * deh;
    int namelen;
    char * p;

    ih = B_N_PITEM_HEAD (bh, 0);
    for (i = 0; i < B_NR_ITEMS (bh); i ++, ih ++) {
	if (!is_direntry_ih (ih))
	    continue;
	deh = B_I_DEH (bh, ih);
	for (j = 0; j < ih_entry_count (ih); j ++, deh ++) {
	    p = name_in_entry (deh, j);
	    namelen = name_length (ih, deh, j);
	    if (namelen == strlen (name) && !strncmp (name, p, namelen)) {
		fprintf (log, "block %lu, item %d, entry %d is %s\n", bh->b_blocknr, i, j, name);fflush (log);
	    }
	}
    }
    return;
}


void do_scan (reiserfs_filsys_t fs)
{
    unsigned long i;
    struct buffer_head * bh;
    int type;
    char * answer = 0;
    size_t n = 0;
    struct key key = {0, 0, };
    unsigned long done, total;


    total = reiserfs_bitmap_ones (debug_bitmap (fs));

    log = fopen ("scan.log", "w+");

    if (debug_mode (fs) == DO_SCAN_FOR_NAME) {
	printf ("What name do you want to look for?");
	getline (&answer, &n, stdin);
	answer [strlen (answer) - 1] = 0;
	printf ("Looking for name \"%s\"..\n", answer);
	key.k_dir_id = 1;
    } else {
	printf ("What key do you want to find: dirid?");
	getline (&answer, &n, stdin);
	key.k_dir_id = atoi (answer);
	printf ("objectid?");
	getline (&answer, &n, stdin);
	key.k_objectid = atoi (answer);
	printf ("looking for (%u %u)\n", key.k_dir_id, key.k_objectid);
    }

    done = 0;
    for (i = 0; i < SB_BLOCK_COUNT (fs); i ++) {
	if (!reiserfs_bitmap_test_bit (debug_bitmap (fs), i))
	    continue;
	bh = bread (fs->s_dev, i, fs->s_blocksize);
	if (!bh) {
	    printf ("could not read block %lu\n", i);
	    continue;
	}
	type = who_is_this (bh->b_data, bh->b_size);
	switch (type) {
	case THE_JDESC:
	    if (!key.k_dir_id)
		printf ("block %lu is journal descriptor\n", i);
	    break;
	case THE_SUPER:
	    if (!key.k_dir_id)
		printf ("block %lu is reiserfs super block\n", i);
	    break;
	case THE_INTERNAL:
	    if (!key.k_dir_id)
		printf ("block %lu is reiserfs internal node\n", i);
	    break;
	case THE_LEAF:
	    if (debug_mode (fs) == DO_SCAN_FOR_NAME) {
		look_for_name (bh, answer);
	    } else if (key.k_dir_id) {
		look_for_key (bh, &key);
	    } else {
		printf ("block %lu is reiserfs leaf node\n", i);
	    }
	    break;
	}
	brelse (bh);
	print_how_far (stderr, &done, total, 1, 0);
    }
}
