/*
 * Copyright 1996, 1997, 1998, 1999 Hans Reiser
 */

/* mkreiserfs is very simple. It supports only 4 and 8K blocks. It skips
   first 64k of device, and then writes the super
   block, the needed amount of bitmap blocks (this amount is calculated
   based on file system size), and root block. Bitmap policy is
   primitive: it assumes, that device does not have unreadable blocks,
   and it occupies first blocks for super, bitmap and root blocks.
   bitmap blocks are interleaved across the disk, mainly to make
   resizing faster. */

//
// FIXME: not 'not-i386' safe
//
#define _GNU_SOURCE

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <asm/types.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/vfs.h>
#include <time.h>
#include <sys/ioctl.h>
#include <sys/mount.h>
#include <linux/major.h>
#include <sys/stat.h>
#include <linux/kdev_t.h>

#include "io.h"
#include "misc.h"
#include "reiserfs_lib.h"
#include "../version.h"


reiserfs_filsys_t fs;


#define print_usage_and_exit() die ("Usage: %s [ -f ] [ -h tea | rupasov | r5 ]"\
				    " [ -v 1 | 2] [-j device [ -o ] [ -c size | -u | -d ] [-r 8 | 4 | 2 ]] [ -q ] device [block-count]\n\n", argv[0])
  /*
   * -f - force
   * -h - hash name
   * -v - version of fs
   * -j - separate journal device
   * -o - reserv journal space in fs
   * -c - journal size
   * -u - use default size of journal
   * -d - use whole device for the journal
   * -r - ratio
   * device - device name
   * block-count - size of filesystem
   */

#define DEFAULT_BLOCKSIZE 4096




struct buffer_head * g_sb_bh;
struct buffer_head * g_bitmap_bh;
struct buffer_head * g_rb_bh;
struct buffer_head * g_journal_bh ;


unsigned long options = 0;

#define M_FORCE  0x1000
#define M_BSIZE  0x2000
#define M_JSIZE  0x4000
#define M_JDEV   0x8000
#define M_JRATIO 0xa000
#define M_HASH   0x0100
#define M_VER    0x0200
#define M_QUIET  0x0400
#define M_RESERV 0x0010
#define M_DEFLT  0x0020
#define M_WHOLE  0x0040


int g_journal_devMajorMinor =0;

int block_size = DEFAULT_BLOCKSIZE;

unsigned long device_size; /* size of "main" device */
unsigned long fs_size; /* size of filesystem */

unsigned long jdevice_size; /* size of "journal" device */
unsigned long journal_size = JOURNAL_BLOCK_COUNT; /* size of journal */

int g_hash = DEFAULT_HASH;
int g_3_6_format = 1; /* new format is default */
int g_ratio = 8;
int g_journal_begin = REISERFS_JOURNAL_OFFSET_IN_BYTES / DEFAULT_BLOCKSIZE;


/* reiserfs needs at least: enough blocks for journal, 64 k at the beginning,
   one block for super block, bitmap block and root block */
static unsigned long min_block_amount (int block_size, unsigned long journal_size)
{
    unsigned long blocks;

    blocks = REISERFS_DISK_OFFSET_IN_BYTES / block_size + 
	1 + 1 + 1 + journal_size; /* super block, bitmap, root,
                                     journal with journal header */
    if (blocks > block_size * 8)
	/* */
	die ("mkreiserfs: journal size specified incorrectly");

    return blocks;
}


/* form super block (old one) */
static void make_super_block (int dev, int g_journal_devMajorMinor)
{
    struct reiserfs_super_block * rs;
    int sb_size = g_3_6_format ? SB_SIZE : SB_SIZE_V1;
    __u32 * oids;


    if (SB_SIZE > block_size)
	die ("mkreiserfs: blocksize (%d) too small", block_size);

    /* get buffer for super block */
    g_sb_bh = getblk (dev, REISERFS_DISK_OFFSET_IN_BYTES / block_size, block_size);

    rs = (struct reiserfs_super_block *)g_sb_bh->b_data;
    set_blocksize (rs, block_size);
    set_block_count (rs, fs_size);
    set_state (rs, REISERFS_VALID_FS);
    set_tree_height (rs, 2);

    set_bmap_nr (rs, (fs_size + (block_size * 8 - 1)) / (block_size * 8));
    set_version (rs, g_3_6_format ? REISERFS_VERSION_2 : REISERFS_VERSION_1);

    /*We can use this field to inform, what version of mkreiserfs used to make this partition*/
    if (g_3_6_format) memcpy (rs->s_mkfs_version, REISERFSPROGS_VERSION,strlen(REISERFSPROGS_VERSION));

    set_hash (rs, g_hash);

    set_journal_dev(rs, g_journal_devMajorMinor) ;
    set_journal_size(rs, journal_size) ;
    

    // the differences between sb V1 and sb V2 are: magic string
    memcpy (rs->s_v1.s_magic, g_3_6_format ? REISER2FS_SUPER_MAGIC_STRING : REISERFS_SUPER_MAGIC_STRING,
	    strlen (g_3_6_format ? REISER2FS_SUPER_MAGIC_STRING : REISERFS_SUPER_MAGIC_STRING));
    // start of objectid map
    oids = (__u32 *)((char *)rs + sb_size);
    
    // max size of objectid map
    rs->s_v1.s_oid_maxsize = cpu_to_le16 ((block_size - sb_size) / sizeof(__u32) / 2 * 2);

    oids[0] = cpu_to_le32 (1);
    oids[1] = cpu_to_le32 (REISERFS_ROOT_OBJECTID + 1);
    set_objectid_map_size (rs, 2);

    mark_buffer_dirty (g_sb_bh);
    mark_buffer_uptodate (g_sb_bh, 1);

    return;
}


void zero_journal_blocks(int dev, int start, int len) {
    int i ;
    
    struct buffer_head *bh ;
    unsigned long done = 0;
    struct reiserfs_journal_header * jh;
    struct reiserfs_super_block * rs = (struct reiserfs_super_block *)g_sb_bh->b_data;
    

    fprintf (stderr, "Initializing journal - ");

    for (i = 0 ; i < len ; i++) {
        print_how_far (stderr, &done, len, 1, 1/*be quiet*/);
        bh = getblk (dev, start + i, block_size) ;
        memset(bh->b_data, 0, block_size) ;
        mark_buffer_dirty(bh) ;
        mark_buffer_uptodate(bh, 1) ;
        bwrite (bh);
        brelse(bh) ;
    }


    /* make journal header */
    bh = getblk (dev, start + i, block_size);
    memset (bh->b_data, 0, block_size) ;
    jh = (struct reiserfs_journal_header *)bh->b_data;


    /* write random value in the super block and journal header so that we
       have a way to check journal device in mounting */
    
    {
        __u32 datum;
        int f_id;

        if( read( f_id=open( "/dev/random", O_RDONLY ), &datum, sizeof (datum) ) == -1 ) {
            perror( "cannot read /dev/random" );
            exit( 1 );
        }
        close(f_id);
        set_journal_magic (rs, datum);
    }

    jh->s_journal_magic          = rs->s_v1.s_journal_magic;
    jh->s_block_count            = rs->s_v1.s_block_count;
    jh->s_journal_block          = rs->s_v1.s_journal_block;
    jh->s_orig_journal_size      = rs->s_v1.s_orig_journal_size;
    memcpy( jh->s_magic, rs->s_v1.s_magic, 10);

    /*theus field will be initialiased with value in 2 stage*/
 
    if (rs_journal_size(rs)/g_ratio > JOURNAL_TRANS_MAX)
        jh->s_journal_trans_max = cpu_to_le32 (JOURNAL_TRANS_MAX) ;
    else
        jh->s_journal_trans_max =  cpu_to_le32(rs_journal_size(rs)/8 );
    
    jh->s_journal_max_batch = cpu_to_le32 (le32_to_cpu(jh->s_journal_trans_max)*JOURNAL_MAX_BATCH/JOURNAL_TRANS_MAX) ;
    jh->s_journal_max_commit_age = cpu_to_le32 (JOURNAL_MAX_COMMIT_AGE) ;
    jh->s_journal_max_trans_age = cpu_to_le32 (JOURNAL_MAX_TRANS_AGE) ;
        
    mark_buffer_dirty(bh) ;
    mark_buffer_uptodate(bh, 1) ;
    bwrite (bh);
    brelse(bh) ;
    printf ("\n"); fflush (stdout);
}


/* this only sets few first bits in bitmap block. Fills not initialized fields
   of super block (root block and bitmap block numbers) */
static void make_bitmap ( int g_journal_devMajorMinor)
{
    struct reiserfs_super_block * rs = (struct reiserfs_super_block *)g_sb_bh->b_data;
    int i, j;
    
    /* get buffer for bitmap block */
    g_bitmap_bh = getblk (g_sb_bh->b_dev, g_sb_bh->b_blocknr + 1, g_sb_bh->b_size);
  
    /* mark, that first 8K of device is busy */
    for (i = 0; i < REISERFS_DISK_OFFSET_IN_BYTES / block_size; i ++)
	set_bit (i, g_bitmap_bh->b_data);
    
    /* mark that super block is busy */
    set_bit (i++, g_bitmap_bh->b_data);

    /* mark first bitmap block as busy */
    set_bit (i ++, g_bitmap_bh->b_data);
  
    /* sb->s_journal_block = g_block_number - JOURNAL_BLOCK_COUNT ; */ /* journal goes at end of disk */ /*????*/
    if (!g_journal_devMajorMinor || options & M_RESERV)
        {
            g_journal_begin=i;
            /* mark journal blocks as busy BUG! we need to check to make sure journal
               will fit in the first bitmap block */
            for (j = 0 ; j < (JOURNAL_BLOCK_COUNT + 1); j++) /* the descriptor block goes after the journal */
                set_bit (i ++, g_bitmap_bh->b_data);
        }
    set_journal_start (rs, g_journal_begin);
    /* and tree root is busy */
    set_bit (i, g_bitmap_bh->b_data);

    set_root_block (rs, i);
    set_free_blocks (rs, rs_block_count (rs) - i - 1);

    /* count bitmap blocks not resides in first s_blocksize blocks - ?? */
    set_free_blocks (rs, rs_free_blocks (rs) - (rs_bmap_nr (rs) - 1));

    mark_buffer_dirty (g_bitmap_bh);
    mark_buffer_uptodate (g_bitmap_bh, 0);

    mark_buffer_dirty (g_sb_bh);
    return;
}


/* form the root block of the tree (the block head, the item head, the
   root directory) */
static void make_root_block (void)
{
    struct reiserfs_super_block * rs = (struct reiserfs_super_block *)g_sb_bh->b_data;
    char * rb;
    struct item_head * ih;

    /* get memory for root block */
    g_rb_bh = getblk (g_sb_bh->b_dev, rs_root_block (rs), rs_blocksize (rs));
    rb = g_rb_bh->b_data;

    /* block head */
    set_leaf_node_level (g_rb_bh);
    set_node_item_number (g_rb_bh, 0);
    set_node_free_space (g_rb_bh, rs_blocksize (rs) - BLKH_SIZE);
    
    /* first item is stat data item of root directory */
    ih = (struct item_head *)(g_rb_bh->b_data + BLKH_SIZE);

    make_dir_stat_data (block_size, g_3_6_format ? KEY_FORMAT_2 : KEY_FORMAT_1,
			REISERFS_ROOT_PARENT_OBJECTID, REISERFS_ROOT_OBJECTID,
			ih, g_rb_bh->b_data + block_size - (g_3_6_format ? SD_SIZE : SD_V1_SIZE));
    set_ih_location (ih, block_size - ih_item_len (ih));

    // adjust block head
    set_node_item_number (g_rb_bh, node_item_number (g_rb_bh) + 1);
    set_node_free_space (g_rb_bh, node_free_space (g_rb_bh) - (IH_SIZE + ih_item_len (ih)));
  

    /* second item is root directory item, containing "." and ".." */
    ih ++;
    ih->ih_key.k_dir_id = cpu_to_le32 (REISERFS_ROOT_PARENT_OBJECTID);
    ih->ih_key.k_objectid = cpu_to_le32 (REISERFS_ROOT_OBJECTID);
    ih->ih_key.u.k_offset_v1.k_offset = cpu_to_le32 (DOT_OFFSET);
    ih->ih_key.u.k_offset_v1.k_uniqueness = cpu_to_le32 (DIRENTRY_UNIQUENESS);
    ih->ih_item_len = cpu_to_le16 (g_3_6_format ? EMPTY_DIR_SIZE : EMPTY_DIR_SIZE_V1);
    ih->ih_item_location = cpu_to_le16 (ih_location (ih-1) - ih_item_len (ih));
    ih->u.ih_entry_count = cpu_to_le16 (2);
    set_key_format (ih, KEY_FORMAT_1);

    if (g_3_6_format)
	make_empty_dir_item (g_rb_bh->b_data + ih_location (ih),
			     REISERFS_ROOT_PARENT_OBJECTID, REISERFS_ROOT_OBJECTID,
			     0, REISERFS_ROOT_PARENT_OBJECTID);
    else
	make_empty_dir_item_v1 (g_rb_bh->b_data + ih_location (ih),
				REISERFS_ROOT_PARENT_OBJECTID, REISERFS_ROOT_OBJECTID,
				0, REISERFS_ROOT_PARENT_OBJECTID);

    // adjust block head
    set_node_item_number (g_rb_bh, node_item_number (g_rb_bh) + 1);
    set_node_free_space (g_rb_bh, node_free_space (g_rb_bh) - (IH_SIZE + ih_item_len (ih)));


    mark_buffer_dirty (g_rb_bh);
    mark_buffer_uptodate (g_rb_bh, 0);
    return;
}


/*
 *  write the super block, the bitmap blocks and the root of the tree
 */
static void write_super_and_root_blocks (int dev, int jdev)
{
    struct reiserfs_super_block * rs = (struct reiserfs_super_block *)g_sb_bh->b_data;
    int i;

    zero_journal_blocks(jdev, rs_journal_start (rs), rs_journal_size(rs)) ;

    /* super block */
    bwrite (g_sb_bh);

    /* bitmap blocks */
    for (i = 0; i < rs_bmap_nr (rs); i ++) {
	if (i != 0) {
	    g_bitmap_bh->b_blocknr = i * rs_blocksize (rs) * 8;
	    memset (g_bitmap_bh->b_data, 0, g_bitmap_bh->b_size);
	    set_bit (0, g_bitmap_bh->b_data);
	}
	if (i == rs_bmap_nr (rs) - 1) {
	    int j;

	    /* fill unused part of last bitmap block with 1s */
	    if (rs_block_count (rs) % (rs_blocksize (rs) * 8))
		for (j = rs_block_count (rs) % (rs_blocksize (rs) * 8); j < rs_blocksize (rs) * 8; j ++) {
		    set_bit (j, g_bitmap_bh->b_data);
		}
	}
	/* write bitmap */
	mark_buffer_dirty (g_bitmap_bh);
	bwrite (g_bitmap_bh);
    }

    /* root block */
    bwrite (g_rb_bh);
    brelse (g_rb_bh);
    brelse (g_bitmap_bh);
    brelse (g_sb_bh);
}


static void report (char * devname,char * j_devname)
{
    struct reiserfs_super_block * rs = (struct reiserfs_super_block *)g_sb_bh->b_data;
    unsigned int i;

    printf ("Creating reiserfs of %s format\n", g_3_6_format ? "3.6" : "3.5");
    printf ("Block size %d bytes\n", rs_blocksize (rs));
    printf ("Block count %d\n", rs_block_count (rs));
    printf ("Used blocks %d\n", rs_block_count (rs) - rs_free_blocks (rs));
    printf ("Free blocks count %d\n", rs_free_blocks (rs));
    printf ("First %ld blocks skipped\n", g_sb_bh->b_blocknr);
    printf ("Super block is in %ld\n", g_sb_bh->b_blocknr);
    printf ("Bitmap blocks (%d) are : \n\t%ld", rs_bmap_nr (rs), g_bitmap_bh->b_blocknr);
    for (i = 1; i < rs_bmap_nr (rs); i ++) {
	printf (", %d", i * rs_blocksize (rs) * 8);
    }
    printf ("\nJournal size %d (blocks %d-%d of file %s)\n",
	    rs_journal_size(rs), rs_journal_start (rs), 
	    rs_journal_start (rs) + rs_journal_size(rs), j_devname?j_devname:devname);
    printf ("Root block %u\n", rs_root_block (rs));
    printf ("Hash function \"%s\"\n", g_hash == TEA_HASH ? "tea" :
	    ((g_hash == YURA_HASH) ? "rupasov" : "r5"));
    fflush (stdout);
}


/* wipe out first 64 k of a device and both possible reiserfs super block */
static void invalidate_other_formats (int dev)
{
    struct buffer_head * bh;
    
    bh = getblk (dev, 0, 64 * 1024);
    memset (bh->b_data, 0, bh->b_size);
    mark_buffer_uptodate (bh, 1);
    mark_buffer_dirty (bh);
    bwrite (bh);
    brelse (bh);
}


static void set_hash_function (char * str)
{
    if (!strcmp (str, "tea"))
	g_hash = TEA_HASH;
    else if (!strcmp (str, "rupasov"))
	g_hash = YURA_HASH;
    else if (!strcmp (str, "r5"))
	g_hash = R5_HASH;
    else
	printf ("mkreiserfs: wrong hash type specified. Using default\n");
}


static void set_reiserfs_version (char * str)
{
    if (!strcmp (str, "1"))
	g_3_6_format = 0;
    else if (!strcmp (str, "2"))
	g_3_6_format = 1;
    else
	printf ("mkreiserfs: wrong reiserfs version specified. Using default 3.5 format\n");
}


static int forced_and_confirmed (int check_force_flag)
{
    if (check_force_flag && !(options & M_FORCE))
	return 0;

    if (check_force_flag)
	fprintf (stderr, "Forced to ");
    if (!user_confirmed (stderr, "continue (y/n)?", "y\n"))
	return 0;

    return 1;
}


/* */
static int dev_check_open(int force, char * device_name, struct stat * st)
{
    int dev;

    if (is_mounted (device_name)) {
	fprintf (stderr, "mkreiserfs: '%s' contains a mounted file system\n",
		 device_name);
	if (!forced_and_confirmed (1/*check force flag*/))
	    exit (-1);
    }

    dev = open (device_name, O_RDWR);
    if (dev == -1)
	die ("mkreiserfs: can not open '%s': %m", device_name);
  
    if (fstat (dev, st) < 0)
	die ("mkreiserfs: unable to stat %s: %m", device_name);

    if (!S_ISBLK (st->st_mode)) {
	fprintf (stderr, "mkreiserfs: %s is not a block special device\n",
		 device_name);
	if (!forced_and_confirmed (1/*check force flag*/))
	    exit (-1);
    } else {
	// from e2progs-1.18/misc/mke2fs.c
	if ((MAJOR (st->st_rdev) == HD_MAJOR && MINOR (st->st_rdev) % 64 == 0) ||
	    (SCSI_BLK_MAJOR (MAJOR(st->st_rdev)) && MINOR (st->st_rdev) % 16 == 0)) {
	    fprintf (stderr, "mkreiserfs: %s is entire device, not just one partition! ", 
		     device_name); 
	    if (forced_and_confirmed (0/*do not check force flag*/))
		exit (-1);
	}
    }
    return dev;
}


static int str2int (char * str)
{
    int val;
    char * tmp;

    val = (int) strtol (str, &tmp, 0);
    if (*tmp)
	die ("mkreiserfs: strtol is unable to make an integer of %s\n", str);
    return val;
}


static void set_journal_device_ratio (char * str)
{
    g_ratio = str2int (str);
      
    if (g_ratio < 2 || g_ratio > 8)
	die ("mkreiserfs: %dk is wrong ratio for journal", g_ratio);

}


/* journal must fit into number of blocks pointed by first bitmap */
static void set_journal_device_size (char * str)
{
    journal_size = str2int (str);

    if (journal_size < JOURNAL_BLOCK_COUNT / 32)
	die ("mkreiserfs: wrong journal size specified: %lu. Should be at least %lu",
	     journal_size, JOURNAL_BLOCK_COUNT / 32);
}


static void check_journal_device (char * jdevice_name)
{
    /* size of device dedicated to hold journal */
    jdevice_size = count_blocks (jdevice_name, block_size, -1);

    if (jdevice_size < JOURNAL_BLOCK_COUNT / 32)
	die ("mkreiserfs: journal device %s is too small (%lu blocks). "
	     "You need %lu at least\n", 
	     jdevice_size, jdevice_name, JOURNAL_BLOCK_COUNT / 32);

    /* does journal device have enough space */
    if (journal_size > jdevice_size - g_journal_begin - 1)
	die ("mkreiserfs: specified journal size %d is too big. Device %s has only %d blocks\n",
	     journal_size, jdevice_name, jdevice_size - g_journal_begin - 1);

    if (options & M_WHOLE)
	journal_size = jdevice_size - g_journal_begin - 1;  /* journal on whole device */
}


int main (int argc, char **argv)
{
    int dev;
    int journal_dev;
    int force = 0;
    struct stat st;
    char * device_name;
    char * jdevice_name = NULL;
    char c;

    print_banner ("mkreiserfs");

    if (argc < 2)
	print_usage_and_exit ();


    while ( ( c = getopt( argc, argv, "fh:v:j:oc:udr:q" ) ) != EOF )
	switch( c )
	{
	case 'f': /* force if file is not a block device or fs is
		     mounted. Confirm still required */
	    force += 1;
	    options |= M_FORCE;
	    break;

	case 'h': /* hash is specified */
	    options |= M_HASH;
	    set_hash_function (optarg);
	    break;
	    
	case 'u': /* ? */
	    options |= M_DEFLT;
	    break;
	    
	case 'd': /* */
	    options |= M_WHOLE;
	    break;
	    
	case 'o': /* ? */
	    options |= M_RESERV;
	    break;

	case 'v': /* filesystem format is specified */
	    options |= M_VER;
	    set_reiserfs_version (optarg);
	    break;

	case 'c':
	    options |= M_JSIZE;
	    set_journal_device_size (optarg);
	    break;
	    
	case 'j': /* journal device is specified */
	    options |= M_JDEV;
	    jdevice_name = optarg;
	    break;

	case 'r': /* ? */
	    options |= M_JRATIO;
	    set_journal_device_ratio (optarg);
	    break;

	default :
	    print_usage_and_exit ();
	}

    /* we are formatting this file */
    device_name = argv [optind];
    device_size = count_blocks (device_name, block_size, -1);

    if (optind == argc - 2) {
	/* number of blocks for filesystem is specified */
	fs_size = str2int (argv[optind + 1]);
	if (fs_size > device_size)
	    die ("mkreiserfs: there are only %lu blocks on %s, you want %lu",
		 device_size, device_name, fs_size);
    } else if (optind == argc - 1) {
	/* number of blocks is not specified */
	fs_size = device_size;
    } else {
	print_usage_and_exit ();
    }


    /* check if filesystem device is mounted and open it */
    dev = dev_check_open (options & M_FORCE, device_name, &st);
    journal_dev = dev;


    if (options & M_JDEV && strcmp (device_name, jdevice_name)) {
	/* journal is to be on separate device */
	check_journal_device (jdevice_name);

	/* check if journal device is mounted and open it */
	journal_dev = dev_check_open (options & M_FORCE, jdevice_name, &st);
	g_journal_devMajorMinor = st.st_rdev;
    }

    if (fs_size < min_block_amount (block_size, dev == journal_dev ? (JOURNAL_BLOCK_COUNT + 1) : 0))
	die ("mkreiserfs: can not create filesystem on that small device (%lu blocks).\n"
	     "It should have at least %lu blocks",
	     device_size, min_block_amount (block_size, 
					    dev == journal_dev ? (JOURNAL_BLOCK_COUNT + 1) : 0));
    

    /* these fill buffers (super block, first bitmap, root block) with
       reiserfs structures */
    make_super_block (dev, g_journal_devMajorMinor);
    make_bitmap (g_journal_devMajorMinor);
    make_root_block ();
  
    report (device_name, jdevice_name);

    printf ("ATTENTION: YOU SHOULD REBOOT AFTER FDISK!\n\t    ALL DATA WILL BE LOST ON '%s'! ", device_name);
    if (dev != journal_dev)
        printf (" and  journal on '%s'(%x)", jdevice_name,  g_journal_devMajorMinor);
    if (!user_confirmed (stderr, "(y/n)", "y\n"))
	die ("mkreiserfs: Disk was not formatted");

    invalidate_other_formats (dev);
    if (g_journal_devMajorMinor)
	invalidate_other_formats (journal_dev);

    write_super_and_root_blocks (dev, journal_dev);

    check_and_free_buffer_mem ();

    printf ("Syncing.."); fflush (stdout);

    close (dev) ;
    close(journal_dev);
    sync ();
 
    printf ("\n\nReiserFS core development sponsored by SuSE Labs (suse.com)\n\n"
	    "Journaling sponsored by MP3.com.\n\n"
	    "To learn about the programmers and ReiserFS, please go to\n"
	    "http://www.devlinux.com/namesys\n\nHave fun.\n\n"); 
    return 0;
}
