/*
 * Copyright 1996-2001 Hans Reiser, licensing governed by ../README
 */

/* mkreiserfs is very simple. It supports only 4 and 8K blocks. It skips
   first 64k of device, and then writes the super
   block, the needed amount of bitmap blocks (this amount is calculated
   based on file system size), and root block. Bitmap policy is
   primitive: it assumes, that device does not have unreadable blocks,
   and it occupies first blocks for super, bitmap and root blocks.
   bitmap blocks are interleaved across the disk, mainly to make
   resizing faster. */

//
// FIXME: not 'not-i386' safe
//
#define _GNU_SOURCE

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <asm/types.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/vfs.h>
#include <time.h>
#include <sys/ioctl.h>
#include <sys/mount.h>
#include <linux/major.h>
#include <sys/stat.h>
#include <linux/kdev_t.h>
#include <sys/utsname.h>
#include <getopt.h>
#include <stdarg.h>

#include "io.h"
#include "misc.h"
#include "reiserfs_lib.h"
#include "../version.h"


char *program_name;

static void message( const char * fmt, ... ) 
	__attribute__ ((format (printf, 1, 2)));

	static void message( const char * fmt, ... ) 
{
    char *buf;
    va_list args;
	
    buf = NULL;
    va_start( args, fmt );
    vasprintf( &buf, fmt, args );
    va_end( args );

    if( buf ) {
	    fprintf( stderr, "%s: %s\n", program_name, buf );
	    free( buf );
    }
}

static void print_usage_and_exit(void)
{
	fprintf(stderr, "Usage: %s [options] "
			" device [block-count]\n"
			"\n"
			"Options:\n\n"
			"\t-b | --block-size (size of file-system block, in bytes)\n"
			"\t-j | --journal-device (path to separate device to hold journal)\n"
			"\t-s | --journal-size (size of journal on separate device, in blocks)\n"
			"\t-o | --offset (offset from the start of journal device, in blocks)\n"
			"\t-t | --transaction-max-size (maximal size of transaction, in blocks)\n"
			"\t-h | --hash rupasov|tea|r5 (hash function to use by default)\n"
			"\t-v | --format 3.5|3.6 (old 3.5 format or newer 3.6)\n"
			"\t-V\n"
			"\t-f\n", program_name);
	exit (1);
}


int Create_default_journal = 1;
int Block_size = 4096;

/* size of journal + 1 block for journal header */
unsigned long Journal_size = JOURNAL_DEFAULT_SIZE + 1;
int Max_trans_size = JOURNAL_TRANS_MAX;
int Hash = DEFAULT_HASH;
int Offset = 16;
char * Version;





/* form super block (old one) */
static void make_super_block (reiserfs_filsys_t * fs)
{
    set_sb_state (fs->fs_ondisk_sb, REISERFS_VALID_FS);
    set_sb_tree_height (fs->fs_ondisk_sb, 2);
    set_sb_hash_code (fs->fs_ondisk_sb, Hash);

	/* store how many blocks are reserved for journal */
	if (Create_default_journal ||
		strcmp (fs->fs_file_name, fs->fs_j_file_name))
		set_sb_reserved_for_journal(fs->fs_ondisk_sb, 0);
	else
		set_sb_reserved_for_journal(fs->fs_ondisk_sb, Journal_size - 1);
}


/* wipe out first 64 k of a device and both possible reiserfs super block */
static void invalidate_other_formats (int dev)
{
    struct buffer_head * bh;
    
    bh = bread (dev, 0, 64 * 1024);
	if (!bh)
		die ("Unable to read first blocks of the device");
    memset (bh->b_data + 1024, 0, bh->b_size - 1024);
    mark_buffer_uptodate (bh, 1);
    mark_buffer_dirty (bh);
    bwrite (bh);
    brelse (bh);
}


void zero_journal (reiserfs_filsys_t * fs)
{
    int i;
    struct buffer_head * bh;
    unsigned long done;
    unsigned long start, len;


    fprintf (stderr, "Initializing journal - ");

    start = get_jp_journal_1st_block (sb_jp (fs->fs_ondisk_sb));
    len = get_jp_journal_size (sb_jp (fs->fs_ondisk_sb));

    done = 0;
    for (i = 0; i < len; i ++) {
        print_how_far (stderr, &done, len, 1, 1/*be quiet*/);
        bh = getblk (fs->fs_journal_dev, start + i, fs->fs_blocksize);
		if (!bh)
			die ("zero_journal: getblk failed");
        memset (bh->b_data, 0, bh->b_size);
        mark_buffer_dirty (bh);
        mark_buffer_uptodate (bh, 1);
        bwrite (bh);
        brelse (bh);
    }

    fprintf (stderr, "\n");
    fflush (stderr);
}


/* this only sets few first bits in bitmap block. Fills not initialized fields
   of super block (root block and bitmap block numbers) */
static void make_bitmap (reiserfs_filsys_t * fs)
{
    struct reiserfs_super_block * sb = fs->fs_ondisk_sb;
    int i;
    unsigned long block;
    int marked;
    

    marked = 0;

    /* mark skipped area and super block */
    for (i = 0; i <= fs->fs_super_bh->b_blocknr; i ++) {
		reiserfs_bitmap_set_bit (fs->fs_bitmap2, i);
		marked ++;
    }

    /* mark bitmaps as used */
    block = fs->fs_super_bh->b_blocknr + 1;
    for (i = 0; i < get_sb_bmap_nr (sb); i ++) {
		reiserfs_bitmap_set_bit (fs->fs_bitmap2, block);
		marked ++;
		if (spread_bitmaps (fs))
			block = (block / (fs->fs_blocksize * 8) + 1) * (fs->fs_blocksize * 8);
		else
			block ++;
    }

    if (journal_is_relocated (fs->fs_ondisk_sb))
		/* root block is right after super block and first bitmap */
		block = fs->fs_super_bh->b_blocknr + 1 + 1;
    else {
		/* journal is on main device */
		for (i = 0; i <= get_jp_journal_size (sb_jp (sb)); i ++) {
			reiserfs_bitmap_set_bit (fs->fs_bitmap2,
									 i + get_jp_journal_1st_block (sb_jp (sb)));
			marked ++;
		}
		block = get_jp_journal_1st_block (sb_jp (sb)) + i;
    }

    reiserfs_bitmap_set_bit (fs->fs_bitmap2, block);
    marked ++;

    set_sb_root_block (sb, block);
    set_sb_free_blocks (sb, get_sb_block_count (sb) - marked);
}


static void set_root_dir_nlink (struct item_head * ih, void * sd)
{
    __u32 nlink;

    nlink = 3;
    set_sd_nlink (ih, sd, &nlink);
}


/* form the root block of the tree (the block head, the item head, the
   root directory) */
static void make_root_block (reiserfs_filsys_t * fs)
{
    struct reiserfs_super_block * sb;
    struct buffer_head * bh;


    sb = fs->fs_ondisk_sb;
    /* get memory for root block */
    bh = getblk (fs->fs_dev, get_sb_root_block (sb), get_sb_block_size (sb));
    if (!bh)
		die ("make_root_block: getblk failed");

    mark_buffer_uptodate (bh, 1);

    make_empty_leaf (bh);
    make_sure_root_dir_exists (fs, set_root_dir_nlink, 0);
    brelse (bh);


    /**/
    mark_objectid_used (fs, REISERFS_ROOT_PARENT_OBJECTID);
    mark_objectid_used (fs, REISERFS_ROOT_OBJECTID);
    
}



static void report (reiserfs_filsys_t * fs)
{
    print_block (stdout, fs, fs->fs_super_bh);
}




static void set_hash_function (char * str)
{
    if (!strcmp (str, "tea"))
		Hash = TEA_HASH;
    else if (!strcmp (str, "rupasov"))
		Hash = YURA_HASH;
    else if (!strcmp (str, "r5"))
		Hash = R5_HASH;
    else
		message("wrong hash type specified. Using default");
}


static void set_reiserfs_version (char * str)
{
    if (!strcmp (str, "3.5"))
		Version = "3.5";
    else  {
		Version = "3.6";
		if (strcmp (str, "3.6"))
			message("wrong reiserfs version specified. Using default 3.6 format");
    }
}



#define check_forcing_ask_confirmation() \
	if (!force)\
	    /* avoid formatting it without being forced */\
	    return 0;\
	if (force == 1) {\
	    if (!user_confirmed (stderr, "Continue (y/n):", "y\n"))\
		return 0;\
	}\


/* we only can use a file for filesystem or journal if it is either not
   mounted block device or regular file and we are forced to use it */
static int can_we_format_it (char * device_name, int force)
{
    mode_t mode;
    dev_t rdev;


    if (is_mounted (device_name)) {
		/* device looks mounted */
	    message("'%s' looks mounted.", device_name);
		check_forcing_ask_confirmation ();
    }

    mode = get_st_mode (device_name);
    rdev = get_st_rdev (device_name);

    if (!S_ISBLK (mode)) {
		/* file is not a block device */
	    message("%s is not a block special device", device_name);
		check_forcing_ask_confirmation ();
    } else {
		/* from e2progs-1.18/misc/mke2fs.c */
		if ((MAJOR (rdev) == HD_MAJOR && MINOR (rdev) % 64 == 0) ||
			(SCSI_BLK_MAJOR (rdev) && MINOR (rdev) % 16 == 0)) {
			/* /dev/hda or similar */
			message("%s is entire device, not just one partition!",
					device_name); 
			check_forcing_ask_confirmation ();
		}
    }

    return 1;
}

#undef check_forcing_ask_confirmation()



static int str2int (char * str)
{
    int val;
    char * tmp;

    val = (int) strtol (str, &tmp, 0);
    if (*tmp)
		die ("%s: strtol is unable to make an integer of %s\n", program_name, str);
    return val;
}


static void set_block_size (char * str)
{
    Block_size = str2int (str);
      
    if (Block_size != 4096)/* && block_size != 8192 && block_size != 1024 && 
							  block_size != 2048)*/
		die ("%s: wrong blocksize %s specified, only 4096 is supported currently", program_name, str);

}


static void set_transaction_max_size (char * str)
{
	Max_trans_size = str2int( str );
}


/* reiserfs_create_journal will check this */
static void set_journal_device_size (char * str)
{
    Journal_size = str2int (str);
/*
    if (Journal_size < JOURNAL_MIN_SIZE)
		die ("%s: wrong journal size specified: %lu. Should be at least %u",
			 program_name, 
			 Journal_size + 1, JOURNAL_MIN_SIZE + 1);
*/
}


/* reiserfs_create_journal will check this */
static void set_offset_in_journal_device (char * str)
{
	Offset = str2int( str );
}


static int is_journal_default (char * name, char * jname)
{
	if (jname && strcmp (name, jname))
		return 0;
	if (Journal_size != JOURNAL_DEFAULT_SIZE + 1)
		return 0;
	if (Max_trans_size != JOURNAL_TRANS_MAX)
		return 0;
	return 1;
}


/* if running kernel is 2.2 - mkreiserfs creates 3.5 format, if 2.4 - 3.6,
   otherwise - mkreiserfs fails */
static int select_format (void)
{
	struct utsname sysinfo;


	if (Version) {
		if (!strcmp (Version, "3.5"))
			return REISERFS_FORMAT_3_5;

		if (strcmp (Version, "3.6")) {
			message ("Unknown fromat %s specified\n", Version);
			exit (1);
		}
		return REISERFS_FORMAT_3_6;
	}
	
	message ("Guessing about desired format.. ");
	
	if (uname (&sysinfo) == -1) {
		message ("could not get system info: %m");
		exit (1);
	}
	
	message ("Kernel %s is running.", sysinfo.release);
	if (!strncmp (sysinfo.release, "2.4", 3))
		return REISERFS_FORMAT_3_6;

	if (strncmp (sysinfo.release, "2.2", 3)) {
		message( "You should run either 2.4 or 2.2 to be able to create reiserfs filesystem or specify desired format with -v");
		exit (1);
	}

    message ("Creating filesystem of format 3.5");
    return REISERFS_FORMAT_3_5;
}


int main (int argc, char **argv)
{
    reiserfs_filsys_t * fs;
    int force;
    char * device_name;
    char * jdevice_name;
    unsigned long fs_size;
    int c;


    program_name = strrchr( argv[ 0 ], '/' );
    program_name = program_name ? ++ program_name : argv[ 0 ];
    
    print_banner (program_name);

    if (argc < 2)
		print_usage_and_exit ();
    
    force = 0;
    fs_size = 0;
    device_name = 0;
    jdevice_name = 0;


    while (1) {
		static struct option options[] = {
			{"block-size", required_argument, 0, 'b'},
			{"journal-device", required_argument, 0, 'j'},
			{"journal-size", required_argument, 0, 's'},
			{"transaction-max-size", required_argument, 0, 't'},
			{"offset", required_argument, 0, 'o'},
			{"hash", required_argument, 0, 'h'},
			{"format", required_argument, 0, 'v'},
			{0, 0, 0, 0}
		};
		int option_index;
      
		c = getopt_long (argc, argv, "b:j:s:t:o:h:v:Vf",
						 options, &option_index);
		if (c == -1)
			break;
	
		switch (c) {
		case 'b': /* --block-size */
			set_block_size (optarg);
			break;

		case 'j': /* --journal-device */
			Create_default_journal = 0;
			jdevice_name = optarg;
			break;

		case 's': /* --journal-size */
			Create_default_journal = 0;
			set_journal_device_size (optarg);	    
			break;

		case 't': /* --transaction-max-size */
			Create_default_journal = 0;
			set_transaction_max_size (optarg);
			break;

		case 'o': /* --offset */
			Create_default_journal = 0;
			set_offset_in_journal_device (optarg);
			break;

		case 'h': /* --hash */
			set_hash_function (optarg);
			break;

		case 'v': /* --format */
			set_reiserfs_version (optarg);
			break;

		case 'V':
			exit (1);

		case 'f':
			force ++;
			break;

		default:
			print_usage_and_exit();
		}
    }


    /* device to be formatted */
    device_name = argv [optind];

	if (is_journal_default (device_name, jdevice_name))
		Create_default_journal = 1;
	
    if (optind == argc - 2) {
		/* number of blocks for filesystem is specified */
		fs_size = str2int (argv[optind + 1]);
    } else if (optind == argc - 1) {
		/* number of blocks is not specified */
		fs_size = count_blocks (device_name, Block_size);
    } else {
		print_usage_and_exit ();
    }


    if (!can_we_format_it (device_name, force))
		return 1;
	
    if (jdevice_name)
		if (!can_we_format_it (jdevice_name, force))
			return 1;

    fs = reiserfs_create (device_name, select_format (), fs_size, Block_size, Create_default_journal);
    if (!fs) {
		return 1;
    }

    if (!reiserfs_create_journal (fs, jdevice_name, Offset, Journal_size, Max_trans_size)) {
		return 1;
    }

    if (!reiserfs_create_ondisk_bitmap (fs)) {
		return 1;
    }

    /* these fill buffers (super block, first bitmap, root block) with
       reiserfs structures */
    make_super_block (fs);
    make_bitmap (fs);
    make_root_block (fs);
  
    report (fs);

    if (force < 2) {
		fprintf (stderr, "ATTENTION: YOU SHOULD REBOOT AFTER FDISK!\n"
				 "\tALL DATA WILL BE LOST ON '%s'", device_name);
		if (jdevice_name && strcmp (jdevice_name, device_name))
			fprintf (stderr, " AND ON JOURNAL DEVICE '%s'", jdevice_name);

		if (!user_confirmed (stderr, "!\nContinue (y/n):", "y\n"))
			return 1;
    }


	invalidate_other_formats (fs->fs_dev);

		
    zero_journal (fs);

    reiserfs_close (fs);

    printf ("Syncing.."); fflush (stdout);
    sync ();
    printf ("ok\n");
 
    printf ("\n\nReiserFS core development sponsored by SuSE Labs (suse.com)\n\n"
			"Journaling sponsored by MP3.com.\n\n"
			"To learn about the programmers and ReiserFS, please go to\n"
			"http://namesys.com\n\nHave fun.\n\n"); 
    return 0;
}

/* 
 * Use BSD fomatting.
 * Local variables:
 * c-indentation-style: "bsd"
 * mode-name: "BSDC"
 * c-basic-offset: 4
 * tab-width: 4
 * End:
 */
