/*
 * Copyright 2000 by Hans Reiser, licensing governed by reiserfs/README
 */
#include "debugreiserfs.h"


reiserfs_filsys_t fs;


#define print_usage_and_exit() die ("Usage: %s [options] \
 device\n\n\
Options:\n\
  -d\tprint block details
  -m\tprint bitmap blocks\n\
  -j\tprint journal\n\
  -o\tprint objectid map\n\n\
  -p\tsend filesystem metadata to stdout\n\
  -s\tscan device either for specific key or for any metadata\n\
  -n\tscan device for specific name in reiserfs directories\n\
  -S\tgo through whole device when running -p, -s or -n\n\
  -b bitmap_file\trunning -p, -s or -n read blocks marked in this bitmap only\n\
  -B block\tblock to print\n\n\
  -q\tno speed info (for -p, -s and -n)\n\n", argv[0]);



#if 1
struct reiserfs_fsstat {
    int nr_internals;
    int nr_leaves;
    int nr_files;
    int nr_directories;
    int nr_unformatted;
} g_stat_info;
#endif



static void print_disk_tree (reiserfs_filsys_t fs, int block_nr)
{
    struct buffer_head * bh;

    bh = bread (fs->s_dev, block_nr, fs->s_blocksize);
    if (!bh) {
	die ("Could not read block %d\n", block_nr);
    }
    if (is_internal_node (bh)) {
	int i;
	struct disk_child * dc;

	g_stat_info.nr_internals ++;
	print_block (stdout, fs, bh, debug_mode (fs) & PRINT_DETAILS, -1, -1);
      
	dc = B_N_CHILD (bh, 0);
	for (i = 0; i <= B_NR_ITEMS (bh); i ++, dc ++)
	    print_disk_tree (fs, dc->dc_block_number);
      
    } else if (is_leaf_node (bh)) {
	g_stat_info.nr_leaves ++;
	print_block (stdout, fs, bh, debug_mode (fs) & PRINT_DETAILS, -1, -1);
    } else {
	print_block (stdout, fs, bh, debug_mode (fs) & PRINT_DETAILS, -1, -1);
	die ("print_disk_tree: bad block type");
    }
    brelse (bh);
}



void pack_one_block (reiserfs_filsys_t fs, unsigned long block);
static void print_one_block (reiserfs_filsys_t fs, int block)
{
    struct buffer_head * bh;
    
    if (test_bit (block % (fs->s_blocksize * 8), 
		  SB_AP_BITMAP (fs)[block / (fs->s_blocksize * 8)]->b_data))
	fprintf (stderr, "%d is used in true bitmap\n", block);
    else
	fprintf (stderr, "%d is free in true bitmap\n", block);
    
    bh = bread (fs->s_dev, block, fs->s_blocksize);
    if (!bh) {
	printf ("print_one_block: bread fialed\n");
	return;
    }

    if (debug_mode (fs) == DO_PACK) {
	pack_one_block (fs, bh->b_blocknr);
	brelse (bh);
	return;
    }

    if (who_is_this (bh->b_data, fs->s_blocksize) != THE_UNKNOWN)
	print_block (stdout, fs, bh, PRINT_DETAILS, -1, -1);
    else
	printf ("Looks like unformatted\n");
    brelse (bh);
    return;
}


static void corrupt_clobber_hash (char * name, struct item_head * ih, 
				  struct reiserfs_de_head * deh)
{
    printf ("\tCorrupting deh_offset of entry \"%s\" of [%u %u]\n", name,
	    ih->ih_key.k_dir_id, ih->ih_key.k_objectid);
    deh->deh_offset = 700;
}


/* this reads list of desired corruptions from stdin and perform the
   corruptions. Format of that list:
   A hash_code
   C name objectid     - 'C'ut entry 'name' from directory item with 'objectid'
   H name objectid     - clobber 'H'hash of entry 'name' of directory 'objectid'
   I item_num pos_in_item  make pos_in_item-th slot of indirect item to point out of device
   O item_num          - destroy item 'O'rder - make 'item_num'-th to have key bigger than 'item_num' + 1-th item
   D item_num          - 'D'elete item_num-th item
   S item_num value    - change file size (item_num-th item must be stat data)
   F item_num value    - change sd_first_direct_byte of stat data
   J item_num objectid
   E name objectid new - change entry's deh_objectid to new
   P                   - print the block
*/
static void do_corrupt_one_block (reiserfs_filsys_t fs)
{
    struct buffer_head * bh;
    int i, j;
    struct item_head * ih;
    int item_num;
    char * line = 0;
    int n = 0;
    char code, name [100];
    __u32 objectid, new_objectid;
    int value;
    int hash_code;
    int pos_in_item;
    unsigned long block;


    block = certain_block (fs);

    if (test_bit (block % (fs->s_blocksize * 8), 
		  SB_AP_BITMAP (fs)[block / (fs->s_blocksize * 8)]->b_data))
	fprintf (stderr, "%lu is used in true bitmap\n", block);
    else
	fprintf (stderr, "%lu is free in true bitmap\n", block);
    
    bh = bread (fs->s_dev, block, fs->s_blocksize);
    if (!bh) {
	printf ("corrupt_one_block: bread fialed\n");
	return;
    }

    if (who_is_this (bh->b_data, fs->s_blocksize) != THE_LEAF) {
	printf ("Can not corrupt not a leaf node\n");
	brelse (bh);
	return;
    }

    printf ("Corrupting block %lu..\n", bh->b_blocknr);

    while (getline (&line, &n, stdin) != -1) {
	switch (line[0]) {
	case '#':
	case '\n':
	    continue;
	case '?':
	    printf ("A hash_code     - reset hAsh code in super block\n"
		    "C name objectid - Cut entry 'name' from directory item with 'objectid'\n"
		    "H name objectid - clobber Hash of entry 'name' of directory 'objectid'\n"
		    "I item_num pos_in_item  make pos_in_tem-th slot of Indirect item to point out of device\n"
		    "O item_num      - destroy item Order - make 'item_num'-th to have key bigger than 'item_num' + 1-th item\n"
		    "D item_num      - Delete item_num-th item\n"
		    "S item_num value - change file Size (item_num-th item must be stat data)\n"
		    "F item_num value - change sd_First_direct_byte of stat data\n"
		    "J item_num objectid - set 'obJectid' of 'item_num'-th item\n"
		    "E name objectid objectid - set deh_objectid of an entry to objectid\n");

	    continue;

	case 'P':
	    print_block (stderr, fs, bh, 3, -1, -1);
	    break;
	    
	case 'A':
	    /* corrupt hash record in super block */
	    if (sscanf (line, "%c %d\n", &code, &hash_code) != 2) {
		printf ("Wrong format \'%c\'\n", line [0]);
		continue;
	    }
	    break;
	    
	case 'C': /* cut entry */
	case 'H': /* make hash wrong */
	    if (sscanf (line, "%c %s %u\n", &code, name, &objectid) != 3) {
		printf ("Wrong format \'%c\'\n", line [0]);
		continue;
	    }
	    break;

	case 'J': /* set objectid : used to simulate objectid sharing problem */
	    if (sscanf (line, "%c %d %d\n", &code, &item_num, &objectid) != 3) {
		printf ("Wrong format \'%c\'\n", line [0]);
		continue;
	    }
	    break;

	case 'E': /* set objectid : used to simulate objectid sharing problem */
	    if (sscanf (line, "%c %s %u %d\n", &code, name, &objectid, &new_objectid) != 4) {
		printf ("Wrong format \'%c\'\n", line [0]);
		continue;
	    }
	    break;

	case 'I': /* break unformatted node pointer */
	    if (sscanf (line, "%c %d %d\n", &code, &item_num, &pos_in_item) != 3) {
		printf ("Wrong format \'%c\'\n", line [0]);
		continue;
	    }
	    break;
	    
	case 'D': /* delete item */
	case 'O': /* make item out of order */
	    if (sscanf (line, "%c %d\n", &code, &item_num) != 2) {
		printf ("Wrong format \'%c\'\n", line [0]);
		continue;
	    }
	    break;
	    
	case 'S': /* corrupt st_size */
	case 'F': /*         st_first_direct_byte */
	    if (sscanf (line, "%c %d %d\n", &code, &item_num, &value) != 3) {
		printf ("Wrong format \'%c\'\n", line [0]);
		continue;
	    }
	    break;
	}
	
	if (code == 'A') {
	    reiserfs_warning (stderr, "Changing %s to %s\n", code2name (rs_hash (fs->s_rs)),
			       code2name (hash_code));
	    set_hash (fs->s_rs, hash_code);
	    mark_buffer_dirty (fs->s_sbh);
	    continue;
	}

	ih = B_N_PITEM_HEAD (bh, 0);
	for (i = 0; i < node_item_number (bh); i ++, ih ++) {
	    struct reiserfs_de_head * deh;

	    if (code == 'I' && i == item_num) {
		if (!is_indirect_ih (ih) || pos_in_item >= I_UNFM_NUM (ih)) {
		    reiserfs_warning (stderr, "Not an indirect item or there is "
				       "not so many unfm ptrs in it\n");
		    continue;
		}
		* ((__u32 *)B_I_PITEM (bh, ih) + pos_in_item) = SB_BLOCK_COUNT(fs) + 100;
		mark_buffer_dirty (bh);
		goto cont;
	    }

	    if (code == 'J' && i == item_num) {
		ih->ih_key.k_objectid = objectid;
		mark_buffer_dirty (bh);
		goto cont;
	    }

	    if (code == 'S' && i == item_num) {
		/* fixme: old stat data only */
		struct stat_data_v1 * sd;

		sd = (struct stat_data_v1 *)B_I_PITEM (bh, ih); 
		reiserfs_warning (stderr, "Changing sd_size of %k from %d to %d\n",
				   &ih->ih_key, sd->sd_size, value);
		sd->sd_size = value;
		mark_buffer_dirty (bh);
		goto cont;		
	    }

	    if (code == 'F' && i == item_num) {
		/* fixme: old stat data only */
		struct stat_data_v1 * sd;

		sd = (struct stat_data_v1 *)B_I_PITEM (bh, ih); 
		reiserfs_warning (stderr, "Changing sd_first_direct_byte of %k from %d to %d\n",
				   &ih->ih_key, sd->sd_first_direct_byte, value);		
		sd->sd_first_direct_byte = value;
		mark_buffer_dirty (bh);
		goto cont;		
	    }

	    if (code == 'D' && i == item_num) {
		delete_item (fs, bh, item_num);
		mark_buffer_dirty (bh);
		goto cont;
	    }

	    if (code == 'O' && i == item_num) {
		/* destroy item order */
		struct key * key;
		if (i == node_item_number (bh) - 1) {
		    printf ("can not destroy order\n");
		    continue;
		}
		key = &(ih + 1)->ih_key;
		ih->ih_key.k_dir_id = key->k_dir_id + 1;
		mark_buffer_dirty (bh);
	    }

	    if (ih->ih_key.k_objectid != objectid || !is_direntry_ih (ih))
		continue;

	    deh = B_I_DEH (bh, ih);

	    for (j = 0; j < ih_entry_count (ih); j ++, deh ++) {
		/* look for proper entry */
		if (name_length (ih, deh, j) != strlen (name) ||
		    strncmp (name, name_in_entry (deh, j), strlen (name)))
		    continue;

		/* ok, required entry found, make a corruption */
		switch (code) {
		case 'C': /* cut entry */
		    cut_entry (fs, bh, i, j, 1);
		    mark_buffer_dirty (bh);

		    if (!B_IS_IN_TREE (bh)) {
			printf ("NOTE: block is deleted from the tree\n");
			exit (0);
		    }
		    goto cont;
		    break;

		case 'H': /* clobber hash */
		    corrupt_clobber_hash (name, ih, deh);
		    goto cont;
		    break;

		case 'E': /* change entry's deh_objectid */
		    deh->deh_objectid = new_objectid;
		    break;

		default:
		    printf ("Unknown command found\n");
		}
		mark_buffer_dirty (bh);
	    }
	}
    cont:
    }
    free (line);
    printf ("Done\n");
    brelse (bh);
    return;
}


/* this reads stdin and recover file of given key:  */
/* the input has to be in the follwong format:
   K dirid objectid
   N name
   B blocknumber
   ..
   then recover_file will read every block, look there specified file and put it into
*/
static void do_recover (reiserfs_filsys_t fs)
{
    char name [100];
    char * line = 0;
    int n = 0;
    int fd;
    struct key key = {0, 0, };
    struct buffer_head * bh;
    struct item_head * ih;
    unsigned long block;
    char code;
    loff_t recovered = 0;
    int i, j;
    reiserfs_bitmap_t bitmap;
    int used, not_used;

    bitmap = reiserfs_create_bitmap (SB_BLOCK_COUNT (fs));
    reiserfs_fetch_disk_bitmap (bitmap, fs);
    /* we check how many blocks recoverd items point to are free or used */
    used = 0;
    not_used = 0;

    fd = 0;
    while (getline (&line, &n, stdin) != -1) {
	if (line [0] == '#' || line [0] == '\n')
	    continue;
	switch (line [0]) {
	case 'K':
	    /* get a key of file which is to be recovered */
	    if (sscanf (line, "%c %u %u\n", &code, &key.k_dir_id, &key.k_objectid) != 3) {
		die ("recover_file: wrong input K format");
	    }
	    printf ("Recovering file (%u, %u)\n", key.k_dir_id, key.k_objectid);
	    break;

	case 'N':
	    /* get a file name */
	    recovered = 0;
	    if (sscanf (line, "%c %s\n", &code, name) != 2) {
		die ("recover_file: wrong input N format");
	    }
	    fd = open (name, O_RDWR | O_CREAT | O_EXCL, 0644);
	    if (fd == -1)
		die ("recover_file: could not create file %s: %s",
		     name,strerror (errno));
	    printf ("Recovering file %s..\n", name);
	    break;

	case 'B':
	    if (!fd)
		die ("recover_file: file name is not specified");
	    if (sscanf (line, "%c %lu\n", &code, &block) != 2) {
		die ("recover_file: wrong input B format");
	    }
	    bh = bread (fs->s_dev, block, fs->s_blocksize);
	    if (!bh) {
		printf ("reading block %lu failed\n", block);
		continue;
	    }

	    printf ("working with block %lu..\n", block);

	    ih = B_N_PITEM_HEAD (bh, 0);
	    for (i = 0; i < node_item_number (bh); i ++, ih ++) {
		__u32 * indirect;
		struct buffer_head * tmp_bh;

		if (!is_indirect_ih (ih) || key.k_dir_id != ih->ih_key.k_dir_id ||
		    key.k_objectid != ih->ih_key.k_objectid)
		    continue;

		indirect = (__u32 *)B_I_PITEM (bh, ih);
		for (j = 0; j < I_UNFM_NUM (ih); j ++) {
		    block = le32_to_cpu (indirect [j]);
		    if (!block)
			continue;
		    tmp_bh = bread (fs->s_dev, block, fs->s_blocksize);
		    if (!tmp_bh) {
			printf ("reading block %Lu failed\n", (loff_t)block * fs->s_blocksize);
			continue;
		    }
		    if (lseek64 (fd, get_offset (&ih->ih_key) + j * fs->s_blocksize - 1,
				 SEEK_SET) == (loff_t)-1) {
			printf ("llseek failed to pos %Ld\n", (loff_t)block * fs->s_blocksize);
			brelse (tmp_bh);
			continue;
		    }
		    if (reiserfs_bitmap_test_bit (bitmap, block))
			used ++;
		    else
			not_used ++;
		    /*printf ("block of file %Ld gets block %lu\n",
		      (get_offset (&ih->ih_key) - 1) / fs->s_blocksize + j, block);*/
		    if (write (fd, tmp_bh->b_data, tmp_bh->b_size) != tmp_bh->b_size) {
			printf ("write failed to pos %Ld\n", (loff_t)block * fs->s_blocksize);
			brelse (tmp_bh);
			continue;
		    }
		    recovered += fs->s_blocksize;
		    brelse (tmp_bh);
		}
	    }
	    brelse (bh);
	    break;
	}
    }
    printf ("recover_file: %Ld bytes recovered of file %s, key %u %u, %d blocks are free and %d are used\n",
	    recovered, name, key.k_dir_id, key.k_objectid, not_used, used);
}


/* debugreiserfs -p or -P compresses reiserfs meta data: super block, journal,
   bitmap blocks and blocks looking like leaves. It may save "bitmap" of
   blocks they packed in the file of special format. Reiserfsck can then load
   "bitmap" saved in that file and build the tree of blocks marked used in
   that "bitmap" */
char * where_to_save;


static char * parse_options (struct debugreiserfs_data * data, int argc, char * argv [])
{
    int c;
    char * tmp;
  
    data->scan_area = USED_BLOCKS;
    data->mode = DO_DUMP;

    while ((c = getopt (argc, argv, "b:cSB:psnrdomjq")) != EOF) {
	switch (c) {
	case 'b':
	    /* will load bitmap from a file and read only blocks
               marked in it. This is for -p and -s */
	    asprintf (&data->input_bitmap, "%s", optarg);
	    data->scan_area = EXTERN_BITMAP;
	    break;

	case 'c':
	    /* mark blocks looking like leaves */
	    asprintf (&data->output_bitmap, "%s", optarg);
	    break;
	    
	case 'S':
	    /* have debugreiserfs -p or -s to read all the device */
	    data->scan_area = ALL_BLOCKS;
	    break;

	case 'B':	/* print a single node */
	    data->block = strtol (optarg, &tmp, 0);
	    if (*tmp)
		die ("parse_options: bad block number");
	    break;

	case 'C':
	    data->mode = DO_CORRUPT;
	    data->block = strtol (optarg, &tmp, 0);
	    if (*tmp)
		die ("parse_options: bad block number");
	    break;
	    
	case 'p':
	    data->mode = DO_PACK;
	    break;

	case 's':
	    /* read the device and print what reiserfs blocks were found */
	    data->mode = DO_SCAN;
	    break;

	case 'n':
	    data->mode = DO_SCAN_FOR_NAME;
	    break;

	case 'r':
	    data->mode = DO_RECOVER;
	    break;

	case 'd':
	    /*  print leaf details */
	    data->options |= PRINT_DETAILS;
	    break;

	case 'o':
	    /* print objectid map */
	    data->options |= PRINT_OBJECTID_MAP;
	    break;

	case 'm':	/* print a block map */
	case 'M':	/* print a block map with details */
	    data->options |= PRINT_BITMAP;
	    break;

	case 'j':
	    data->options |= PRINT_JOURNAL;
	    break;

	case 'q':
	    /* this makes packing to not show speed info during -p or -P */
	    data->options |= BE_QUIET;
	    break;
	}
    }

    if (optind != argc - 1)
	/* only one non-option argument is permitted */
	print_usage_and_exit();
  
    return argv[optind];
}





void pack_partition (reiserfs_filsys_t fs);

static void do_pack (reiserfs_filsys_t fs)
{
    if (certain_block (fs))
	pack_one_block (fs, certain_block (fs));
    else
	pack_partition (fs);
	
}

/* FIXME: statistics does not work */
static void do_dump_tree (reiserfs_filsys_t fs)
{
    if (certain_block (fs)) {
	print_one_block (fs, certain_block (fs));
	return;
    }

    print_block (stdout, fs, SB_BUFFER_WITH_SB (fs));
    
    if (data (fs)->options & PRINT_JOURNAL)
	print_journal (fs);
    
    if (data (fs)->options & PRINT_OBJECTID_MAP)
	print_objectid_map (stdout, fs);
    
    if (data (fs)->options & PRINT_BITMAP)
	print_bmap (stdout, fs, 1/*opt_print_block_map == 1 ? 1 : 0*/);
    
    if (data (fs)->options & PRINT_DETAILS) {
	print_disk_tree (fs, SB_ROOT_BLOCK (fs));
	
	/* print the statistic */
	printf ("File system uses %d internal + %d leaves + %d unformatted nodes = %d blocks\n",
		g_stat_info.nr_internals, g_stat_info.nr_leaves, g_stat_info.nr_unformatted, 
		g_stat_info.nr_internals + g_stat_info.nr_leaves + g_stat_info.nr_unformatted);
    }
}

FILE * log;

static void look_for_key (struct buffer_head * bh, struct key * key)
{
    int i, j;
    struct item_head * ih;
    struct reiserfs_de_head * deh;


    ih = B_N_PITEM_HEAD (bh, 0);
    for (i = 0; i < B_NR_ITEMS (bh); i ++, ih ++) {
	if ((ih->ih_key.k_dir_id == key->k_dir_id || (int)key->k_dir_id == -1) &&
	    ih->ih_key.k_objectid == key->k_objectid) {
	    reiserfs_warning (log, "%d-th item of block %lu is item of file %K: %H\n",
			      i, bh->b_blocknr, key, ih);
	}
	if (!is_direntry_ih (ih))
	    continue;
	deh = B_I_DEH (bh, ih);
	for (j = 0; j < ih_entry_count (ih); j ++, deh ++) {
	    if ((deh->deh_dir_id == key->k_dir_id || (int)key->k_dir_id == -1) &&
		deh->deh_objectid == key->k_objectid) {
		reiserfs_warning (log, "dir item %d (%H) of block %lu has "
				  "entry (%d-th) %.*s pointing to %K\n",
				  i, ih, bh->b_blocknr, j,
				  name_length (ih, deh, j), name_in_entry (deh, j), key);
	    }
	}	
    }
    return;
}


static void look_for_name (struct buffer_head * bh, char * name)
{
    int i, j;
    struct item_head * ih;
    struct reiserfs_de_head * deh;
    int namelen;
    char * p;

    ih = B_N_PITEM_HEAD (bh, 0);
    for (i = 0; i < B_NR_ITEMS (bh); i ++, ih ++) {
	if (!is_direntry_ih (ih))
	    continue;
	deh = B_I_DEH (bh, ih);
	for (j = 0; j < ih_entry_count (ih); j ++, deh ++) {
	    p = name_in_entry (deh, j);
	    namelen = name_length (ih, deh, j);
	    if (namelen == strlen (name) && !strncmp (name, p, namelen)) {
		fprintf (log, "block %lu, item %d, entry %d is %s\n", bh->b_blocknr, i, j, name);fflush (log);
	    }
	}
    }
    return;
}




static void do_scan (reiserfs_filsys_t fs)
{
    unsigned long i;
    struct buffer_head * bh;
    int type;
    char * answer = 0;
    size_t n = 0;
    struct key key = {0, 0, };
    unsigned long done, total;


    total = reiserfs_bitmap_ones (debug_bitmap (fs));

    log = fopen ("scan.log", "w+");

    if (debug_mode (fs) == DO_SCAN_FOR_NAME) {
	printf ("What name do you want to look for?");
	getline (&answer, &n, stdin);
	answer [strlen (answer) - 1] = 0;
	printf ("Looking for name \"%s\"..\n", answer);
	key.k_dir_id = 1;
    } else {
	printf ("What key do you want to find: dirid?");
	getline (&answer, &n, stdin);
	key.k_dir_id = atoi (answer);
	printf ("objectid?");
	getline (&answer, &n, stdin);
	key.k_objectid = atoi (answer);
	printf ("looking for (%u %u)\n", key.k_dir_id, key.k_objectid);
    }

    done = 0;
    for (i = 0; i < SB_BLOCK_COUNT (fs); i ++) {
	if (!reiserfs_bitmap_test_bit (debug_bitmap (fs), i))
	    continue;
	bh = bread (fs->s_dev, i, fs->s_blocksize);
	if (!bh) {
	    printf ("could not read block %lu\n", i);
	    continue;
	}
	type = who_is_this (bh->b_data, bh->b_size);
	switch (type) {
	case THE_JDESC:
	    if (!key.k_dir_id)
		printf ("block %lu is journal descriptor\n", i);
	    break;
	case THE_SUPER:
	    if (!key.k_dir_id)
		printf ("block %lu is reiserfs super block\n", i);
	    break;
	case THE_INTERNAL:
	    if (!key.k_dir_id)
		printf ("block %lu is reiserfs internal node\n", i);
	    break;
	case THE_LEAF:
	    if (debug_mode (fs) == DO_SCAN_FOR_NAME) {
		look_for_name (bh, answer);
	    } else if (key.k_dir_id) {
		look_for_key (bh, &key);
	    } else {
		printf ("block %lu is reiserfs leaf node\n", i);
	    }
	    break;
	}
	brelse (bh);
	print_how_far (stderr, &done, total, 1, 0);
    }
}


static void init_bitmap (reiserfs_filsys_t fs)
{
    unsigned long block_count;

    block_count = SB_BLOCK_COUNT (fs);

    switch (scan_area (fs)) {
    case ALL_BLOCKS:
	debug_bitmap (fs) = reiserfs_create_bitmap (block_count);
	reiserfs_bitmap_fill (debug_bitmap (fs));
	reiserfs_warning (stderr, "Whole device (%d blocks) is to be scanned\n", 
			  reiserfs_bitmap_ones (debug_bitmap (fs)));	
	break;
	
    case USED_BLOCKS:
	reiserfs_warning (stderr, "Loading on-disk bitmap .. ");
	debug_bitmap (fs) = reiserfs_create_bitmap (block_count);
	reiserfs_fetch_disk_bitmap (debug_bitmap (fs), fs);
	reiserfs_warning (stderr, "%d bits set - done\n",
			  reiserfs_bitmap_ones (debug_bitmap (fs)));
	break;
	
    case EXTERN_BITMAP:
	debug_bitmap (fs) = reiserfs_bitmap_load (input_bitmap_file_name(fs));
	if (!debug_bitmap (fs))
	    reiserfs_panic ("could not load fitmap from \"%s\"", 
			    input_bitmap_file_name(fs));
	break;

    default:
	reiserfs_panic ("No area to scan specified");
    }
}


/* FIXME: need to open reiserfs filesystem first */
int main (int argc, char * argv[])
{
    char * file_name;
    int error;
    struct debugreiserfs_data * data;

    print_banner ("debugreiserfs");
 
    data = getmem (sizeof (struct debugreiserfs_data));
    file_name = parse_options (data, argc, argv);

    fs = reiserfs_open (file_name, O_RDONLY, &error, data);
    if (!fs) {
	fprintf (stderr, "\n\ndumpreiserfs: can not open reiserfs on \"%s\": %s\n\n",
		 file_name, error ? strerror (error) : "there is no one");
	return 0;
    }

    init_bitmap (fs);

    switch (debug_mode (fs)) {
    case DO_PACK:
    case DO_PACK_ALL:
	do_pack (fs);
	break;

    case DO_CORRUPT:
	reiserfs_reopen (fs, O_RDWR);
	do_corrupt_one_block (fs);
	break;

    case DO_DUMP:
	do_dump_tree (fs);
	break;

    case DO_SCAN:
    case DO_SCAN_FOR_NAME:
	do_scan (fs);
	break;

    case DO_RECOVER:
	do_recover (fs);
	break;

    case DO_TEST:
	/*do_test (fs);*/
	break;
    }

    reiserfs_close (fs);
    return 0;
}
