/* 
 *  fsys_reiser4.c -- reiser4 filesystem support.
 *  Copyright (C) 2000, 2001   Free Software Foundation, Inc.
 *  
 *  GRUB  --  GRand Unified Bootloader
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef FSYS_REISER4
#include "shared.h"
#include "filesys.h"

#define ENABLE_MINIMAL
#include <reiser4/libreiser4.h>

static reiser4_fs_t *fs = NULL;
static aal_device_t *dev = NULL;
static reiser4_object_t *object = NULL;

/* Read callback of grub specific device. It uses devread() for reading passed
   @count of device blocks starting from @blk to passed @buff. */
static errno_t grub_dev_read(aal_device_t *device,
			     void *buff, blk_t blk,
			     count_t count)
{
	unsigned int size;
	unsigned int factor;
	unsigned int sector;

	/* Calculating actual sector and size in bytes to be read from
	   device. */
	factor = device->blksize / SECTOR_SIZE;
	sector = (unsigned int)blk << aal_log2(factor);
	size = (unsigned int)count * (SECTOR_SIZE * factor);
	
	/* Reading from the current device */
        if (!devread(sector, 0, size, buff))
    	        return -EIO;
		
	return 0;
}

/* Length callback of grub device */
static count_t grub_dev_len(aal_device_t *device) {
	unsigned int factor;

	/* Getting partition length in device blocks */
	factor = device->blksize / SECTOR_SIZE;
	return (part_length >> aal_log2(factor));
}

/*
  Initializing grub device abstraction instance. It will use devread and friends
  for providing needed functionality.
*/
struct aal_device_ops grub_dev_ops = {
	.read   = grub_dev_read,
	.len    = grub_dev_len
};

/* Initializes reiser4 */
static int reiser4_init(void) {
	extern aal_hash_table_t *plugins;
	
	plugins = NULL;

	/* Initializing memory manager */
	aal_mem_init((void *)FSYS_BUF, FSYS_BUFLEN);

	/* Initializing device abstraction on current device GRUB uses. */
	if (!(dev = aal_device_open(&grub_dev_ops, NULL,
				    SECTOR_SIZE, 0)))
	{
		return 0;
	}

	/* Initializing libreiser4 (plugins, etc) */
	return !libreiser4_init();
}

#define MEMORY_WATERMARK 8192

/* Memory pressure detect function. */
static int mpressure_detect(reiser4_tree_t *tree) {
	return (aal_mem_free() <= MEMORY_WATERMARK);
}

/* Reiser4 mount() routine */
int reiser4_mount(void) {
	
	/* Initialize all reiser4 related stuff first */
	if (!reiser4_init())
		return 0;
	
	/* Open filesystem on @dev. */
	if (!(fs = reiser4_fs_open(dev)))
		return 0;

	fs->tree->mpc_func = mpressure_detect;
	
	object = NULL;
	return 1;
}

/* Reiser4 read() handler */
int reiser4_read(char *buf, int len) {
	int64_t read;

	if (object == NULL)
		return 0;

	/* Seet at current position denoted by @filepos */
	if (objplug(object)->o.object_ops->seek) {
		plug_call(objplug(object)->o.object_ops,
			  seek, object->ent, filepos);
	}
	
	/* Reading current file data starting from @filepos */
	disk_read_func = disk_read_hook;
	read = objplug(object)->o.object_ops->read ?
		plug_call(objplug(object)->o.object_ops, read, 
			  object->ent, buf, len) : -EINVAL;
	disk_read_func = NULL;

	if (read < 0) {
		errnum = ERR_FSYS_CORRUPT;
		return 0;
	}

    	filepos += read;
	return read;
}

/* Reiser4 file open() routine */
int reiser4_dir(char *dirname) {
	char *ch;
	
	if (fs == NULL)
		return 0;

	if (object != NULL) {
		plug_call(objplug(object)->o.object_ops, 
			  close, object->ent);
		aal_free(object);
		object = NULL;
	}

	/* Cutting out string after first space character */
    	if ((ch = aal_strchr(dirname, ' ')))
		*ch = '\0';
		
	/* This function is also called for getting directory list for
	   maintaining the bash-like completion. */
#ifndef STAGE1_5
	if (print_possibilities) {
		char entry[256];
		entry_hint_t entry_hint;
		
		/* Getting last part of name (jsut after last '/') */
		if (*(dirname + aal_strlen(dirname) - 1) != '/') {
		
			if (!(ch = aal_strrchr(dirname, '/'))) {
				errnum = ERR_BAD_FILETYPE;
				return 0;
			}

			aal_strncpy(entry, ch + 1, sizeof(entry));
			*(ch + 1) = '\0';
		} else {
			aal_memset(entry, 0, sizeof(entry));
		}

		/* Open obejct by @dirname */
		if (!(object = reiser4_semantic_open(fs->tree, dirname, 
						     NULL, 1))) 
		{
			errnum = ERR_FILE_NOT_FOUND;
			return 0;
		}

		/* Checking if it is a directory object */
		if (object->ent->opset.plug[OPSET_OBJ]->id.group != DIR_OBJECT)
		{
			/* If not, cutting out last '/' character */
    			if ((ch = aal_strrchr(dirname, '/')))
				*ch = '\0';

			/* Close current object */
			plug_call(objplug(object)->o.object_ops, 
				  close, object->ent);
			aal_free(object);
			return 0;
		}
		
		/* Reading the opened directory to build the completion list. */
		if (objplug(object)->o.object_ops->readdir) {
			while (plug_call(objplug(object)->o.object_ops, readdir,
					 object->ent, &entry_hint) > 0) 
			{
				if (substring(entry, entry_hint.name) <= 0) {
					if (print_possibilities > 0)
						print_possibilities = 
							-print_possibilities;
				
					print_a_completion(entry_hint.name);
				}
			}
		}
	} else {
#endif
		/* This is the case when resier4_dir() is called for open the
		   file @dirname, not for building completion list. */
		if (!(object = reiser4_semantic_open(fs->tree, dirname,
						     NULL, 1))) 
		{
			errnum = ERR_FILE_NOT_FOUND;
			return 0;
		}
		
		if (object->ent->opset.plug[OPSET_OBJ]->id.group != REG_OBJECT)
		{
			errnum = ERR_BAD_FILETYPE;
			return 0;
		}
		
		/* Initializing GRUB global variables @filepos and @filemax. */
		filepos = 0;
		filemax = reiser4_object_size(object);
	
		return 1;
#ifndef STAGE1_5
	}

	return 1;
#endif

	errnum = ERR_FILE_NOT_FOUND;
	return 0;
}

/* Returns how many sectors may be used for embeding reiser4_stage1_5 in teh
   case of installing GRUB to partition instead of MBR. */
int reiser4_embed (int *start_sector, int needed_sectors) {
	*start_sector = 1;
	return needed_sectors <= ((REISER4_MASTER_OFFSET >> SECTOR_BITS) - 1);
}
#endif /* FSYS_REISER4 */
