/* Copyright 2001, 2002, 2003 by Hans Reiser, licensing governed by
 * reiser4/README */

/* Handling of "pseudo" files representing unified access to meta data in
   reiser4. See pseudo.c for more comments. */

#if !defined( __REISER4_PSEUDO_H__ )
#define __REISER4_PSEUDO_H__

#include "../plugin_header.h"
#include "../../key.h"

#include <linux/fs.h>
#include <linux/seq_file.h>

typedef enum {
	PSEUDO_READ_NONE,
	PSEUDO_READ_SEQ,
	PSEUDO_READ_SINGLE,
	PSEUDO_READ_FORWARD
} pseudo_read_type;

typedef enum {
	PSEUDO_WRITE_NONE,
	PSEUDO_WRITE_STRING,
	PSEUDO_WRITE_FORWARD
} pseudo_write_type;

/* low level operations on the pseudo files.

   Methods from this interface are directly callable by reiser4 system call.

   This operation structure looks suspiciously like yet another plugin
   type. Doing so would simplify some things. For example, there are already
   functions to look up plugin by name, dynamic loading is planned, etc.

*/
struct pseudo_plugin;
typedef struct pseudo_plugin pseudo_plugin;
struct pseudo_plugin {
	plugin_header h;

	int (*try) (pseudo_plugin *pplug,
		    const struct inode *parent, const char *name);
	/* lookup method applicable to this pseudo file by method name.
	
	   This is for something like "foo/..acl/dup", here "../acl" is the
	   name of a pseudo file, and "dup" is name of an operation (method)
	   applicable to "../acl". Once "..acl" is resolved to ACL object,
	   ->lookup( "dup" ) can be called to get operation.
	
	*/
	int (*lookup)(struct inode *parent, struct dentry ** dentry);

	oid_t (*makeid)(void);

	umode_t lookup_mode;

	/* NOTE-NIKITA some other operations. Reiser4 syntax people should
	   add something here. */

	pseudo_read_type read_type;
	union {
		struct seq_operations ops;
		int (*single_show) (struct seq_file *, void *);
		ssize_t (*read)(struct file *, char __user *, size_t , loff_t *);

	} read;

	pseudo_write_type write_type;
	union {
		int (*gets)(struct file *, const char *);
		ssize_t (*write)(struct file *,
				 const char __user *, size_t , loff_t *);
	} write;
	int (*readdir)(struct file *f, void *dirent, filldir_t filld);
};

/* portion of reiser4_inode specific for pseudo files */
typedef struct pseudo_info {
	/* pseudo file plugin controlling this file */
	pseudo_plugin *plugin;
	/* host object, for /etc/passwd/..oid, this is pointer to inode of
	 * /etc/passwd */
	struct inode  *host;
	/* for private use of pseudo file plugin */
	unsigned long  datum;
} pseudo_info_t;

extern int lookup_pseudo_file(struct inode *parent, struct dentry **dentry);

typedef enum {
	PSEUDO_UID_ID,
	PSEUDO_GID_ID,
	PSEUDO_RWX_ID,
	PSEUDO_OID_ID,
	PSEUDO_KEY_ID,
	PSEUDO_SIZE_ID,
	PSEUDO_NLINK_ID,
	PSEUDO_LOCALITY_ID,
	PSEUDO_PAGECACHE_ID,
	PSEUDO_PAGECACHE_STAT_ID,
	PSEUDO_PSEUDOS_ID,
	PSEUDO_BMAP_ID,
	PSEUDO_READDIR_ID,
	PSEUDO_PLUGIN_ID,
	PSEUDO_PLUGINS_ID,
	PSEUDO_PLUGIN_FIELD_ID,
	PSEUDO_ITEMS_ID,
	PSEUDO_NEW_ID,
	LAST_PSEUDO_ID
} reiser4_pseudo_id;

/* __REISER4_PSEUDO_H__ */
#endif

/* Make Linus happy.
   Local variables:
   c-indentation-style: "K&R"
   mode-name: "LC"
   c-basic-offset: 8
   tab-width: 8
   fill-column: 120
   scroll-step: 1
   End:
*/
