/* Copyright 2001, 2002, 2003 by Hans Reiser, licensing governed by
 * reiser4/README */

/* Handling of "pseudo" files representing unified access to meta data in
   reiser4. */

/*
 * See http://namesys.com/v4/v4.html, and especially
 * http://namesys.com/v4/pseudo.html for basic information about reiser4
 * pseudo files, access to meta-data, reiser4() system call, etc.
 *
 * Pseudo files should be accessible from both reiser4() system call and
 * normal POSIX calls.
 *
 * OVERVIEW
 *
 *     Pseudo files provide access to various functionality through file
 *     system name space. As such they are similar to pseudo file systems
 *     already existing in UNIX and Linux: procfs, sysfs, etc. But pseudo
 *     files are embedded into name space of Reiser4---real block device based
 *     file system, and are more tightly integrated with it. In particular,
 *     some pseudo files are "attached" to other files (either "real" or also
 *     pseudo), by being accessible through path names of the form
 *
 *         "a/b/c/..something"
 *
 *     Here object accessible through "a/b/c/..something" is attached to the
 *     object accessible through "a/b/c" , and the latter is said to be the
 *     "host" object of the former.
 *
 *     Object can have multiple pseudo files attached to it, distinguished by
 *     the last component of their names "..something", "..somethingelse",
 *     etc.
 *
 *     (Note however, that ".." prefix is just a convention, and it is not
 *     necessary that all pseudo file names started with it.)
 *
 *     Moreover, in addition to the purely pseudo files (that is, file system
 *     objects whose content (as available through read(2) system call) is not
 *     backed by any kind of persistent storage), extended file attributes
 *     (see attr(5) on Linux, and http://acl.bestbits.at/) including security
 *     attributes such as ACLs are also available through file system name
 *     space.
 *
 *     As a result each file system object has a sub-name-space rooted at it,
 *     which is in striking contrast with traditional UNIX file system, where
 *     only directories has sub-objects and all other types of files (regular,
 *     FIFO-s, devices, and symlinks) are leaves.
 *
 *     For the sake of objectivity it should be mentioned that this is not
 *     _completely_ new development in file system design, see
 *     http://docs.sun.com/db/doc/816-0220/6m6nkorp9?a=view
 *
 *     In particular, as each object has sub-objects, name space tree is
 *     infinite in both extent (number of reachable objects) and depth.
 *
 *     Some pseudo files are "built-in". They are present as sub-objects in
 *     each file system object, unless specifically disabled.
 *
 *     Built-in pseudo files are implemented in this file and described at
 *     http://namesys.com/v4/pseudo.html
 *
 * IMPLEMENTATION
 *
 *     Pseudo files are implemented as normal inodes, living in the same super
 *     block as other inodes for reiser4 file system. Their inode numbers are
 *     generated by fs/inode.c:new_inode() function and are not persistent (in
 *     the sense that they are not guaranteed to be the same after
 *     remount). To avoid clashes with "normal" inodes, all pseudo inodes are
 *     placed into otherwise unused locality (for example, 0), hence allowing
 *     reiser4_inode_find_actor() to tell them from normal inodes.
 *
 *     All pseudo inodes share the same object plugin
 *     PSEUDO_FILE_PLUGIN_ID. In pseudo-inode specific part of reiser4_inode
 *     (pseudo_info), two things are stored:
 *
 *         1. pointer to the inode of the "host object" (for /a/b/c/..acl,
 *         /a/b/c is the host object)
 *
 *         2. pointer to pseudo plugin, used by PSEUDO_FILE_PLUGIN_ID to
 *         implement VFS operations.
 *
 *     This design has following advantages:
 *
 *         1. provides for ease addition of new pseudo files without going
 *         through writing whole new object plugin.
 *
 *         2. allows sys_reiser4() to be implemented by directory invoking
 *         pseudo plugin methods.
 *
 */

#include "../../inode.h"
#include "../../debug.h"
#include "../plugin.h"

#include "pseudo.h"

static int init_pseudo(struct inode *parent, struct inode *pseudo,
		       pseudo_plugin *pplug, const char *name);

static struct inode *add_pseudo(struct inode *parent,
				pseudo_plugin *pplug, struct dentry **d);

static void pseudo_set_datum(struct inode *pseudo, unsigned long datum)
{
	reiser4_inode_data(pseudo)->file_plugin_data.pseudo_info.datum = datum;
}

/*
 * try to look up built-in pseudo file by its name.
 */
reiser4_internal int
lookup_pseudo_file(struct inode *parent, struct dentry **dentry)
{
	reiser4_plugin *plugin;
	const char     *name;
	struct inode   *pseudo;
	int             result;

	assert("nikita-2999", parent != NULL);
	assert("nikita-3000", dentry != NULL);

	/* if pseudo files are disabled for this file system bail out */
	if (reiser4_is_set(parent->i_sb, REISER4_NO_PSEUDO))
		return RETERR(-ENOENT);

	name = (*dentry)->d_name.name;
	pseudo = ERR_PTR(-ENOENT);
	/* scan all pseudo file plugins and check each */
	for_all_plugins(REISER4_PSEUDO_PLUGIN_TYPE, plugin) {
		pseudo_plugin *pplug;

		pplug = &plugin->pseudo;
		if (pplug->try != NULL && pplug->try(pplug, parent, name)) {
			pseudo = add_pseudo(parent, pplug, dentry);
			break;
		}
	}
	if (!IS_ERR(pseudo))
		result = 0;
	else
		result = PTR_ERR(pseudo);
	return result;
}

static struct inode *add_pseudo(struct inode *parent,
				pseudo_plugin *pplug, struct dentry **d)
{
	struct inode *pseudo;

	pseudo = new_inode(parent->i_sb);
	if (pseudo != NULL) {
		int result;

		result = init_pseudo(parent, pseudo, pplug, (*d)->d_name.name);
		if (result != 0)
			pseudo = ERR_PTR(result);
		else
			*d = d_splice_alias(pseudo, *d);
	} else
		pseudo = ERR_PTR(RETERR(-ENOMEM));
	return pseudo;
}


/*
 * initialize pseudo file @pseudo to be child of @parent, with plugin @pplug
 * and name @name.
 */
static int
init_pseudo(struct inode *parent, struct inode *pseudo,
	    pseudo_plugin *pplug, const char *name)
{
	int result;
	reiser4_inode *idata;
	reiser4_object_create_data data;
	static const oid_t pseudo_locality = 0x0ull;

	idata = reiser4_inode_data(pseudo);
	idata->locality_id = pseudo_locality;
	idata->file_plugin_data.pseudo_info.host   = parent;
	idata->file_plugin_data.pseudo_info.plugin = pplug;

	data.id   = PSEUDO_FILE_PLUGIN_ID;
	data.mode = pplug->lookup_mode;

	plugin_set_file(&idata->pset, file_plugin_by_id(PSEUDO_FILE_PLUGIN_ID));
	/* if plugin has a ->lookup method, it means that @pseudo should
	 * behave like directory. */
	if (pplug->lookup != NULL)
		plugin_set_dir(&idata->pset,
			       dir_plugin_by_id(PSEUDO_DIR_PLUGIN_ID));

	result = inode_file_plugin(pseudo)->set_plug_in_inode(pseudo,
							      parent, &data);
	if (result != 0) {
		warning("nikita-3203", "Cannot install pseudo plugin");
		print_plugin("plugin", pseudo_plugin_to_plugin(pplug));
		return result;
	}

	/* inherit permission plugin from parent */
	grab_plugin(idata, reiser4_inode_data(parent), perm);

	pseudo->i_nlink = 1;
	/* insert inode into VFS hash table */
	insert_inode_hash(pseudo);
	return 0;
}

/* helper function: return host object of @inode pseudo file */
static struct inode *get_inode_host(struct inode *inode)
{
	return reiser4_inode_data(inode)->file_plugin_data.pseudo_info.host;
}

/* helper function: return host object by file descriptor */
static struct inode *get_pseudo_host(struct file *file)
{
	struct inode *inode;

	inode = file->f_dentry->d_inode;
	return get_inode_host(inode);
}

/* helper function: return host object by seq_file */
static struct inode *get_seq_pseudo_host(struct seq_file *seq)
{
	struct file *file;

	file = seq->private;
	return get_pseudo_host(file);
}


static int try_by_label(pseudo_plugin *pplug,
			const struct inode *parent, const char *name)
{
	return !strcmp(name, pplug->h.label);
}

static int show_uid(struct seq_file *seq, void *cookie)
{
	seq_printf(seq, "%lu", (long unsigned)get_seq_pseudo_host(seq)->i_uid);
	return 0;
}

static int check_perm(struct inode *inode)
{
	if (IS_RDONLY(inode))
		return RETERR(-EROFS);
	if (IS_IMMUTABLE(inode) || IS_APPEND(inode))
		return RETERR(-EPERM);
	return 0;
}

static int update_ugid(struct dentry *dentry, struct inode *inode,
		       uid_t uid, gid_t gid)
{
	int result;

	/* logic COPIED from fs/open.c:chown_common() */
	result = check_perm(inode);
	if (result == 0) {
		struct iattr newattrs;

		newattrs.ia_valid =  ATTR_CTIME;
		if (uid != (uid_t) -1) {
			newattrs.ia_valid |= ATTR_UID;
			newattrs.ia_uid = uid;
		}
		if (gid != (uid_t) -1) {
			newattrs.ia_valid |= ATTR_GID;
			newattrs.ia_gid = gid;
		}
		if (!S_ISDIR(inode->i_mode))
			newattrs.ia_valid |= ATTR_KILL_SUID|ATTR_KILL_SGID;
		down(&inode->i_sem);
		result = notify_change(dentry, &newattrs);
		up(&inode->i_sem);
	}
	return result;
}

static int get_uid(struct file *file, const char *buf)
{
	uid_t uid;
	int result;

	if (sscanf(buf, "%i", &uid) == 1) {
		struct inode *host;

		host = get_pseudo_host(file);
		result = update_ugid(file->f_dentry->d_parent, host, uid, -1);
	} else
		result = RETERR(-EINVAL);
	return result;
}

static int show_gid(struct seq_file *seq, void *cookie)
{
	seq_printf(seq, "%lu", (long unsigned)get_seq_pseudo_host(seq)->i_gid);
	return 0;
}

static int get_gid(struct file *file, const char *buf)
{
	gid_t gid;
	int result;

	if (sscanf(buf, "%i", &gid) == 1) {
		struct inode *host;

		host = get_pseudo_host(file);
		result = update_ugid(file->f_dentry->d_parent, host, -1, gid);
	} else
		result = RETERR(-EINVAL);
	return result;
}

static int show_oid(struct seq_file *seq, void *cookie)
{
	seq_printf(seq, "%llu", get_inode_oid(get_seq_pseudo_host(seq)));
	return 0;
}

static int show_key(struct seq_file *seq, void *cookie)
{
	char buf[KEY_BUF_LEN];
	reiser4_key key;

	sprintf_key(buf, build_sd_key(get_seq_pseudo_host(seq), &key));
	seq_printf(seq, "%s", buf);
	return 0;
}

static int show_size(struct seq_file *seq, void *cookie)
{
	seq_printf(seq, "%lli", get_seq_pseudo_host(seq)->i_size);
	return 0;
}

static int show_nlink(struct seq_file *seq, void *cookie)
{
	seq_printf(seq, "%u", get_seq_pseudo_host(seq)->i_nlink);
	return 0;
}

static int show_locality(struct seq_file *seq, void *cookie)
{
	seq_printf(seq, "%llu",
		   reiser4_inode_data(get_seq_pseudo_host(seq))->locality_id);
	return 0;
}

static int show_rwx(struct seq_file *seq, void *cookie)
{
	umode_t      m;

	m = get_seq_pseudo_host(seq)->i_mode;
	seq_printf(seq, "%#ho %c%c%c%c%c%c%c%c%c%c",
		   m,

		   S_ISREG(m) ? '-' :
		   S_ISDIR(m) ? 'd' :
		   S_ISCHR(m) ? 'c' :
		   S_ISBLK(m) ? 'b' :
		   S_ISFIFO(m) ? 'p' :
		   S_ISLNK(m) ? 'l' :
		   S_ISSOCK(m) ? 's' : '?',

		   m & S_IRUSR ? 'r' : '-',
		   m & S_IWUSR ? 'w' : '-',
		   m & S_IXUSR ? 'x' : '-',

		   m & S_IRGRP ? 'r' : '-',
		   m & S_IWGRP ? 'w' : '-',
		   m & S_IXGRP ? 'x' : '-',

		   m & S_IROTH ? 'r' : '-',
		   m & S_IWOTH ? 'w' : '-',
		   m & S_IXOTH ? 'x' : '-');
	return 0;
}

static int get_rwx(struct file *file, const char *buf)
{
	umode_t rwx;
	int result;

	if (sscanf(buf, "%hi", &rwx) == 1) {
		struct inode *host;

		host = get_pseudo_host(file);
		result = check_perm(host);
		if (result == 0) {
			struct iattr newattrs;

			down(&host->i_sem);
			if (rwx == (mode_t) -1)
				rwx = host->i_mode;
			newattrs.ia_mode =
				(rwx & S_IALLUGO) | (host->i_mode & ~S_IALLUGO);
			newattrs.ia_valid = ATTR_MODE | ATTR_CTIME;
			result = notify_change(file->f_dentry->d_parent,
					       &newattrs);
			up(&host->i_sem);
		}
	} else
		result = RETERR(-EINVAL);
	return result;
}

static unsigned long list_length(const struct list_head *head)
{
	struct list_head *scan;
	unsigned long length;

	length = 0;
	list_for_each(scan, head)
		++ length;
	return length;
}

static void * pseudos_start(struct seq_file *m, loff_t *pos)
{
	if (*pos >= LAST_PSEUDO_ID)
		return NULL;
	return pseudo_plugin_by_id(*pos);
}

static void pseudos_stop(struct seq_file *m, void *v)
{
}

static void * pseudos_next(struct seq_file *m, void *v, loff_t *pos)
{
	++ (*pos);
	return pseudos_start(m, pos);
}

static int pseudos_show(struct seq_file *m, void *v)
{
	pseudo_plugin *pplug;

	pplug = v;
	if (pplug->try != NULL)
		seq_printf(m, "%s\n", pplug->h.label);
	return 0;
}

static void * bmap_start(struct seq_file *m, loff_t *pos)
{
	struct inode *host;

	host = get_seq_pseudo_host(m);
	if (*pos << host->i_blkbits >= host->i_size)
		return NULL;
	else
		return (void *)((unsigned long)*pos + 1);
}

static void bmap_stop(struct seq_file *m, void *v)
{
}

static void * bmap_next(struct seq_file *m, void *v, loff_t *pos)
{
	++ (*pos);
	return bmap_start(m, pos);
}

extern int reiser4_lblock_to_blocknr(struct address_space *mapping,
				     sector_t lblock, reiser4_block_nr *blocknr);


static int bmap_show(struct seq_file *m, void *v)
{
	sector_t lblock;
	int result;
	reiser4_block_nr blocknr;

	lblock = ((sector_t)(unsigned long)v) - 1;
	result = reiser4_lblock_to_blocknr(get_seq_pseudo_host(m)->i_mapping,
					   lblock, &blocknr);
	if (result == 0) {
		if (blocknr_is_fake(&blocknr))
			seq_printf(m, "%#llx\n", blocknr);
		else
			seq_printf(m, "%llu\n", blocknr);
	}
	return result;
}

typedef struct readdir_cookie {
	tap_t       tap;
	coord_t     coord;
	lock_handle lh;
} readdir_cookie;

static int is_host_item(struct inode *host, coord_t *coord)
{
	if (item_type_by_coord(coord) != DIR_ENTRY_ITEM_TYPE)
		return 0;
	if (!inode_file_plugin(host)->owns_item(host, coord))
		return 0;
	return 1;
}

static void finish(readdir_cookie *c)
{
	if (c != NULL && !IS_ERR(c)) {
		tap_done(&c->tap);
		kfree(c);
	}
}

static void * readdir_start(struct seq_file *m, loff_t *pos)
{
	struct inode   *host;
	readdir_cookie *c;
	dir_plugin     *dplug;
	reiser4_key     dotkey;
	struct qstr     dotname;
	int             result;
	loff_t          entryno;

	host = get_seq_pseudo_host(m);
	dplug = inode_dir_plugin(host);

	dotname.name = ".";
	dotname.len  = 1;

	down(&host->i_sem);
	if (dplug == NULL) {
		finish(NULL);
		return NULL;
	}

	dplug->build_entry_key(host, &dotname, &dotkey);

	c = kmalloc(sizeof *c, GFP_KERNEL);
	if (c == NULL) {
		finish(NULL);
		return ERR_PTR(RETERR(-ENOMEM));
	}

	result = object_lookup(host,
			       &dotkey,
			       &c->coord,
			       &c->lh,
			       ZNODE_READ_LOCK,
			       FIND_EXACT,
			       LEAF_LEVEL,
			       LEAF_LEVEL,
			       CBK_READDIR_RA,
			       NULL);

	tap_init(&c->tap, &c->coord, &c->lh, ZNODE_READ_LOCK);
	if (result == 0)
		result = tap_load(&c->tap); {
		if (result == 0) {
			for (entryno = 0; entryno != *pos; ++ entryno) {
				result = go_next_unit(&c->tap);
				if (result == -E_NO_NEIGHBOR) {
					finish(c);
					return NULL;
				}
				if (result != 0)
					break;
				if (!is_host_item(host, c->tap.coord)) {
					finish(c);
					return NULL;
				}
			}
		}
	}
	if (result != 0) {
		finish(c);
		return ERR_PTR(result);
	} else
		return c;
}

static void readdir_stop(struct seq_file *m, void *v)
{
	up(&get_seq_pseudo_host(m)->i_sem);
	finish(v);
}

static void * readdir_next(struct seq_file *m, void *v, loff_t *pos)
{
	readdir_cookie *c;
	struct inode   *host;
	int result;

	c = v;
	++ (*pos);
	host = get_seq_pseudo_host(m);
	result = go_next_unit(&c->tap);
	if (result == 0) {
		if (!is_host_item(host, c->tap.coord)) {
			finish(c);
			return NULL;
		} else
			return v;
	} else {
		finish(c);
		return ERR_PTR(result);
	}
}

static int readdir_show(struct seq_file *m, void *v)
{
	readdir_cookie *c;
	item_plugin *iplug;
	char *name;
	char buf[DE_NAME_BUF_LEN];

	c = v;
	iplug = item_plugin_by_coord(&c->coord);

	name = iplug->s.dir.extract_name(&c->coord, buf);
	assert("nikita-3221", name != NULL);
	seq_printf(m, "%s/", name);
	return 0;
}

typedef struct plugin_entry {
	const char *name;
	int         offset;
} plugin_entry;

#define PLUGIN_ENTRY(field)			\
{						\
	.name = #field,				\
	.offset = offsetof(plugin_set, field)	\
}

#define PSEUDO_ARRAY_ENTRY(idx, aname)		\
[idx] = {					\
	.name = aname,				\
	.offset = idx				\
}

static plugin_entry pentry[] = {
	PLUGIN_ENTRY(file),
	PLUGIN_ENTRY(dir),
	PLUGIN_ENTRY(perm),
	PLUGIN_ENTRY(formatting),
	PLUGIN_ENTRY(hash),
	PLUGIN_ENTRY(sd),
	PLUGIN_ENTRY(dir_item),
	PLUGIN_ENTRY(crypto),
	PLUGIN_ENTRY(digest),
	PLUGIN_ENTRY(compression),
	{
		.name = NULL,
		.offset = 0
	}
};

typedef enum {
	PFIELD_TYPEID,
	PFIELD_ID,
	PFIELD_LABEL,
	PFIELD_DESC
} plugin_field;

static plugin_entry fentry[] = {
	PSEUDO_ARRAY_ENTRY(PFIELD_TYPEID, "type_id"),
	PSEUDO_ARRAY_ENTRY(PFIELD_ID, "id"),
	PSEUDO_ARRAY_ENTRY(PFIELD_LABEL, "label"),
	PSEUDO_ARRAY_ENTRY(PFIELD_DESC, "desc"),
	{
		.name   = NULL,
		.offset = 0
	},
};

static int show_plugin(struct seq_file *seq, void *cookie)
{
	struct inode   *host;
	struct file    *file;
	struct inode   *inode;
	reiser4_plugin *plug;
	plugin_entry   *entry;
	int             idx;
	plugin_set     *pset;

	file  = seq->private;
	inode = file->f_dentry->d_inode;

	/* foo is grandparent of foo/..plugin/file  */
	host  = get_inode_host(get_inode_host(inode));
	idx   = reiser4_inode_data(inode)->file_plugin_data.pseudo_info.datum;
	entry = &pentry[idx];
	pset  = reiser4_inode_data(host)->pset;
	plug  = *(reiser4_plugin **)(((char *)pset) + entry->offset);

	if (plug != NULL)
		seq_printf(seq, "%i %s %s",
			   plug->h.id, plug->h.label, plug->h.desc);
	return 0;
}

static int array_lookup_pseudo(struct inode *parent, struct dentry ** dentry,
			       plugin_entry *array, pseudo_plugin *pplug)
{
	int result;
	int idx;
	struct inode *pseudo;

	pseudo = ERR_PTR(-ENOENT);
	for (idx = 0; array[idx].name != NULL; ++ idx) {
		if (!strcmp((*dentry)->d_name.name, array[idx].name)) {
			pseudo = add_pseudo(parent, pplug, dentry);
			break;
		}
	}
	if (IS_ERR(pseudo))
		result = PTR_ERR(pseudo);
	else {
		result = 0;
		pseudo_set_datum(pseudo, idx);
	}
	return result;
}

static int array_readdir_pseudo(struct file *f, void *dirent, filldir_t filld,
				plugin_entry *array, int size)
{
	loff_t off;
	ino_t  ino;

	off = f->f_pos;
	if (off < 0)
		return 0;

	/* for god's sake, why switch(loff_t) requires __cmpdi2? */
	switch ((int)off) {
	case 0:
		ino = f->f_dentry->d_inode->i_ino;
		if (filld(dirent, ".", 1, off, ino, DT_DIR) < 0)
			break;
		++ off;
		/* fallthrough */
	case 1:
		ino = parent_ino(f->f_dentry);
		if (filld(dirent, "..", 2, off, ino, DT_DIR) < 0)
			break;
		++ off;
		/* fallthrough */
	default:
		for (; off < size + 1; ++ off) {
			const char *name;

			name = array[off - 2].name;
			if (filld(dirent, name, strlen(name),
				  off, off + (long)f, DT_REG) < 0)
				break;
		}
	}
	f->f_pos = off;
	return 0;
}


static int lookup_plugin_field(struct inode *parent, struct dentry ** dentry)
{
	return array_lookup_pseudo(parent, dentry, fentry,
				   pseudo_plugin_by_id(PSEUDO_PLUGIN_FIELD_ID));
}

static int show_plugin_field(struct seq_file *seq, void *cookie)
{
	struct inode   *host;
	struct inode   *parent;
	struct file    *file;
	struct inode   *inode;
	reiser4_plugin *plug;
	plugin_entry   *entry;
	int             pidx;
	int             idx;
	plugin_set     *pset;

	file  = seq->private;
	inode = file->f_dentry->d_inode;

	parent = get_inode_host(inode);
	/* foo is grand-grand-parent of foo/..plugin/hash/id  */
	host  = get_inode_host(get_inode_host(parent));
	pidx  = reiser4_inode_data(parent)->file_plugin_data.pseudo_info.datum;
	idx   = reiser4_inode_data(inode)->file_plugin_data.pseudo_info.datum;
	entry = &pentry[pidx];
	pset  = reiser4_inode_data(host)->pset;
	plug  = *(reiser4_plugin **)(((char *)pset) + entry->offset);

	if (plug != NULL) {
		switch (idx) {
		case PFIELD_TYPEID:
			seq_printf(seq, "%i", plug->h.type_id);
			break;
		case PFIELD_ID:
			seq_printf(seq, "%i", plug->h.id);
			break;
		case PFIELD_LABEL:
			seq_printf(seq, "%s", plug->h.label);
			break;
		case PFIELD_DESC:
			seq_printf(seq, "%s", plug->h.desc);
			break;
		}
	}

	return 0;
}

static int readdir_plugin_field(struct file *f, void *dirent, filldir_t filld)
{
	return array_readdir_pseudo(f, dirent, filld,
				    fentry, sizeof_array(fentry));
}

static int lookup_plugins(struct inode *parent, struct dentry ** dentry)
{
	return array_lookup_pseudo(parent, dentry, pentry,
				   pseudo_plugin_by_id(PSEUDO_PLUGIN_ID));
}

static int readdir_plugins(struct file *f, void *dirent, filldir_t filld)
{
	return array_readdir_pseudo(f, dirent, filld,
				    pentry, sizeof_array(pentry));
}

typedef enum {
	PAGECACHE_NRPAGES,
	PAGECACHE_CLEAN,
	PAGECACHE_DIRTY,
	PAGECACHE_LOCKED,
	PAGECACHE_IO
} pagecache_stat;

static plugin_entry pagecache_entry[] = {
	PSEUDO_ARRAY_ENTRY(PAGECACHE_NRPAGES, "nrpages"),
	PSEUDO_ARRAY_ENTRY(PAGECACHE_CLEAN, "clean"),
	PSEUDO_ARRAY_ENTRY(PAGECACHE_DIRTY, "dirty"),
	PSEUDO_ARRAY_ENTRY(PAGECACHE_LOCKED, "locked"),
	PSEUDO_ARRAY_ENTRY(PAGECACHE_IO, "io"),
	{
		.name   = NULL,
		.offset = 0
	},
};

static int show_pagecache(struct seq_file *seq, void *cookie)
{
	struct inode *host;
	struct address_space *as;

	unsigned long nrpages;
	unsigned long clean;
	unsigned long dirty;
	unsigned long locked;
	unsigned long io;

	host = get_seq_pseudo_host(seq);
	
	as = host->i_mapping;
	spin_lock(&as->page_lock);
	nrpages = as->nrpages;
	clean   = list_length(&as->clean_pages);
	dirty   = list_length(&as->dirty_pages);
	locked  = list_length(&as->locked_pages);
	io      = list_length(&as->io_pages);
	spin_unlock(&as->page_lock);

	seq_printf(seq, "%lu %lu %lu %lu %lu",
		   nrpages, clean, dirty, locked, io);
	return 0;
}

static int readdir_pagecache(struct file *f, void *dirent, filldir_t filld)
{
	return array_readdir_pseudo(f, dirent, filld,
				    pagecache_entry,
				    sizeof_array(pagecache_entry));
}

static int lookup_pagecache(struct inode *parent, struct dentry ** dentry)
{
	return array_lookup_pseudo(parent, dentry, pagecache_entry,
				   pseudo_plugin_by_id(PSEUDO_PAGECACHE_STAT_ID));
}

static int show_pagecache_stat(struct seq_file *seq, void *cookie)
{
	struct inode   *host;
	struct file    *file;
	struct inode   *inode;
	int             idx;

	struct address_space *as;

	file  = seq->private;
	inode = file->f_dentry->d_inode;

	/* foo is grand-parent of foo/..pagecache/dirty  */
	host  = get_inode_host(get_inode_host(inode));
	idx   = reiser4_inode_data(inode)->file_plugin_data.pseudo_info.datum;
	as    = host->i_mapping;
	spin_lock(&as->page_lock);
	switch (idx) {
	case PAGECACHE_NRPAGES:
		seq_printf(seq, "%lu", as->nrpages);
		break;
	case PAGECACHE_CLEAN:
		seq_printf(seq, "%lu", list_length(&as->clean_pages));
		break;
	case PAGECACHE_DIRTY:
		seq_printf(seq, "%lu", list_length(&as->dirty_pages));
		break;
	case PAGECACHE_LOCKED:
		seq_printf(seq, "%lu", list_length(&as->locked_pages));
		break;
	case PAGECACHE_IO:
		seq_printf(seq, "%lu", list_length(&as->io_pages));
		break;
	}
	spin_unlock(&as->page_lock);

	return 0;
}

static void * items_start(struct seq_file *m, loff_t *pos)
{
	struct inode   *host;
	readdir_cookie *c;
	file_plugin    *fplug;
	reiser4_key     headkey;
	int             result;
	loff_t          entryno;

	host = get_seq_pseudo_host(m);
	fplug = inode_file_plugin(host);

	down(&host->i_sem);
	if (fplug->key_by_inode == NULL) {
		finish(NULL);
		return NULL;
	}

	fplug->key_by_inode(host, 0, &headkey);

	c = kmalloc(sizeof *c, GFP_KERNEL);
	if (c == NULL) {
		finish(NULL);
		return ERR_PTR(RETERR(-ENOMEM));
	}

	result = object_lookup(host,
			       &headkey,
			       &c->coord,
			       &c->lh,
			       ZNODE_READ_LOCK,
			       FIND_MAX_NOT_MORE_THAN,
			       TWIG_LEVEL,
			       LEAF_LEVEL,
			       0,
			       NULL);

	tap_init(&c->tap, &c->coord, &c->lh, ZNODE_READ_LOCK);
	if (result == 0)
		result = tap_load(&c->tap); {
		if (result == 0) {
			for (entryno = 0; entryno != *pos; ++ entryno) {
				result = go_next_unit(&c->tap);
				if (result == -E_NO_NEIGHBOR) {
					finish(c);
					return NULL;
				}
				if (result != 0)
					break;
				if (!fplug->owns_item(host, c->tap.coord)) {
					finish(c);
					return NULL;
				}
			}
		}
	}
	if (result != 0) {
		finish(c);
		return ERR_PTR(result);
	} else
		return c;
}

static void items_stop(struct seq_file *m, void *v)
{
	up(&get_seq_pseudo_host(m)->i_sem);
	finish(v);
}

static void * items_next(struct seq_file *m, void *v, loff_t *pos)
{
	readdir_cookie *c;
	struct inode   *host;
	int result;

	c = v;
	++ (*pos);
	host = get_seq_pseudo_host(m);
	result = go_next_unit(&c->tap);
	if (result == 0) {
		if (!inode_file_plugin(host)->owns_item(host, c->tap.coord)) {
			finish(c);
			return NULL;
		} else
			return v;
	} else {
		finish(c);
		return ERR_PTR(result);
	}
}

static int items_show(struct seq_file *m, void *v)
{
	readdir_cookie *c;
	item_plugin    *iplug;
	char            buf[KEY_BUF_LEN];
	reiser4_key     key;


	c = v;
	iplug = item_plugin_by_coord(&c->coord);

	sprintf_key(buf, unit_key_by_coord(&c->coord, &key));
	seq_printf(m, "%s %s ", buf, iplug->h.label);
	if (iplug->b.show != NULL)
		iplug->b.show(m, &c->coord);
	seq_printf(m, "\n");
	return 0;
}

static int get_new(struct file *file, const char *buf)
{
	int result;

	if (strchr(buf, '/') == NULL) {
		result = RETERR(-ENOSYS);
	} else
		result = RETERR(-EINVAL);
	return result;
}

pseudo_plugin pseudo_plugins[LAST_PSEUDO_ID] = {
	[PSEUDO_UID_ID] = {
			 .h = {
			       .type_id = REISER4_PSEUDO_PLUGIN_TYPE,
			       .id = PSEUDO_UID_ID,
			       .pops = NULL,
			       .label = "..uid",
			       .desc = "returns owner",
			       .linkage = TYPE_SAFE_LIST_LINK_ZERO
			 },
			 .try         = try_by_label,
			 .lookup      = NULL,
			 .lookup_mode = S_IFREG | S_IRUGO | S_IWUSR,
			 .read_type   = PSEUDO_READ_SINGLE,
			 .read        = {
				 .single_show = show_uid
			 },
			 .write_type  = PSEUDO_WRITE_STRING,
			 .write       = {
				 .gets        = get_uid
			 }
	},
	[PSEUDO_GID_ID] = {
			 .h = {
			       .type_id = REISER4_PSEUDO_PLUGIN_TYPE,
			       .id = PSEUDO_GID_ID,
			       .pops = NULL,
			       .label = "..gid",
			       .desc = "returns group",
			       .linkage = TYPE_SAFE_LIST_LINK_ZERO
			 },
			 .try         = try_by_label,
			 .lookup      = NULL,
			 .lookup_mode = S_IFREG | S_IRUGO | S_IWUSR,
			 .read_type   = PSEUDO_READ_SINGLE,
			 .read        = {
				 .single_show = show_gid
			 },
			 .write_type  = PSEUDO_WRITE_STRING,
			 .write       = {
				 .gets        = get_gid
			 }
	},
	[PSEUDO_RWX_ID] = {
			 .h = {
			       .type_id = REISER4_PSEUDO_PLUGIN_TYPE,
			       .id = PSEUDO_RWX_ID,
			       .pops = NULL,
			       .label = "..rwx",
			       .desc = "returns rwx permissions",
			       .linkage = TYPE_SAFE_LIST_LINK_ZERO
			 },
			 .try         = try_by_label,
			 .lookup      = NULL,
			 .lookup_mode = S_IFREG | S_IRUGO | S_IWUSR,
			 .read_type   = PSEUDO_READ_SINGLE,
			 .read        = {
				 .single_show = show_rwx
			 },
			 .write_type  = PSEUDO_WRITE_STRING,
			 .write       = {
				 .gets        = get_rwx
			 }
	},
	[PSEUDO_OID_ID] = {
			 .h = {
			       .type_id = REISER4_PSEUDO_PLUGIN_TYPE,
			       .id = PSEUDO_OID_ID,
			       .pops = NULL,
			       .label = "..oid",
			       .desc = "returns object id",
			       .linkage = TYPE_SAFE_LIST_LINK_ZERO
			 },
			 .try         = try_by_label,
			 .lookup      = NULL,
			 .lookup_mode = S_IFREG | S_IRUGO,
			 .read_type   = PSEUDO_READ_SINGLE,
			 .read        = {
				 .single_show = show_oid
			 },
			 .write_type  = PSEUDO_WRITE_NONE
	},
	[PSEUDO_KEY_ID] = {
			 .h = {
			       .type_id = REISER4_PSEUDO_PLUGIN_TYPE,
			       .id = PSEUDO_KEY_ID,
			       .pops = NULL,
			       .label = "..key",
			       .desc = "returns object's key",
			       .linkage = TYPE_SAFE_LIST_LINK_ZERO
			 },
			 .try         = try_by_label,
			 .lookup      = NULL,
			 .lookup_mode = S_IFREG | S_IRUGO,
			 .read_type   = PSEUDO_READ_SINGLE,
			 .read        = {
				 .single_show = show_key
			 },
			 .write_type  = PSEUDO_WRITE_NONE
	},
	[PSEUDO_SIZE_ID] = {
			 .h = {
			       .type_id = REISER4_PSEUDO_PLUGIN_TYPE,
			       .id = PSEUDO_SIZE_ID,
			       .pops = NULL,
			       .label = "..size",
			       .desc = "returns object's size",
			       .linkage = TYPE_SAFE_LIST_LINK_ZERO
			 },
			 .try         = try_by_label,
			 .lookup      = NULL,
			 .lookup_mode = S_IFREG | S_IRUGO,
			 .read_type   = PSEUDO_READ_SINGLE,
			 .read        = {
				 .single_show = show_size
			 },
			 .write_type  = PSEUDO_WRITE_NONE
	},
	[PSEUDO_NLINK_ID] = {
			 .h = {
			       .type_id = REISER4_PSEUDO_PLUGIN_TYPE,
			       .id = PSEUDO_NLINK_ID,
			       .pops = NULL,
			       .label = "..nlink",
			       .desc = "returns nlink count",
			       .linkage = TYPE_SAFE_LIST_LINK_ZERO
			 },
			 .try         = try_by_label,
			 .lookup      = NULL,
			 .lookup_mode = S_IFREG | S_IRUGO,
			 .read_type   = PSEUDO_READ_SINGLE,
			 .read        = {
				 .single_show = show_nlink
			 },
			 .write_type  = PSEUDO_WRITE_NONE
	},
	[PSEUDO_LOCALITY_ID] = {
			 .h = {
			       .type_id = REISER4_PSEUDO_PLUGIN_TYPE,
			       .id = PSEUDO_LOCALITY_ID,
			       .pops = NULL,
			       .label = "..locality",
			       .desc = "returns object's locality",
			       .linkage = TYPE_SAFE_LIST_LINK_ZERO
			 },
			 .try         = try_by_label,
			 .lookup      = NULL,
			 .lookup_mode = S_IFREG | S_IRUGO,
			 .read_type   = PSEUDO_READ_SINGLE,
			 .read        = {
				 .single_show = show_locality
			 },
			 .write_type  = PSEUDO_WRITE_NONE
	},
	[PSEUDO_PAGECACHE_ID] = {
			 .h = {
			       .type_id = REISER4_PSEUDO_PLUGIN_TYPE,
			       .id = PSEUDO_PAGECACHE_ID,
			       .pops = NULL,
			       .label = "..pagecache",
			       .desc = "returns page cache stats",
			       .linkage = TYPE_SAFE_LIST_LINK_ZERO
			 },
			 .try         = try_by_label,
			 .lookup      = lookup_pagecache,
			 .lookup_mode = S_IFREG | S_IRUGO | S_IXUGO,
			 .read_type   = PSEUDO_READ_SINGLE,
			 .read        = {
				 .single_show = show_pagecache
			 },
			 .write_type  = PSEUDO_WRITE_NONE,
			 .readdir     = readdir_pagecache
	},
	[PSEUDO_PAGECACHE_STAT_ID] = {
			 .h = {
			       .type_id = REISER4_PSEUDO_PLUGIN_TYPE,
			       .id = PSEUDO_PAGECACHE_STAT_ID,
			       .pops = NULL,
			       .label = "pagecache stat",
			       .desc = "pagecache stat",
			       .linkage = TYPE_SAFE_LIST_LINK_ZERO
			 },
			 .try         = NULL,
			 .lookup      = NULL,
			 .lookup_mode = S_IFREG | S_IRUGO,
			 .read_type   = PSEUDO_READ_SINGLE,
			 .read        = {
				 .single_show = show_pagecache_stat
			 },
			 .write_type  = PSEUDO_WRITE_NONE
	},
	[PSEUDO_PSEUDOS_ID] = {
			 .h = {
			       .type_id = REISER4_PSEUDO_PLUGIN_TYPE,
			       .id = PSEUDO_PSEUDOS_ID,
			       .pops = NULL,
			       .label = "..pseudo",
			       .desc = "returns a list of pseudo files",
			       .linkage = TYPE_SAFE_LIST_LINK_ZERO
			 },
			 .try         = try_by_label,
			 .lookup      = NULL,
			 .lookup_mode = S_IFREG | S_IRUGO,
			 .read_type   = PSEUDO_READ_SEQ,
			 .read        = {
				 .ops = {
					 .start = pseudos_start,
					 .stop  = pseudos_stop,
					 .next  = pseudos_next,
					 .show  = pseudos_show
				 }
			 },
			 .write_type  = PSEUDO_WRITE_NONE
	},
	[PSEUDO_BMAP_ID] = {
			 .h = {
			       .type_id = REISER4_PSEUDO_PLUGIN_TYPE,
			       .id = PSEUDO_BMAP_ID,
			       .pops = NULL,
			       .label = "..bmap",
			       .desc = "returns a list blocks for this file",
			       .linkage = TYPE_SAFE_LIST_LINK_ZERO
			 },
			 .try         = try_by_label,
			 .lookup      = NULL,
			 .lookup_mode = S_IFREG | S_IRUGO,
			 .read_type   = PSEUDO_READ_SEQ,
			 .read        = {
				 .ops = {
					 .start = bmap_start,
					 .stop  = bmap_stop,
					 .next  = bmap_next,
					 .show  = bmap_show
				 }
			 },
			 .write_type  = PSEUDO_WRITE_NONE
	},
	[PSEUDO_READDIR_ID] = {
			 .h = {
			       .type_id = REISER4_PSEUDO_PLUGIN_TYPE,
			       .id = PSEUDO_READDIR_ID,
			       .pops = NULL,
			       .label = "..readdir",
			       .desc = "returns a list of names in the dir",
			       .linkage = TYPE_SAFE_LIST_LINK_ZERO
			 },
			 .try         = try_by_label,
			 .lookup      = NULL,
			 .lookup_mode = S_IFREG | S_IRUGO,
			 .read_type   = PSEUDO_READ_SEQ,
			 .read        = {
				 .ops = {
					 .start = readdir_start,
					 .stop  = readdir_stop,
					 .next  = readdir_next,
					 .show  = readdir_show
				 }
			 },
			 .write_type  = PSEUDO_WRITE_NONE
	},
	[PSEUDO_PLUGIN_ID] = {
			 .h = {
			       .type_id = REISER4_PSEUDO_PLUGIN_TYPE,
			       .id = PSEUDO_PLUGIN_ID,
			       .pops = NULL,
			       .label = "plugin",
			       .desc = "plugin",
			       .linkage = TYPE_SAFE_LIST_LINK_ZERO
			 },
			 .try         = NULL,
			 .lookup      = lookup_plugin_field,
			 .lookup_mode = S_IFREG | S_IRUGO | S_IXUGO,
			 .read_type   = PSEUDO_READ_SINGLE,
			 .read        = {
				 .single_show = show_plugin
			 },
			 .write_type  = PSEUDO_WRITE_NONE,
			 .readdir     = readdir_plugin_field
	},
	[PSEUDO_PLUGINS_ID] = {
			 .h = {
			       .type_id = REISER4_PSEUDO_PLUGIN_TYPE,
			       .id = PSEUDO_PLUGINS_ID,
			       .pops = NULL,
			       .label = "..plugin",
			       .desc = "list of plugins",
			       .linkage = TYPE_SAFE_LIST_LINK_ZERO
			 },
			 .try         = try_by_label,
			 .lookup      = lookup_plugins,
			 .lookup_mode = S_IFREG | S_IRUGO | S_IXUGO,
			 .read_type   = PSEUDO_READ_NONE,
			 .write_type  = PSEUDO_WRITE_NONE,
			 .readdir     = readdir_plugins
	},
	[PSEUDO_PLUGIN_FIELD_ID] = {
			 .h = {
			       .type_id = REISER4_PSEUDO_PLUGIN_TYPE,
			       .id = PSEUDO_PLUGIN_ID,
			       .pops = NULL,
			       .label = "plugin-field",
			       .desc = "plugin field",
			       .linkage = TYPE_SAFE_LIST_LINK_ZERO
			 },
			 .try         = NULL,
			 .lookup      = NULL,
			 .lookup_mode = S_IFREG | S_IRUGO,
			 .read_type   = PSEUDO_READ_SINGLE,
			 .read        = {
				 .single_show = show_plugin_field
			 },
			 .write_type  = PSEUDO_WRITE_NONE,
			 .readdir     = NULL
	},
	[PSEUDO_ITEMS_ID] = {
			 .h = {
			       .type_id = REISER4_PSEUDO_PLUGIN_TYPE,
			       .id = PSEUDO_ITEMS_ID,
			       .pops = NULL,
			       .label = "..items",
			       .desc = "returns a list of items for this file",
			       .linkage = TYPE_SAFE_LIST_LINK_ZERO
			 },
			 .try         = try_by_label,
			 .lookup      = NULL,
			 .lookup_mode = S_IFREG | S_IRUGO,
			 .read_type   = PSEUDO_READ_SEQ,
			 .read        = {
				 .ops = {
					 .start = items_start,
					 .stop  = items_stop,
					 .next  = items_next,
					 .show  = items_show
				 }
			 },
			 .write_type  = PSEUDO_WRITE_NONE
	},
	[PSEUDO_NEW_ID] = {
			 .h = {
			       .type_id = REISER4_PSEUDO_PLUGIN_TYPE,
			       .id = PSEUDO_NEW_ID,
			       .pops = NULL,
			       .label = "..new",
			       .desc = "creates new file in the host",
			       .linkage = TYPE_SAFE_LIST_LINK_ZERO
			 },
			 .try         = try_by_label,
			 .lookup      = NULL,
			 .lookup_mode = S_IFREG | S_IWUSR,
			 .read_type   = PSEUDO_READ_NONE,
			 .read        = {
				 .single_show = show_rwx
			 },
			 .write_type  = PSEUDO_WRITE_STRING,
			 .write       = {
				 .gets        = get_new
			 }
	},
};

/* Make Linus happy.
   Local variables:
   c-indentation-style: "K&R"
   mode-name: "LC"
   c-basic-offset: 8
   tab-width: 8
   fill-column: 120
   scroll-step: 1
   End:
*/
