/*
%%% copyright-cmetz-97
This software is Copyright 1997-1998 by Craig Metz, All Rights Reserved.
The Inner Net License Version 2 applies to this software.
You should have received a copy of the license with this software. If
you didn't get a copy, you may request one from <license@inner.net>.

*/
#define NAME "netd"
#define USAGE "[-C config file]"

#include <sys/types.h>
#include <stdio.h>
#include <pwd.h>
#include <grp.h>
#include <netdb.h>
#include <unistd.h>
#include <syslog.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <signal.h>
#include <unistd.h>
#include <ctype.h>
#include <setjmp.h>
#include <limits.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "support.h"
#include "inner.h"

int minfd = INT_MAX, maxfd = -1;
fd_set allfds;
fd_set servicefds;
fd_set larvalfds;

#define INSTANCES_MAX 128
#define USERNAME_MAX 32

struct prefix {
  struct prefix *next;
  int line;
  union sockaddr_union addr;
  union sockaddr_union mask;
};

struct rule {
  struct rule *next;
  int line;
  struct prefix *prefixes;
  int instances;
  int ident;
  int checkname;
  int downgrade;
  char *argv[10];
  char clbuf[256];
  int ninstances;
  uid_t uid;
  gid_t gid;
  char user[USERNAME_MAX];
  char directory[PATH_MAX];
  mode_t umask;
};

struct service {
  struct service *next;
  int line;
  struct rule *rules;
  int socktype;
  fd_set fds;
} *services = NULL;

#define FDTOSERVICE_PRIME 17
struct fdtoservice {
  struct fdtoservice *next;
  int fd;
  struct service *service;
} *fdtoservices[FDTOSERVICE_PRIME];

#define PIDTORULE_PRIME 17
struct pidtorule {
  struct pidtorule *next;
  pid_t pid;
  int fd;
  struct rule *rule;
} *pidtorules[PIDTORULE_PRIME];

struct larvalmsg {
  pid_t pid;
  struct rule *rule;
};

#define CF_DIR "/etc/"

int handshaketime = 5;

#define CF_TOPLEVEL 0
#define CF_SERVICE  1
#define CF_RULE     2

#define CF_TOKEN_END 0
#define CF_TOKEN_SERVICE 1
#define CF_TOKEN_RULE 2
#define CF_TOKEN_BIND 3
#define CF_TOKEN_SOCKTYPE 4
#define CF_TOKEN_FAMILY 5
#define CF_TOKEN_QUEUELEN 6
#define CF_TOKEN_PREFIX 7
#define CF_TOKEN_RUN 8
#define CF_TOKEN_INSTANCES 9
#define CF_TOKEN_USER 10
#define CF_TOKEN_DIRECTORY 11
#define CF_TOKEN_CHECKNAME 12
#define CF_TOKEN_IDENT 13
#define CF_TOKEN_DOWNGRADE 14
#define CF_TOKEN_UMASK 15
struct nrl_nametonum cf_tokens[] = {
  { CF_TOKEN_END, "}", 1 },
  { CF_TOKEN_SERVICE, "service", 2 },
  { CF_TOKEN_RULE, "rule", 2 },
  { CF_TOKEN_BIND, "bind", 2 },
  { CF_TOKEN_SOCKTYPE, "socktype", 2 },
  { CF_TOKEN_FAMILY, "family", 2 },
  { CF_TOKEN_QUEUELEN, "queuelen", 2 },
  { CF_TOKEN_PREFIX, "prefix", 2 },
  { CF_TOKEN_RUN, "run", 0 },
  { CF_TOKEN_INSTANCES, "instances", 2 },
  { CF_TOKEN_USER, "user", 2 },
  { CF_TOKEN_DIRECTORY, "directory", 2 },
  { CF_TOKEN_CHECKNAME, "checkname", 2 },
  { CF_TOKEN_IDENT, "ident", 2 },
  { CF_TOKEN_DOWNGRADE, "downgrade", 2 },
  { CF_TOKEN_UMASK, "umask", 2 },
  { -1, NULL, 0 }
};

#define CF_STRENGTH_SKIP 0
#define CF_STRENGTH_TRY 1
#define CF_STRENGTH_REQUIRE 2
struct nrl_nametonum cf_strengths[] = {
  { CF_STRENGTH_SKIP, "skip", 0 },
  { CF_STRENGTH_TRY, "try", 0 },
  { CF_STRENGTH_REQUIRE, "require", 0 },
  { -1, NULL, 0 }
};

char buffer[256];

void turn()
{
  struct pidtorule **pp, *pidtorule;
  pid_t pid;
  struct rule *rule;

  while(1) {
    if ((pid = waitpid(-1, NULL, WNOHANG)) < 0) {
      if (errno == ECHILD)
	break;
      lprintf(LP_SYSTEM, "turn: waitpid");
#if DEBUG
      if (debug)
	abort();
#endif /* DEBUG */
      continue;
    };
    if (!pid)
      break;
    for (pp = &(pidtorules[pid % PIDTORULE_PRIME]); *pp && ((*pp)->pid != pid); pp = &((*pp)->next));
    if (!*pp) {
#if DEBUG
      if (debug) {
	lprintf(LP_DEBUG, "turn: internal error: unknown child %d died", pid);
	abort();
      };
#endif /* DEBUG */
      continue;
    };
    pidtorule = *pp;
    *pp = (*pp)->next;
    if (pidtorule->fd >= 0) {
      close(pidtorule->fd);
      FD_CLR(pidtorule->fd, &larvalfds);
      FD_CLR(pidtorule->fd, &allfds);
    };
    if (!(rule = pidtorule->rule)) {
      free(pidtorule);
#if DEBUG
      if (debug)
        lprintf(LP_DEBUG, "turn: turned larval child %d", pid);
#endif /* DEBUG */
      continue;
    };
    free(pidtorule);
    if (--rule->ninstances < 0)
      rule->ninstances = 0;
#if DEBUG
      if (debug)
        lprintf(LP_DEBUG, "turn: turned child %d of rule at line %d", pid, rule->line);
#endif /* DEBUG */
  };
};

void sigchldhandler(int unused)
{
#if DEBUG
  if (debug)
    lprintf(LP_DEBUG, "sigchld: entering handler");
#endif /* DEBUG */

  turn();

#if DEBUG
  if (debug)
    lprintf(LP_DEBUG, "sigchld: leaving handler");
#endif /* DEBUG */
};

void sigalrmhandler_exit(int unused)
{
  lprintf(LP_ERROR, "child: parent failed to return handshake in time");
#if DEBUG
  if (debug)
    abort();
#endif /* DEBUG */
  exit(1);
};

jmp_buf timeout_env;
void sigalrmhandler_timeout(int i)
{
  lprintf(LP_ERROR, "child: operation timed out");
  longjmp(timeout_env, i);
};

#if DEBUG
void sigusr1handler(int unused)
{
  struct stat stat;
  int i;
  char buf[128];

  if (!debug)
    return;

  lprintf(LP_DEBUG, "sigusr1: entering handler");
  
  for (i = 0; i < sysconf(_SC_OPEN_MAX); i++) {
    if (!fstat(i, &stat)) {
      strcpy(buf, "fd %d is a ");
      if (S_ISLNK(stat.st_mode))
	strcat(buf, "symlink");
      if (S_ISREG(stat.st_mode))
	strcat(buf, "file ");
      if (S_ISDIR(stat.st_mode))
	strcat(buf, "directory ");
      if (S_ISCHR(stat.st_mode))
	strcat(buf, "character device ");
      if (S_ISBLK(stat.st_mode))
	strcat(buf, "block device ");
      if (S_ISFIFO(stat.st_mode))
	strcat(buf, "FIFO/pipe ");
      if (S_ISSOCK(stat.st_mode))
	strcat(buf, "socket ");
      lprintf(LP_DEBUG, buf, i);
      lprintf(LP_DEBUG, "dev=%d ino=%d mode=%o nlink=%d uid=%d gid=%d rdev=%d size=%d blksize=%d blocks=%d atime=%d mtime=%d ctime=%d\n", i, stat.st_dev, stat.st_ino, stat.st_mode, stat.st_nlink, stat.st_uid, stat.st_gid, stat.st_rdev, stat.st_size, stat.st_blksize, stat.st_blocks, stat.st_atime, stat.st_mtime, stat.st_ctime);
    };
  };
  lprintf(LP_DEBUG, "sigusr1: leaving handler");
};
#endif /* DEBUG */

int main(int argc, char **argv)
{
  char *c;
  int i, j;

  INIT(argc, argv);

  for (i = sysconf(_SC_OPEN_MAX); i > 2; i--)
    close(i);

  openlog(myname, LOG_PID, LOG_DAEMON);
  inner_lprintf_opts_callback(inner_lprintf_callback_syslog);

  sprintf(buffer, CF_DIR "%s.conf", myname);

  while ((i = getopt(argc, argv, STDOPTS_FLAGS "C:")) != EOF) {
    switch(i) {
      case 'C':
	if (strlen(optarg) >= (sizeof(buffer) - 1)) {
	  lprintf(LP_ERROR, "config file pathname too long");
	  exit(1);
	};
	strncpy(buffer, optarg, sizeof(buffer));
	buffer[sizeof(buffer)-1] = 0;
	break;
      STDOPTS_CASES
    };
  };

#if DEBUG
  if (debug < 4)
#endif /* DEBUG */
  {
    if ((i = fork()) < 0) {
      lprintf(LP_SYSTEM, "fork");
      exit(1);
    };
    if (i)
      exit(0);
    if (setsid() < 0) {
      lprintf(LP_SYSTEM, "setsid");
#if DEBUG
      if (debug)
        abort();
#endif /* DEBUG */
    };
    signal(SIGHUP, SIG_IGN);
    signal(SIGINT, SIG_IGN);
    signal(SIGQUIT, SIG_IGN);
    signal(SIGTSTP, SIG_IGN);
    chdir("/");
    umask(0077);
  };

  {
    FILE *f;
    char *c, *c2;
    struct service **ps = &services;
    int argc;
    char *argv[10];
    struct passwd *passwd;

#define STATESTACK_MAX 4
#if DEBUG
#define PUSHSTATE(x) \
    if (++statesp >= STATESTACK_MAX) { \
      lprintf(LP_INTERNAL, "push %d overruns state stack", x); \
      exit(1); \
    }; \
    if (debug > 2) \
      lprintf(LP_DEBUG, "push %d/%d -> %d/%d", statestack[statesp-1].state, statesp-1, x, statesp); \
    statestack[statesp] = statestack[statesp-1]; \
    statestack[statesp].state = x; \
    st = &(statestack[statesp]);
#define POPSTATE \
    if (--statesp < 0) { \
      lprintf(LP_INTERNAL, "pop %d underruns state stack", statestack[0].state); \
      exit(1); \
    }; \
    if (debug > 2) \
      lprintf(LP_DEBUG, "pop %d/%d -> %d/%d", statestack[statesp+1].state, statesp+1, statestack[statesp].state, statesp); \
    st = &(statestack[statesp]);
#else /* DEBUG */
#define PUSHSTATE(x) \
    if (++statesp >= STATESTACK_MAX) { \
      lprintf(LP_INTERNAL, "push %d overruns state stack", x); \
      exit(1); \
    }; \
    statestack[statesp] = statestack[statesp-1]; \
    statestack[statesp].state = x; \
    st = &(statestack[statesp]);
#define POPSTATE \
    if (--statesp < 0) { \
      lprintf(LP_INTERNAL, "pop %d underruns state stack", statestack[0].state); \
      exit(1); \
    }; \
    st = &(statestack[statesp]);
#endif /* DEBUG */

    struct statevars {
      int state;

      int sline;
      int bindline;
      int family;
      int socktype;
      int queuelen;
      char shbuf[NI_MAXHOST];
      char sbuf[NI_MAXSERV];

      int rline;
      int instances;
      char rhbuf[NI_MAXHOST];
      char user[USERNAME_MAX];
      char directory[PATH_MAX];
      char *argv[10], clbuf[256];
      uid_t uid;
      gid_t gid;
      int ident;
      int checkname;
      int downgrade;
      mode_t umask;
    } statestack[STATESTACK_MAX], *st = &(statestack[0]);
    int statesp = 0, uid;
    struct rule *srules, **pr;
    struct prefix *prefixes, **pp;
    char *inputfile_name;
    unsigned int inputfile_line;

    memset(statestack, 0, sizeof(statestack));
    setpwent();

    if (uid = getuid()) {
      if (!(passwd = getpwuid(uid))) {
	lprintf(LP_SYSTEM, "config: getpwuid");
	exit(1);
      };
    } else {
      if (!(passwd = getpwnam("nobody"))) {
	lprintf(LP_SYSTEM, "config: getpwnam(\"nobody\")");
	exit(1);
      };
    };

    strcpy(st->user, passwd->pw_name);
    st->uid = passwd->pw_uid;
    st->gid = passwd->pw_gid;

    strcpy(st->directory, "");

    st->ident = CF_STRENGTH_TRY;
    st->checkname = CF_STRENGTH_TRY;

    st->umask = 0077;

    if (!(f = fopen(buffer, "r"))) {
      lprintf(LP_SYSTEM, "fopen(\"%s\", \"r\")", buffer);
      exit(1);
    };

    inputfile_name = strdup(buffer);
    inputfile_line = 0;
    inner_lprintf_opts_file(inputfile_name, &inputfile_line);

    memset(fdtoservices, 0, sizeof(fdtoservices));

    FD_ZERO(&allfds);
    FD_ZERO(&servicefds);

    while(fgets(buffer, sizeof(buffer), f)) {
      inputfile_line++;
      if (c = strchr(buffer, '#'))
	*c = 0;
      if (c = strchr(buffer, ';'))
	*c = 0;
      if (c = strchr(buffer, '\n'))
	*c = 0;

      for (c = buffer; isspace(*c); c++);

#if DEBUG
      if ((debug > 3) && buffer[0])
	lprintf(LP_DEBUG, "%s", c);
#endif /* DEBUG */

      if (!(argv[argc = 0] = strtok(c, " \t\n")))
	continue;

      if ((i = nrl_nametonum(cf_tokens, argv[0])) < 0) {
	lprintf(LP_SYNTAX, "%s unknown or not valid here", argv[0]);
	exit(1);
      };

      while((argv[++argc] = strtok(NULL, " \t\n")));

      if ((j = nrl_numtoflags(cf_tokens, i)) && (argc != j)) {
	lprintf(LP_SYNTAX, "invalid number of arguments (%d should be %d)", argc-1, j-1);
	exit(1);
      };

      switch(i) {
        case CF_TOKEN_END:
	  switch(st->state) {
	    case CF_TOPLEVEL:
	      lprintf(LP_SYNTAX, "close-brace not allowed here");
	      exit(1);
	    case CF_SERVICE:
	      if (!st->bindline) {
		inputfile_line = st->sline;
		lprintf(LP_SYNTAX, "no bind statement");
		exit(1);
	      };

	      if (!(*ps = malloc(sizeof(struct service)))) {
		lprintf(LP_SYSTEM, "malloc(%d)", sizeof(struct service));
		exit(1);
	      };
	      memset(*ps, 0, sizeof(struct service));
	      (*ps)->line = st->sline;
	      (*ps)->rules = srules;
	      (*ps)->socktype = st->socktype;

	      {
		struct addrinfo req, *ai, *ai2;
		struct fdtoservice **pf;
		int k;

		memset(&req, 0, sizeof(struct addrinfo));
		req.ai_flags = AI_PASSIVE;
		req.ai_socktype = st->socktype;
		req.ai_family = st->family;

		if (i = getaddrinfo(st->shbuf, st->sbuf, &req, &ai)) {
                  inputfile_line = st->bindline;
		  lprintf(LP_ERROR, "getaddrinfo(%s.%s): %s(%d)", st->shbuf, st->sbuf, gai_strerror(i), i);
		  exit(1);
		};

		FD_ZERO(&((*ps)->fds));
		k = 0;
		for (ai2 = ai; ai2; ai2 = ai2->ai_next) {
		  if ((i = socket(ai2->ai_family, ai2->ai_socktype, ai2->ai_protocol)) < 0) {
#if DEBUG
		    if (debug)
		      lprintf(LP_DEBUG, "socket: %s(%d)", strerror(errno), errno);
#endif /* DEBUG */
		    continue;
		  };
                  j = 1;
		  if (setsockopt(i, SOL_SOCKET, SO_REUSEADDR, &j, sizeof(int)) < 0) {
#if DEBUG
		    if (debug) {
		      lprintf(LP_DEBUG, "setsockopt(..., SO_REUSEADDR, ...): %s(%d)", strerror(errno), errno);
		      abort();
		    };
#endif /* DEBUG */
		    close(i);
		    continue;
		  };
		  if (bind(i, ai2->ai_addr, ai2->ai_addrlen) < 0) {
#if DEBUG
		    if (debug)
		      lprintf(LP_DEBUG, "bind: %s(%d)", strerror(errno), errno);
#endif /* DEBUG */
		    close(i);
		    continue;
		  };
		  if ((listen(i, st->queuelen) < 0) && (errno != EOPNOTSUPP)) {
#if DEBUG
		    if (debug) {
		      lprintf(LP_DEBUG, "listen: %s(%d)", strerror(errno), errno);
		      abort();
		    };
#endif /* DEBUG */
		    close(i);
		    continue;
		  };
		  if (fcntl(i, F_SETFD, 1) < 0) {
#if DEBUG
		    if (debug) {
		      lprintf(LP_DEBUG, "fcntl(..., F_SETFD, 1): %s(%d)", strerror(errno), errno);
		      abort();
		    };
#endif /* DEBUG */
		    close(i);
		    continue;
		  };
		  if ((j = fcntl(i, F_GETFL)) < 0) {
#if DEBUG
		    if (debug) {
		      lprintf(LP_DEBUG, "fcntl(..., F_GETFL): %s(%d)", strerror(errno), errno);
		      abort();
		    };
#endif /* DEBUG */
		    close(i);
		    continue;
		  };
		  if (fcntl(i, F_SETFL, j | O_NONBLOCK) < 0) {
#if DEBUG
		    if (debug) {
		      lprintf(LP_DEBUG, "fcntl(..., F_SETFL, ...): %s(%d)", strerror(errno), errno);
		      abort();
		    };
#endif /* DEBUG */
		    close(i);
		    continue;
		  };

		  for (pf = &(fdtoservices[i % FDTOSERVICE_PRIME]); *pf; pf = &((*pf)->next));
		  if (!(*pf = malloc(sizeof(struct fdtoservice)))) {
#if DEBUG
		    if (debug) {
		      lprintf(LP_DEBUG, "malloc: %s(%d)", strerror(errno), errno);
		      abort();
		    };
#endif /* DEBUG */
		    close(i);
		    continue;
		  };
		  memset(*pf, 0, sizeof(struct fdtoservice));
		  (*pf)->fd = i;
		  (*pf)->service = *ps;

		  FD_SET(i, &((*ps)->fds));
		  FD_SET(i, &allfds);
		  FD_SET(i, &servicefds);
		  if (i < minfd)
		    minfd = i;
		  if (i > maxfd)
		    maxfd = i;
		  k++;
		};

		freeaddrinfo(ai);

		if (!k) {
		  inputfile_line = st->sline;
		  lprintf(LP_SYNTAX, "no sockets result from definition");
		  exit(1);
		};
	      };

	      ps = &((*ps)->next);
	      POPSTATE;
	      break;
	    case CF_RULE:
	      if (!st->argv[0]) {
		inputfile_line = st->rline;
		lprintf(LP_SYNTAX, "no run statement");
		exit(1);
	      };

	      if (!(*pr = malloc(sizeof(struct rule)))) {
		inputfile_line = st->rline;
		lprintf(LP_SYSTEM, "malloc");
		exit(1);
	      };
	      memset(*pr, 0, sizeof(struct rule));
	      (*pr)->line = st->rline;
	      (*pr)->prefixes = prefixes;
	      (*pr)->checkname = st->checkname;
	      (*pr)->ident = st->ident;
	      (*pr)->downgrade = st->downgrade;
	      (*pr)->instances = st->instances;
	      (*pr)->uid = st->uid;
	      (*pr)->gid = st->gid;
	      strcpy((*pr)->user, st->user);
	      strcpy((*pr)->directory, st->directory);
	      
	      for (i = 0, c = (*pr)->clbuf; st->argv[i]; i++) {
		(*pr)->argv[i] = c;
		strcpy(c, st->argv[i]);
		c += strlen(c);
		c++;
	      };

	      pr = &((*pr)->next);
	      POPSTATE;
	      break;
	    default:
	      lprintf(LP_INTERNAL, "invalid state %d", st->state);
	      exit(1);
	  };
	  break;
        case CF_TOKEN_SERVICE:
	  if (st->state != CF_TOPLEVEL) {
	    lprintf(LP_SYNTAX, "can't start a service here");
	    exit(1);
	  };
	  srules = NULL;
	  pr = &srules;
	  PUSHSTATE(CF_SERVICE);
	  st->sline = inputfile_line;
	  break;
        case CF_TOKEN_RULE:
	  if (st->state != CF_SERVICE) {
	    lprintf(LP_SYNTAX, "can't start a rule here");
	    exit(1);
	  };
	  prefixes = NULL;
	  pp = &prefixes;
	  PUSHSTATE(CF_RULE);
	  st->rline = inputfile_line;
	  break;
        case CF_TOKEN_BIND:
	  st->bindline = inputfile_line;
	  if (c2 = strrchr(argv[1], '.')) {
	    c = argv[1];
	    *(c2++) = 0;
	  } else {
	    c2 = c;
	    c = "*";
	  };
	  if (strlen(c) >= NI_MAXHOST) {
	    lprintf(LP_SYNTAX, "address \"%s\" is too long", c);
	    exit(1);
	  };
	  if (strlen(c2) >= NI_MAXSERV) {
	    lprintf(LP_SYNTAX, "service \"%s\" is too long", c2);
	    exit(1);
	  };
	  strcpy(st->shbuf, c);
	  strcpy(st->sbuf, c2);
	  break;
        case CF_TOKEN_SOCKTYPE:
	  if ((st->socktype = nrl_socktypenametonum(argv[1])) < 0) {
	    lprintf(LP_SYNTAX, "%s is an unknown/invalid socktype", argv[1]);
	    exit(1);
	  };
	  break;
        case CF_TOKEN_FAMILY:
          if (!strcmp(argv[1], "*"))
            st->family = 0;
          else
	    if ((st->family = nrl_afnametonum(argv[1])) < 0) {
	      lprintf(LP_SYNTAX, "%s is an unknown/invalid address family", argv[1]);
	      exit(1);
	    };
	  break;
        case CF_TOKEN_QUEUELEN:
	  st->queuelen = strtoul(argv[1], &c, 10);
	  if ((*c < 0) || (st->queuelen < 1) || (st->queuelen > 1000)) {
	    lprintf(LP_SYNTAX, "%s is an invalid queue length", argv[1]);
	    exit(1);
	  };
	  break;
        case CF_TOKEN_PREFIX:
	  {
	    struct addrinfo req, *ai, *ai2;
	    uint8_t *a, *m, *d;
	    union sockaddr_union mask;
	    int prefixlen;

	    if (!strcmp(st->rhbuf, "*")) {
	      prefixlen = 0;
	    } else {
	      if (c = strrchr(argv[1], '/')) {
		*(c++) = 0;
		prefixlen = strtoul(c, &c2, 10);
		if (*c2) {
		  lprintf(LP_SYNTAX, "%s is an invalid prefix length", c);
		  exit(1);
		};
	      } else {
		prefixlen = -1;
	      };
	    };

	    memset(&req, 0, sizeof(struct addrinfo));
	    req.ai_flags = AI_PASSIVE;
	    req.ai_socktype = st->socktype;
	    req.ai_family = st->family;

	    if (i = getaddrinfo(argv[1], NULL, &req, &ai)) {
	      lprintf(LP_ERROR, "getaddrinfo %s.*: %s(%d)", argv[1], gai_strerror(i), i);
	      exit(1);
	    };

	    for (ai2 = ai; ai2; ai2 = ai2->ai_next) {
	      if (nrl_build_samask((struct sockaddr *)&mask, ai2->ai_family, prefixlen)) {
#if DEBUG
		if (debug)
		  lprintf(LP_DEBUG, "can't build %d-bit mask for %s sockaddr", prefixlen, nrl_afnumtoname(ai2->ai_family));
#endif /* DEBUG */
		continue;
	      };
	      if (!(*pp = malloc(sizeof(struct prefix)))) {
		lprintf(LP_SYSTEM, "malloc(%d)", sizeof(struct prefix));
		exit(1);
	      };
	      memset(*pp, 0, sizeof(struct prefix));

	      a = (uint8_t *)ai2->ai_addr;
	      m = (uint8_t *)&mask;
	      d = (uint8_t *)&((*pp)->addr);
	      for (i = 0; i < ai2->ai_addrlen; i++)
		*(d++) = *(a++) & *(m++);

	      (*pp)->line = inputfile_line;
	      memcpy(&((*pp)->mask), &mask, ai2->ai_addrlen);

	      pp = &(*pp)->next;
	    };

	    freeaddrinfo(ai);
	  };
	  break;
        case CF_TOKEN_RUN:
	  {
	    char *c = st->clbuf;

	    if (argc < 2) {
	      lprintf(LP_SYNTAX, "no command given to run");
	      exit(1);
	    };
	    
	    if (argc > 10) {
	      lprintf(LP_SYNTAX, "too many tokens in command line (%d)", argc - 1);
	      exit(1);
	    };

	    for (i = 0; i < argc - 1; i++) {
	      st->argv[i] = c;
	      strcpy(c, argv[i+1]);
	      c += strlen(c);
	      c++;
	    };
	  };
	  break;
        case CF_TOKEN_INSTANCES:
	  st->instances = strtoul(argv[1], &c, 10);
	  if ((*c < 0) || (st->instances > INSTANCES_MAX)) {
	    lprintf(LP_SYNTAX, "%s is an invalid number of instances", argv[1]);
	    exit(1);
	  };
	  break;
        case CF_TOKEN_USER:
	  if ((strlen(argv[1]) >= USERNAME_MAX) || !(passwd = getpwnam(argv[1]))) {
	    lprintf(LP_SYNTAX, "%s is an invalid user name", argv[1]);
	    exit(1);
	  };
	  if ((uid != passwd->pw_uid) && uid) {
	    lprintf(LP_ERROR, "only the superuser can change user ids");
	    exit(1);
	  };
	  strcpy(st->user, argv[1]);
	  st->uid = passwd->pw_uid;
	  st->gid = passwd->pw_gid;
	  break;
        case CF_TOKEN_DIRECTORY:
	  if (strlen(argv[1]) >= PATH_MAX) {
	    lprintf(LP_SYNTAX, "%s is too long", argv[1]);
	    exit(1);
	  };
	  strcpy(st->directory, argv[1]);
	  break;
        case CF_TOKEN_CHECKNAME:
	  if ((st->checkname = nrl_nametonum(cf_strengths, argv[1])) < 0) {
	    lprintf(LP_SYNTAX, "%s is an unknown strength", argv[1]);
	    exit(1);
	  };
	  break;
        case CF_TOKEN_IDENT:
	  if ((st->ident = nrl_nametonum(cf_strengths, argv[1])) < 0) {
	    lprintf(LP_SYNTAX, "%s is an unknown strength", argv[1]);
	    exit(1);
	  };
	  break;
        case CF_TOKEN_DOWNGRADE:
	  if ((st->downgrade = nrl_nametonum(cf_strengths, argv[1])) < 0) {
	    lprintf(LP_SYNTAX, "%s is an unknown strength", argv[1]);
	    exit(1);
	  };
	  break;
        case CF_TOKEN_UMASK:
	  st->umask = strtoul(argv[1], &c, 8);
	  if ((*c < 0) || (st->umask > 0777)) {
	    lprintf(LP_SYNTAX, "%s is an invalid umask", argv[1]);
	    exit(1);
	  };
          break;
        default:
	  lprintf(LP_INTERNAL, "invalid token %d", i);
	  exit(1);
      };
    };
    fclose(f);

    inner_lprintf_opts_file(NULL, NULL);
    free(inputfile_name);
  };

  if (maxfd < 0) {
    lprintf(LP_SYNTAX, "no services");
    exit(1);
  };

  endpwent();
  nrl_domainname();

#if DEBUG
  if (debug)
    lprintf(LP_DEBUG, "entering main loop");
#endif /* DEBUG */

  {
    struct sigaction sa;

    memset(&sa, 0, sizeof(struct sigaction));

    sa.sa_handler = sigchldhandler;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = SA_RESTART;

    if (sigaction(SIGCHLD, &sa, NULL) < 0) {
      lprintf(LP_SYSTEM, "sigaction");
      exit(1);
    };

#if DEBUG
    if (debug) {
      sa.sa_handler = sigusr1handler;
      sigemptyset(&sa.sa_mask);
      sa.sa_flags = SA_RESTART;
    
      if (sigaction(SIGUSR1, &sa, NULL) < 0) {
        lprintf(LP_SYSTEM, "sigaction");
        exit(1);
      };
    };
#endif /* DEBUG */
  };

  {
    fd_set rfds, efds;
    struct rule *rule;
    struct fdtoservice *fdtoservice;
    union sockaddr_union addr;
    int addrlen;
    int fds[2];
    struct larvalmsg larvalmsg;
    pid_t pid;
    struct pidtorule *pidtorule, **pp;
    sigset_t sigset;
    struct timeval timeval, timeval2;

    FD_ZERO(&larvalfds);

    sigemptyset(&sigset);
    sigaddset(&sigset, SIGCHLD);
    sigaddset(&sigset, SIGUSR1);

    if (sigprocmask(SIG_UNBLOCK, &sigset, NULL) < 0) {
      lprintf(LP_SYSTEM, "sigprocmask(SIG_UNBLOCK, ...)");
      exit(1);
    };

    memset(&timeval, 0, sizeof(timeval));
    timeval.tv_usec = 100;

    while (1) {
      memcpy(&rfds, &allfds, sizeof(fd_set));
      memcpy(&efds, &allfds, sizeof(fd_set));
      memcpy(&timeval2, &timeval, sizeof(struct timeval));

#if DEBUG
      if (debug)
	lprintf(LP_DEBUG, "main: entering select");
#endif /* DEBUG */

      if (sigprocmask(SIG_UNBLOCK, &sigset, NULL) < 0) {
	lprintf(LP_SYSTEM, "sigprocmask(SIG_UNBLOCK, ...)");
	exit(1);
      };

      if (select(maxfd + 1, &rfds, NULL, &efds, NULL) < 0) {
	if ((errno == EINTR) || (errno == EAGAIN))
	  continue;
	lprintf(LP_SYSTEM, "select");
#if DEBUG
	if (debug)
	  abort();
#endif /* DEBUG */
	exit(1);
      };

      if (sigprocmask(SIG_BLOCK, &sigset, NULL) < 0) {
	lprintf(LP_SYSTEM, "sigprocmask(SIG_BLOCK, ...)");
	exit(1);
      };

#if DEBUG
      if (debug)
	lprintf(LP_DEBUG, "back from select");
#endif /* DEBUG */

      for (i = 0; i <= maxfd; i++) {
	if (!FD_ISSET(i, &rfds))
	  continue;
	if (FD_ISSET(i, &servicefds)) {
	  for (fdtoservice = fdtoservices[i % FDTOSERVICE_PRIME]; fdtoservice && (fdtoservice->fd != i); fdtoservice = fdtoservice->next);
	  if (!fdtoservice) {
	    lprintf(LP_INTERNAL, "fd %d not in fdtoservice map?", i);
#if DEBUG
	    if (debug)
	      abort();
#endif /* DEBUG */
	    continue;
	  };

#if DEBUG
	  if (debug)
	    lprintf(LP_DEBUG, "matched service at line %d", fdtoservice->service->line);
#endif /* DEBUG */

	  if (socketpair(PF_UNIX, SOCK_DGRAM, 0, fds) < 0) {
#if DEBUG
	    if (debug) {
	      lprintf(LP_DEBUG, "main: socketpair: %s(%d)", strerror(errno), errno);
	      abort();
	    };
#endif /* DEBUG */
	    continue;
	  };
	  if (fcntl(fds[0], F_SETFD, 1) < 0) {
#if DEBUG
	    if (debug) {
	      lprintf(LP_DEBUG, "main: fcntl(..., F_SETFD, 1): %s(%d)", strerror(errno), errno);
	      abort();
	    };
#endif /* DEBUG */
	    close(fds[0]);
	    close(fds[1]);
	    continue;
	  };
	  if (fcntl(fds[1], F_SETFD, 1) < 0) {
#if DEBUG
	    if (debug) {
	      lprintf(LP_DEBUG, "main: fcntl(..., F_SETFD, 1): %s(%d)", strerror(errno), errno);
	      abort();
	    };
#endif /* DEBUG */
	    close(fds[0]);
	    close(fds[1]);
	    continue;
	  };
	  if ((j = accept(i, (struct sockaddr *)&addr, &addrlen)) < 0) {
#if DEBUG
	    if (debug) {
	      lprintf(LP_DEBUG, "main: accept: %s(%d)", strerror(errno), errno);
	      abort();
	    };
#endif /* DEBUG */
	    close(fds[0]);
	    close(fds[1]);
	    continue;
	  };
#if DEBUG
	  if (debug < 4)
#endif /* DEBUG */
	  {
	    if ((pid = fork()) < 0) {
#if DEBUG
	      if (debug) {
		lprintf(LP_DEBUG, "main: fork: %s(%d)", strerror(errno), errno);
		abort();
	      };
#endif /* DEBUG */
	      close(j);
	      close(fds[0]);
	      close(fds[1]);
	      continue;
	    };
	    if (pid) {
	      close(j);
	      close(fds[1]);
#if DEBUG
	      if (debug)
		lprintf(LP_DEBUG, "main: forked pid %d, fd %d", pid, fds[0]);
#endif /* DEBUG */
	      
	      for (pp = &(pidtorules[pid % PIDTORULE_PRIME]); *pp; pp = &((*pp)->next));
	      if (!(*pp = malloc(sizeof(struct pidtorule)))) {
		lprintf(LP_ERROR, "main: malloc: %s(%d)", strerror(errno), errno);
#if DEBUG
		if (debug)
		  abort();
#endif /* DEBUG */
		continue;
	      };
	      memset(*pp, 0, sizeof(struct pidtorule));
	      (*pp)->pid = pid;
	      (*pp)->fd = fds[0];
	      
	      FD_SET(fds[0], &larvalfds);
	      FD_SET(fds[0], &allfds);
	      if (i < minfd)
		minfd = i;
	      if (i > maxfd)
		maxfd = i;
	      
	      continue;
	    };
	  };

          signal(SIGALRM, sigalrmhandler_exit);
          alarm(15);

	  close(fds[0]);
	  for (i = 0; i <= maxfd; i++)
	    if (FD_ISSET(i, &allfds))
	      close(i);

	  {
	    union sockaddr_union maskedaddr;
	    uint8_t *a, *m, *d;
	    char laddr[NI_MAXHOST], faddr[NI_MAXHOST];
	    char lport[NI_MAXSERV], fport[NI_MAXSERV];
	    char fname[NI_MAXHOST], identname[32];
	    int checkname = 0, ident = 0, downgrade = 0;
	    int timer;
	    struct prefix *prefix;

	    addrlen = sizeof(union sockaddr_union);
	    if (getpeername(j, (struct sockaddr *)&addr, &addrlen) < 0) {
	      lprintf(LP_ERROR, "child: getpeername: %s(%d)", strerror(errno), errno);
	      exit(1);
	    };

#if DEBUG
	    if (debug > 2) {
	      lprintf(LP_DEBUG, "child: addrlen=%d, addr:", addrlen);
	      inner_dump_sockaddr((struct sockaddr *)&addr, addrlen);
	    };
#endif /* DEBUG */

	    i = sizeof(union sockaddr_union);
	    if (getsockname(j, (struct sockaddr *)&maskedaddr, &i) < 0) {
	      lprintf(LP_ERROR, "child: getsockname: %s(%d)", strerror(errno), errno);
	      exit(1);
	    };

	    if (getnameinfo((struct sockaddr *)&addr, addrlen, faddr, sizeof(faddr), fport, sizeof(fport), NI_NUMERICHOST | NI_NUMERICSERV)) {
	      lprintf(LP_ERROR, "child: getnameinfo failed");
#if DEBUG
	      if (debug)
		abort();
#endif /* DEBUG */
	      exit(1);
	    };

	    if (getnameinfo((struct sockaddr *)&maskedaddr, i, laddr, sizeof(laddr), lport, sizeof(lport), NI_NUMERICHOST | NI_NUMERICSERV)) {
	      lprintf(LP_ERROR, "child: getnameinfo failed");
#if DEBUG
	      if (debug)
		abort();
#endif /* DEBUG */
	      exit(1);
	    };

	    lprintf(LP_INFO, "connection from %s.%s to %s.%s", faddr, fport, laddr, lport);

	    for (rule = fdtoservice->service->rules; rule; rule = rule->next) {
#if DEBUG
	      if (debug > 2)
		lprintf(LP_DEBUG, "child: trying rule at %d");
#endif /* DEBUG */

	      if (rule->prefixes) {
		for (prefix = rule->prefixes; prefix; prefix = prefix->next) {
#if DEBUG
		  if (debug > 2) {
		    lprintf(LP_DEBUG, "child: trying prefix at %d:", prefix->line);
		    inner_dump_sockaddr((struct sockaddr *)&prefix->addr, 0);
		    inner_dump_sockaddr((struct sockaddr *)&prefix->mask, SA_LEN((struct sockaddr *)&prefix->addr));
		  };
#endif /* DEBUG */

		  if (addr.sa.sa_family != prefix->addr.sa.sa_family)
		    continue;

		  d = (uint8_t *)&maskedaddr;
		  a = (uint8_t *)&addr;
		  m = (uint8_t *)&prefix->mask;

		  for (i = 0; i < SA_LEN((struct sockaddr *)&addr); i++)
		    *(d++) = *(a++) & *(m++);

#if DEBUG
		  if (debug > 2) {
		    lprintf(LP_DEBUG, "child: prefix at %d: incoming prefix:", prefix->line);
		    inner_dump_sockaddr((struct sockaddr *)&maskedaddr, 0);
		  };
#endif /* DEBUG */

		  if (memcmp(&maskedaddr, &prefix->addr, SA_LEN((struct sockaddr *)&maskedaddr))) {
#if DEBUG
		    if (debug)
		      lprintf(LP_DEBUG, "child: prefix at %d: test failed", prefix->line);
#endif /* DEBUG */
		    continue;
		  };
		  break;
		};
		if (!prefix) {
#if DEBUG
		  if (debug)
		    lprintf(LP_DEBUG, "child: rule at %d: prefix test failed", rule->line);
#endif /* DEBUG */
		  continue;
		};
	      };

	      if (rule->checkname) {
		if (!checkname) {
		  checkname = 1;
		  if (!setjmp(timeout_env)) {
		    timer = alarm(0);
		    signal(SIGALRM, sigalrmhandler_timeout);
		    alarm(60);
#if DEBUG
		    if (debug > 2) {
		      lprintf(LP_DEBUG, "child: addrlen=%d, addr:", addrlen);
		      inner_dump_sockaddr((struct sockaddr *)&addr, addrlen);
		    };
#endif /* DEBUG */
		    if (i = getnameinfo((struct sockaddr *)&addr, addrlen, fname, sizeof(fname), NULL, 0, NI_NAMEREQD)) {
#if DEBUG
		      if (debug)
			lprintf(LP_DEBUG, "child: rule at %d: checkname: getnameinfo failed -- no reverse DNS entry present? (i)", rule->line, i);
#endif /* DEBUG */
		    } else {
		      struct addrinfo req, *ai;
		      union sockaddr_union cmpsa;
		      memset(&req, 0, sizeof(struct addrinfo));

#if DEBUG
		      if (debug)
			lprintf(LP_DEBUG, "child: rule at %d: checkname: returned name is %s", rule->line, fname);
#endif /* DEBUG */

		      if ((addr.sa.sa_family == AF_INET6) && IN6_IS_ADDR_V4MAPPED(&addr.sin6.sin6_addr)) {
#if DEBUG
			if (debug > 2)
			  lprintf(LP_DEBUG, "child: rule at %d: checkname: using IPv4-in-IPv6 mapped comparison", rule->line, fname);
#endif /* DEBUG */
		      	req.ai_family = AF_INET;
			memset(&cmpsa.sin, 0, sizeof(struct sockaddr_in));
#if SALEN
			cmpsa.sa.sa_len = sizeof(struct sockaddr_in);
#endif /* SALEN */
			cmpsa.sa.sa_family = AF_INET;
			cmpsa.sin.sin_addr.s_addr = ((uint32_t *)&addr.sin6.sin6_addr)[3];
		      } else {
#if DEBUG
			if (debug > 2)
			  lprintf(LP_DEBUG, "child: rule at %d: checkname: using normal comparison", rule->line, fname);
#endif /* DEBUG */
			req.ai_family = addr.sa.sa_family;
			memcpy(&cmpsa, &addr, SA_LEN(&addr.sa));
		      };

		      if (i = getaddrinfo(fname, fport, &req, &ai)) {
#if DEBUG
			if (debug)
			  lprintf(LP_DEBUG, "child: rule at %d: checkname: getaddrinfo %s.%s: %s(%d)", rule->line, fname, fport, gai_strerror(i), i);
#endif /* DEBUG */
		      } else {
			struct addrinfo *ai2;
#if DEBUG
			if (debug > 2) {
			  lprintf(LP_DEBUG, "child: rule at %d: checkname: comparing with:", rule->line);
			  inner_dump_sockaddr(&cmpsa.sa, SA_LEN(&cmpsa.sa));
			};
#endif /* DEBUG */
			for (ai2 = ai; ai2; ai2 = ai2->ai_next) {
#if DEBUG
			  if (debug > 2) {
			    lprintf(LP_DEBUG, "child: rule at %d: checkname: ai_addrlen = %d, ai_addr:", rule->line, ai2->ai_addrlen);
			    inner_dump_sockaddr(ai2->ai_addr, ai2->ai_addrlen);
			  };
#endif /* DEBUG */
			  if (!inner_addrcmp(ai2->ai_addr, &cmpsa.sa)) {
#if DEBUG
			    if (debug)
			      lprintf(LP_DEBUG, "child: rule at %d: checkname succeeded", rule->line);
#endif /* DEBUG */
			    break;
			  };
			};
			if (ai2)
			  checkname = 2;
#if DEBUG
			else
			  if (debug >= 0) {
                            lprintf(LP_DEBUG, "child: checkname failed; debriefing data:");
		            lprintf(LP_DEBUG, "child: addrlen=%d, addr:", addrlen);
		            inner_dump_sockaddr((struct sockaddr *)&addr, addrlen);
		            for (ai2 = ai; ai2; ai2 = ai2->ai_next) {
			      lprintf(LP_DEBUG, "child: rule at %d: checkname: ai_addrlen = %d, ai_addr:", rule->line, ai2->ai_addrlen);
			      inner_dump_sockaddr(ai2->ai_addr, ai2->ai_addrlen);
			    };
			  };
#endif /* DEBUG */
			freeaddrinfo(ai);
		      };
		    };
		  };
		  alarm(0);
		  signal(SIGALRM, sigalrmhandler_exit);
		  alarm(timer);
		};

		if (checkname < rule->checkname) {
#if DEBUG
		  if (debug)
		    lprintf(LP_DEBUG, "child: rule at %d: checkname: test failed (%d < %d)", rule->line, checkname, rule->checkname);
#endif /* DEBUG */
		  continue;
		};
	      };

	      if (rule->ident) {
		if (!ident) {
		  if (!setjmp(timeout_env)) {
		    struct addrinfo req, *lai, *fai, *ai2;
		    int k;

		    timer = alarm(0);
		    signal(SIGALRM, sigalrmhandler_timeout);
		    alarm(5);

		    memset(&req, 0, sizeof(struct addrinfo));

		    ident = 1;
		    req.ai_family = addr.sa.sa_family;
		    req.ai_socktype = SOCK_STREAM;

		    if (i = getaddrinfo(faddr, "ident", &req, &fai)) {
#if DEBUG
		      if (debug)
			lprintf(LP_DEBUG, "child: rule at %d: ident: getaddrinfo %s.ident: %s(%d)", rule->line, faddr, gai_strerror(i), i);
#endif /* DEBUG */
		      goto ident_end;
		    };

		    if (i = getaddrinfo(laddr, NULL, &req, &lai)) {
#if DEBUG
		      if (debug)
			lprintf(LP_DEBUG, "child: rule at %d: ident: getaddrinfo %s.*: %s(%d)", rule->line, laddr, gai_strerror(i), i);
#endif /* DEBUG */
		      goto ident_end;
		    };

		    sprintf(buffer, "%s,%s\r\n", fport, lport);
		    k = strlen(buffer);

		    for (ai2 = fai; ai2; ai2 = ai2->ai_next) {
		      if ((i = socket(ai2->ai_family, ai2->ai_socktype, ai2->ai_protocol)) < 0) {
#if DEBUG
			if (debug)
			  lprintf(LP_DEBUG, "child: rule at %d: ident: socket: %s(%d)", rule->line, strerror(errno), errno);
#endif /* DEBUG */
			continue;
		      };
		      if (connect(i, ai2->ai_addr, ai2->ai_addrlen) < 0) {
#if DEBUG
			if (debug)
			  lprintf(LP_DEBUG, "child: rule at %d: ident: connect: %s(%d)", rule->line, strerror(errno), errno);
#endif /* DEBUG */
			close(i);
			continue;
		      };
		      if (write(i, buffer, k) != k) {
#if DEBUG
			if (debug)
			  lprintf(LP_DEBUG, "child: rule at %d: ident: write: %s(%d)", rule->line, strerror(errno), errno);
#endif /* DEBUG */
			close(i);
			continue;
		      };
		      for (c = buffer; c < (buffer + sizeof(buffer) - 1); c++) {
			if (read(i, c, 1) != 1) {
#if DEBUG
			  if (debug)
			    lprintf(LP_DEBUG, "child: rule at %d: ident: read: %s(%d)", rule->line, strerror(errno), errno);
#endif /* DEBUG */
			  break;
			};
			if ((*c == '\r') || (*c == '\n')) {
			  *c = 0;
			  break;
			};
		      };
		      close(i);
		      if (*c) {
#if DEBUG
			if (debug)
			  lprintf(LP_DEBUG, "child: rule at %d: ident: hit error or end of buffer", rule->line);
#endif /* DEBUG */
			continue;
		      };
#if DEBUG
		      if (debug)
			lprintf(LP_DEBUG, "child: rule at %d: ident: got %s", rule->line, buffer);
#endif /* DEBUG */
		      if (!(c = strchr(buffer, ':'))) {
#if DEBUG
			if (debug)
			  lprintf(LP_DEBUG, "child: rule at %d: ident: syntax error in response", rule->line, strerror(errno), errno);
#endif /* DEBUG */
			continue;
		      };
		      c++;
		      while(*c && isspace(*c)) c++;
		      if (!*c) {
#if DEBUG
			if (debug)
			  lprintf(LP_DEBUG, "child: rule at %d: ident: syntax error in response", rule->line, strerror(errno), errno);
#endif /* DEBUG */
			continue;
		      };
		      if (strncasecmp(c, "userid", 6)) {
#if DEBUG
			if (debug)
			  lprintf(LP_DEBUG, "child: rule at %d: ident: response was probably an error", rule->line, strerror(errno), errno);
#endif /* DEBUG */
			continue;
		      };
		      c += 6;
		      while(*c && isspace(*c)) c++;
		      if (*(c++) != ':') {
#if DEBUG
			if (debug)
			  lprintf(LP_DEBUG, "child: rule at %d: ident: syntax error in response", rule->line, strerror(errno), errno);
#endif /* DEBUG */
			continue;
		      };
		      if (!(c = strchr(c, ':'))) {
#if DEBUG
			if (debug)
			  lprintf(LP_DEBUG, "child: rule at %d: ident: syntax error in response", rule->line, strerror(errno), errno);
#endif /* DEBUG */
			continue;
		      };
		      c++;
		      while(*c && isspace(*c)) c++;
		      if (!*c) {
#if DEBUG
			if (debug)
			  lprintf(LP_DEBUG, "child: rule at %d: ident: syntax error in response", rule->line, strerror(errno), errno);
#endif /* DEBUG */
			continue;
		      };
#if DEBUG
		      if (debug)
			lprintf(LP_DEBUG, "child: ident: %s", c);
#endif /* DEBUG */

		      strncpy(identname, c, sizeof(identname)-1);
		      identname[sizeof(identname)-1] = 0;

		      ident = 2;
		      break;
		    };
		    freeaddrinfo(lai);
		    freeaddrinfo(fai);
		  };
ident_end:
		  alarm(0);
		  signal(SIGALRM, sigalrmhandler_exit);
		  alarm(timer);
		};

		if (ident < rule->ident) {
#if DEBUG
		  if (debug)
		    lprintf(LP_DEBUG, "child: rule at %d: ident: test failed (%d < %d)", rule->line, ident, rule->ident);
#endif /* DEBUG */
		  continue;
		};
	      };

	      if (rule->downgrade) {
		if (!downgrade) {
		  /* XXX */
		  if ((addr.sa.sa_family == AF_INET6) && IN6_IS_ADDR_V4MAPPED(&addr.sin6.sin6_addr))
		    downgrade = 2;
		  else
		    downgrade = 1;
		};
		if (downgrade < rule->downgrade) {
#if DEBUG
		  if (debug)
		    lprintf(LP_DEBUG, "child: rule at %d: downgrade: test failed (%d < %d)", rule->line, downgrade, rule->downgrade);
#endif /* DEBUG */
		  continue;
		};
	      };

	      if ((rule->instances > 0) && (rule->ninstances >= rule->instances)) {
#if DEBUG
		if (debug)
		  lprintf(LP_DEBUG, "child: rule at %d: instances test failed (%d >= %d)", rule->line, rule->ninstances, rule->instances);
#endif /* DEBUG */
		continue;
	      };
	      break;
	    };

	    if (!rule) {
#if DEBUG
	      if (debug) {
		lprintf(LP_DEBUG, "child: no rules matched");
		abort();
	      };
#endif /* DEBUG */
	      exit(1);
	    };

	    if (checkname < 2)
	      strcpy(fname, faddr);

	    if (rule->uid != getuid()) {
	      if (setgid(rule->gid) < 0) {
		lprintf(LP_ERROR, "child: setgid: %s(%d)", strerror(errno), errno);
		exit(1);
	      };
	      if (initgroups(rule->user, rule->gid) < 0) {
		lprintf(LP_ERROR, "child: initgroups: %s(%d)", strerror(errno), errno);
		exit(1);
	      };
	      if (setuid(rule->uid) < 0) {
		lprintf(LP_ERROR, "child: setuid: %s(%d)", strerror(errno), errno);
		exit(1);
	      };
	    };

	    memset(&larvalmsg, 0, sizeof(struct larvalmsg));
	    larvalmsg.pid = getpid();
	    larvalmsg.rule = rule;
	    if (write(fds[1], &larvalmsg, sizeof(larvalmsg)) != sizeof(larvalmsg)) {
	      lprintf(LP_ERROR, "child: write: %s(%d)", strerror(errno), errno);
	      exit(1);
	    };

#if 0
	    alarm(2);
	    if (read(fds[1], &larvalmsg, sizeof(larvalmsg)) != sizeof(larvalmsg)) {
	      lprintf(LP_ERROR, "child: read: %s(%d)", strerror(errno), errno);
	      exit(1);
	    };
	    close(fds[1]);

	    if ((larvalmsg.pid != getpid()) || (larvalmsg.rule != rule)) {
	      lprintf(LP_ERROR, "child: internal error: echoed handshake doesn't match");
	      exit(1);
	    };
#endif /* 0 */

	    /* XXX */
	    if (downgrade == CF_STRENGTH_REQUIRE) {
	      int i = AF_INET;
	      if (setsockopt(j, IPPROTO_IPV6, IPV6_ADDRFORM, &i, sizeof(int))) {
		lprintf(LP_ERROR, "child: setsockopt(..., IPPROT_IPV6, IPV6_ADDRFORM, ...): %s(%d)", strerror(errno), errno);
		exit(1);
	      };
	    };

	    if (dup2(j, 0) < 0) {
	      lprintf(LP_ERROR, "child: dup2(..., 0): %s(%d)", strerror(errno), errno);
	      exit(1);
	    };
	    if (dup2(j, 1) < 0) {
	      lprintf(LP_ERROR, "child: dup2(..., 1): %s(%d)", strerror(errno), errno);
	      exit(1);
	    };
	    if (dup2(j, 2) < 0) {
	      lprintf(LP_ERROR, "child: dup2(..., 2): %s(%d)", strerror(errno), errno);
	      exit(1);
	    };
	    close(j);

	    if (rule->directory[0])
	      if (chdir(rule->directory) < 0) {
		lprintf(LP_ERROR, "child: chdir: %s(%d)", strerror(errno), errno);
		exit(1);
	      };

            umask(rule->umask);

	    if (setsid() < 0) {
	      lprintf(LP_ERROR, "child: setsid: %s(%d)", strerror(errno), errno);
	      exit(1);
	    };

	    if (sigprocmask(SIG_UNBLOCK, &sigset, NULL) < 0) {
	      lprintf(LP_ERROR, "child: sigprocmask(SIG_UNBLOCK, ...): %s(%d)", strerror(errno), errno);
	      exit(1);
	    };

	    if (ident < 2)
	      lprintf(LP_INFO, "%s -> %s", fname, rule->argv[0]);
	    else
	      lprintf(LP_INFO, "%s@%s -> %s", identname, fname, rule->argv[0]);
	    };
	    closelog();
#if DEBUG
	    if (!debug)
#endif /* DEBUG */
	      for (i = 3; i < sysconf(_SC_OPEN_MAX); i++)
		close(i);
	    alarm(0);
	    execv(rule->argv[0], rule->argv);
	    lprintf(LP_ERROR, "child: exec %s: %s(%d)", rule->argv[0], strerror(errno), errno);
#if DEBUG
	    if (debug)
	      abort();
#endif /* DEBUG */
	    exit(1);
	};

	if (FD_ISSET(i, &larvalfds)) {
	  if ((j = read(i, &larvalmsg, sizeof(struct larvalmsg))) != sizeof(struct larvalmsg)) {
	    if (j < 0)
	      lprintf(LP_ERROR, "main: read: %s(%d)", strerror(errno), errno);
	    else
	      lprintf(LP_ERROR, "main: read only %d/%d bytes of child's message", j, sizeof(struct larvalmsg));
#if DEBUG
	    if (debug)
	      abort();
#endif /* DEBUG */
	    close(i);
	    FD_CLR(i, &larvalfds);
	    FD_CLR(i, &allfds);
	    continue;
	  };
#if DEBUG
	  if (debug)
	    lprintf(LP_DEBUG, "main: got handshake from child %d", larvalmsg.pid);
#endif /* DEBUG */

	  for (pp = &(pidtorules[larvalmsg.pid % PIDTORULE_PRIME]); *pp && ((*pp)->pid != larvalmsg.pid); pp = &((*pp)->next));

	  if (((pid = waitpid(larvalmsg.pid, NULL, WNOHANG) < 0) && (errno == ECHILD)) || (pid == larvalmsg.pid)) {
#if DEBUG
	    if (debug)
	      lprintf(LP_DEBUG, "main: ... but it appears to have already died");
#endif /* DEBUG */
	    
	    if (!*pp) {
#if DEBUG
	      if (debug)
		lprintf(LP_DEBUG, "main: ... and has already been cleaned up");
#endif /* DEBUG */
	      continue;
	    };
	    pidtorule = *pp;
	    *pp = (*pp)->next;
	    close(pidtorule->fd);
	    free(pidtorule);
#if DEBUG
	    if (debug)
	      lprintf(LP_DEBUG, "main: ... and wasn't already cleaned up");
#endif /* DEBUG */
	    continue;
	  };

	  if (pid < 0) {
	    lprintf(LP_ERROR, "main: waitpid(%d, ...): %s(%d)", larvalmsg.pid, strerror(errno), errno);
#if DEBUG
	    if (debug)
	      abort();
#endif /* DEBUG */
	    close(i);
	    FD_CLR(i, &larvalfds);
	    FD_CLR(i, &allfds);
	    continue;
	  };

	  if (!*pp) {
#if DEBUG
	    if (debug)
	      lprintf(LP_DEBUG, "main: internal error: fd %d points to unknown child %d? killing...", i, pid);
#endif /* DEBUG */

	    if (kill(pid, SIGKILL) < 0)
	      lprintf(LP_ERROR, "main: couldn't kill child %d", pid);

#if DEBUG
	    if (debug)
	      abort();
#endif /* DEBUG */
	    close(i);
	    FD_CLR(i, &larvalfds);
	    FD_CLR(i, &allfds);
	    continue;
	  };

#if 0
	  j = write(i, &larvalmsg, sizeof(struct larvalmsg));
#endif /* 0 */
	  close(i);
	  FD_CLR(i, &larvalfds);
	  FD_CLR(i, &allfds);
	  (*pp)->fd = -1;

#if 0
	  if (j != sizeof(struct larvalmsg)) {
	    if (j < 0)
	      lprintf(LP_ERROR, "main: write: %s(%d)", strerror(errno), errno);
	    else
	      lprintf(LP_ERROR, "main: wrote only %d/%d bytes back to child", j, sizeof(struct larvalmsg));
	    if (kill(pid, SIGKILL) < 0) {
	      lprintf(LP_ERROR, "main: couldn't kill child %d", pid);
	    };
#if DEBUG
	    if (debug)
	      abort();
#endif /* DEBUG */
	    continue;
	  };
#endif /* 0 */

	  (*pp)->rule = larvalmsg.rule;
	  larvalmsg.rule->ninstances++;
	};
      };
    };
  };
};
