/*
%%% copyright-cmetz-96
This software is Copyright 1996-1998 by Craig Metz, All Rights Reserved.
The Inner Net License Version 2 applies to this software.
You should have received a copy of the license with this software. If
you didn't get a copy, you may request one from <license@inner.net>.

*/
#define NAME "gai"
#define USAGE "[-n <name>] [-s <service>] [-f <family>] [-t <socktype>] [-p protocol] [-F <flags>]"

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "support.h"
#include "inner.h"

extern char *optarg;

char buffer[1024];

static struct nrl_nametonum flagsnametonum[] = {
  { AI_PASSIVE, "passive", 0 },
  { AI_CANONNAME, "canonname", 0 },
  { AI_NUMERICHOST, "numerichost", 0 },
  { AI_EXT, "ext", 0 },
  { 0, NULL, -1 }
};

static struct nrl_nametonum socktypenametonum[] = {
  { SOCK_RAW, "raw", 0 },
  { SOCK_DGRAM, "dgram", 0 },
  { SOCK_STREAM, "stream", 0 },
  { 0, NULL, -1 }
};

int main(int argc, char **argv)
{
  int i, j;
  struct addrinfo *ai = NULL, **pai = &ai;
  struct addrinfo req;
  char *name = NULL, *service = NULL;
  char hbuf[2][NI_MAXHOST], sbuf[2][NI_MAXSERV];

  INIT(argc, argv);

  memset(&req, 0, sizeof(req));

  while((i = getopt(argc, argv, STDOPTS_FLAGS "F:n:s:f:t:p:x")) != -1) {
    switch(i) {
      case 'n':
        name = optarg;
        break;
      case 's':
        service = optarg;
        break;
      case 'f':
        if (!(req.ai_family = nrl_afnametonum(optarg))) {
          lprintf(LP_ERROR, "unknown/invalid address family '%s'", optarg);
          exit(1);
        };
        break;
      case 't':
        if (!(req.ai_socktype = nrl_nametonum(socktypenametonum, optarg))) {
          lprintf(LP_ERROR, "unknown/invalid socket type '%s'", optarg);
          exit(1);
        };
        break;
      case 'p':
        {
	char *c;
        req.ai_protocol = strtoul(optarg, &c, 0);
        if (*c) {
          struct protoent *protoent;
          if (!(protoent = getprotobyname(optarg))) {
            lprintf(LP_ERROR, "unknown/invalid protocol '%s'", optarg);
            exit(1);
	  };
          req.ai_protocol = protoent->p_proto;
	};
	};
        break;
      case 'F':
	{
	  char *c = optarg, *c2;

	  while(c2 = strtok(c, ",:")) {
	    if ((i = nrl_nametonum(flagsnametonum, c2)) < 0) {
	      fprintf(stderr, "%s: unknown flag: '%s'\n", myname, c2);
	      exit(1);
	    };
	    req.ai_flags |= i;
	    c = NULL;
	  };
	};
	break;
      case 'x':
        pai = NULL;
        break;
      STDOPTS_CASES;
    }
  }

  i = getaddrinfo(name, service, &req, pai);
  
  printf("getaddrinfo() returned: %s(%d).\n", gai_strerror(i), i);
  
  if (!i) {
    struct addrinfo *ai2 = ai;

    i = 0;
    for (; ai; ai = ai->ai_next) {
      printf("\nrecord %d:\n", i++);

      printf("ai_flags:      %x(%d)[", ai->ai_flags, ai->ai_flags);
      {
	char *c = "";

        for (j = 0; j < sizeof(flagsnametonum)/sizeof(struct nrl_nametonum);
	    j++) {
          if (!(ai->ai_flags & flagsnametonum[j].num))
	    continue;
          printf("%s%s", c, flagsnametonum[j].name);
	  c = ",";
	};
      };
      printf("]\n");

      printf("ai_family:     %s(%d)\n", nrl_afnumtoname(ai->ai_family), ai->ai_family);
      printf("ai_socktype:   %s(%d)\n", nrl_socktypenumtoname(ai->ai_socktype), ai->ai_socktype);
      printf("ai_protocol:   %x(%d)\n", ai->ai_protocol, ai->ai_protocol);
      printf("ai_addrlen:    %x(%d)\n", ai->ai_addrlen, ai->ai_addrlen);
      if (j = getnameinfo(ai->ai_addr, ai->ai_addrlen, hbuf[0], sizeof(hbuf[0]), sbuf[0], sizeof(sbuf[0]), 0)) {
        lprintf(LP_ERROR, "getnameinfo failed, returning %s(%d)", gai_strerror(j), j);
	continue;
      };
      if (j = getnameinfo(ai->ai_addr, ai->ai_addrlen, hbuf[1], sizeof(hbuf[1]), sbuf[1], sizeof(sbuf[1]), NI_NUMERICHOST | NI_NUMERICSERV)) {
        lprintf(LP_ERROR, "getnameinfo failed, returning %s(%d)", gai_strerror(j), j);
	continue;
      };
      printf("ai_addr:       %s.%s(%s.%s)\n", hbuf[0], sbuf[0], hbuf[1], sbuf[1]);
      printf("ai_canonname:  %s\n", ai->ai_canonname);
      printf("ai_next:       %p\n", ai->ai_next);

      fflush(stdout);
    };

    freeaddrinfo(ai2);
  };

  return i;
}
