/*
%%% copyright-cmetz-97
This software is Copyright 1997-1998 by Craig Metz, All Rights Reserved.
The Inner Net License Version 2 applies to this software.
You should have received a copy of the license with this software. If
you didn't get a copy, you may request one from <license@inner.net>.

*/
#define NAME "fingerd"
#define USAGE "[-l] [-m mail host]"

#include <sys/types.h>
#include <sys/socket.h>
#include <pwd.h>
#include <grp.h>
#include <netdb.h>
#include <unistd.h>
#include <syslog.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <ctype.h>

#include "support.h"
#include "inner.h"

struct gidtoname_cache {
  struct gidtoname_cache *next;
  gid_t gid;
  char name[32];
};

struct gidtoname_cache *gidtoname_cache = NULL;

char *gidtoname(gid_t gid)
{
  struct gidtoname_cache **g, *g2;

  for (g = &(gidtoname_cache); *g; g = &((*g)->next)) {
    if ((*g)->gid == gid) {
      g2 = *g;
      *g = g2->next;
      g2->next = gidtoname_cache;
      gidtoname_cache = g2;
      return g2->name;
    };
  };

  if (!(g2 = malloc(sizeof(struct gidtoname_cache)))) {
    syslog(LOG_ERR, "malloc failed");
    exit(1);
  };

  {
    struct group *gr;
    if (!(gr = getgrgid(gid))) {
      syslog(LOG_ERR, "can't find name for gid %d", gid);
      exit(1);
    };
    strncpy(g2->name, gr->gr_name, sizeof(g2->name)-1);
    g2->name[sizeof(g2->name)-1] = 0;
  };

  g2->gid = gid;
  g2->next = gidtoname_cache;
  gidtoname_cache = g2;

  return g2->name;
};

int main(int argc, char **argv)
{
  char buffer[32];
  int i;
  char *key = NULL;
  char *userpart = NULL;
  char *mailhost = NULL;
  struct inner_netline_state *readstate = NULL, *writestate = NULL;
  char peer[PEERBUF_SZ] = "";

  INIT(argc, argv);

  openlog(inner_myname, LOG_NDELAY | LOG_PID, LOG_DAEMON);
  inner_lprintf_opts_callback(inner_lprintf_callback_syslog);

  if (inner_getpeer(peer) < 0)
    exit(1);

  mailhost = nrl_domainname();

  while ((i = getopt(argc, argv, STDOPTS_FLAGS "m:")) != EOF) {
    switch(i) {
      case 'm':
	mailhost = optarg;
	break;
      case 'l':
        inner_readnetline_log++;
        inner_writenetline_log++;
	break;
      STDOPTS_CASES
    };
  };

  if (!mailhost) {
    syslog(LOG_ERR, "can't determine mail host");
    exit(1);
  };

  if ((i = inner_readnetline(STDIN_FILENO, buffer, &readstate)) < 0) {
    syslog(LOG_ERR, "inner_readnetline failed");
    exit(1);
  };

  syslog(LOG_INFO, "query on '%s' from %s", buffer, peer);
  
  {
    char *c;
    for (c = buffer; *c; c++) {
      if (!isalnum(*c) && (*c != ' ') && (*c != '/') && (*c != '+') && (*c != '-') && (*c != '_'))
	goto badsyntax;
    };
    
    if (!(c = strtok(buffer, " ")))
      goto nokey;
    
    if (*c == '/')
      if (!(c = strtok(buffer, " ")))
	goto nokey;
    
    key = c;
    
    if (c = strchr(key, '+')) {
      *(c++) = 0;
      userpart = c;
    };
  };

  {
    char *matches[8];
    struct group *g;
    int j, fuzzy = 1;

    memset(matches, 0, sizeof(matches));

    if (!(g = getgrnam("user"))) {
      syslog(LOG_ERR, "No group named 'user'");
      exit(1);
    };

    if (strlen(key) < 4)
      fuzzy = 0;

    for (i = j = 0; g->gr_mem[i]; i++)
      if ((fuzzy && strstr(g->gr_mem[i], key)) || (!fuzzy && strcmp(g->gr_mem[i], key))) {
	matches[j++] = g->gr_mem[i];
	if (j == (sizeof(matches)/sizeof(char *)))
	  goto toomany;
      };

    if (!j)
      goto nomatches;

    qsort((void *)matches, j, sizeof(char *), (void *)strcmp);

    {
      struct passwd *p;
      char planbuf[1024], plan[] = "/.plan";
      FILE *f;
      int userpartlen;

      if (userpart)
        userpartlen = strlen(userpart);
      else
        userpartlen = 0;

      for (i = 0; i < j; i++) {
	if (!(p = getpwnam(matches[i]))) {
	  syslog(LOG_ERR, "'%s' is a member of group 'user', but not in password file!");
	  continue;
	};
	printf("%s [%s] %s@%s\r\n", p->pw_gecos, gidtoname(p->pw_gid), p->pw_name, mailhost);
        fflush(stdout);
	if (strlen(p->pw_dir) + userpartlen > sizeof(planbuf) - sizeof(plan))
	  continue;

        f = NULL;

        if (userpart) {
	  strcpy(planbuf, p->pw_dir);
	  strcat(planbuf, plan);
          strcat(planbuf, "+");
          strcat(planbuf, userpart);
          f = fopen(planbuf, "r");
        };

        if (!f) {
	  strcpy(planbuf, p->pw_dir);
	  strcat(planbuf, plan);
          f = fopen(planbuf, "r");
        };

	if (f) {
	  fputs("\r\n", stdout);
	  while(fgets(planbuf, sizeof(planbuf), f)) {
	    fputs(" ", stdout);
	    fputs(planbuf, stdout);
	  };
	  fclose(f);
	  fflush(stdout);
	};
	write(STDOUT_FILENO, "\r\n", 2);
      };
    };
  };

  exit(0);

badsyntax:
  {
    char msg[] = "Syntax error in your query.\r\n";
    write(STDOUT_FILENO, msg, sizeof(msg)-1);
    exit(0);
  };

nokey:
  {
    char msg[] = "You did not specify a search key.\r\n";
    write(STDOUT_FILENO, msg, sizeof(msg)-1);
    exit(0);
  };

toomany:
  {
    char msg[] = "Too many matches; use a more specific search key.\r\n";
    write(STDOUT_FILENO, msg, sizeof(msg)-1);
    exit(0);
  };

nomatches:
  {
    char msg[] = "No matches.\r\n";
    write(STDOUT_FILENO, msg, sizeof(msg)-1);
    exit(0);
  };
};
