/*
%%% copyright-cmetz-97
This software is Copyright 1997-1998 by Craig Metz, All Rights Reserved.
The Inner Net License Version 2 applies to this software.
You should have received a copy of the license with this software. If
you didn't get a copy, you may request one from <license@inner.net>.

*/
#define NAME "fdsniff"
#define USAGE "[-q]"

#include <stdio.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <dirent.h>
#include <netdb.h>

#include "support.h"
#include "inner.h"

int inodestofind = 0;

struct inotoname {
  struct inotoname *next;
  ino_t ino;
  char *name;
} *inotonames = NULL;

struct dirname {
  struct dirname *next;
  char *name;
} *dirnames = NULL, **pd = &dirnames;

void inodesearch(char *dirname)
{
  DIR *dir;
  struct dirent *dirent;
  struct inotoname *inotoname;
  struct stat stat;
  char buffer[2048];

  if (!(dir = opendir(dirname))) {
    if ((errno == EACCES) || (errno == ENOENT))
      return;
    lprintf(LP_SYSTEM, "opendir(%s)", dirname);
    exit(1);
  };

  while((dirent = readdir(dir))) {
    if (!strcmp(dirent->d_name, ".") || !strcmp(dirent->d_name, ".."))
      continue;
    if (!strcmp(dirname, "/") && !strcmp(dirent->d_name, "dev"))
      continue;
    if (snprintf(buffer, sizeof(buffer), "%s%s%s", dirname, (dirname[strlen(dirname)-1] == '/') ? "" : "/", dirent->d_name) == sizeof(buffer)) {
      lprintf(LP_INTERNAL, "snprintf: exceeded buffer size, dirname=%s, d_name=%s", dirname, dirent->d_name);
      continue;
    };
    for (inotoname = inotonames; inotoname; inotoname = inotoname->next) {
      if (!inotoname->name && (inotoname->ino == dirent->d_ino)) {
        if (!(inotoname->name = strdup(buffer))) {
	  lprintf(LP_SYSTEM, "strdup(\"%s\")", buffer);
          exit(1);
        };
	if (--inodestofind <= 0)
	  return;
      };
    };
    if (lstat(buffer, &stat) < 0) {
      if ((errno == EACCES) || (errno == ENOENT))
	continue;
      lprintf(LP_SYSTEM, "lstat(%s, ...)", buffer);
      exit(1);
    };
    if (S_ISDIR(stat.st_mode)) {
      if (!(*pd = malloc(sizeof(struct dirname)))) {
	lprintf(LP_SYSTEM, "malloc(%d)", sizeof(struct dirname));
	exit(1);
      };
      memset(*pd, 0, sizeof(struct dirname));
      if (!((*pd)->name = malloc(strlen(buffer)+1))) {
	lprintf(LP_SYSTEM, "malloc(%d)", strlen(buffer)+1);
	exit(1);
      };
      strcpy((*pd)->name, buffer);
      pd = &((*pd)->next);
    };
  };

  if (closedir(dir) < 0) {
    lprintf(LP_SYSTEM, "closedir");
    exit(1);
  };
};

int main(int argc, char **argv)
{
  struct stat stat;
  int i, j, maxfd;
  int quick = 0;

  INIT(argc, argv);

  while ((i = getopt(argc, argv, STDOPTS_FLAGS "ql")) != EOF) {
    switch(i) {
      case 'q':
        quick = 1;
	break;
      case 'l':
	inner_lprintf_opts_callback(inner_lprintf_callback_syslog);
	break;
      STDOPTS_CASES
    };
  };

  if (!quick) {
    struct inotoname **pi;

    for (i = 0, inodestofind = 0, maxfd = 0; i < sysconf(_SC_OPEN_MAX); i++) {
      if (!fstat(i, &stat)) {
	maxfd = i;
	for (pi = &inotonames; *pi && ((*pi)->ino != stat.st_ino); pi = &((*pi)->next));
	if (!*pi) {
	  if (!(*pi = malloc(sizeof(struct inotoname)))) {
	    lprintf(LP_SYSTEM, "malloc(%d)", sizeof(struct inotoname));
	    exit(1);
	  };
	  memset(*pi, 0, sizeof(struct inotoname));
	  (*pi)->ino = stat.st_ino;
	  inodestofind++;
	};
      };
    };

    if (inodestofind) {
      printf("%s: searching for %d inode%s... ", myname, inodestofind, (inodestofind > 1) ? "s" : "");
      fflush(stdout);
      
      inodesearch("/dev");
      if (inodestofind)
        inodesearch("/");
      
      {
	struct dirname *dirname;
	while(dirnames) {
	  dirname = dirnames;
	  if (!(dirnames = dirnames->next))
	    pd = &dirnames;
	  
	  inodesearch(dirname->name);
	  free(dirname->name);
	  free(dirname);
	};
      };
      printf("\n");
    };
  };

  {
    struct inotoname *inotoname;
    char hbuf[64], sbuf[32];
    union sockaddr_union laddr, faddr;
    fd_set printed;
    int numprinted, last;
    struct stat stat2;

    FD_ZERO(&printed);

    for (i = 0; i <= maxfd; i++) {
      if (FD_ISSET(i, &printed))
        continue;
      if (fstat(i, &stat))
	continue;
      
      numprinted = last = 0;
      
      for (j = i + 1; j <= maxfd; j++) {
        if (fstat(j, &stat2))
          continue;
        if (stat2.st_ino != stat.st_ino)
          continue;
	FD_SET(j, &printed);
	if (!numprinted++)
	  printf("fds %d", i);
	if (last)
	  printf(", %d", last);
	last = j;
      };
      
      if (numprinted) {
	if (numprinted > 1)
	  printf(", and %d are a ", last);
	else
	  printf(" and %d are a ", last);
      } else
        printf("fd %d is a ", i);
      
      if (S_ISLNK(stat.st_mode))
        printf("symlink");
      if (S_ISREG(stat.st_mode))
        printf("file ");
      if (S_ISDIR(stat.st_mode))
        printf("directory ");
      if (S_ISCHR(stat.st_mode))
        printf("character device ");
      if (S_ISBLK(stat.st_mode))
        printf("block device ");
      if (S_ISFIFO(stat.st_mode))
        printf("FIFO/pipe ");
      if (S_ISSOCK(stat.st_mode)) {
        printf("socket ");
	j = sizeof(union sockaddr_union);
	if (getsockname(i, (struct sockaddr *)&laddr, &j) < 0) {
	  lprintf(LP_SYSTEM, "getsockname");
	  exit(1);
	};
	j = sizeof(union sockaddr_union);
	if (getpeername(i, (struct sockaddr *)&faddr, &j) < 0) {
	  lprintf(LP_SYSTEM, "getpeername");
	  exit(1);
	};
	if (getnameinfo((struct sockaddr *)&laddr, SA_LEN((struct sockaddr *)&laddr), hbuf, sizeof(hbuf), sbuf, sizeof(sbuf), 0)) {
	  lprintf(LP_ERROR, "getnameinfo failed");
	  exit(1);
	};
	printf("(local: %s.%s", hbuf, sbuf);
	if (getnameinfo((struct sockaddr *)&laddr, SA_LEN((struct sockaddr *)&laddr), hbuf, sizeof(hbuf), sbuf, sizeof(sbuf), NI_NUMERICHOST | NI_NUMERICSERV)) {
	  lprintf(LP_ERROR, "getnameinfo failed");
	  exit(1);
	};
	printf("(%s.%s) ", hbuf, sbuf);
	if (getnameinfo((struct sockaddr *)&faddr, SA_LEN((struct sockaddr *)&faddr), hbuf, sizeof(hbuf), sbuf, sizeof(sbuf), 0)) {
	  lprintf(LP_ERROR, "getnameinfo failed");
	  exit(1);
	};
	printf("foreign: %s.%s", hbuf, sbuf);
	if (getnameinfo((struct sockaddr *)&faddr, SA_LEN((struct sockaddr *)&faddr), hbuf, sizeof(hbuf), sbuf, sizeof(sbuf), NI_NUMERICHOST | NI_NUMERICSERV)) {
	  lprintf(LP_ERROR, "getnameinfo failed");
	  exit(1);
	};
	printf("(%s.%s)) ", hbuf, sbuf);
      };

      if (stat.st_ino) {
        for (inotoname = inotonames; inotoname && (inotoname->ino != stat.st_ino); inotoname = inotoname->next);
        if (inotoname && inotoname->name)
          printf("(%s) ", inotoname->name);
      };
      printf("\n  dev=%d ino=%d mode=%o nlink=%d uid=%d gid=%d rdev=%d size=%d blksize=%d blocks=%d atime=%d mtime=%d ctime=%d\n", stat.st_dev, (int)stat.st_ino, stat.st_mode, stat.st_nlink, stat.st_uid, stat.st_gid, stat.st_rdev, (int)stat.st_size, (int)stat.st_blksize, (int)stat.st_blocks, (int)stat.st_atime, (int)stat.st_mtime, (int)stat.st_ctime);
    };
  };
  return 0;
};
