/*
%%% copyright-cmetz-98
This software is Copyright 1998 by Craig Metz, All Rights Reserved.
The Inner Net License Version 2 applies to this software.
You should have received a copy of the license with this software. If
you didn't get a copy, you may request one from <license@inner.net>.

*/

#define NAME "daemon-watcher"
#define USAGE "[-a] <pid file> <command>"

#include <sys/types.h>
#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#if __linux__
#include <getopt.h>
#endif /* __linux__ */
#include <limits.h>
#include <sys/wait.h>
#include <syslog.h>
#include <fcntl.h>
#include <unistd.h>

#include "inner.h"

void sigchld_handler(int arg)
{
  pid_t pid;
  int status;

  if (pid = waitpid(-1, &status, WNOHANG)) {
    if (pid < 0) {
      lprintf(LP_SYSTEM, "waitpid(-1, &status, 0)");
      return;
    }
    if (WIFSIGNALED(status)) {
      lprintf(LP_ERROR, "child %d died of signal %d", pid, WTERMSIG(status));
      return;
    };
    if (WEXITSTATUS(status)) {
      lprintf(LP_ERROR, "child %d exited with status %d", pid, WEXITSTATUS(status));
      return;
    };
  }
};

int main(int argc, char *argv[])
{
  int i, aflag = 0;
  char *c, *pidfilename;
  FILE *f;
  pid_t pid;
  char buffer[16];
  unsigned long ul;
  struct sigaction restart, norestart;

  INIT(argc, argv);

  while((i = getopt(argc, argv, STDOPTS_FLAGS "a")) != -1) {
    switch(i) {
      case 'a':
	aflag = 1;
	break;
      STDOPTS_CASES;
    };
  };

  if ((argc - optind) != 2)
    usage();

  for (i = 2; i < sysconf(_SC_OPEN_MAX); i++)
    close(i);

  openlog(inner_myname, LOG_NDELAY | LOG_PID, LOG_DAEMON);
  inner_lprintf_opts_callback(inner_lprintf_callback_syslog);

  if ((i = open("/dev/null", O_RDWR)) < 0) {
    lprintf(LP_SYSTEM, "open(\"/dev/null\", O_RDWR)");
    exit(1);
  };

  {
    int j;

    for (j = 0; j < 3; j++) {
      close(j);
      if (dup2(i, j) < 0) {
        lprintf(LP_SYSTEM, "dup2(%d, %d)", i, j);
	exit(1);
      };
    };
  };

  close(i);

  if (fork())
    exit(1);

  setsid();

  pidfilename = argv[optind++];

  memset(&norestart, 0, sizeof(struct sigaction));
  norestart.sa_handler = &sigchld_handler;
  memcpy(&restart, &norestart, sizeof(struct sigaction));
  restart.sa_flags = SA_RESTART;

  if (sigaction(SIGCHLD, &restart, NULL)) {
    lprintf(LP_SYSTEM, "sigaction(SIGCHLD, &restart, NULL)");
    exit(1);
  };

  if (aflag)
    goto getpid;

  while(1) {
#if DEBUG
    if (debug)
      lprintf(LP_DEBUG, "(re-)starting \"%s\"", argv[optind]);
#endif /* DEBUG */

    if (pid = fork()) {
      if (pid < 0) {
        lprintf(LP_SYSTEM, "fork()");
        exit(1);
      };
    } else {
      execv(argv[optind], &argv[optind]);
      lprintf(LP_SYSTEM, "execv(%s, ...)", argv[optind]);
      exit(1);
    }

    sleep(60);

getpid:
#if DEBUG
    if (debug)
      lprintf(LP_DEBUG, "getting pid from \"%s\"", pidfilename);
#endif /* DEBUG */

    for (i = 0; i < 3; sleep(5), i++) {
      if (!(f = fopen(pidfilename, "r"))) {
	if (errno != ESRCH) {
	  lprintf(LP_SYSTEM, "fopen(\"%s\", \"r\")", pidfilename);
	  exit(1);
	};
        continue;
      };

      c = fgets(buffer, sizeof(buffer)-1, f);
      fclose(f);
      if (!c) {
	lprintf(LP_SYSTEM, "fgets(buffer, %d, f)", sizeof(buffer)-1);
	continue;
      };

      buffer[sizeof(buffer)-1] = 0;
      ul = strtoul(buffer, &c, 10);
      if (!ul || (ul == ULONG_MAX)) {
	lprintf(LP_ERROR, "\"%s\" is not a valid pid number", buffer);
        continue;
      };

      pid = (pid_t)ul;
      if (kill(pid, 0)) {
	if (errno != ESRCH) {
	  lprintf(LP_SYSTEM, "kill(%d, 0)", pid);
	  exit(1);
	};
        continue;
      }

      goto watchit;
    }

    lprintf(LP_ERROR, "unable to get pid from \"%s\"", pidfilename);
    exit(1);

watchit:
#if DEBUG
    if (debug)
      lprintf(LP_DEBUG, "entering main loop, watching pid %d", pid);
#endif /* DEBUG */

    while(!kill(pid, 0)) {
      if (sigaction(SIGCHLD, &norestart, NULL)) {
        lprintf(LP_SYSTEM, "sigaction(SIGCHLD, &norestart, NULL)");
        exit(1);
      };

      sleep(15);

      if (sigaction(SIGCHLD, &restart, NULL)) {
        lprintf(LP_SYSTEM, "sigaction(SIGCHLD, &restart, NULL)");
        exit(1);
      };
    };

#if DEBUG
    if (debug)
      lprintf(LP_DEBUG, "exiting main loop, watching pid %d", pid);
#endif /* DEBUG */

    if (errno != ESRCH) {
      lprintf(LP_SYSTEM, "kill(%d, 0)", pid);
      exit(1);
    }
  };
};
