/*
%%% copyright-cmetz-97
This software is Copyright 1997-1998 by Craig Metz, All Rights Reserved.
The Inner Net License Version 2 applies to this software.
You should have received a copy of the license with this software. If
you didn't get a copy, you may request one from <license@inner.net>.

*/

#include <stdio.h>
#include <sys/types.h>
#include <syslog.h>
#include <errno.h>
#include <stdarg.h>

#include "inner.h"

#define INNER_LPRINTF_INITIALIZED 0x8000
#define INNER_LPRINTF_INTERNALFLAGS (INNER_LPRINTF_INITIALIZED)

#define INNER_LPRINTF_INIT(opts) \
  if (!((opts)->flags & INNER_LPRINTF_INITIALIZED)) { \
    memset((opts), 0, sizeof(struct inner_lprintf_opts)); \
    (opts)->flags = INNER_LPRINTF_INITIALIZED | INNER_LPRINTF_MYNAME; \
    if (inner_debug) \
      (opts)->flags |= INNER_LPRINTF_SHOWDEBUG | INNER_LPRINTF_SHOWSOURCE; \
    (opts)->callback_print = print_stdio; \
  }

struct inner_lprintf_opts {
  unsigned int flags;
  void (*callback_print)(int, char *);

  char *inputfile;
  unsigned int *inputline;

  char *sourcefilefunction;
  unsigned int sourceline;
};

static struct inner_lprintf_opts inner_lprintf_opts;

#if DEBUG
extern int inner_debug;
#endif /* DEBUG */

static void print_stdio(int type, char *message)
{
  FILE *file;

  switch(type) {
    case INNER_LPRINTF_INFO:
      file = stdout;
      break;
    default:
      file = stderr;
  };

  fprintf(file, "%s\n", message);
};

static struct inner_lprintf_callback callback_stdio = {
  0, 0, &print_stdio };
struct inner_lprintf_callback *inner_lprintf_callback_stdio = &callback_stdio;

static void print_syslog(int type, char *message)
{
  int priority;

  switch(type) {
    case INNER_LPRINTF_INFO:
      priority = LOG_INFO;
      break;
    case INNER_LPRINTF_WARNING:
      priority = LOG_WARNING;
      break;
    case INNER_LPRINTF_ERROR:
      priority = LOG_ERR;
      break;
    case INNER_LPRINTF_SYNTAX:
      priority = LOG_ERR;
      break;
    case INNER_LPRINTF_INTERNAL:
      priority = LOG_ERR;
      break;
    case INNER_LPRINTF_SYSTEM:
      priority = LOG_ERR;
      break;
    case INNER_LPRINTF_DEBUG:
      priority = LOG_DEBUG;
      break;
    default:
      priority = LOG_ERR;
  };

  syslog(priority, message);
};

static struct inner_lprintf_callback callback_syslog = {
  INNER_LPRINTF_MYNAME, 0, &print_syslog };
struct inner_lprintf_callback *inner_lprintf_callback_syslog = &callback_syslog;

void inner_lprintf_opts_file(char *name, unsigned int *line)
{
  struct inner_lprintf_opts *opts = &inner_lprintf_opts;

  INNER_LPRINTF_INIT(opts);

  if (name && line) {
    opts->inputfile = name;
    opts->inputline = line;
  } else {
    opts->inputfile = NULL;
    opts->inputline = NULL;
  };
};

void inner_lprintf_opts_source(char *filefunction, unsigned int line)
{
  struct inner_lprintf_opts *opts = &inner_lprintf_opts;

  INNER_LPRINTF_INIT(opts);

  if (filefunction && line) {
    opts->sourcefilefunction = filefunction;
    opts->sourceline = line;
  } else {
    opts->sourcefilefunction = NULL;
    opts->sourceline = 0;
  };
};

void inner_lprintf_opts_callback(struct inner_lprintf_callback *callback)
{
  struct inner_lprintf_opts *opts = &inner_lprintf_opts;
  unsigned int mask;

  INNER_LPRINTF_INIT(opts);

  if (!(callback && callback->print))
    callback = inner_lprintf_callback_stdio;

  opts->callback_print = callback->print;
  mask = callback->mask & ~INNER_LPRINTF_INTERNALFLAGS;
  opts->flags = (opts->flags & (INNER_LPRINTF_INTERNALFLAGS | ~mask)) | (callback->flags & mask);
};

void inner_lprintf_opts_flags(unsigned int mask, unsigned int flags)
{
  struct inner_lprintf_opts *opts = &inner_lprintf_opts;

  INNER_LPRINTF_INIT(opts);

  mask &= ~INNER_LPRINTF_INTERNALFLAGS;
  opts->flags = (opts->flags & (INNER_LPRINTF_INTERNALFLAGS | ~mask)) | (flags & mask);
};

void inner_lprintf(int type, char *format, ...)
{
  va_list ap;
  char *prepend, *append;
  char buffer[1024], error[256];
  int buflen;
  int i;
  struct inner_lprintf_opts *opts = &inner_lprintf_opts;

  INNER_LPRINTF_INIT(opts);

  switch(type) {
    case INNER_LPRINTF_INFO:
      prepend = append = NULL;
      break;
    case INNER_LPRINTF_WARNING:
      prepend = "warning: ";
      append = NULL;
      break;
    case INNER_LPRINTF_ERROR:
      prepend = "error: ";
      append = NULL;
      break;
    case INNER_LPRINTF_SYNTAX:
      prepend = "syntax error: ";
      append = NULL;
      break;
    case INNER_LPRINTF_INTERNAL:
      prepend = "internal error: ";
      append = NULL;
      break;
    case INNER_LPRINTF_SYSTEM:
      snprintf(error, sizeof(error), ": %s(%d)", strerror(errno), errno);
      prepend = "system error: ";
      append = error;
      break;
    case INNER_LPRINTF_DEBUG:
      if (!(opts->flags & INNER_LPRINTF_SHOWDEBUG))
	return;
      prepend = append = NULL;
      break;
    default:
      snprintf(error, sizeof(error), "(inner_lprintf got invalid type %d) ", type);
      prepend = error;
      append = NULL;
      break;
  };

  buflen = 0;

  if ((opts->flags & INNER_LPRINTF_MYNAME) && inner_myname) {
    if ((i = snprintf(buffer + buflen, sizeof(buffer) - buflen, "%s: ", inner_myname)) < 0)
      goto overrun;
    buflen += i;
  };

  if ((opts->flags & INNER_LPRINTF_SHOWSOURCE) && opts->sourcefilefunction && opts->sourceline) {
    if ((i = snprintf(buffer + buflen, sizeof(buffer) - buflen, "%s:%d: ", opts->sourcefilefunction, opts->sourceline)) < 0)
      goto overrun;
    buflen += i;
  };

  if (opts->inputfile && opts->inputline && *opts->inputline) {
    if ((i = snprintf(buffer + buflen, sizeof(buffer) - buflen, "%s:%d: ", opts->inputfile, *opts->inputline)) < 0)
      goto overrun;
    buflen += i;
  };

  if (prepend) {
    if ((i = snprintf(buffer + buflen, sizeof(buffer) - buflen, prepend)) < 0)
      goto overrun;
    buflen += i;
  };

  va_start(ap, format);
  if ((i = vsnprintf(buffer + buflen, sizeof(buffer) - buflen, format, ap)) < 0)
    goto overrun;
  va_end(ap);
  if ((i > 0) && (*(buffer + buflen + i - 1)) == '\n')
    i--;
  buflen += i;

  if (append) {
    if ((i = snprintf(buffer + buflen, sizeof(buffer) - buflen, append)) < 0)
      goto overrun;
    buflen += i;
  };

  (*opts->callback_print)(type, buffer);
  return;

overrun:
  (*opts->callback_print)(INNER_LPRINTF_INTERNAL, "inner_lprintf: internal buffer overrun");
  return;
};
