/*
%%% portions-copyright-cmetz-96
Portions of this software are Copyright 1996-1998 by Craig Metz, All Rights
Reserved. The Inner Net License Version 2 applies to these portions of
the software.
You should have received a copy of the license with this software. If
you didn't get a copy, you may request one from <license@inner.net>.

%%% copyright-nrl-95
This software is Copyright 1995-1998 by Randall Atkinson, Ronald Lee,
Daniel McDonald, Bao Phan, and Chris Winters. All Rights Reserved. All
rights under this copyright have been assigned to the US Naval Research
Laboratory (NRL). The NRL Copyright Notice and License Agreement Version
1.1 (January 17, 1995) applies to this software.
You should have received a copy of the license with this software. If you
didn't get a copy, you may request one from <license@ipv6.nrl.navy.mil>.

*/

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/param.h>
#include <netdb.h>
#include <string.h>
#include <sys/un.h>
#include <netinet/in.h>

#include "support.h"

#define min(a,b) ((a) <= (b) ? (a) : (b))

static char buffer[12];
static char hextab[] = { '0', '1', '2', '3', '4', '5', '6', '7',
                         '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };

static struct nrl_nametonum afnametonum[] = {
  { AF_INET, "inet", sizeof(struct sockaddr_in) },
#if INET6
  { AF_INET6, "inet6", sizeof(struct sockaddr_in6) },
#endif /* INET6 */
  { AF_LOCAL, "local", sizeof(struct sockaddr_un) },
  { AF_UNIX, "unix", sizeof(struct sockaddr_un) },
  { 0, NULL, -1 }
};

static struct nrl_nametonum socktypenametonum[] = {
  { SOCK_RAW, "raw", 0 },
  { SOCK_STREAM, "stream", 0 },
  { SOCK_DGRAM, "dgram", 0 },
#ifndef SOCK_UNSPEC
#define SOCK_UNSPEC 0
#endif /* SOCK_UNSPEC */
  { SOCK_UNSPEC, "unspec", 0 },
  { 0, NULL, -1 }
};

int nrl_nametonum(tab, name)
struct nrl_nametonum *tab;
char *name;
{
  int i, j = -1;

  i = strlen(name);

  for (; tab->name; tab++) {
    if (!strcasecmp(tab->name, name))
      return(tab->num);
    if (i && !strncmp(tab->name, name, i))
      if (j < 0)
        j = tab->num;
      else {
        i = 0; 
        j = -1;
      };
  };

  return j;
}

char *nrl_numtoname(tab, num)
struct nrl_nametonum *tab;
int num;
{
  for (; tab->name; tab++)
    if (tab->num == num)
      return(tab->name);

  return NULL;
}

int nrl_numtoflags(tab, num)
struct nrl_nametonum *tab;
int num;
{
  for (; tab->name; tab++)
    if (tab->num == num)
      return(tab->flags);

  return -1;
}

int nrl_afnametonum(name)
char *name;
{
  int i = nrl_nametonum(afnametonum, name);
  if ((i < 0) && isdigit(*name))
    i = atoi(name);
  return i;
}

char *nrl_afnumtoname(int num)
{
  char *c = nrl_numtoname(afnametonum, num);
 
  if (!c) {
    strcpy(buffer, "af ");
    buffer[3] = hextab[(num & 0xf0) >> 4];
    buffer[4] = hextab[num & 0x0f];
    buffer[5] = 0;
    return buffer;
  };

  return c;
};

int nrl_socktypenametonum(char *name)
{
  int i = nrl_nametonum(socktypenametonum, name);
  if ((i < 0) && isdigit(*name))
    i = atoi(name);
  return i;
};

char *nrl_socktypenumtoname(int num)
{
  char *c = nrl_numtoname(socktypenametonum, num);
 
  if (!c) {
    strcpy(buffer, "socktype ");    
    buffer[9] = hextab[(num & 0xf0) >> 4];
    buffer[10] = hextab[num & 0x0f];
    buffer[11] = 0;
    return buffer;
  };

  return c;
};

static char *domain = NULL;
static char domainbuffer[MAXHOSTNAMELEN] = "";

char *nrl_domainname(void)
{
  static int first = 1;

  if (first) {
    char *c, buf[MAXHOSTNAMELEN];
    struct hostent *h;

    first = 0;

    if ((h = gethostbyname("localhost")) && (c = strchr(h->h_name, '.')))
      return strcpy(domain = domainbuffer, ++c);

    if (!gethostname(domainbuffer, sizeof(domainbuffer))) {
      if (c = strchr(domainbuffer, '.'))
        return (domain = ++c);

      if ((h = gethostbyname(domainbuffer)) && (c = strchr(h->h_name, '.')))
        return strcpy(domain = domainbuffer, ++c);
    }

    {
      struct in_addr in_addr;

      in_addr.s_addr = htonl(0x7f000001);

      if ((h = gethostbyaddr((const char *)&in_addr, sizeof(struct in_addr), AF_INET)) && (c = strchr(h->h_name, '.')))
        return strcpy(domain = domainbuffer, ++c);
    }

    return NULL;
  };

  return domain;
};

int nrl_build_samask(struct sockaddr *mask, int af, int len)
{
  u_int8_t *p;
  if (len < -1)
    return -1;
  switch(af) {
#if INET6
    case AF_INET6:
      if (len > 128)
        return -1;
      if (len == -1)
        len = 128;
      p = (u_int8_t *)&(((struct sockaddr_in6 *)mask)->sin6_addr);
      memset(mask, 0, sizeof(struct sockaddr_in6));
#if 0
      ((struct sockaddr_in6 *)mask)->sin6_port = ~1;
#endif /* 0 */
      break;
#endif /* INET6 */
    case AF_INET:
      if (len > 32)
        return -1;
      if (len == -1)
        len = 32;
      p = (u_int8_t *)&(((struct sockaddr_in *)mask)->sin_addr);
      memset(mask, 0, sizeof(struct sockaddr_in));
#if 0
      ((struct sockaddr_in *)mask)->sin_port = ~1;
#endif /* 0 */
      break;
#if LOCAL
    case AF_LOCAL:
      len = 0;
      memset(mask, 0, sizeof(struct sockaddr_in));
      break;
#endif /* LOCAL */
    default:
      return -1;
  };
#if SALEN
  mask->sa_len = ~1;
#endif /* SALEN */
  mask->sa_family = ~1;
  while(len >= 8) {
    *(p++) = 0xff;
    len -= 8;
  };
  while(len--)
    *p = (*p << 1) | 1;
  return 0;
};

int nrl_prefix_length(struct sockaddr *mask, int af)
{
  int len = 0;
  int i, l;
  u_int8_t *p;

  switch(af) {
  case AF_INET:
#if SALEN
    l = min(mask->sa_len - 4, sizeof(struct in_addr));
#else /* SALEN */
    l = sizeof(struct in_addr);
#endif /* SALEN */
    p = (u_int8_t *)&((struct sockaddr_in *)mask)->sin_addr;
    break;
#if INET6
  case AF_INET6:
#if SALEN
    l = min(mask->sa_len - 8, sizeof(struct in6_addr));
#else /* SALEN */
    l = sizeof(struct in6_addr);
#endif /* SALEN */
    p = (u_int8_t *)&((struct sockaddr_in6 *)mask)->sin6_addr;
    break;
#endif /* INET6 */
  default:
    return -1;
  }

  while ((l > 0) && (*p == 0xff)) {
    len += 8;
    p++;
    l--;
  }
  if (l < 1)
    return len;

  if (*p) {
    i = *(p++);
    l--;
    while(i & 0x80) {
      len++;
      i = (i << 1) & 0xff;
    }
    if (i)
      return -1;
  }

  while(l-- > 0)
    if (*(p++))
      return -1;

  return len;
}

unsigned int __sa_len(const struct sockaddr *sockaddr)
{
  switch(sockaddr->sa_family) {
    case AF_UNIX:
      return sizeof(struct sockaddr_un);
    case AF_INET:
      return sizeof(struct sockaddr_in);
    case AF_INET6:
      return sizeof(struct sockaddr_in6);
    default:
      return 0;
  };
};
