/*
%%% portions-copyright-cmetz-96
Portions of this software are Copyright 1996-1998 by Craig Metz, All Rights
Reserved. The Inner Net License Version 2 applies to these portions of
the software.
You should have received a copy of the license with this software. If
you didn't get a copy, you may request one from <license@inner.net>.

%%% copyright-nrl-95
This software is Copyright 1995-1998 by Randall Atkinson, Ronald Lee,
Daniel McDonald, Bao Phan, and Chris Winters. All Rights Reserved. All
rights under this copyright have been assigned to the US Naval Research
Laboratory (NRL). The NRL Copyright Notice and License Agreement Version
1.1 (January 17, 1995) applies to this software.
You should have received a copy of the license with this software. If you
didn't get a copy, you may request one from <license@ipv6.nrl.navy.mil>.

*/

#include <sys/types.h>
#include <sys/socket.h>

#include <netinet/in.h>
#ifndef linux
#include <netns/ns.h>

struct iso_addr;
char *iso_ntoa __P((struct iso_addr *));
struct sockaddr_dl;
char *link_ntoa __P((struct sockaddr_dl *));
#endif /* linux */

#include <stdio.h>
#include <string.h>
#include <arpa/inet.h>

#include "support.h"

static char inet_ntop_buffer[INET6_ADDRSTRLEN];

#ifndef min
#define min(x,y) ( ((x) < (y)) ? (x) : (y) )
#endif /* min */

#if inet_ntop == __inet_ntop
#undef inet_ntop

const char *inet_ntop(int af, const void *ap, char *cp, size_t len);

const char *__inet_ntop(int af, const void *ap, char *cp, size_t len)
{
  return inet_ntop(af, ap, cp, len);
};
#endif /* inet_ntop == __inet_ntop */

#if INET6
static char hextab[] = { '0', '1', '2', '3', '4', '5', '6', '7',
                         '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };

extern const struct in6_addr nrl_in6addr_mapped;

static void __in6_ntoa(struct in6_addr *addr, char *buffer)
{
  char *op = buffer;
  uint8_t *ap;

  /*
   * Treat v4-mapped and v4-compatible differently.
   * (In fact v4-mapped, may want to just return the v4 address.)
   */
  if (IN6_IS_ADDR_V4MAPPED(addr)) {
    strcpy(op,"::ffff:");
    op += 7;
    inet_ntop(AF_INET, &addr->s6_addr[12], op, 38);
  }
  if (IN6_IS_ADDR_V4COMPAT(addr)) {
    strcpy(op,"::");
    op += 2;
    inet_ntop(AF_INET, &addr->s6_addr[12], op, 43);
  }

  *op = '\0';
  ap = (uint8_t *)addr;

  {
    char unpack[4];
    int i, j, k;
    int start = 0, len = 0;
    unsigned short *sp;

    for (i = j = k = 0, sp = (unsigned short *)ap; i < 8; i++) {
      if (*(sp++)) {
	if (j >= len) {
	  start = k;
	  len = j;
	  j = 0;
	}
      } else {
	if (!(j++))
	  k = i;
      }
    }
    if (j >= len) {
      start = k;
      len = j;
    }

    if (len < 2)
      start = 9;

    if (!start)
      *(op++) = ':';

    i = 0;
    while(i < 8) {
      if (i == start) {
	*(op++) = ':';
	ap += len * 2;
	i += len;
	continue;
      }

      unpack[0] = ((*(char *)ap) & 0xf0) >> 4;
      unpack[1] =  (*(char *)ap) & 0x0f; ap++;
      unpack[2] = ((*(char *)ap) & 0xf0) >> 4;
      unpack[3] =  (*(char *)ap) & 0x0f; ap++;

      for (j = 0; (j < 3) && !unpack[j]; j++);
      while(j < 4)
	*(op++) = hextab[unpack[j++]];
      
      *(op++) = ':';
      i++;
    }
    if ((start + len) == 8) 
      *(op++) = '0';
    else
      --op;
    *op = 0;
  }
}
#endif /* INET6 */

#ifndef INET_ADDRSTRLEN
#define INET_ADDRSTRLEN sizeof("255.255.255.255")
#endif /* INET_ADDRSTRLEN */

const char *inet_ntop(int af, const void *ap, char *cp, size_t len)
{
  char *ptr;
  int maxlen;

  ptr = NULL;
  if (!cp) {
    cp = inet_ntop_buffer;
    if (len)
      len = min(len, sizeof(inet_ntop_buffer));
    else
      len = sizeof(inet_ntop_buffer);
  }

  switch(af) {
  case AF_INET:
    {
      char buffer[INET_ADDRSTRLEN];
      const uint8_t *p = ap;

      snprintf(buffer, INET_ADDRSTRLEN, "%d.%d.%d.%d", p[0], p[1], p[2], p[3]);
      len = min(len, INET_ADDRSTRLEN);
      strncpy(cp, buffer, len - 1);
      cp[len - 1] = 0;
      return cp;
    };
    break;

#if INET6
  case AF_INET6:
    {
      char buffer[INET6_ADDRSTRLEN];

      __in6_ntoa((struct in6_addr *)ap, buffer);
      len = min(len, INET6_ADDRSTRLEN);
      strncpy(cp, buffer, len - 1);
      cp[len - 1] = 0;
      return cp;
    };
    break;
#endif /* INET6 */

#ifndef linux
#if defined(AF_ISO) && !__FreeBSD__
  case AF_ISO:
    ptr = iso_ntoa((struct iso_addr *)ap);
    /* len? */
    break;
#endif /* defined(AF_ISO) && !__FreeBSD__ */

#ifdef AF_NS
  case AF_NS:
    ptr = ns_ntoa(*((struct ns_addr *)ap));
    /* len? */
    break;
#endif /* AF_NS */

#ifdef AF_LINK
  case AF_LINK:
    ptr = link_ntoa((struct sockaddr_dl *)ap);
    /* len? */
    break;
#endif /* AF_LINK */
#endif /* linux */
  }

  if (ptr) {
    strncpy(cp, ptr, len - 1);
    cp[len - 1] = 0;
    return cp;
  }

  return NULL;
}
