#ifndef EXT2_FS_C_H
#define EXT2_FS_C_H

#include <linux/ioctl.h>
#include <linux/types.h>

#ifdef __KERNEL__
# include <linux/config.h>
/* If defined, compress each cluster as soon as we get to the end of a
   whole cluster, when writing.  (If undefined, we wait until
   ext2_release_file() or the like.) */
# define EXT2_COMPRESS_WHEN_CLU

/* Some working area details are yet to be worked out, so I'm using this
   set of wrappers. */
/* # define ext2_rd_wa (ext2_wa[0]) */
# define ext2_rd_wa_lock (ext2_wa_lock[0])
# define ext2_lock_rd_wa() ext2_lock_wa(0)
# define ext2_unlock_rd_wa() ext2_unlock_wa(0)
# define ext2_rd_wa_ix 0
# ifdef CONFIG_EXT2_SEPARATE_WORK_AREAS
#  define EXT2_N_WA 2
#  define ext2_wr_wa_ix 1
#  define ext2_wr_wa_lock (ext2_wa_lock[1])
#  define ext2_lock_wr_wa() ext2_lock_wa(1)
#  define ext2_unlock_wr_wa() ext2_unlock_wa(1)
# else
#  define EXT2_N_WA 1
#  define ext2_wr_wa_ix 0
#  define ext2_wr_wa_lock (ext2_wa_lock[0])
#  define ext2_lock_wr_wa() ext2_lock_wa(0)
#  define ext2_unlock_wr_wa() ext2_unlock_wa(0)
# endif
#endif /* __KERNEL__ */

#define EXT2_IOC_GETCLUSTERSIZE 	_IOR('c', 0, long)
#define EXT2_IOC_SETCLUSTERSIZE 	_IOW('c', 0, long)
#define EXT2_IOC_GETCOMPRMETHOD 	_IOR('c', 1, long)
#define EXT2_IOC_SETCOMPRMETHOD 	_IOW('c', 1, long)
#define EXT2_IOC_GETFIRSTCLUSTERSIZE 	_IOR('c', 2, long)
#define EXT2_IOC_RECOGNIZE_COMPRESSED	_IOW('c', 2, long)
#define EXT2_IOC_GETCLUSTERBIT		_IOR('c', 3, long)
#define EXT2_IOC_GETCOMPRRATIO		_IOR('c', 4, long)
/* Don't use _IOW('c', {5,6}, long), as these are used by old
   e2compress binaries as SETCLUSTERBIT and CLRCLUSTERBIT
   respectively. */

/* EXT2_xxxx_ALG is an index into ext2_algorithm_table[] defined in
   fs/ext2/compress.c. */
/* N.B. Don't change these without also changing the table in
   compress.c.  Be careful not to break binary compatibility.
   (EXT2_NONE_ALG and EXT2_UNDEF_ALG are safe from binary
   compatibility problems, though, so they can safely be renumbered --
   and indeed probably should be if you do add another algorithm.) */
#define EXT2_LZV1_ALG 0
#define EXT2_LZRW3A_ALG 1
#define EXT2_GZIP_ALG 2
#define EXT2_BZIP2_ALG 3
#define EXT2_LZO_ALG 4
#define EXT2_NONE_ALG 5
#define EXT2_UNDEF_ALG 6
#define EXT2_N_ALGORITHMS 5 /* Count of "real" algorithms.  Excludes
			       `none' and `undef'. */

/* EXT2_xxxx_METH is an index into ext2_method_table[] defined in
   fs/ext2/compress.c. */
/* N.B. Don't change these without also changing the table in
   compress.c. */
#define EXT2_LZV1_METH 0
#define EXT2_AUTO_METH 1
#define EXT2_DEFER_METH 2
#define EXT2_NEVER_METH 3
#define EXT2_BZIP2_METH 4
#define EXT2_LZRW3A_METH 8
#define EXT2_LZO1X_1_METH 10
#define EXT2_GZIP_1_METH 16
#define EXT2_GZIP_2_METH 17
#define EXT2_GZIP_3_METH 18
#define EXT2_GZIP_4_METH 19
#define EXT2_GZIP_5_METH 20
#define EXT2_GZIP_6_METH 21
#define EXT2_GZIP_7_METH 22
#define EXT2_GZIP_8_METH 23
#define EXT2_GZIP_9_METH 24
#define EXT2_N_METHODS 32 /* Don't change this unless you know what
			      you're doing.  In particular, it's tied
			      to the width of the algorithm field
			      in i_flags.*/

#ifdef __KERNEL__
# ifdef CONFIG_EXT2_DEFAULT_COMPR_METHOD_NONE
#  define EXT2_DEFAULT_COMPR_METHOD	EXT2_DEFER_METH
# elif defined (CONFIG_EXT2_DEFAULT_COMPR_METHOD_LZO)
#  define EXT2_DEFAULT_COMPR_METHOD	EXT2_LZO1X_1_METH
#  ifndef CONFIG_EXT2_HAVE_LZO
#   error "Default algorithm (lzo) is not compiled in."
#  endif
# elif defined (CONFIG_EXT2_DEFAULT_COMPR_METHOD_LZV1)
#  define EXT2_DEFAULT_COMPR_METHOD	EXT2_LZV1_METH
#  ifndef CONFIG_EXT2_HAVE_LZV1
#   error "Default algorithm (lzv1) is not compiled in."
#  endif
# elif defined (CONFIG_EXT2_DEFAULT_COMPR_METHOD_LZRW3A)
#  define EXT2_DEFAULT_COMPR_METHOD	EXT2_LZRW3A_METH
#  ifndef CONFIG_EXT2_HAVE_LZRW3A
#   error "Default algorithm (lzrw3a) is not compiled in."
#  endif
# elif defined (CONFIG_EXT2_DEFAULT_COMPR_METHOD_GZIP1)
#  define EXT2_DEFAULT_COMPR_METHOD	EXT2_GZIP_1_METH
# elif defined (CONFIG_EXT2_DEFAULT_COMPR_METHOD_GZIP2)
#  define EXT2_DEFAULT_COMPR_METHOD	EXT2_GZIP_2_METH
# elif defined (CONFIG_EXT2_DEFAULT_COMPR_METHOD_GZIP3)
#  define EXT2_DEFAULT_COMPR_METHOD	EXT2_GZIP_3_METH
# elif defined (CONFIG_EXT2_DEFAULT_COMPR_METHOD_GZIP4)
#  define EXT2_DEFAULT_COMPR_METHOD	EXT2_GZIP_4_METH
# elif defined (CONFIG_EXT2_DEFAULT_COMPR_METHOD_GZIP5)
#  define EXT2_DEFAULT_COMPR_METHOD	EXT2_GZIP_5_METH
# elif defined (CONFIG_EXT2_DEFAULT_COMPR_METHOD_GZIP6)
#  define EXT2_DEFAULT_COMPR_METHOD	EXT2_GZIP_6_METH
# elif defined (CONFIG_EXT2_DEFAULT_COMPR_METHOD_GZIP7)
#  define EXT2_DEFAULT_COMPR_METHOD	EXT2_GZIP_7_METH
# elif defined (CONFIG_EXT2_DEFAULT_COMPR_METHOD_GZIP8)
#  define EXT2_DEFAULT_COMPR_METHOD	EXT2_GZIP_8_METH
# elif defined (CONFIG_EXT2_DEFAULT_COMPR_METHOD_GZIP9)
#  define EXT2_DEFAULT_COMPR_METHOD	EXT2_GZIP_9_METH
# elif defined (CONFIG_EXT2_DEFAULT_COMPR_METHOD_BZIP2)
#  define EXT2_DEFAULT_COMPR_METHOD	EXT2_BZIP2_METH
#  ifndef CONFIG_EXT2_HAVE_BZIP2
#   error "Default algorithm (bzip2) is not compiled in."
#  endif
# else
#  error "No default compression algorithm."
# endif
# if EXT2_DEFAULT_COMPR_METHOD >= EXT2_GZIP_1_METH && EXT2_DEFAULT_COMPR_METHOD <= EXT2_GZIP_9_METH
#  ifndef CONFIG_EXT2_HAVE_GZIP
#   error "Default algorithm (gzip) is not compiled in."
#  endif
# endif

# if defined (CONFIG_EXT2_DEFAULT_CLUSTER_BITS_2)
#  define EXT2_DEFAULT_LOG2_CLU_NBLOCKS	2
# elif defined (CONFIG_EXT2_DEFAULT_CLUSTER_BITS_3)
#  define EXT2_DEFAULT_LOG2_CLU_NBLOCKS	3
# elif defined (CONFIG_EXT2_DEFAULT_CLUSTER_BITS_4)
#  define EXT2_DEFAULT_LOG2_CLU_NBLOCKS	4
# elif defined (CONFIG_EXT2_DEFAULT_CLUSTER_BITS_5)
#  define EXT2_DEFAULT_LOG2_CLU_NBLOCKS	5
# else
#  error "No default cluster size."
# endif

# define EXT2_DEFAULT_CLU_NBLOCKS	(1 << EXT2_DEFAULT_LOG2_CLU_NBLOCKS)

# if (EXT2_LZV1_ALG != 0) || (EXT2_BZIP2_ALG != 3) || (EXT2_LZO_ALG != 4) || (EXT2_N_ALGORITHMS != 5)
#  error "this code needs changing; but then, you shouldn't be messing with algorithm ids anyway unless you are very careful to protect disk format compatibility"
# endif
# ifdef CONFIG_EXT2_HAVE_LZV1
#  define _ext2_lzv1_builtin (1 << EXT2_LZV1_ALG)
# else
#  define _ext2_lzv1_builtin 0
# endif
# define _ext2_lzrw3a_builtin 0
# ifdef CONFIG_EXT2_HAVE_GZIP
#  define _ext2_gzip_builtin (1 << EXT2_GZIP_ALG)
# else
#  define _ext2_gzip_builtin 0
# endif
# ifdef CONFIG_EXT2_HAVE_BZIP2
#  define _ext2_bzip2_builtin (1 << EXT2_BZIP2_ALG)
# else
#  define _ext2_bzip2_builtin 0
# endif
# ifdef CONFIG_EXT2_HAVE_LZO
#  define _ext2_lzo_builtin (1 << EXT2_LZO_ALG)
# else
#  define _ext2_lzo_builtin 0
# endif

# ifdef CONFIG_EXT2_HAVE_LZV1_MODULE
#  define _ext2_lzv1_module (1 << EXT2_LZV1_ALG)
# else
#  define _ext2_lzv1_module 0
# endif
# define _ext2_lzrw3a_module 0
# ifdef CONFIG_EXT2_HAVE_GZIP_MODULE
#  define _ext2_gzip_module (1 << EXT2_GZIP_ALG)
# else
#  define _ext2_gzip_module 0
# endif
# ifdef CONFIG_EXT2_HAVE_BZIP2_MODULE
#  define _ext2_bzip2_module (1 << EXT2_BZIP2_ALG)
# else
#  define _ext2_bzip2_module 0
# endif
# ifdef CONFIG_EXT2_HAVE_LZO_MODULE
#  define _ext2_lzo_module (1 << EXT2_LZO_ALG)
# else
#  define _ext2_lzo_module 0
# endif

# define EXT2_ALGORITHMS_MODULE  (_ext2_lzv1_module | _ext2_lzrw3a_module | _ext2_gzip_module | _ext2_bzip2_module | _ext2_lzo_module)
# define EXT2_ALGORITHMS_BUILTIN  (_ext2_lzv1_builtin | _ext2_lzrw3a_builtin | _ext2_gzip_builtin | _ext2_bzip2_builtin | _ext2_lzo_builtin)

# if EXT2_ALGORITHMS_MODULE & EXT2_ALGORITHMS_BUILTIN
#  error "Arithmetic error?  Some algorithm appears to be both built-in and a module."
# endif

/* EXT2_ALGORITHMS_SUPP is what we test when mounting a filesystem.
   See fs/ext2/super.c. */
# define EXT2_ALGORITHMS_SUPP (EXT2_ALGORITHMS_MODULE | EXT2_ALGORITHMS_BUILTIN)
# if EXT2_ALGORITHMS_SUPP == 0
#  error "You must select at least one compression algorithm."
# endif

/* Cluster head on disk.  Little-endian. */
struct ext2_cluster_head {
  __u16 magic;		/* == EXT2_COMPRESS_MAGIC_04X. */
  __u8  method;		/* compression method id. */
  __u8  holemap_nbytes;	/* length of holemap[] array */
  __u32 checksum;	/* adler32 checksum.  Checksum covers all fields
			   below this one, and the compressed data. */
  __u32 ulen;		/* size of uncompressed data */
  __u32 clen;		/* size of compressed data (excluding cluster head) */
  __u8  holemap[0];     /* bitmap describing where to put holes. */
};


# define EXT2_MAX_CLUSTER_BYTES		(32*1024)
# define EXT2_LOG2_MAX_CLUSTER_BYTES	(5 + 10)


struct ext2_wa_S {
  __u8 u[EXT2_MAX_CLUSTER_BYTES];  /* Uncompressed data. */
  __u8 c[EXT2_MAX_CLUSTER_BYTES];  /* Compressed data. */
  __u8 heap[1];  /* Heap: working space for de/compression routines. */
};
extern struct ext2_wa_S *ext2_rd_wa;
extern size_t ext2_rd_wa_size;
# ifdef CONFIG_EXT2_SEPARATE_WORK_AREAS
extern struct ext2_wa_S *ext2_wr_wa;
extern size_t ext2_wr_wa_size;
# else
#  define ext2_wr_wa ext2_rd_wa
#  define ext2_wr_wa_size ext2_rd_wa_size
# endif

# define EXT2_CLEANUP_FL		0x40 /* See Readme.e2compr */
# define EXT2_COMPRESS_MAGIC_04X	0x9ec7
# define EXT2_MAX_CLUSTER_BLOCKS	32
# define EXT2_ECOMPR			EIO
/* A cluster is considered compressed iff the block number for the
   last block of that cluster is EXT2_COMPRESSED_BLKADDR.  If this
   changes then check if there's anywhere that needs a cpu_to_le32()
   conversion. */
# define EXT2_COMPRESSED_BLKADDR	0xffffffff 

# define ROUNDUP_DIV(_n, _d) ((_n) ? 1 + (((_n) - 1) / (_d)) : 0)
# define ROUNDUP_RSHIFT(_n, _b) ((_n) ? 1 + (((_n) - 1) >> (_b)) : 0)

# if defined(EXT2_NDIR_BLOCKS) && (EXT2_NDIR_BLOCKS != 12)
#  error "e2compr currently assumes that EXT2_NDIR_BLOCKS is 12."
/* If EXT2_NDIR_BLOCKS changes then change the definitions of
   ext2_first_cluster_nblocks() and friends, and search the patch for
   anywhere where 12 is hard-coded.  (At the time of writing, it's
   only hard-coded in ext2_first_cluster_nblocks().)  What we want to
   achieve is for clusters not to straddle address blocks.  Apart from
   performance, some code in compress.c (search for `straddle')
   assumes this. */
# endif

/* I like these names better. */
# define EXT2_MAX_CLU_NBYTES EXT2_MAX_CLUSTER_BYTES
# define EXT2_LOG2_MAX_CLU_NBYTES EXT2_LOG2_MAX_CLUSTER_BYTES
# define EXT2_MAX_CLU_NBLOCKS EXT2_MAX_CLUSTER_BLOCKS

#else /* !__KERNEL__ */

/* Cluster head on disk, for e2compr versions before 0.4.0.  I'm
   leaving this here so tht as I may make e2compress able to read
   old-style e2compr files. */
struct ext2_cluster_head_03x {
  __u16 magic;			/* == EXT2_COMPRESS_MAGIC_03X */
  __u16 len;			/* size of uncompressed data */
  __u16 compr_len;		/* size of compressed data */
  __u8  method;			/* compress method */
  __u8  reserved_0;
  __u32 bitmap;			/* block bitmap */
  __u32 reserved_2;		/* 0 or adler32 checksum of
				   _compressed_ data */
};
# define EXT2_COMPRESS_MAGIC_03X	0x8ec7 /* Head magic number
						  for e2compr versions
						  before 0.4.0. */
#endif /* !__KERNEL__ */

#ifdef __KERNEL__
# include <linux/fs.h>
# include <linux/assert.h>


# define EXT2_ALG_INIT_COMPRESS		1
# define EXT2_ALG_INIT_DECOMPRESS	2

extern int    ext2_get_cluster_blocks (struct inode*, u32, struct buffer_head**);
extern int    ext2_decompress_cluster (struct inode*, u32);
extern int    ext2_compress_cluster (struct inode*, u32);
extern int    ext2_decompress_inode (struct inode*);
extern int    ext2_cleanup_compressed_inode (struct inode*);
extern void   ext2_update_comprblk (struct inode *);

extern void   ext2_lock_rd_wa_uninterruptible(void);
extern int    ext2_lock_wa (unsigned);
extern void   ext2_unlock_wa (unsigned);

extern size_t ext2_decompress_blocks    (struct inode*, struct buffer_head**, int, size_t);
extern int    ext2_count_blocks		(struct inode*);
extern int    ext2_recognize_compressed (struct inode *, unsigned cluster);
extern unsigned long ext2_adler32	(unsigned long, unsigned char*, int);

extern size_t ext2_iLZV1   (int);
extern size_t ext2_iLZV2   (int);
extern size_t ext2_iNONE   (int);
extern size_t ext2_iGZIP   (int);
extern size_t ext2_iBZIP2  (int);
extern size_t ext2_iLZO    (int);

extern size_t ext2_wLZV1   (__u8*, __u8*, void*, size_t, size_t, int);
extern size_t ext2_wLZV2   (__u8*, __u8*, void*, size_t, size_t, int);
extern size_t ext2_wNONE   (__u8*, __u8*, void*, size_t, size_t, int);
extern size_t ext2_wGZIP   (__u8*, __u8*, void*, size_t, size_t, int);
extern size_t ext2_wBZIP2  (__u8*, __u8*, void*, size_t, size_t, int);
extern size_t ext2_wLZO    (__u8*, __u8*, void*, size_t, size_t, int);

extern size_t ext2_rLZV1   (__u8*, __u8*, void*, size_t, size_t, int);
extern size_t ext2_rLZV2   (__u8*, __u8*, void*, size_t, size_t, int);
extern size_t ext2_rNONE   (__u8*, __u8*, void*, size_t, size_t, int);
extern size_t ext2_rGZIP   (__u8*, __u8*, void*, size_t, size_t, int);
extern size_t ext2_rBZIP2  (__u8*, __u8*, void*, size_t, size_t, int);
extern size_t ext2_rLZO    (__u8*, __u8*, void*, size_t, size_t, int);

struct ext2_algorithm { 
	char	*name; 
	int	avail;
	size_t (*init)		(int); 
	size_t (*compress)	(__u8*, __u8*, void*, size_t, size_t, int); 
	size_t (*decompress)	(__u8*, __u8*, void*, size_t, size_t, int);
};

struct ext2_method {
	unsigned  alg;
	int	  xarg;
};

extern int ext2_register_compression_module(unsigned alg,
				size_t compress_memory,
				size_t decompress_memory,
		                struct ext2_algorithm *new_algorithm);
extern void ext2_unregister_compression_module(unsigned alg);

# define ext2_first_cluster_nblocks(_i) ((_i)->u.ext2_i.i_clu_nblocks > 4 && (_i)->i_sb->s_blocksize < 4096 ? 12 : 4)
# define ext2_block_to_cluster(_i,_b)	((_b) < ext2_first_cluster_nblocks(_i) ? 0 : (((_b) - ext2_first_cluster_nblocks(_i)) >> (_i)->u.ext2_i.i_log2_clu_nblocks) + 1)
# define ext2_offset_to_cluster(_i,_o)	ext2_block_to_cluster((_i), ((_o) >> (_i)->i_sb->s_blocksize_bits))
# define ext2_n_clusters(_i)	((_i)->i_size ? ext2_offset_to_cluster((_i), (_i)->i_size - 1) + 1 : 0)
# define ext2_cluster_block0(_i,_c)	((_c) ? ext2_first_cluster_nblocks(_i) + (((_c) - 1) << (_i)->u.ext2_i.i_log2_clu_nblocks) : 0)
# define ext2_cluster_nblocks(_i,_c)	((_c) ? (_i)->u.ext2_i.i_clu_nblocks : ext2_first_cluster_nblocks(_i))
# define ext2_cluster_offset(_i,_c)	((_c) ? ext2_cluster_block0((_i), (_c)) << (_i)->i_sb->s_blocksize_bits : 0)

extern inline int
ext2_offset_is_clu_boundary(struct inode *inode, u32 off)
{
	if (off & (inode->i_sb->s_blocksize - 1))
		return 0;
	if (off == 0)
		return 1;
	off >>= inode->i_sb->s_blocksize_bits;
	if (off < ext2_first_cluster_nblocks(inode))
		return 0;
	off -= ext2_first_cluster_nblocks(inode);
	return !(off & (inode->u.ext2_i.i_clu_nblocks - 1));
}

struct ext2_wa_contents_S {
	ino_t ino;
	kdev_t dev;
	unsigned cluster;
};
extern struct ext2_wa_contents_S ext2_rd_wa_ucontents;
# ifdef CONFIG_EXT2_SEPARATE_WORK_AREAS
extern struct ext2_wa_contents_S ext2_wr_wa_ucontents;
# else
#  define ext2_wr_wa_ucontents ext2_rd_wa_ucontents
# endif

extern struct ext2_algorithm ext2_algorithm_table[];
extern struct ext2_method ext2_method_table[];

/* Both of these return -errno if error, 0 if not compressed, positive
   if compressed.  (You should use the macro unless you've already
   tested COMPRBLK.) */
extern int ext2_cluster_is_compressed_fn (struct inode *inode, __u32 cluster);
extern inline int ext2_cluster_is_compressed (struct inode *inode, __u32 cluster)
{
	if ((inode->u.ext2_i.i_flags & EXT2_COMPRBLK_FL) == 0)
		return 0;
	return ext2_cluster_is_compressed_fn (inode, cluster);
}

# define ext2_compression_error(_i)	((_i)->u.ext2_i.i_flags & EXT2_ECOMPR_FL)
# define ext2_compression_disabled(_i)	((_i)->u.ext2_i.i_flags & EXT2_NOCOMPR_FL)
# define ext2_compression_enabled(_i)	(! ext2_compression_disabled (_i))

#endif /* __KERNEL__ */



#ifdef __KERNEL__
# ifdef CONFIG_EXT2_COMPRESS
#  define HOLE_BLKADDR(_b) \
	(((_b) == 0) \
	 || ((_b) == EXT2_COMPRESSED_BLKADDR))
# else
#  define HOLE_BLKADDR(_b) ((_b) == 0)
# endif

/* For some reason or other, I see code like `if (le32_to_cpu(tmp) !=
   0)' around in the kernel.  So far I haven't checked whether or not
   the compiler knows that the swab can be dropped. */
# if defined(EXT2_COMPRESSED_BLKADDR) && EXT2_COMPRESSED_BLKADDR != 0xffffffff
/* This may be a false positive; the "correct" test would be `if
   defined(CONFIG_EXT2_COMPRESS)', but if this test does succeed, then
   there is at least cause to have a look around. */
#  error "Next bit of code is wrong."
# endif

# define HOLE_BLKADDR_SWAB32(_b) HOLE_BLKADDR(_b)
#endif /* __KERNEL__ */


#endif /* EXT2_FS_C_H */
