/*
 *	e2ratio.c	Simple program to display the compression ratio.
 *
 *	Copyright (C) 1995, Antoine Dumesnil de Maricourt.
 *
 *	This file can be redistributed under the terms of the GNU General
 *	Public License.
 */

#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <malloc.h>

#include <sys/ioctl.h>
#include <sys/param.h>
#include <sys/stat.h>
#include "ext2_fs_c.h"
#include "exit.h"

static const char *prog_name = "e2ratio";
static int every_file  = 0;
static int summarize   = 0;
static int show_ratio  = 1;
static int show_size   = 1;

static void
print_version_string (void)
{
  printf ("e2ratio " VERSION "\n");
}

static int
select_file (const struct dirent *d)
{
  /* Display all but `.' and `..'. */
  return ((d->d_name[0] != '.')
	  || ((d->d_name[1] != '\0')
	      && ((d->d_name[1] != '.') || (d->d_name[2] != '\0'))));
}

struct Ratio {
  long size[2];
};

static struct Ratio
get_ratio (char *path, int len, int echo)
{
  struct stat  st;
  struct Ratio ratio;
  char const *msg;

  ratio.size[0] = ratio.size[1] = 0;

  if (lstat (path, &st) == -1)
    {
      msg = "can't stat file";
      goto error;
    }

  if (S_ISDIR (st.st_mode))
    {
      struct dirent **lst;
      int i, count;

      if ((count = scandir (path, &lst, select_file, NULL)) < 0)
	{
	  msg = "can't scan directory";
	  goto error;
	}

      /*
       *	Do every file in directory.
       */

      if (len > 0)
	{
	  path[len++] = '/';
	  path[len] = '\0';
	}

      for (i = 0; i < count; i++)
	{
	  struct Ratio sub;

	  strcpy (path + len, lst[i]->d_name);
	  sub = get_ratio (path, len + strlen (lst[i]->d_name), 0);

	  ratio.size[0] += sub.size[0];
	  ratio.size[1] += sub.size[1];
	}

      /*
       *	Free the list
       */

      for (i = 0; i < count; i++)
	free (lst[i]);

      free (lst);
      path[len-1] = '\0'; 

      ratio.size[0] += st.st_blocks;
      ratio.size[1] += st.st_blocks;
    }
  else if (S_ISREG (st.st_mode))
    {
      long block[2];
      int  fd;

      if ((fd = open (path, O_RDONLY)) == -1)
	{
	  msg = "can't open file";
	  goto error;
	}

      if (ioctl (fd, EXT2_IOC_GETCOMPRRATIO, block) == -1)
	{
	  close (fd);
	  msg = "can't compute ratio";
	  goto error;
	}

      close (fd);

      ratio.size[0] += block[0];
      ratio.size[1] += block[1];
    }
  else
    {
      /*
       *	Neither a directory nor a regular file.
       */
      
      ratio.size[0] += st.st_blocks;
      ratio.size[1] += st.st_blocks;
    }

  /*
   *	Display the ratio.
   */

  if (echo || every_file || (!summarize && S_ISDIR (st.st_mode)))
    {
      /* todo: We should behave as `du' wrt POSIXLY_CORRECT
	 environment variable.  Of course POSIX doesn't specify
	 lsattr behaviour, but people who get this behaviour from
	 `du' would probably want the same behaviour here.  Same
	 goes for lsattr. */
      if (show_size)
	printf ("%-7ld %-7ld ", ratio.size[0] / 2, ratio.size[1] / 2);

      if (show_ratio)
	{
	  if (ratio.size[0] == 0)
	    printf ("  -     ");
	  else
	    printf ("%5.1f%%  ",
		    100.0 * (double) ratio.size[1] / ratio.size[0]);
	}
      else
	{
	  if (ratio.size[0] == 0)
	    printf ("  -    ");
	  else
	    printf ("%5.2f  ", (double) ratio.size[0] / ratio.size[1]);
	}

      printf ("%s\n", path);
    }

  return ratio;

error:
  {
    int n = errno;

    fprintf (stderr, "%s: %s: %s (%s)\n", prog_name, path, msg, strerror (n));
    return ratio;
  }
}


static void
help(void)
{
  printf ("Usage: %s [-abfls] [files ...]\n"
	  "Show compressed disk usage information.\n"
	  "\n"
	  "  -a         Display every file.\n"
	  "  -b         Don't display file size, just the ratio.\n"
	  "  -f         Display compression factor instead of ratio.\n"
	  "  -l         Display file size.  (This is the default.)\n"
	  "  -s         Summarize: display only a total for each argument.\n"
	  "  --help     Display this message, and exit immediately.\n"
	  "  --version  Display version information, and exit immediately.\n"
	  "\n"
	  "Report bugs to e2compr@e2compr.memalpha.cx\n",
	  prog_name);

  exit (0);
}

static void
usage (void)
{
  fprintf (stderr, 
	   "usage: %s [-abfls] [files ...]\n"
	   "Try `%s --help' for more information.\n",
	   prog_name, prog_name);

  exit (1);
}

int
main (int argc, char* argv[])
{
  char path[MAXPATHLEN];
  char c;
  int i;

  if (argc && *argv)
    prog_name = *argv;

  if ((argc >= 2) && (strcmp(argv[1], "--help") == 0))
    help();

  if ((argc == 2) && (strcmp (argv[1], "--version") == 0))
    {
      print_version_string();
      exit (0);
    }

  while ((c = getopt (argc, argv, "abfls")) != EOF)
    switch (c) {
    case 'a' : every_file  = 1;	break;
    case 'b' : show_size   = 0;	break;
    case 'f' : show_ratio  = 0;	break;
    case 'l' : show_size   = 1;	break;
    case 's' : summarize   = 1;	break;
    default  : usage ();
    }

  if (every_file && summarize)
    {
      fprintf (stderr, "%s: cannot both summarize and show all entries.\n"
	       "Try `%s --help' for more information.\n",
	       prog_name, prog_name);
      return 1;
    }

  if (optind < argc)
    for (i = optind; i < argc; i++)
      {
	strcpy (path, argv[i]);
	get_ratio (path, strlen (argv[i]), 1);
      }
  else
    {
      path[0] = '.';
      path[1] = '\0';
      get_ratio (path, 1, 1);
    }

  exit (0);
}
