/* e2bitmap.c: Show which clusters of a file are ext2-compressed.
   Copyright (C) 1998, 1999 Peter Moulder

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   The author can be contacted at <pjm@e2compr.memalpha.cx> or

      Peter Moulder
      1/36 Auburn Grove
      Hawthorn East, 3123
      Australia

 */


#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <errno.h>
#include <string.h>
#include <linux/fs.h>  /* FIGETBSZ */
#include <linux/ext2_fs.h> /* EXT2_IOC_GETFLAGS */
#include "ext2_fs_c.h"
#include "clu-calc.h"
#include "exit.h"

static char *prog_name;

static char *version_string = "e2bitmap " VERSION;

static void
usage (FILE *fp, int rc)
{
  fprintf (fp,
	   "Usage: %s <files>\n"
	   "Shows the cluster bitmap for each regular file on the command line.\n",
	   prog_name);
  exit (rc);
}


static inline unsigned
max (unsigned a, unsigned b)
{
  return ((a > b)
	  ? a
	  : b);
}

int
main (int argc, char **argv)
{
  int rc = 0;
  prog_name = *argv++;

  if (argc == 1)
    {
      (void) fprintf (stderr, "%s: no arguments.\nTry `--help'.\n",
		      prog_name);
      exit (EXIT_FAILURE);
    }

  if (strcmp (*argv, "--help") == 0)
    {
      puts (version_string);
      usage (stdout, 0);
    }

  if (strcmp (*argv, "--version") == 0)
    {
      puts (version_string);
      exit(0);
    }

  if (**argv == '-')
    {
      if (strcmp (*argv, "--") == 0)
	argv++;
      else
	{
	  (void) fprintf (stderr,
			  "%s: unrecognised option `%s'\nTry `--help'.\n",
			  prog_name, *argv);
	  exit(1);
	}
    }
	  
  for (; *argv != NULL; ++argv)
    {
      int ifd;
      unsigned long value, clu_nblocks, blk_nchars;
      int i, nclusters, ncompressed;
      struct stat st;

      if ((lstat (*argv, &st) == -1)
	  || (!S_ISREG (st.st_mode)))
	{
	  /* Ignore it. */
	  continue;
	}

      ifd = open (*argv, O_RDONLY);
      if ((ifd == -1)
	  || (ioctl (ifd, EXT2_IOC_GETFLAGS, &value) == -1))
	{
	  perror (prog_name);
	  if (ifd != -1)
	    close (ifd);
	  rc |= 1;
	  continue;
	}

      if ((value & EXT2_COMPRBLK_FL) == 0)
	{
	  printf ("%s: uncompressed\n", *argv);
	  close (ifd);
	  continue;
	}

      if (ioctl (ifd, EXT2_IOC_GETCLUSTERSIZE, &clu_nblocks) == -1)
	{
	  (void) fprintf (stderr, "%s: can't read cluster size on `%s': %s\n",
			  prog_name, *argv, strerror(errno));
	  rc |= 1;
	  (void) close (ifd);
	  continue;
	}

      if (ioctl (ifd, FIGETBSZ, &blk_nchars))
	{
	  (void) fprintf (stderr, "%s: can't read block size on `%s': %s\n",
			  prog_name, *argv, strerror(errno));
	  rc |= 1;
	  (void) close (ifd);
	  continue;
	}

#if HAVE_E2COMPR_03 && HAVE_E2COMPR_04
      nclusters = max((st.st_size
		       ? ((st.st_size - 1) / (clu_nblocks * blk_nchars)) + 1
		       : 0),
		      ext2fs_n_clusters(clu_nblocks, blk_nchars, st.st_size));
#elif HAVE_E2COMPR_03
      nclusters = (st.st_size
		   ? ((st.st_size - 1) / (clu_nblocks * blk_nchars)) + 1
		   : 0);
#else
      nclusters = ext2fs_n_clusters(clu_nblocks, blk_nchars, st.st_size);
#endif

      ncompressed = 0;
      printf ("%s: ", *argv);
      for (i = 0; i < nclusters; i++)
	{
	  value = i;
	  ioctl(ifd, EXT2_IOC_GETCLUSTERBIT, &value);
	  if (value)
	    {
	      putchar ('1');
	      ncompressed++;
	    }
	  else
	    putchar ('0');
	}
      printf (" (%.2f%%)\n", 100.0 * ncompressed / nclusters);
      close (ifd);
    }

  exit (rc);
}
