/* clear-e2c.c: Remove the e2compr flag from ext2 superblock.
   Copyright (C) 1999 Peter Moulder

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   The author can be contacted at <pjm@e2compr.memalpha.cx> or

      Peter Moulder
      1/36 Auburn Grove
      Hawthorn East, 3123
      Australia

 */


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include "exit.h"

static char const * const version_string = "clear-e2c " VERSION "\n";

#define EXT2FS_FEATURE_INCOMPAT_COMPRESSION 0x0001
#include <linux/ext2_fs.h> /* EXT2_FEATURE_INCOMPAT_COMPRESSION */
#ifdef EXT2_FEATURE_INCOMPAT_COMPRESSION
# if (EXT2FS_FEATURE_INCOMPAT_COMPRESSION) != (EXT2_FEATURE_INCOMPAT_COMPRESSION)
#  error "Kernel using non-standard value of EXT2_FEATURE_INCOMPAT_COMPRESSION"
# endif
#endif

static char const *program_name;
static int verbose_flag = 0;

static int
usage (FILE *fp, int rc)
{
  fprintf (fp, "Usage: %s [ --verbose | -v ] <ext2-filesystem> ...\n"
	   "E.g. %s /dev/hda1\n"
	   "Clears the flag saying that the filesystem uses e2compr.\n"
	   "Note: It is best to umount the filesystem in question before using this program.\n"
	   "Note2: This program does not actually decompress any files.  To do that, use:\n"
	   "  find <mountpoints> -xdev -attr +cB -print0 | xargs -0r chattr -c &&\n"
	   "   umount <mountpoints> &&\n"
	   "   %s <devices>\n",
	   program_name, program_name, program_name);
  return rc;
}

#define streq(_a, _b) (strcmp ((_a), (_b)) == 0)

static int
doit (char const *filename)
{
  int fd, rc;
  char flags; /* N.B. just the low byte.  (This saves us swabbing.) */
  unsigned char magic[2];

  fd = open (filename, O_RDWR);
  if (fd < 0)
    {
      perror (filename);
      return 1;
    }

  rc = lseek (fd, 270 * 4, SEEK_SET);
  if (rc == (off_t) (-1))
    {
      fprintf (stderr, "%s: seek %s: %s\n", program_name, filename, strerror (errno));
      (void) close (fd);
      return 1;
    }

  rc = read (fd, magic, 2);
  if (rc != 2)
    {
      fprintf (stderr, "%s: read %s: %s\n", program_name, filename, strerror (errno));
      (void) close (fd);
      return 1;
    }

  if ((magic[0] != 0x53) ||
      (magic[1] != 0xEF))
    {
      fprintf (stderr, "%s: %s: Not an ext2 filesystem: bad magic number.\n",
	       program_name, filename);
      (void) close (fd);
      return 1;
    }

  rc = lseek (fd, 280 * 4, SEEK_SET);
  if (rc == (off_t) (-1))
    {
      fprintf (stderr, "%s: seek %s: %s\n", program_name, filename, strerror (errno));
      (void) close (fd);
      return 1;
    }

  rc = read (fd, &flags, 1);
  if (rc != 1)
    {
      fprintf (stderr, "%s: read %s: %s\n", program_name, filename, strerror (errno));
      (void) close (fd);
      return 1;
    }

  if (!(flags & EXT2FS_FEATURE_INCOMPAT_COMPRESSION))
    {
      if (verbose_flag)
	printf ("e2compr flag of %s already cleared, ignoring.\n", filename);
      goto out;
    }

  rc = lseek (fd, 280 * 4, SEEK_SET);
  if (rc == (off_t) (-1))
    {
      fprintf (stderr, "%s: seek %s: %s\n", program_name, filename, strerror (errno));
      (void) close (fd);
      return 1;
    }

  flags &= ~EXT2FS_FEATURE_INCOMPAT_COMPRESSION;
  rc = write (fd, &flags, 1);
  if (rc != 1)
    {
      fprintf (stderr, "%s: write %s: %s\n", program_name, filename, strerror (errno));
      (void) close (fd);
      return 1;
    }

  if (verbose_flag)
    printf ("Successfully cleared e2compr flag on %s\n", filename);

 out:
  rc = close (fd);
  if (rc != 0)
    {
      fprintf (stderr, "%s: close %s: %s", program_name, filename, strerror (errno));
      return 1;
    }
  return 0;
}


int
main (int argc, char **argv)
{
  int rc = 0;

  program_name = argv[0];
  if (argc == 2)
    {
      if (streq (argv[1], "--help"))
	exit (usage (stdout, 0));
      if (streq (argv[1], "--version"))
	exit (puts (version_string) < 0);
    }

  if (argc < 2)
    return usage (stderr, 1);

  while (streq (argv[1], "--verbose") || streq (argv[1], "-v"))
    {
      verbose_flag++;
      ++argv;
    }

  if (argv[1][0] == '-')
    return usage (stderr, 1);

  for (++argv; *argv != NULL; ++argv)
    rc |= doit (*argv);
  exit (rc);
}
