/**************************************************************************\
*          Copyright (c) 1995 INRIA Sophia Antipolis, FRANCE.              *
*                                                                          *
* Permission to use, copy, modify, and distribute this material for any    *
* purpose and without fee is hereby granted, provided that the above       *
* copyright notice and this permission notice appear in all copies.        *
* WE MAKE NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS     *
* MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS   *
* OR IMPLIED WARRANTIES.                                                   *
\**************************************************************************/
/**************************************************************************\
*  File    :             	                			   *
*  Date    : 1997/03/03		           				   *
*  Author  : Martin May         					   *
*--------------------------------------------------------------------------*
*  Description : Second version                                            *
*                                                                          *
*                                                                          *
*                                                                          *
*--------------------------------------------------------------------------*
*        Name	        |    Date   |          Modification                *
*--------------------------------------------------------------------------*
*                       |           |                                      *
*                       |           |                                      *
\**************************************************************************/

#include <linux/string.h>
#include <linux/time.h>
#include <linux/malloc.h>
#include <linux/mm.h>
#include <linux/ip.h>
#include <linux/netdevice.h>
#include <net/tc_global.h>


#include <net/tc_types.h>
#include <net/tc_queue.h>
#include <net/tc_hash.h>

#include <net/tc_wfq.h>
#include <net/tc_wfq_var.h>

/* global variables */
static struct fs_pkt_t *fsp;
struct fsched_t *fsched = NULL;

struct fs_ses_t *fs_malloc_ses(void) 
{
    struct fs_ses_t *fss;
    if ( (fss=(struct fs_ses_t*) new_kmem_zalloc(sizeof(struct fs_ses_t),GFP_ATOMIC)) ) {
        bzero((char *)fss,sizeof(struct fs_ses_t));
    }
    return(fss);
}
void fs_free_ses(struct fs_ses_t *fss)
{
   kmem_free(fss,sizeof(struct fs_ses_t) );
}

long fs_gettime()
{
   struct timeval tv;
   
   do_gettimeofday(&tv);
   return (tv.tv_sec);
}

static void fq_reset(struct fs_ctx_t *fsc)
{
    int i;
    struct fs_ses_t *fss;

    for(i=fsc->headq.count,fss=(struct fs_ses_t *)fsc->headq.hd;
        i;
        i--,fss=fss->next)
        fss->fss_F_1=0;
    fsc->fsc_v=0;
}
      
static void fq_housek(struct fs_ctx_t *fsc)
{
    int i;
    struct fs_ses_t *fss;
    
    for(i=fsc->headq.count;i;i--) {
        fss=(struct fs_ses_t *)fsc->headq.hd->prev;
        if ((fsched->fs_ctime-fss->fss_last)>=fsc->fsc_oldp) {
            dequeuep(&fsc->headq,(struct elemq_t *)fss);
            hash_delete_tc(&fsc->rooth,fss);
            fs_free_ses(fss);
        } else
            break;
    }
    if (fsc->fsc_v>=MAX_FS_TAG)
        fq_reset(fsc);
}

void tc_init_wfq(struct device *dev)
{
   struct fs_ctx_t *fsc;
   struct fs_ses_t *fss;
   
   /* init the global fsched structure */
   fsched = (struct fsched_t *) new_kmem_zalloc(sizeof(struct fsched_t), GFP_KERNEL);
   if (fsched == NULL)
      printk(KERN_WARNING "Not Enough Memory for fsched on interface %s\n",dev->name);
      
   bzero((char *)fsched, sizeof(struct fsched_t));
   fsched->fs_max_ctx = MAX_FS_CTX;
   fsched->fs_max_ses = MAX_FS_SES;
   fsched->fs_ctime = fs_gettime();
   
   /* init the context for the device */
   fsc = (struct fs_ctx_t *) new_kmem_zalloc(sizeof(struct fs_ctx_t), 11);
   bzero((char *)fsc, sizeof(struct fs_ctx_t));
   if (fsc == NULL)
      printk(KERN_WARNING "Not Enough Memory for FQ Context on interface %s\n",dev->name);
   fsc->fsc_max_ses=MAX_FS_SES;
   fsc->fsc_cycle=FS_CYCLE;
   fsc->fsc_oldp=FS_OLDP;
   fsc->fsc_up_mask=FS_UP_MASK;
   
   fsc->fsc_last=fsched->fs_ctime;
   fsc->fsc_flag=0;
   fsc->fsc_max_dump=99;
   fsc->fsc_usr0=0;
   fsc->fsc_usr1=0;

   /* init counters */
   fsc->ip = 0;
   fsc->ip6 = 0;
   fsc->arp = 0;
   fsc->others = 0;
   
   fsc->fs_head = NULL;
   fsc->fs_tail = NULL;
   
   /* allocate memory for the FIFO session */
   fss = fs_malloc_ses();
   if (fss == NULL)
      printk(KERN_WARNING "Not Enough Memory for FQ FIFO Session on interface %s\n",dev->name);
   fss->af_family=dev->family;
   fss->proto=0; /* to change XXX */
   fss->fss_last=fsched->fs_ctime;
   fss->key0 = 1;
   fss->key1 = 1;
   fss->id=0;
   fss->fss = fss;

   enqueuef(&fsc->headq,(struct elemq_t*)fss);
   /*   hash_insert_tc(&fsc->fifoses , fss);*/

   fsc->fifoses = fss;
   
   fsc->fsc_dev = dev;
   dev->if_ptr = fsc;
#ifdef WFQ_DEBUG
   printk(KERN_WARNING "TC: wfq_init  %s\n",dev->name);
#endif
}

struct fs_ses_t *fq_lookup(struct fs_ctx_t *fsc,struct sk_buff *m)
{
    u_int32 key0,key1;
    char            *data0, *data1;
    struct fs_ses_t *fss;
    struct iphdr    *ip;
    u_int32         fs_stat=0;
    char proto=0;

    switch (ntohs(m->protocol)){
    case ETH_P_IP:{
        ip = (struct iphdr *)m->nh.iph;
        data0=(char*)&ip->saddr;/* src & dst IP addresses */
        data1=((char*)((u_int32*)ip+ip->ihl))+2; /* src & dst ports */
        proto=ip->protocol;
        if (proto==IPPROTO_UDP || proto==IPPROTO_TCP) {
            key0=fss_key(data0,data1);
            key1=ntohs(*(u_int16*)data1);
        } else {
            key1=0;
            key0=fss_key(data0,(char *)&key1);

        }
        break;
    }
    default:{
        /* XXX We should never come here */
        key0=0x710a3f8b;
        key1=*(u_int32*)&data0[4];
    }
    }

    
    if ( (fss=(struct fs_ses_t *)hash_lookup(&fsc->rooth,key0,key1,&fs_stat)) ) {
       /* Session was looked up */
       /*        printk(KERN_WARNING " TC: nach fss hash lookup \n");*/
        fss=(struct fs_ses_t *)((char *)fss - sizeof(void *)*2);
    } else {
#ifdef WFQ_DEBUG
       printk(KERN_WARNING " TC: new session lookup \n");
#endif
       /* Session doesn't exist yet, create a new one */
       if (fsc->headq.count<fsc->fsc_max_ses && (fss=fs_malloc_ses())) {
	  /* Allocate a new session */
       } else {
	  /* Free the LRU session */
	  fss=(struct fs_ses_t*)dequeuel(&fsc->headq);
	  hash_delete_tc(&fsc->rooth,fss);
	  bzero((char*)fss,sizeof(struct fs_ses_t));
	  fsc->fsc_free_lru++;
       }
        fsched->fs_ctime=fs_gettime();
       /* Init the new fss */
       fss->id=fsc->fsc_ses_id++;
       fss->key0=key0;
       fss->key1=key1;
       fss->af_family=m->dev->family;
       fss->proto=m->protocol;
       fss->fss_last=fsched->fs_ctime;
       bcopy(data0,fss->data,8);
       if (key1)
	  bcopy(data1,&fss->data[8],2);
       else
	  bzero(&fss->data[8],2);
       
       fss->fss = fss;
       
       enqueuef(&fsc->headq,(struct elemq_t*)fss);
       hash_insert_tc(&fsc->rooth,fss);
#ifdef WFQ_DEBUG
       if (fs_tst(DEBUG_LOOKUP))
	  printk(KERN_WARNING  "lookup new: fss:%u (%08x)\n",fss->id,(int)fss);
#endif
    }

#ifdef WFQ_DEBUG
    if (fs_tst(DEBUG_LOOKUP))
        printk(KERN_WARNING "lookup: fss:%u AF:%u\n",fss->id,(u_int)m->dev->family);
#endif
    return(fss);
}

void fq_enqueue(struct fs_ctx_t *fsc, 
		struct fs_ses_t *fss,
		struct sk_buff *skb)
{
   int    f;
   struct fs_pkt_t *f0, *f1;
   struct fs_pkt_t *fsp;

   fsp = (struct fs_pkt_t *) new_kmem_zalloc(sizeof(struct fs_pkt_t),GFP_ATOMIC);

   /* from fq_tag */   
   fsp->fsp_magic=FS_MAGIC;
   fsp->fsp_fss=fss;
   fsp->fsp_F_1=fss->fss_F_1;
   /*   printk(KERN_WARNING " TC: enqueue2 \n");*/

   fsp->skbp = skb; /* Pointer to the Packet in the queue */
   
   /* Using always K=1 is fairer, but using a value of K, allows to
    * bursts arrived just before another given session creation,
    * to be served at a higher rate. */
   if (fss->fss_F_1 > fsc->fsc_v)
      fsp->fsp_F=fss->fss_F_1=skb->len + fss->fss_F_1;
   else
      fsp->fsp_F=fss->fss_F_1=skb->len + fsc->fsc_v;

   f = fsp->fsp_F;

   /* end fq_tag - start fq_enqueue */

   for (f0 = NULL, f1 = fsc->fs_head;
	f1 && f >= f1->fsp_F;
	f0 = f1, f1 = f1->next);

   fsp->next = f1;
   fsp->prev = f0;
   if (f0)                 /* fsp < f1 && fsp is not the first packet */
      f0->next = fsp;
   else                    /* fsp < f1 && fsp is the fist packet */
      fsc->fs_head = fsp;
   if (f1)                 /* fsp < f1 && fsp is not the last packet */
      f1->prev = fsp; 
   else                    /* fsp is the last packet */
      fsc->fs_tail = fsp;
   
   /*   fsc->fsc_dev->buffs->qlen++;*/
   fsc->fsc_npkts++;
/*   skb->list->qlen++;*/
   fsched->fs_ctime=fs_gettime();

   if ( !(++fss->fss_n & fsc->fsc_up_mask) ) {
      queue_up(&fsc->headq,(struct elemq_t *)fss); /* Ups the session */
      fss->fss_last=fsched->fs_ctime; /* Update last use */
   }
}

struct sk_buff *fq_dequeue(struct fs_ctx_t *fsc)
{
   struct fs_pkt_t *fsp;
   struct sk_buff *skb;

   if ((fsp=fsc->fs_head) != NULL){
      if (fsp->next == NULL){
	 fsc->fs_tail = NULL;
	 fsc->fs_head = NULL;
      }
      else {
	 fsc->fs_head = fsp->next;
      /* fsp->next->prev = NULL;*/
      }
      fsp->next = NULL;
      fsc->fsc_npkts--;
      
      if (fsp->fsp_magic == FS_MAGIC){
	 /* update v(t) */
	 fsc->fsc_v = fsp->fsp_F;
	 
	 /* update per session counters */
	 fsp->fsp_fss->fss_npkts++;
#ifdef WFQ_DEBUG
	 if ((fsp->fsp_fss->fss_npkts % 1000)== 0)
	    printk(KERN_WARNING " TC: pkts %d\n",fsp->fsp_fss->fss_npkts);
#endif
	 fsp->fsp_fss->fss_nbytes += fsp->skbp->len;
	 
      } else
	 printk(KERN_WARNING " TC: fq_dequeue: no magic found!\n");
      
      fsched->fs_ctime = fs_gettime();
      if (fsc->fsc_npkts == 0){
	 if ((fsched->fs_ctime - fsc->fsc_last) >= fsc->fsc_cycle) {
	    fq_housek(fsc);
	    fsc->fsc_last=fsched->fs_ctime;
	 }
      }

      skb = fsp->skbp;
      kfree(fsp);

      /* unlink the skb in the skb list */
      if (skb->list){
	 if (skb->next && skb->prev){
	    skb->next->prev = skb->prev;
	    skb->prev->next = skb->next;
	 }
	 else {
	    if (!skb->next && skb->prev)
	       skb->prev->next = NULL;
	    if (skb->next && !skb->prev)
	       skb->list->next = skb->next;
	    if (!skb->next && !skb->prev)
	       skb->list->next = NULL;
	 }
      }
      /*      skb->list->qlen--;*/
      return (skb);
      
   } else {
      printk(KERN_WARNING " TC:Head = NULL\n");
      return (NULL);
   }
}


void fq_ask_state(struct fs_ctx_t *fsc)
{
   printk(KERN_WARNING "WFQ-INFO: for interface %s \n",fsc->fsc_dev->name);
   printk(KERN_WARNING "WFQ-INFO: inqueue= %d\n", fsc->fsc_npkts);
   printk(KERN_WARNING "WFQ-INFO: queued= %d\n",fsc->fifoses->fss_npkts);
   printk(KERN_WARNING "WFQ-INFO: dropped= %d\n",fsc->fifoses->fss_drop);
   printk(KERN_WARNING "WFQ-INFO: ip= %d ip6= %d arp= %d others= %d \n",
	  fsc->ip, fsc->ip6, fsc->arp, fsc->others);
}





