/**************************************************************************\
*          Copyright (c) 1995 INRIA Sophia Antipolis, FRANCE.              *
*                                                                          *
* Permission to use, copy, modify, and distribute this material for any    *
* purpose and without fee is hereby granted, provided that the above       *
* copyright notice and this permission notice appear in all copies.        *
* WE MAKE NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS     *
* MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS   *
* OR IMPLIED WARRANTIES.                                                   *
\**************************************************************************/
/**************************************************************************\
*  File    :             	                			   *
*  Date    : 1997/01/27		           				   *
*  Author  : Martin May  						   *
*--------------------------------------------------------------------------*
*  Description : generic Traffic Control kernel for Linux. Catch the ioctl *
*  passed from the user space and make the modifications for the           *
*  scheduler.                                                              *
*                                                                          *
*--------------------------------------------------------------------------*
*        Name	        |    Date   |          Modification                *
*--------------------------------------------------------------------------*
*  tc_main.c            |           |                                      *
*                       |           |                                      *
\**************************************************************************/


#include <sys/param.h>
#include <sys/socket.h>
#include <sys/errno.h>
#include <sys/time.h>
#include <linux/time.h>
#include <linux/skbuff.h>
#include <linux/sockios.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>

#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>

#include <syslog.h>

#include <net/tc_global.h>
#include <net/tc_hash.h>
#include <net/tc_1bit.h>

#ifndef TRUE
#define TRUE	1
#endif

#ifndef FALSE
#define FALSE	0
#endif


int 
tc_ioctl_cbq(struct tcreq *tc)
{
   int error = 0;
   struct device *dev;

   switch (tc->iq_function) {
   case IF_ENABLE:{
      dev = dev_get(tc->iq_name);
      tc_init_cbq(dev);
      dev->if_tc_enabled = tc->iq_flags;
#ifdef CBQ_DEBUG1
      printk(KERN_WARNING "CBQ: Interface %s %d  ENABLED CBQ\n",dev->name, dev->if_tc_enabled );
#endif
      break;
   }
   case IF_DISABLE:{
      dev = dev_get(tc->iq_name);
      dev->if_tc_enabled = 0;
#ifdef CBQ_DEBUG1
      printk(KERN_WARNING "CBQ: Interface %s %d  DISABLED CBQ\n",dev->name, dev->if_tc_enabled );
#endif
      break;
   }
   case IF_ADDFILT:
#ifdef CBQ_DEBUG1
      printk(KERN_WARNING "CBQ: Add Filter\n");
#endif
      tc_add_filter(tc);
      break;
   case IF_DELFILT:
#ifdef CBQ_DEBUG1
      printk(KERN_WARNING "CBQ: Delete Filter\n");
#endif
      tc_del_filter(tc);
      break;
   case IF_ADDFLOW:
#ifdef CBQ_DEBUG1
      printk(KERN_WARNING "CBQ: Add Flow\n");
#endif
      tc_add_flow(tc);
      break;
   case IF_MODFLOW:
#ifdef CBQ_DEBUG1
      printk(KERN_WARNING "CBQ: Modify Flow\n");
#endif
      tc_mod_flow(tc);
      break;
   case IF_DELFLOW:
#ifdef CBQ_DEBUG1
      printk(KERN_WARNING "CBQ: Delete Flow (doing nothing)\n");
#endif
      break;
   case IF_ADDSHARE:{
#ifdef CBQ_DEBUG1
      printk(KERN_WARNING "CBQ: Add Share\n");
#endif
      tc_add_share(tc);
      break;
   }
   case IF_ASKSTATE:{
#ifdef CBQ_DEBUG1
      printk(KERN_WARNING "CBQ: Ask State\n");
#endif
      dev = dev_get(tc->iq_name);
      tc_ask_state(dev->if_ptr);
      break;
   }
   case IF_RESET:
      printk(KERN_WARNING "CBQ: RESET (doing nothing)\n\n");
      break;
   default:
      error = -EINVAL;
      printk(KERN_WARNING "CBQ: Unknown Value\n");
      break;
   }
   return error;
}
	

int 
tc_ioctl_wfq(struct tcreq *tc)
{
   int error = 0;
   struct device *dev;
   
   switch (tc->iq_function) {
   case IF_ENABLE:{
      dev = dev_get(tc->iq_name);
      tc_init_wfq(dev);
      dev->if_tc_enabled = tc->iq_flags;
#ifdef WFQ_DEBUG
      printk(KERN_WARNING "WFQ: Interface %s %d  ENABLED WFQ \n",dev->name, dev->if_tc_enabled );
#endif
      break;
   }
   case IF_DISABLE:{
      dev = dev_get(tc->iq_name);
      dev->if_tc_enabled = 0;
#ifdef WFQ_DEBUG
      printk(KERN_WARNING "WFQ: Interface %s %d  DISABLED WFQ\n",dev->name, dev->if_tc_enabled );
#endif
      break;
   }
   case IF_ASKSTATE:{
#ifdef WFQ_DEBUG
      printk(KERN_WARNING "WFQ: Ask State\n");
#endif
      dev = dev_get(tc->iq_name);
      fq_ask_state(dev->if_ptr);
      break;
   }
   default:
      error = -EINVAL;
      printk(KERN_WARNING "Unknown Value\n");
      break;
   }
   return error;
}

int 
tc_ioctl_test(struct tcreq *tc)
{
   int error = 0;
   struct device *dev;
   
   switch (tc->iq_function) {
   case IF_ENABLE:{
      dev = dev_get(tc->iq_name);
      dev->if_tc_enabled = tc->iq_flags;
      printk(KERN_WARNING "WFQ: Interface %s %d  ENABLED TEST \n",dev->name, dev->if_tc_enabled );
      break;
   }
   case IF_DISABLE:{
      dev = dev_get(tc->iq_name);
      dev->if_tc_enabled = 0;
      printk(KERN_WARNING "WFQ: Interface %s %d  DISABLED TEST\n",dev->name, dev->if_tc_enabled );
      break;
   }
   default:
      error = -EINVAL;
      printk(KERN_WARNING "WFQ: Unknown Value\n");
      break;
   }
   return error;
}	 

void
onebit_ask_state(struct device *dev)
{
   struct tc_1bit_info *tc1p;
   tc1p = (struct tc_1bit_info *)dev->if_ptr;
   
   printk(KERN_WARNING "1BIT-INFO: Queueing Information Queue Lenght:\n");
   printk(KERN_WARNING "1BIT-INFO: Lenght high=%d  Lengh low=%d\n",skb_queue_len(dev->buffs),skb_queue_len(dev->buffs+1));
   
   printk(KERN_WARNING "1BIT-INFO: queued_high=%d  queued_low=%d \n",
	  tc1p->queued_high,tc1p->queued_low);
   printk(KERN_WARNING "1BIT-INFO: dropped_high=%d dropped_low=%d \n",
	  tc1p->dropped_high,tc1p->dropped_low);
   printk(KERN_WARNING "1BIT-INFO: npkts_high=%d npkts_low=%d \n",
	  tc1p->npkts_high,tc1p->npkts_low);
   printk(KERN_WARNING "1BIT-INFO: nbytes_high=%d nbytes_low=%d \n",
	  tc1p->nbytes_high,tc1p->nbytes_low);
}   

int 
tc_ioctl_1bit(struct tcreq *tc)
{
   int error = 0;
   struct device *dev;
   struct tc_1bit_info *tc1p;
   
   
   switch (tc->iq_function) {
   case IF_ENABLE:{
      dev = dev_get(tc->iq_name);
      dev->if_tc_enabled = tc->iq_flags;
      tc1p = (struct tc_1bit_info *)new_kmem_zalloc(sizeof(struct tc_1bit_info),GFP_ATOMIC);
      if (tc1p == NULL){
	 printk(KERN_WARNING "1BIT tc_init: malloc said NO %d \n",sizeof(struct tc_1bit_info));
      }
      tc1p->queued_high= 0;
      tc1p->queued_low= 0;
      tc1p->dropped_high= 0;
      tc1p->dropped_low= 0;
      tc1p->npkts_high= 0;
      tc1p->npkts_low= 0;
      tc1p->nbytes_high= 0;
      tc1p->nbytes_low= 0;
	 
      dev->if_ptr = tc1p;
      
      printk(KERN_WARNING "1BIT: Interface %s %d  ENABLED 1BIT \n",dev->name, dev->if_tc_enabled );
      break;
   }
   case IF_DISABLE:{
      dev = dev_get(tc->iq_name);
      dev->if_tc_enabled = 0;
      printk(KERN_WARNING "1BIT: Interface %s %d  DISABLED 1BIT\n",dev->name, dev->if_tc_enabled );
      break;
   }
   case IF_ASKSTATE:{
      dev = dev_get(tc->iq_name);
      onebit_ask_state(dev);
      break;
   }
   default:
      error = -EINVAL;
      printk(KERN_WARNING "1BIT: Unknown Value\n");
      break;
   }
   return error;
}	 

int 
tc_ioctl_2bit(struct tcreq *tc)
{
   int error = 0;
   struct device *dev;
   struct tc_1bit_info *tc1p;
   
   
   switch (tc->iq_function) {
   case IF_ENABLE:{
      dev = dev_get(tc->iq_name);
      dev->if_tc_enabled = tc->iq_flags;
      tc1p = (struct tc_1bit_info *)new_kmem_zalloc(sizeof(struct tc_1bit_info),GFP_ATOMIC);
      if (tc1p == NULL){
	 printk(KERN_WARNING "1BIT tc_init: malloc said NO %d \n",sizeof(struct tc_1bit_info));
      }
      tc1p->queued_high= 0;
      tc1p->queued_low= 0;
      tc1p->dropped_high= 0;
      tc1p->dropped_low= 0;
      tc1p->npkts_high= 0;
      tc1p->npkts_low= 0;
      tc1p->nbytes_high= 0;
      tc1p->nbytes_low= 0;
	 
      dev->if_ptr = tc1p;
      
      printk(KERN_WARNING "1BIT: Interface %s %d  ENABLED 1BIT \n",dev->name, dev->if_tc_enabled );
      break;
   }
   case IF_DISABLE:{
      dev = dev_get(tc->iq_name);
      dev->if_tc_enabled = 0;
      printk(KERN_WARNING "1BIT: Interface %s %d  DISABLED 1BIT\n",dev->name, dev->if_tc_enabled );
      break;
   }
   case IF_ASKSTATE:{
      dev = dev_get(tc->iq_name);
      onebit_ask_state(dev);
      break;
   }
   default:
      error = -EINVAL;
      printk(KERN_WARNING "1BIT: Unknown Value\n");
      break;
   }
   return error;
}	 
