/**************************************************************************\
*          Copyright (c) 1995 INRIA Sophia Antipolis, FRANCE.              *
*                                                                          *
* Permission to use, copy, modify, and distribute this material for any    *
* purpose and without fee is hereby granted, provided that the above       *
* copyright notice and this permission notice appear in all copies.        *
* WE MAKE NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS     *
* MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS   *
* OR IMPLIED WARRANTIES.                                                   *
\**************************************************************************/
/**************************************************************************\
*  File    :             	                			   *
*  Date    : 1997/03/03		           				   *
*  Author  : Martin May         					   *
*--------------------------------------------------------------------------*
*  Description : Second version                                            *
*                                                                          *
*                                                                          *
*                                                                          *
*--------------------------------------------------------------------------*
*        Name	        |    Date   |          Modification                *
*--------------------------------------------------------------------------*
* Laurent Pautet        | 1998/02/04| Adding IPv6 management for T.C.      *
*                       |           |                                      *
\**************************************************************************/

#include <sys/param.h>
#include <sys/socket.h>
#include <sys/errno.h>
#include <sys/time.h>
#include <linux/time.h>
#include <linux/sockios.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/skbuff.h>


#include <net/tc_global.h>
#include <net/tc_hash.h>
#include <net/tc_cbq.h>

#ifndef TRUE
#define TRUE	1
#endif

#ifndef FALSE
#define FALSE	0
#endif

extern void rmc_delete_class(struct tc_ifdat *ifdat, struct tc_class *cl);

double pow(double x, double y)
{
   double z ;
   double r = 0;
   for ( z = 1 ; z <= y ; z ++){
      r =  r * x;
   }
   return (r);
}

void tc_init_cbq(struct device *dev)
{
   struct tc_ifdat *tcdp;
   int i;
   
   /* init the interface information structure tc_ifdat */
   tcdp = (struct tc_ifdat *)new_kmem_zalloc(sizeof(struct tc_ifdat),GFP_ATOMIC);
   if (tcdp == NULL){
      printk(KERN_WARNING "CBQ tc_init: malloc said NO %d \n",sizeof(struct tc_ifdat));
   }
   
   tcdp->rooth.size = 0;
   tcdp->rooth.count = 0;
   
   tcdp->rootclass = NULL;
   tcdp->defaultclass = NULL;
   tcdp->csum = 0;
   
   /* init counters */
   tcdp->queued7 = 0;
   tcdp->queued6 = 0;
   tcdp->queued5 = 0;
   tcdp->queued4 = 0;
   tcdp->queued3 = 0;
   tcdp->queued2 = 0;
   tcdp->queued1 = 0;
   tcdp->queued0 = 0;
   tcdp->drops7  = 0;
   tcdp->drops6  = 0;
   tcdp->drops5  = 0;
   tcdp->drops4  = 0;
   tcdp->drops3  = 0;
   tcdp->drops2  = 0;
   tcdp->drops1  = 0;
   tcdp->drops0  = 0;
   
   for (i = 0 ; i < (RM_MAXPRIO+1) ; i++)
       {
	 tcdp->classlist[i] = NULL;
	 tcdp->classes[i]   = NULL;
       }

   /* call the init function of the CBQ scheduler */
   rmc_init (dev, tcdp, 800, dev->start, dev->tx_queue_len-5, 2);
   
   tcdp->class = tcdp->defaultclass;
   
   tcdp->class_table = (struct tc_class **) new_kmem_zalloc
      (sizeof(struct tc_class *) * TC_MAX_CLASS,GFP_ATOMIC);
   for (i=0 ; i<TC_MAX_CLASS ; i++)
      tcdp->class_table[i] = NULL;
   
   if (tcdp->class_table == NULL)
      printk(KERN_WARNING "CBQ tc_init: malloc said NO to class_tablee \n");
   
   tcdp->filter_handels = 0;
   tcdp->bandwidth = 10000000;
   
   /* attach the state to the interface info structure */
   tcdp->tc_share = NULL;
   
   /* attach the interface info to the device structure */
   dev->if_ptr = tcdp;
}


/*
 * The function return a pointer to the class 
 * corresponding to the classhandle
 */

struct tc_class *class_h_2_class_p(struct tc_ifdat  *ifdp, 
				   u_long handle, int *error)
{
   
   switch (handle) {
   case NULL_CLASS_HANDLE:
      return ((struct tc_class *)NULL);
      
   case ROOT_CLASS_HANDLE:
      if (!ifdp->rootclass){
	 printk(KERN_WARNING "CBQ: no ROOT_CLASS_HANDLE \n");
	 *error = EINVAL;
      }
      return (ifdp->rootclass);
      
   case DEFAULT_CLASS_HANDLE:
      if (!ifdp->defaultclass){
	 printk(KERN_WARNING "CBQ: no DEFAULT_CLASS_HANDLE \n");
	 *error = EINVAL;
      }
      return (ifdp->defaultclass);

   default:
      if (handle >= TC_MAX_CLASS){
	 printk(KERN_WARNING "CBQ: CLASS_HANDLE > MAX_CLASS\n");
	 *error = ENXIO;
      }
      else
	 return (ifdp->class_table[handle]);
   }
   /* unreached */
   return (NULL);
}


/*
 * The function finds the lowest free
 * handle number in the class tabel for a flow
 */


u_long
tc_alloc_class_handle(tcdp, error)
   struct tc_ifdat  *tcdp;
   int              *error;
{
   int     i;
   u_long  handle = 0;
   
   for (i = 0; i < TC_MAX_CLASS; i++) {
      if (tcdp->class_table[i] == NULL)
	 break;
   }
   
   if (i >= TC_MAX_CLASS)
      *error = ENOSPC;

   handle = (u_long)i;

   return (handle);
}



int 
tc_add_share(struct tcreq *tc)
{
   int              error = 0;
   struct tc_class  *cl, *parent, *borrow;
   struct device    *dev;
   struct tc_ifdat  *ifdp;
   u_int32          prio, phandle;
   u_int32          maxidle, offtime, nspbyte;
   u_int            handle;
   struct tc_share  *tcshare;
   
#ifdef CBQ_DEBUG2
   printk(KERN_WARNING "TC add share: Start\n");
#endif
   dev = dev_get(tc->iq_name);
   ifdp = dev->if_ptr;
   if (ifdp == NULL) {
      printk(KERN_WARNING "TC add share: Not initialized\n");
      return (3);
   }
   
   tcshare = (struct tc_share *)new_kmem_zalloc
      (sizeof(struct tc_share),GFP_ATOMIC);
   
   if (tcshare == NULL)
      printk(KERN_WARNING "TC add share: No memory\n");
   
   prio = tc->iq_clval.tc_prio;
   
   phandle = ROOT_CLASS_HANDLE;

   /* get class pointers */
   parent = class_h_2_class_p(ifdp, phandle, &error);
   if (error) {
      printk(KERN_WARNING "TC add share: class_h_2_class_p \n");
      return (error);
   }

   borrow = parent;
   
   maxidle = 5;
   offtime = 3;
   nspbyte = ((NS_PER_SEC ) / tc->iq_clval.tc_bandwidth) *8 ;

   printk(KERN_WARNING "CBQ: add share: prio = %d\n",prio);
   printk(KERN_WARNING "CBQ: add share: bw=%d nspb=%d \n",tc->iq_clval.tc_bandwidth , nspbyte); 
   printk(KERN_WARNING "CBQ: add share: borrow  = %p\n",borrow); 
   
   cl = rmc_newclass(prio, ifdp, nspbyte, 
		     rmc_delay_action, TC_MAX_QUEUE_LEN,
		     parent, borrow,
		     maxidle, offtime);
   
   
   ifdp->nr_of_classes++;
   handle = tc_alloc_class_handle(ifdp, &error);
   if (error){
      printk(KERN_WARNING "CBQ: alloc_class_handle: error \n");
      return (error);
   }
   
   ifdp->class_table[handle] = cl;
   tc->iq_handle = handle;

   tcshare->bandwidth = tc->iq_clval.tc_bandwidth;
   tcshare->allocated = 0;
   tcshare->prio      = prio;
   tcshare->handle    = handle;
   tcshare->flinfo    = NULL;
   tcshare->maxdelay  = 200;
   tcshare->av_pkt_size = 1000;

   if (ifdp->tc_share == NULL)
      ifdp->tc_share = tcshare;
   else {
      tcshare->next = ifdp->tc_share;
      ifdp->tc_share = tcshare;
   }
   
   printk(KERN_WARNING "CBQ: Handle for the new class %d \n",handle);

   return (error);
}

void tc_ask_state(struct tc_ifdat  *tcdp)
{
   printk(KERN_WARNING "CBQ-INFO: defqueue= %d act queue=%d\n",
	  tcdp->defaultclass->qcnt, tcdp->class->qcnt );
   printk(KERN_WARNING "CBQ:Q 7=%d 6=%d 5=%d 4=%d 3=%d 2=%d 1=%d o= %d \n",
	  tcdp->queued7, tcdp->queued6, tcdp->queued5, 
	  tcdp->queued4, tcdp->queued3, tcdp->queued2, 
	  tcdp->queued1, tcdp->queued0);
   printk(KERN_WARNING "CBQ:D 7=%d 6=%d 5=%d 4=%d 3=%d 2=%d 1=%d o= %d \n",
	  tcdp->drops7, tcdp->drops6, tcdp->drops5,
	  tcdp->drops4, tcdp->drops3, tcdp->drops2,
	  tcdp->drops1, tcdp->drops0);
   /*   printk(KERN_WARNING "CBQ-INFO: QL=%d \n", tcdp->dev->tx_queue_len);*/
}


/*
 * This function creates a new node in 
 * the class tree. The function is called by the
 * ioctl function and passes the information for
 * the new entry.
 *
 * Return 0 if succeded
 *        1 in case of error
 */



int tc_add_flow(struct tcreq *tc)
{
   int              error = 0;
   struct tc_class  *cl, *parent, *borrow;
   struct device    *dev;
   struct tc_ifdat  *ifdp;
   u_int32           prio;
   u_int32           maxidle, offtime;
   int               handle;
   u_int32           req_bandwidth, req_nspb;
   struct tc_share  *tcshare;
   double            fraction, gain,gb, gb1;
   u_int32           nspb, burst;
   u_int32           packetst; /* packet sending time */
   u_int32           maxq, maxdelay;
   struct tc_flow_info *flp;
   
   dev = dev_get(tc->iq_name);
   ifdp = dev->if_ptr;
   
   if (ifdp == NULL) {
      printk(KERN_WARNING "TC add flow: Not initialized\n");
      return (3);
   }
   
   if (ifdp->tc_share == NULL){
      printk(KERN_WARNING "CBQ Add Flow: No Share Class created\n");
      tcshare = NULL;
      return (error);
   }
   else
      tcshare = ifdp->tc_share;
   
   
   nspb = (NS_PER_SEC/ifdp->bandwidth) *8;
   
   switch (tc->iq_fs.fs_tos){
   case TOS_GUARANTEED:{
      maxdelay = tcshare->maxdelay;
      maxq     = TC_MAX_QUEUE_LEN;
      req_bandwidth = tc->iq_fs.fs_tspec.ts_tb.tbf_rate + tc->iq_fs.fs_tspec.ts_tb.tbf_bkt;
      if ((tcshare->allocated +req_bandwidth) > tcshare->bandwidth){
	 printk(KERN_WARNING "CBQ Add Flow: AC said NO for GS req: %d \n",req_bandwidth);
	 return (error);
      }
      break;
   }
   case TOS_CLOAD:{
      maxdelay = tc->iq_fs.fs_rspec.rs_p.p_dly;
      maxq     = ( (u_int32) maxdelay * NS_PER_MS / (nspb * tcshare->av_pkt_size) );
      req_bandwidth = tc->iq_fs.fs_tspec.ts_tb.tbf_rate ;
      if ((tcshare->allocated + req_bandwidth) > tcshare->bandwidth){
	 printk(KERN_WARNING "CBQ Add Flow: AC said NO for CLOAD req: %d \n",req_bandwidth);
	 return (error);
      } 
      break;
   default:
      printk(KERN_WARNING"CBQ: ADD_FLOW: hopefully unreached\n");
      return (error);
   }
   }
#ifdef CBQ_DEBUG2
   printk(KERN_WARNING "CBQ Add Flow: maxdelay=%d  maxq=%d\n",maxdelay,maxq);
#endif   
   
   /* calculate values for the new flow */
   prio = tcshare->prio;
   burst = tc->iq_fs.fs_tspec.ts_tb.tbf_bkt;
   fraction = ((double) req_bandwidth / (double) tcshare->bandwidth);
#ifdef CBQ_DEBUG1
   printk(KERN_WARNING"CBQ: ADD_FLOW:  bw = %d\n",req_bandwidth);
#endif 
#ifdef CBQ_DEBUG2
   printk(KERN_WARNING"CBQ: ADD_FLOW:  fraction3 = %d\n", (int) ((1-fraction)/fraction));
#endif   
   gain = (1.0 - (1.0 / pow(2, RM_FILTER_GAIN)));
   /*
     printk(KERN_WARNING"CBQ: ADD_FLOW:  gain =%d  %d\n",  gain,pow(2, RM_FILTER_GAIN) );
     printk(KERN_WARNING"CBQ: ADD_FLOW:  gain =%x  %x\n",  gain,pow(2, RM_FILTER_GAIN) );
     */
   packetst = tcshare->av_pkt_size * nspb * (int)((1-fraction)/fraction);
   printk(KERN_WARNING"CBQ: ADD_FLOW:  packetst =%d\n",packetst);
   gb = pow(gain, burst);
   gb1 = pow(gain,burst-1);
   maxidle = packetst * (int)(1 - gb) / gb;
   offtime = packetst * (int)( 1 + 1/(1-gain) * (1 - gb1)/gb1 );
   printk(KERN_WARNING"CBQ: ADD_FLOW: maxidle %d offtime %d \n",maxidle, offtime);
   
   if (maxq  < 10)
      maxq = 10;
   req_nspb = ( NS_PER_SEC/  req_bandwidth) *8;
   
   if (maxq > TC_MAX_QUEUE_LEN)
      maxq = TC_MAX_QUEUE_LEN;
   
#ifdef CBQ_DEBUG2
   printk(KERN_WARNING"CBQ: ADD_FLOW: maxq= %d maxidle= %d  offtime= %d \n",maxq,maxidle,offtime);
#endif   
   
   parent = class_h_2_class_p(ifdp, tcshare->handle, &error);
   borrow = parent;
   if (error){
      printk(KERN_WARNING"CBQ: ADD_FLOW:class_h_2... failed \n");
      return (error);
   }
      
   cl = rmc_newclass(tcshare->prio, ifdp,
		     (u_int) fabs(req_nspb), rmc_delay_action,
		     (u_int) fabs(maxq), parent, borrow,
		     (u_int) fabs(maxidle), (u_int) fabs(offtime));
   
   ifdp->nr_of_flows++;
   handle = tc_alloc_class_handle(ifdp, &error);
   if (error){
      printk(KERN_WARNING"CBQ: ADD_FLOW:alloc_class_han... failed\n");
      return (error);
   }
   
   ifdp->class_table[handle] = cl;
   ifdp->nr_of_classes++;
   
   flp = (struct tc_flow_info *) new_kmem_zalloc
      (sizeof(struct tc_flow_info),GFP_ATOMIC);
   if (flp == NULL) {
      printk(KERN_WARNING"CBQ: ADD_FLOW: Malloc failed \n");
      return (error);
   }
   flp->tbf_bkt = tc->iq_fs.fs_tspec.ts_tb.tbf_bkt;
   flp->tbf_rate = tc->iq_fs.fs_tspec.ts_tb.tbf_rate;
   flp->tbf_buf = tc->iq_fs.fs_tspec.ts_tb.tbf_buf;
   flp->bandwidth = req_bandwidth;
   flp->handle = handle;
   flp->num_filt = 0;
   flp->finfo = NULL;
   flp->next = tcshare->flinfo;
   
   cl->flow = (struct tc_flow_info *) flp;
   
   printk(KERN_WARNING"CBQ: ADD_FLOW: AC said YES for bw=%d handle=%d\n",req_bandwidth, handle);
   printk(KERN_WARNING"CBQ: ADD_FLOW: AC said YES for nspb=%d tccp=%p \n",req_nspb, cl);
   
   tcshare->allocated += req_bandwidth;
   tcshare->flinfo = (struct tc_flow_info *) flp;
   
   tc->iq_handle = handle;
   
   return (error);
}



int tc_mod_flow(struct tcreq *tc)
{
   int              error = 0;
   struct tc_class  *cl, *old, *parent, *borrow;
   struct device    *dev;
   struct tc_ifdat  *ifdp;
   u_int32           prio;
   u_int32           maxidle, offtime;
   int               handle;
   u_int32           req_bandwidth, req_nspb;
   struct tc_share  *tcshare;
   double            fraction, gain,gb, gb1;
   u_int32           nspb, burst;
   u_int32           packetst; /* packet sending time */
   u_int32           maxq, maxdelay;
   struct tc_flow_info *flp,*hp;
   
   printk(KERN_WARNING "CBQ MOD Flow: START\n");
   dev = dev_get(tc->iq_name);
   ifdp = dev->if_ptr;
   
   if (ifdp == NULL) {
      printk(KERN_WARNING "CBQ Mod Flow: Not initialized\n");
      return (3);
   }
   
   if (ifdp->tc_share == NULL){
      printk(KERN_WARNING "CBQ Mod Flow: No Share Class created\n");
      tcshare = NULL;
      return (error);
   }
   else
      tcshare = ifdp->tc_share;
   
   flp = tcshare->flinfo;
   printk(KERN_WARNING "CBQ MOD Flow: 2 frp=%p fh=%d tc=%d\n",flp,flp->handle,tc->iq_handle);

   while (flp->handle != tc->iq_handle){
	if (flp->next != NULL){
	   hp = flp;
	   flp = flp->next;
	}
	else {
	   printk(KERN_WARNING "CBQ MOD FLOW: No such reservation !!\n");
	   return (3);
	}
   }
   old = class_h_2_class_p(ifdp, tc->iq_handle, &error);
   
   nspb = (NS_PER_SEC/ifdp->bandwidth) *8;
   
   switch (tc->iq_fs.fs_tos){
   case TOS_GUARANTEED:{
      maxdelay = tcshare->maxdelay;
      maxq     = TC_MAX_QUEUE_LEN;
      req_bandwidth = tc->iq_fs.fs_tspec.ts_tb.tbf_rate + tc->iq_fs.fs_tspec.ts_tb.tbf_bkt;
      if ((tcshare->allocated - flp->bandwidth + req_bandwidth) > tcshare->bandwidth){
	 printk(KERN_WARNING "CBQ MOD Flow: AC said NO for GS req: %d \n",req_bandwidth);
	 return (error);
      }
      break;
   }
   case TOS_CLOAD:{
      maxdelay = tc->iq_fs.fs_rspec.rs_p.p_dly;
      maxq     = ( (u_int32) maxdelay * NS_PER_MS / (nspb * tcshare->av_pkt_size) );
      req_bandwidth = tc->iq_fs.fs_tspec.ts_tb.tbf_rate ;
      if ((tcshare->allocated - flp->bandwidth + req_bandwidth) > tcshare->bandwidth){
	 printk(KERN_WARNING "CBQ MOD Flow: AC said NO for CLOAD req: %d \n",req_bandwidth);
	 return (error);
      } 
      break;
   default:
      printk(KERN_WARNING"CBQ: MOD_FLOW: hopefully unreached\n");
      return (error);
   }
   }
#ifdef CBQ_DEBUG2
   printk(KERN_WARNING "CBQ MOD Flow: maxdelay=%d  maxq=%d\n",maxdelay,maxq);
#endif   
   
   /* calculate values for the new flow */
   prio = tcshare->prio;
   burst = tc->iq_fs.fs_tspec.ts_tb.tbf_bkt;
   fraction = ((double) req_bandwidth / (double) tcshare->bandwidth);
#ifdef CBQ_DEBUG1
   printk(KERN_WARNING"CBQ: MOD_FLOW:  bw = %d\n",req_bandwidth);
#endif 
#ifdef CBQ_DEBUG2
   printk(KERN_WARNING"CBQ: MOD_FLOW:  fraction3 = %d\n", (int) ((1-fraction)/fraction));
#endif   
   gain = (1.0 - (1.0 / pow(2, RM_FILTER_GAIN)));
   /*
     printk(KERN_WARNING"CBQ: MOD_FLOW:  gain =%d  %d\n",  gain,pow(2, RM_FILTER_GAIN) );
     printk(KERN_WARNING"CBQ: MOD_FLOW:  gain =%x  %x\n",  gain,pow(2, RM_FILTER_GAIN) );
     */
   packetst = tcshare->av_pkt_size * nspb * (int)((1-fraction)/fraction);
   printk(KERN_WARNING"CBQ: MOD_FLOW:  packetst =%d\n",packetst);
   gb = pow(gain, burst);
   gb1 = pow(gain,burst-1);
   maxidle = packetst * (int)(1 - gb) / gb;
   offtime = packetst * (int)( 1 + 1/(1-gain) * (1 - gb1)/gb1 );
   printk(KERN_WARNING"CBQ: MOD_FLOW: maxidle %d offtime %d \n",maxidle, offtime);
   
   if (maxq  < 10)
      maxq = 10;
   req_nspb = ( NS_PER_SEC/  req_bandwidth) *8;
   
   if (maxq > TC_MAX_QUEUE_LEN)
      maxq = TC_MAX_QUEUE_LEN;
   
#ifdef CBQ_DEBUG2
   printk(KERN_WARNING"CBQ: MOD_FLOW: maxq= %d maxidle= %d  offtime= %d \n",maxq,maxidle,offtime);
#endif   
   
   parent = class_h_2_class_p(ifdp, tcshare->handle, &error);
   borrow = parent;
   if (error){
      printk(KERN_WARNING"CBQ: MOD_FLOW:class_h_2... failed \n");
      return (error);
   }
      
   cl = rmc_newclass(tcshare->prio, ifdp,
		     (u_int) fabs(req_nspb), rmc_delay_action,
		     (u_int) fabs(maxq), parent, borrow,
		     (u_int) fabs(maxidle), (u_int) fabs(offtime));
   
   
   flp->tbf_bkt = tc->iq_fs.fs_tspec.ts_tb.tbf_bkt;
   flp->tbf_rate = tc->iq_fs.fs_tspec.ts_tb.tbf_rate;
   flp->tbf_buf = tc->iq_fs.fs_tspec.ts_tb.tbf_buf;
   
   cl->flow = (struct tc_flow_info *) flp;

   memcpy(old, cl, sizeof(struct tc_class));
   
   printk(KERN_WARNING"CBQ: MOD_FLOW: AC said YES for bw=%d handle=%d\n",req_bandwidth, handle);
   printk(KERN_WARNING"CBQ: MOD_FLOW: AC said YES for nspb=%d tccp=%p \n",req_nspb, cl);
   
   tcshare->allocated = tcshare->allocated - flp->bandwidth + req_bandwidth;
   tc->iq_handle = handle;
   flp->bandwidth = req_bandwidth;
   
   return (error);
}

/*
 * This function adds a new filter to the filter list.
 * The new filters are added in front of the list.
 *
 * Return 0 if succeded
 *        1 in case of error
 */



int tc_add_filter(struct tcreq *tc)
{
   int              error = 0;
   struct tc_class  *tccp;
   struct tc_ifdat  *tcdp;
   struct device    *dev;
   long             handle;
   struct tc_flow_info *flow;
   struct tc_filterinfo *filtinfo, *hp;
   __u32          key0,key1;
   char           *data0, *data1;
   struct elemh_t *elem;
   
   
   dev = dev_get(tc->iq_name);
   tcdp = dev->if_ptr;
   
   handle = tc->iq_handle;
#ifdef CBQ_DEBUG2
   printk(KERN_WARNING"CBQ: ADD_FILTER: handle=%d\n",handle);
#endif   
   
   tccp = class_h_2_class_p(tcdp, handle, &error);
   printk(KERN_WARNING"CBQ: ADD_FILTER: tccp=%p\n",tccp);
   
   flow = tccp->flow;
#ifdef CBQ_DEBUG2
   printk(KERN_WARNING"CBQ: ADD_FILTER: flow=%p\n",flow);
#endif   
   
   filtinfo = (struct tc_filterinfo *) new_kmem_zalloc
      (sizeof(struct tc_filterinfo),GFP_ATOMIC);
   if (filtinfo == NULL) {
      printk(KERN_WARNING"CBQ: ADD_FILTER: Malloc failed \n");
      return (error);
   } 
   
   /* Computing of the H-Keys depending on IP version : */
   switch (tc->iq_filt.f_pf) {
   case PF_INET:
     /* key0 is XOR of src & dest adresses, and src & dst port */ 
     /* key1 is src & dst port */
     data0 = (char*) &tc->iq_filt.f_ipv4_srcaddr;     
     data1 = (char*) &tc->iq_filt.f_ipv4_srcport;
#ifdef CBQ_DEBUG2
     printk(KERN_WARNING"CBQ: ADD_FILTER: set data0=%d data1=%d\n",data0, data1);
#endif   
     key0=fss_key(data0,data1);
     key1=*((__u32*)data1);
     break;
   case PF_INET6:
     /* key0 is XOR of src & dst adresses, and flowID */
     /* key1 is flowID with version and priority  */
     printk(KERN_WARNING"CBQ: ADD_FILTER: IPV6 Filter processing.\n");	  
     data0 = (char*) (tc->iq_filt.f_ipv6_srcaddr.s6_addr32);
     data1 = (char*) &tc->iq_filt.f_ipv6_vrprflow;
#ifdef CBQ_DEBUG2
     printk(KERN_WARNING"CBQ: ADD_FILTER: set data0=%x data1=%x\n",data0, data1);
#endif   
     key0=fss_key6(data0,data1);
     printk (KERN_WARNING "CBQ: ADD_FILTER: [key0=]->%x\n",key0);
     key1=*((__u32*)data1); 
     break;
   default:
     printk(KERN_WARNING"CBQ: ADD_FILTER: unknown packet family : %d\n",tc->iq_filt.f_pf);
     return (error);
     break;
   }
   
   flow->num_filt++;
   filtinfo->filter_handle = flow->num_filt;
   filtinfo->key0 = key0;
   filtinfo->key1 = key1;
   filtinfo->next = NULL;
   
   if (flow->finfo == NULL){
#ifdef CBQ_DEBUG2
      printk(KERN_WARNING"CBQ: ADD_FILTER: first filter key0=%x key1=%x\n",key0,key1 );
#endif   
      flow->finfo = filtinfo;
   }
   else {
#ifdef CBQ_DEBUG2
      printk(KERN_WARNING"CBQ: ADD_FILTER: add to  filterlist key0=%x key1=%x\n",key0,key1 );
#endif   
      for (hp = flow->finfo ; hp->next != NULL ; hp = hp->next);
	hp->next = filtinfo;
   }
   
   elem = (struct elemh_t *) new_kmem_zalloc
      (sizeof(struct elemh_t),GFP_ATOMIC);
   if (elem == NULL) {
      printk(KERN_WARNING"CBQ: ADD_FILTER: Malloc failed \n");
      return (error);
   }
   
   elem->key0 = key0;
   elem->key1 = key1;
   elem->tccp = tccp;
   
   hash_insert (&tcdp->rooth, elem);
   tc->iq_fhdl = filtinfo->filter_handle ;
   printk(KERN_WARNING"CBQ: ADD_FILTER: add filter key0=%x key1=%x\n",key0,key1 );
   
   return (error);
}
   


int tc_del_filter(struct tcreq *tc)
{
   int              error = 0;
   struct tc_class  *tccp;
   struct tc_ifdat  *tcdp;
   struct device    *dev;
   long             fhandle;
   struct tc_flow_info *flow;
   struct tc_filterinfo *filtinfo, *hp;
   __u32          key0,key1;
   char           *data0, *data1;
   struct elemh_t *elem;
   __u32          fs_stat=0;
   
   dev = dev_get(tc->iq_name);
   tcdp = dev->if_ptr;
   
   if (tcdp == NULL) {
      printk(KERN_WARNING "TC DEL flow: Not initialized\n");
      return (3);
   }

   /* Computing of the H-Keys depending on IP version : */
   switch (tc->iq_filt.f_pf) {
   case PF_INET:
     /* key0 is XOR of src & dest adresses, and src & dst port */ 
     /* key1 is src & dst port */
   data0 = (char*) &tc->iq_filt.f_ipv4_srcaddr;
   data1 = (char*) &tc->iq_filt.f_ipv4_srcport;
#ifdef CBQ_DEBUG2
   printk(KERN_WARNING"CBQ: DEL_FILTER: set data0=%d data1=%d\n",data0, data1);
#endif   
   key0=fss_key(data0,data1);
   key1=*((__u32*)data1);
   case PF_INET6:
     /* key0 is XOR of src & dst adresses, and flowID */
     /* key1 is flowID with version and priority  */
     printk(KERN_WARNING"CBQ: DEL_FILTER: IPV6 Filter processing.\n");
     data0 = (char*)  (tc->iq_filt.f_ipv6_srcaddr.s6_addr32);
     data1 = (char*)  &tc->iq_filt.f_ipv6_vrprflow;
#ifdef CBQ_DEBUG2
     printk(KERN_WARNING"CBQ: DEL_FILTER: set data0=%u data1=%u\n",data0, data1);
#endif   
     key0=fss_key6(data0,data1);
     key1=*((__u32*)data1); 
     break;
   default:
     printk(KERN_WARNING"CBQ: DEL_FILTER: unknown packet family : %d\n",tc->iq_filt.f_pf);
     return (error);
     break;
   }

   if ( (elem = hash_lookup(&tcdp->rooth,key0,key1,&fs_stat)) ){
#ifdef CBQ_DEBUG1
      printk(KERN_WARNING "TC DEL flow: delete hash entry\n");
#endif   

      tccp = elem->tccp;
      flow = tccp->flow;
      fhandle = tc->iq_fhdl;
      for ( hp = flow->finfo; hp->filter_handle != fhandle; hp=hp->next, filtinfo=hp);
      filtinfo->next = hp->next;
      kmem_free(hp, sizeof(hp));
      flow->num_filt--;
      hash_delete (&tcdp->rooth, elem, key0);
   }
   else {
      printk(KERN_WARNING "TC DEL flow: No corresponding hash entry key0=%d key1=%d\n",key0,key1 );
      return (1);
   }
   return (error);
}
