/*
 * DRET-IPv6  An implementation of the TCP/IP protocol suite for the LINUX
 *              operating system.  INET6 is implemented using the BSD Socket
 *              interface as the means of communication with the user level.
 *
 * Contacts:    
 *              INRIA      <Christophe.Diot@sophia.inria.fr>
 *              MASI       <Eric.Horlait@masi.ibp.fr>
 *              This software has been developped with the financial support
 *              of DRET (French Military Research Agency).
 *
 * Version:     $Id$  
 *      
 * Authors:     
 *              Frank Zago      <Zago@masi.ibp.fr>
 *
 *
 * Fixes:
 *		Pascal Anelli	<Anelli@masi.ibp.fr> add compatibility with AF_INET socket
 *
 * Description:
 *              UDP for IPv6.
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *              This notice must be copied with the distributed package.
 *
 *              Portions of the software are derived from  various 
 *              networking code publicly available, mainly:
 *                      NRL IPv6 code 
 *                      NetBSD code from INRIA (Francis.Dupont@inria.fr)
 *                      Pedro Roque's Linux version (Roque@di.fc.up.pt)
 *
 */

/** Contents *************************************************************

   udp6_sendmsg,
   udp6_recvmsg,
   udp6_close,
   udp6_connect,
   udp6_rcv_skb
   udp6_init
   udp6_getfragcsum
   udp6_ioctl
   udp6_rcv
   udp6_err
   
 ************************************************************************/

#include <asm/uaccess.h>
#include <linux/config.h>
#include <linux/in.h>
#include <linux/skbuff.h>
#include <linux/udp.h>
#include <linux/socket.h>
#include <linux/proc_fs.h>
#include <linux/uio.h>

#include <linux/dret/icmpv6.h>
#include <linux/dret/ipv6_debug.h>

#include <net/sock.h>
#include <net/checksum.h>
#include <net/udp.h>

#include <net/dret/protocol6.h>
#include <net/dret/ipv6.h>
#include <net/dret/icmpv6.h>
#include <net/dret/udp6.h>
#include <net/dret/proc6.h>
#include <net/dret/af_inet6.h>
#include <net/dret/route6.h>

#define min(a,b)	((a)<(b)?(a):(b))


#ifdef IPV6_DEBUG_UDP
#define DEBUG(format, args...) printk(KERN_DEBUG format,##args)
#define PRINT_ADDR(a) printk(KERN_DEBUG  "[UDP6]%X:%X:%X:%X\n",\
			     (a)->s6_addr32[0], (a)->s6_addr32[1], \
			     (a)->s6_addr32[2], (a)->s6_addr32[3]);
#else
#define DEBUG(format, args...) 
#define PRINT_ADDR(a)
#endif

struct udp_mib udp6_statistics;

struct sock * udp6_sock_array[SOCK_ARRAY_SIZE];



/* FIXME: socket cache not implemented yet. 
   Should use a better structure than the old one. */



/*
 * Copy and checksum a UDP packet from user space into a buffer. 
 * We still have to do the planning to get ip_build_xmit to spot direct 
 * transfer to network card and provide an additional callback mode
 * for direct user->board I/O transfers. That one will be fun.
 */
static void udp6_getfragcsum(const void *fkhdr, struct in6_addr *saddr, 
			     char *dest, unsigned int offset, 
			     unsigned int fraglen)
{
   struct udp6_fake_header *ufh = (struct udp6_fake_header *) fkhdr;

   if (offset)
      ufh->csum = csum_partial_copy_fromiovecend(dest,
						 ufh->iov,
					         offset- sizeof(struct udphdr),
						 fraglen,
						 ufh->csum);
   else {
      __u32 csum = 0;
      /* The first fragment, so we finish to compute the checksum
       * and we fit the udp header. */
      csum = csum_partial_copy_fromiovecend(dest + sizeof(struct udphdr),
					    ufh->iov,
					    0,
					    fraglen - sizeof(struct udphdr),
					    csum);
					    
      csum = csum_partial_copy((void *) &ufh->uh, dest, 
      			       sizeof(struct udphdr),
			       csum);
			       
      csum = csum_ipv6_magic(saddr, 
                             ufh->daddr, fraglen, IPPROTO_UDP, 
                             csum);
      /* RFC 1883 8.1 */
      if (!(csum))
	 csum = 0xffff;
      ((struct udphdr *) dest)->check = csum;
   }
}



	               /**==============================**/
	       /**=============================================**/
	       /**	      Handle by socket                 **/
	       /**=============================================**/
	               /**==============================**/


static void udp6_close(struct sock *sk, unsigned long timeout)
{
   lock_sock(sk);

   sk->state = TCP_CLOSE;
   
   /* FIXME
   	struct ipv6_pinfo *np = &sk->net_pinfo.af_inet6;

	if (np->dest)
	{
		ipv6_dst_unlock(np->dest);
	}
   */
   
   release_sock(sk);
   sk->dead = 1;
   destroy_sock(sk);
}



int udp6_connect(struct sock *sk, struct sockaddr *uaddr, int sock_len)
{
   struct sockaddr_in6 *sin6 = (struct sockaddr_in6 *) uaddr;
   struct ipv6_pinfo   *af_inet6;


   if (sock_len != sizeof(struct sockaddr_in6))
       return (-EINVAL);

   if (sin6->sin6_family && sin6->sin6_family != AF_INET6)
      return (-EAFNOSUPPORT);

   if (IS_ANYADDR6(sin6->sin6_addr))
      return (-EADDRNOTAVAIL);	/* don't know where to send */

   af_inet6 = &sk->net_pinfo.af_inet6;

   if (IS_MAPPEDADDR6 (sin6->sin6_addr))
	{
		struct sockaddr_in sin;
		int err;

		sin.sin_family = AF_INET;
		sin.sin_addr.s_addr = sin6->sin6_addr.s6_addr32[3];

		err = udp_connect(sk, (struct sockaddr*) &sin, sizeof(sin));
		
		if (err < 0)
			return err;
			
		if(IS_ANYADDR6(af_inet6->pi6_fl.fll_sa)){
			af_inet6->pi6_fl.fll_sa.s6_addr32[0] = 0;
			af_inet6->pi6_fl.fll_sa.s6_addr32[1] = 0;
			af_inet6->pi6_fl.fll_sa.s6_addr32[2] = 
					           __constant_htonl(0x0000FFFF);
			af_inet6->pi6_fl.fll_sa.s6_addr32[4] = sk->saddr;
		}

		if(IS_ANYADDR6(af_inet6->rcv_saddr)){
			af_inet6->rcv_saddr.s6_addr32[0] = 0;
			af_inet6->rcv_saddr.s6_addr32[1] = 0;
			af_inet6->rcv_saddr.s6_addr32[2] = 
					           __constant_htonl(0x0000FFFF);
			af_inet6->rcv_saddr.s6_addr32[4] = sk->rcv_saddr;
		}
	}
	
   COPY_ADDR6(af_inet6->pi6_fl.fll_da, sin6->sin6_addr);

	/* 
	 *	Check for a route to destination
	 */

   if (!route_check(&sin6->sin6_addr, 0))
		return -ENETUNREACH;

	/* FIXME put the saddr in af_inet6 in the address is unspecified from
	   the device
	if(IS_ANYADDR6(af_inet6->saddr)) {
		ipv6_addr_copy(&np->saddr, &ifa->addr);
	}
*/

   sk->dummy_th.dest = sin6->sin6_port;
   sk->state = TCP_ESTABLISHED;
   return (0);
}


/*
 *    IOCTL requests applicable to the UDP protocol
 */

int udp6_ioctl(struct sock *sk, int cmd, unsigned long arg)
{
   int err;
   switch (cmd) {
   case TIOCOUTQ:
      {
	 unsigned long amount;

	 if (sk->state == TCP_LISTEN)
	    return (-EINVAL);
	 amount = sock_wspace(sk);
	 err = put_user(amount, (unsigned long *) arg);
	 return (err);
      }

   case TIOCINQ:
      {
	 struct sk_buff *skb;
	 unsigned long amount;

	 if (sk->state == TCP_LISTEN)
	    return (-EINVAL);
	 amount = 0;
	 skb = skb_peek(&sk->receive_queue);
	 if (skb != NULL) {
	    /*
	     * We will only return the amount
	     * of this packet since that is all
	     * that will be read.
	     */
	    amount = skb->len - sizeof(struct udphdr);
	 }
	 err = put_user(amount, (unsigned long *) arg);
	 return (err);
      }

   default:
      return (-EINVAL);
   }
   return (0);
}

/*
 * Send UDP frames.
 */

static int udp6_sendmsg(struct sock *sk, struct msghdr *msg, int len)
{
   struct ipv6_pinfo *info = &sk->net_pinfo.af_inet6;
   struct sockaddr_in6 *sin6 = (struct sockaddr_in6 *) msg->msg_name;
   struct udp6_fake_header ufh;
   struct cmsghdr  *cmsgptr;
   struct in6_addr *saddr = NULL; 
   struct options   ip6opt;
   struct flowlist *fll = NULL;
   
   int ulen = len + sizeof(struct udphdr);
   int err, hasopt = 0;
   int clen = msg->msg_controllen; 
   int flags = msg->msg_flags;
   int noblock = msg->msg_flags & MSG_DONTWAIT;
   
   /* 
    *    Check the flags. We support no flags for UDP sending
    */
   if (msg->msg_flags & ~(MSG_DONTROUTE|MSG_DONTWAIT)) {
     DEBUG ("[UDP6] Error : Flag not supported\n");
     return(-EINVAL);
   }
   /* FIXME: should we test for a size that does not fit into 
    * a jumbo packet :) ? 
    */

   /*
    *    Get and verify the address. 
    */   
   if (sin6) {
     if (msg->msg_namelen != sizeof(struct sockaddr_in6)) {
       DEBUG ("[UDP6] Error : Size passed != sizeof(struct sockaddr_in6)\n ");
       return (-EINVAL);
     }
     if (sin6->sin6_family && sin6->sin6_family != AF_INET6) {
       DEBUG ("[UDP6] Error : sin6_familiy != AF_INET6\n");
       return (-EINVAL);
     }
     if ((ufh.uh.dest = sin6->sin6_port) == 0) {
       DEBUG ("[UDP6] Error : sin6_port is null\n");
       return (-EINVAL);
     }

     ufh.daddr = &sin6->sin6_addr;
      
     /* Flow label ? */
     if (sin6->sin6_flowinfo) {
       for (fll = info->fl; fll; fll = fll->fll_next)
	 if(fll->fll_label == sin6->sin6_flowinfo)
	   break;
       if (fll == NULL) {
	 DEBUG ("[UDP6] Error : flow label not found in f. l. list \n");
	 return (-EINVAL);
       }
     }
   } else {
     if (sk->state != TCP_ESTABLISHED) {
       DEBUG ("[UDP6] Error : socket opened for TCP !\n");
       return (-EINVAL);
     }
     ufh.uh.dest = sk->dummy_th.dest;	/* already in network order 
      					   set at udp6_connect */
     ufh.daddr = &info->pi6_fl.fll_da;      
     /* Flow label ? */
     if(info->pi6_fl.fll_label)
       fll = &info->pi6_fl;
   }
   if (IS_MAPPEDADDR6(*ufh.daddr))
     {
       struct sockaddr_in sin;
		
       sin.sin_family = AF_INET;
       sin.sin_addr.s_addr = ufh.daddr->s6_addr32[3];
       
       return udp_sendmsg(sk, msg, len);
     }

   if (IS_ANYADDR6(*ufh.daddr)) {
     DEBUG ("[UDP6] Error : don't know where to send the pascket \n");   
     return (-EINVAL);		/* don't know where to send */
   }

   ufh.uh.source = sk->dummy_th.source;
   ufh.uh.len    = htons(ulen);
   ufh.uh.check  = 0;
   ufh.iov       = msg->msg_iov;

   if (fll) {
      err = ipv6_flow_xmit(sk, udp6_getfragcsum, &ufh,
			   ulen, fll, flags, IPPROTO_UDP, noblock);
      return(err);
   } else {
      /* 
       * Any ancillary data ? 
       *
       * (bb) Should this be made more generic ?... and located at
       * the af_inet6 level for instance...
       * Same code must be written for UDP, TCP...
       * As of Nov 20, 96:
       *    source address is handled.
       */
      if (clen > 0) {
	 for (cmsgptr = CMSG_FIRSTHDR(msg); 
	      (cmsgptr != NULL) && (clen > 0);
	      cmsgptr = CMSG_NXTHDR(msg, cmsgptr)) {
	    /* Source addr. info */
	    if (cmsgptr->cmsg_level == IPPROTO_IPV6) {
	       switch(cmsgptr->cmsg_type) {
	       case IPV6_PKTINFO:
		  saddr = &(((struct in6_pktinfo *)cmsgptr->cmsg_data)->ipi6_addr);	     
		  break;
	       case IPV6_HOPLIMIT:
		  break;
	       case IPV6_NEXTHOP:
		  break;
	       case IPV6_HOPOPTS:
		  break;
	       case IPV6_DSTOPTS:
		  break;
	       case IPV6_SRCRT:
		  /* Will be read and copied by ipv6_build_xmit */
		  DEBUG("[UDP6]udp6_sendmsg: IPV6_SRCRT\n");
		  ip6opt.opt_srcrt = (struct ipv6_rt_hdr *)(cmsgptr + 1);
		  ip6opt.opt_srcrt_len = cmsgptr->cmsg_len - sizeof(struct cmsghdr);
		  hasopt ++;
		  break;
	       default:
	       }
	    } else {
	       DEBUG("[UDP6]udp6_sendmsg: ancillary data with bad level\n");
	    }
	    clen -= cmsgptr->cmsg_len;
	 }
      }
      
      if (hasopt)
	 err = ipv6_build_xmit(sk, udp6_getfragcsum, &ufh,
			       ulen, ufh.daddr, saddr, &ip6opt,
			       flags, IPPROTO_UDP, noblock);
      else
	 err = ipv6_build_xmit(sk, udp6_getfragcsum, &ufh,
			       ulen, ufh.daddr, saddr, NULL,
			       flags, IPPROTO_UDP, noblock);      
   }
   
   if (CMSG_FIRSTHDR(msg))
      kfree_s((void *) CMSG_FIRSTHDR(msg), msg->msg_controllen);

   if (err < 0)
      return (err);
   else {
      udp6_statistics.UdpOutDatagrams++;
      return (len);
   }
}


/*
 *    This should be easy, if there is something there we\
 *      return it, otherwise we block.
 */

int udp6_recvmsg(struct sock *sk, 
                 struct msghdr *msg, 
                 int len,
		 int noblock, 
		 int flags, 
		 int *addr_len)
{
   int copied = 0;
   int err;
   struct sk_buff *skb = NULL;
   struct sockaddr_in6 *sin6 = (struct sockaddr_in6 *) msg->msg_name;
   struct ipv6_pinfo *np = &sk->net_pinfo.af_inet6;

   /*
    *    Check any passed addresses
    */
   if (addr_len)
      *addr_len = sizeof(struct sockaddr_in6);

   /*
    *    From here the generic datagram does a lot of the work. Come
    *      the finished NET3, it will do _ALL_ the work!
    */
   skb = skb_recv_datagram(sk, flags, noblock, &err);
   if (skb == NULL)
      return -EFAULT;

   copied = min(len, skb->len - sizeof(struct udphdr));

   /*
    *    FIXME : should use udp header size info value 
    */
   skb_copy_datagram_iovec(skb, sizeof(struct udphdr), msg->msg_iov, copied);
   sk->stamp = skb->stamp;

   /* 
    * Fill in cmsghdr structure
    */
   {
      struct cmsghdr *cmsg;
      int clen = msg->msg_controllen;
      __u16 tlen = 0;
     
      if (np->pktinfo) {
	 struct {
	    struct cmsghdr cmsg;
	    struct in6_pktinfo pktinfo;
	 } message;
	 clen -= sizeof(message);
	 tlen += sizeof(message);
	 
	 if (clen < 0)
	    return -ENOMEM;

	 message.cmsg.cmsg_level = IPPROTO_IPV6;
	 message.cmsg.cmsg_type = IPV6_PKTINFO;
	 message.cmsg.cmsg_len = sizeof(message);
	 message.pktinfo.ipi6_ifindex = skb->dev->ifindex;
	 COPY_ADDR6(message.pktinfo.ipi6_addr, skb->nh.ipv6h->saddr /*saddr6*/);

	 err = copy_to_user(msg->msg_control, (char *) &message, sizeof(message));
	 if (err)
	    return -EFAULT;
      }
      
      if (np->recvsrcrt && 
	  np->pi6_fl.fll_opt.opt_srcrt_len) {  
	 int llen = np->pi6_fl.fll_opt.opt_srcrt_len + sizeof(struct cmsghdr);
	    
	 cmsg = (struct cmsghdr *)((__u8 *)msg->msg_control + tlen);
	 clen -= llen;
	 tlen += llen;
	 if (clen < 0)
	    return -ENOMEM;
	 
	 cmsg->cmsg_level = IPPROTO_IPV6;
	 cmsg->cmsg_type  = IPV6_SRCRT;
	 cmsg->cmsg_len   = np->pi6_fl.fll_opt.opt_srcrt_len;
	 copy_to_user(cmsg+1, np->pi6_fl.fll_opt.opt_srcrt, llen);
      }
   } 
   /* Copy the address. */
   if (sin6) {
      sin6->sin6_family = AF_INET6;
      sin6->sin6_port = skb->h.uh->source;
      COPY_ADDR6(sin6->sin6_addr, skb->nh.ipv6h->saddr);
   }
   skb_free_datagram(sk, skb);
   return (copied);
}



/* 
 * Backlog 
 */
int udp6_rcv_skb(struct sock *sk, struct sk_buff *skb)
{
   /*
    *    Charge it to the socket, dropping if the queue is full.
    */
   if (__sock_queue_rcv_skb(sk, skb) < 0) {
      udp6_statistics.UdpInErrors++;
      ip6_statistics.IpInDiscards++;
      ip6_statistics.IpInDelivers--;
      skb->sk = NULL;
      kfree_skb(skb, FREE_WRITE);
      return 0;
   }
   udp6_statistics.UdpInDatagrams++;
   return 0;
}


struct proto udp6_prot =
{
   udp6_close,
   udp6_connect,
   NULL,	/* accept */
   NULL,	/* retransmit */
   NULL,	/* write_wakeup */
   NULL,	/* read_wakeup */
   datagram_poll,  /* poll */
   udp6_ioctl,
   NULL,   	/* No special init when the socket is created */
   NULL,        /* destroy */
   NULL,	/* shutdown */
   ipv6_setsockopt,
   ipv6_getsockopt,
   udp6_sendmsg,
   udp6_recvmsg,
   NULL,			/* No special bind function */
   udp6_rcv_skb,
   128,
   0,
   "UDP",
   0, 0,
   udp6_sock_array
};


	               /**==============================**/
	       /**=============================================**/
	       /**	      Handle by interrupt              **/
	       /**=============================================**/
	               /**==============================**/



/*
 *    All we need to do is get the socket, and then do a checksum. 
 */
int udp6_rcv(struct sk_buff *skb, 
	     struct device *dev, 
	     struct in6_addr *saddr,
	     struct in6_addr *daddr, 
	     struct options *opt,
	     unsigned short len,
	     int redo,
	     struct inet6_protocol *protocol)
{
   struct sock *sk;
   struct udphdr *uh;
   unsigned int ulen;
   int addrtype;
   __u32 csum;

   /*
    * If we're doing a "redo" (the socket was busy last time
    * around), we can just queue the packet now..
    */
/* 
   if (redo) {
   udp6_queue_rcv_skb(skb->sk, skb);
   return 0;
   }
 */

   DEBUG("[UDP]udp6_rcv: IN\n");  
   ip6_statistics.IpInDelivers++;

   /*
    *    Validate the packet and the UDP length.
    */
   uh = (struct udphdr *) skb->h.uh;
   ulen = ntohs(uh->len);

   if (ulen > len || len < sizeof(*uh) || ulen < sizeof(*uh)) {
      DEBUG("[UDP]udp6_rcv: short packet: %d/%d\n", ulen, len);
      udp6_statistics.UdpInErrors++;
      kfree_skb(skb, FREE_WRITE);
      return (0);
   }
   
   csum = 0;
   switch (skb->ip_summed) {
   case CHECKSUM_NONE:
      csum = csum_partial((char *) uh, ulen, csum);
   case CHECKSUM_HW:
      csum = csum_ipv6_magic(saddr, daddr, ulen, IPPROTO_UDP, csum);
   }
   
   if (csum) {
      DEBUG("[UDP]udp6_rcv: bad checksum. From port %d to port %d ulen %d\n",
	     ntohs(uh->source),
	     ntohs(uh->dest),
	     ulen);
      udp6_statistics.UdpInErrors++;
      kfree_skb(skb, FREE_WRITE);
      return (0);
   }
   
   addrtype = get_ipv6_addr_type(daddr);

   /*
    *    These are supposed to be switched. 
    */
   /*COPY_ADDR6(skb->daddr6, *saddr);
     COPY_ADDR6(skb->saddr6, *daddr);
   */
   
   len = ulen;
   skb->dev = dev;
   skb_trim(skb, len);
   DEBUG("[UDP]udp6_rcv: From port %d to port %d ulen %d\n",
	  ntohs(uh->source),
	  ntohs(uh->dest),
	  ulen);  

   sk = get_sock6(&udp6_prot, uh->dest, saddr, uh->source, daddr);

   if (sk == NULL) {
      udp6_statistics.UdpNoPorts++;
      DEBUG("[UDP]udp6_rcv: Can't get socket!\n");     
      icmpv6_send_error(skb, ICMP6_UNREACH, ICMP6_UNREACH_PORT, 0, dev);
      skb->sk = NULL;
      kfree_skb(skb, FREE_WRITE);
      return (0);
   }
   
   skb->sk = sk;
   if (sk->users) {
      __skb_queue_tail(&sk->back_log, skb);
      return (0);
   }
   
   udp6_rcv_skb(sk, skb);
   return (0);
}


void udp6_err(int type, int code, unsigned char *buff, __u32 info,
	       struct in6_addr *saddr, struct in6_addr *daddr,
	       struct inet6_protocol *protocol)
{
	struct sock *sk;
	struct udphdr *uh;
	int err;
	
	uh = (struct udphdr *) buff;

        sk = get_sock6(&udp6_prot, uh->dest, saddr, uh->source, daddr);
   
	if (sk == NULL)
	{
		printk(KERN_DEBUG "icmp for unkown sock\n");
		return;
	}

	if (icmpv6_err_convert(type, code, &err))
	{
		if(sk->bsdism && sk->state!=TCP_ESTABLISHED)
			return;
		
		sk->err = err;
		sk->error_report(sk); /* sock_def_callback1 */
	}
	else
		sk->err_soft = err;
}



void udp6_init(struct net_proto_family *ops)
{
   int i;

   /* Fill its sockarray */
   for (i = 0; i < SOCK_ARRAY_SIZE; i++)
      udp6_prot.sock_array[i] = NULL;

#ifdef CONFIG_PROC_FS
   /* Register in /proc/net/udp6 */
   proc_net6_register(&(struct proc_dir_entry) {
		      PROC_NET6_UDP6, 4, "udp6",
		      S_IFREG | S_IRUGO, 1, 0, 0,
		      0, &proc_net6_inode_operations,
		      udp6_get_info
		      });
#endif
}



