
#include <linux/sched.h>
#include <asm/uaccess.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/net.h>
#include <linux/route.h>
#include <linux/netdevice.h>
#include <linux/dret/ipv6_debug.h>
#include <linux/dret/netdevice6.h>
#include <linux/dret/table.h>
#include <linux/in.h>
#include <net/netlink.h>
#include <net/sock.h>
#include <net/snmp.h>

#include <net/dret/ipv6.h>
#include <net/dret/nd.h>
#include <net/dret/protocol6.h>
#include <net/dret/route6.h>
#include <net/dret/addrconf.h>
#include <net/dret/sit.h>
#include <net/dret/table.h>
#include <net/dret/table_tools.h>

#ifdef IPV6_DEBUG_TABLE
#define DEBUG(format, args...) printk(KERN_DEBUG format,##args)
#define PRINT_ADDR(a) printk(KERN_DEBUG "[TABLE]%X:%X:%X:%X\n",\
			     (a)->s6_addr32[0], (a)->s6_addr32[1], \
			     (a)->s6_addr32[2], (a)->s6_addr32[3]);
#else
#define DEBUG(format, args...)
#define PRINT_ADDR(a)
#endif

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  
          IPV6 TABLES
  
 %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
struct gen_table *ipv6_tables[8] = {
   &routing_table,
   &nd_table,
   &flow_table,
};

struct table_leaf *old_leaves;

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

   TABLE MANAGEMENT

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% */
/*--START FUNCTION--(leaf_release)-------------------------------------------
  
  Detach a leaf from the table.
  Leaf is destroyed if reference count reaches 0.

----------------------------------------------------------------------------*/
__inline__ void leaf_release(struct gen_table  *table, 
			     struct table_leaf *tl) {
   DEBUG("[TABLE]leaf_release: IN table = %p tl = %p ref = %d\n", 
	 table, tl, ((struct rt_dest *)tl->tl_data)->rt_ref);
   /* Look at ref count */ 
   if (((struct rt_dest *)tl->tl_data)->rt_ref == 0) {
      kfree_s(tl->tl_data, tl->tl_len);
      kfree(tl);
      atomic_dec(&(table->gt_count));
      DEBUG("[TABLE]leaf_release: OUT released leaf\n");
      return;
   } 
   /* Link to garbage list */
   tl->tl_next = old_leaves;
   old_leaves  = tl;
   /* This list should be run when timer expires */
   DEBUG("[TABLE]leaf_release: OUT tl_ref = %d\n", 
	 ((struct rt_dest *)tl->tl_data)->rt_ref);
   
}
/*--END FUNCTION--(leaf_release----------------------------------------------*/


/*--START FUNCTION--(table_add_1)--------------------------------------------
 |
 | Adds a leaf into a binary tree table.
 |
 | Input: an ipv6 address *addr.
 |        its prefix length
 |        a leaf structure
 |        the table to update.
 | Return: 0
 |
 ------------------------------------------------------------------------ */
int table_add_1(struct in6_addr *addr, int plen, 
		void *data, __u32 len, struct gen_table *table) {
   struct table_node *fn;
   struct table_node *pn = NULL;
   struct table_node *in;
   struct table_node *ln;
   struct table_leaf *lf;
   __u32 bit;
   __u32 dir = 0;

   DEBUG("[TABLE]table_add_1: IN table = %p\n", table);
   /* Form leaf */
   lf = kmalloc(sizeof(struct table_leaf), GFP_ATOMIC);
   if (lf == NULL)
      return (-ENOMEM);
   lf->tl_node = NULL;
   lf->tl_next = NULL;
   lf->tl_len  = len;
   lf->tl_data = data;
   
   /* default route */
   if ((SAME_ADDR6(*addr, in6addr_any)) && (plen == 0)) {   
      DEBUG("[TABLE]table_add1: OUT default entry\n");
      lf->tl_next = table->gt_default;
      table->gt_default = lf;
      return 0;
   }
   
   /* start at root */
   fn = table->gt_root;		
   for (;;) {
      if (fn == NULL) {
	 /* insert data */
	 ln = kmalloc(sizeof(struct table_node), GFP_ATOMIC);
	 if (ln == NULL)
	    return (-ENOMEM);

	 memset(ln, 0, sizeof(struct table_node));
	 ln->t_parent    = pn;
	 ln->t_leaf      = lf; 
	 ln->t_bit       = plen;
	 ln->t_flags     = RTN_BACKTRACK;
	 COPY_ADDR6(ln->t_key, *addr);
	 lf->tl_node = ln; /* back pointer */
	 

	 if (dir)
	    pn->t_right = ln;
	 else
	    pn->t_left  = ln;
	 
	 DEBUG("[TABLE]table_add_1: OUT\n");
	 DEBUG("[TABLE](pn) parent bit %d parent key:\n", pn->t_bit);
	 PRINT_ADDR(&pn->t_key);
	 DEBUG("[TABLE](ln) new leaf bit %d new leaf key:\n", ln->t_bit);
	 /*, (dir? "right":"left"));*/
	 PRINT_ADDR(&ln->t_key); 
	 atomic_inc(&table->gt_count);
	 return (0);
      }
      /* 
       * fn != NULL 
       */
      if (plen >= fn->t_bit && pfx_match(&fn->t_key, addr, fn->t_bit)) {
	 if (plen == fn->t_bit) {
	    /* clean up an intermediate node */
	    if ((fn->t_flags & RTN_BACKTRACK) == 0) {
	       /*leaf_release(fn->t_leaf);
	       fn->t_leaf = NULL;*/
	       fn->t_flags |= RTN_BACKTRACK;
	    } else {
	       DEBUG("[TABLE]table_add_1: OUT route w. same pfx already registered\n");
               return 0;
	    }
	    /* 
	     * 11/21/96 Eliminating redundant routes and identical routes 
	     *          with different devices.
	     *
	     for (l1 = fn->leaf; l1; l1 = l1->next){
	     if (SAME_ADDR6(rt->rd, l1->rd) &&
	     (rt->dev == l1->dev)) {
	     printk("Route already registered \n");
	     return 0;
	     }
	     }*/
	    lf->tl_node = fn;
	    lf->tl_next = fn->t_leaf;	    
	    fn->t_leaf = lf; 
	    COPY_ADDR6(fn->t_key, *addr);
	    DEBUG("[TABLE]table_add_1: OUT (uin)\n");
	    DEBUG("[TABLE](fn) new leaf bit %d new leaf key:\n", fn->t_bit);
	    PRINT_ADDR(&fn->t_key);
	    return 0;
	 }
	 /* 
	  * plen > ref. bit, 
	  * input addr does match leaf addr up to ref bit
	  */
	 
	 /* walk down on tree */
	 pn  = fn;
	 dir = addr_bit(addr, fn->t_bit);
	 fn  = dir ? fn->t_right : fn->t_left;
	 DEBUG("[TABLE]Plen %d Bit %d, going %d\n", 
	       plen, pn->t_bit, dir);
	 continue;
      }
      /*
       * plen < rn_bit... don't have a common prefix anymore
       * or we have a less significant route.
       * insert an intermediate node on the list.
       * this new rt_node will point to the rt_dest we add
       * and the current rt_node
       */
      pn = fn->t_parent;

      /* find 1st bit in difference between the 2 addrs 
       * starting at the reference bit of the parent node.
       */
      bit = addr_diff(addr, &fn->t_key, pn->t_bit);
      if (plen > bit) {
	 /*           (pn)
	  *           / or\ 
	  *          (in)  ...
	  *         /    \
	  *       (ln)   (fn) 
	  */
	 /* allocate new table_node */
	 in = kmalloc(sizeof(struct table_node), GFP_ATOMIC);
	 if (in == NULL)
	    return (-ENOMEM);
	 /* leaf node */
	 ln = kmalloc(sizeof(struct table_node), GFP_ATOMIC);
	 if (ln == NULL) {
	    kfree(in);
	    return (-ENOMEM);
	 }
	 memset(in, 0, sizeof(struct table_node));
	 atomic_inc(&table->gt_count);
	 memset(ln, 0, sizeof(struct table_node));
	 atomic_inc(&table->gt_count);
	 /* 
	  * new intermediate node. 
	  * RTN_BACKTRACK will be off since that an address 
	  * that chooses one of the branches would not match 
	  * less specific routes in the other branch
	  */
	 in->t_parent = pn;
	 in->t_bit    = bit;	        /* diff btw addr & fn->... */	 
	 /* in->t_leaf   = leaf; atomic_inc(&leaf->tl_count); */    
	 COPY_ADDR6(in->t_key, *addr);
	 if (addr_bit(addr, bit)) {
	    in->t_right = ln;
	    in->t_left = fn;
	 } else {
	    in->t_left = ln;
	    in->t_right = fn;
	 }
	 
	 /* update parent pointer */
	 if (dir)
	    pn->t_right = in;
	 else
	    pn->t_left = in;

	 /*
	  * New leaf
	  */
	 ln->t_parent = in;
	 ln->t_bit    = plen;	      
	 ln->t_leaf   = lf; /* atomic_inc(&leaf->tl_count);*/
	 /* We may backtrack and find a less specific route */
	 ln->t_flags  = RTN_BACKTRACK;
	 COPY_ADDR6(ln->t_key, *addr);
	 
	 lf->tl_node = ln;
	 fn->t_parent = in;
	 
	 DEBUG("[ROUTE]rt_add_1: OUT\n");
	 DEBUG("[TABLE](pn) parent bit %d parent key:\n", pn->t_bit);
	 PRINT_ADDR(&pn->t_key);
	 DEBUG("[TABLE](in) int. bit %d int. key:\n", in->t_bit);
	 /*, (dir? "right":"left"));*/
	 PRINT_ADDR(&in->t_key);
	 DEBUG("[TABLE](ln) new leaf bit %d new leaf key:\n", ln->t_bit);
	 /*, (addr_bit(addr, bit)? "right":"left"));*/
	 PRINT_ADDR(&ln->t_key);
	 DEBUG("[TABLE](fn) moved leaf bit %d moved leaf key:\n", fn->t_bit); 
	 /*, (addr_bit(addr, bit)? "left":"right"));*/
	 PRINT_ADDR(&fn->t_key);
	 return 0;
      }
      /*   plen <= bit where the mismatch occurs.  
       *                 (pn)
       *                /
       *              (ln)
       *             /   \
       *           (fn)   NULL
       */
      ln = kmalloc(sizeof(struct table_node), GFP_ATOMIC);
      if (ln == NULL)
	 return (-ENOMEM);
      memset(ln, 0, sizeof(struct table_node));
      atomic_inc(&table->gt_count);
      
      ln->t_parent = pn;
      ln->t_bit    = plen;
      ln->t_flags  = RTN_BACKTRACK;
      ln->t_leaf   = lf; /* atomic_inc(&leaf->tl_count); */
      COPY_ADDR6(ln->t_key, *addr);
      if (addr_bit(&fn->t_key, plen))
	 ln->t_right = fn;
      else
	 ln->t_left = fn;
      
      lf->tl_node = ln;

      if (dir)
	 pn->t_right = ln;
      else
	 pn->t_left = ln;

      fn->t_parent = ln;
      
      DEBUG("[ROUTE]rt_add_1: OUT\n");
      DEBUG("[TABLE](pn) parent bit %d parent key:\n", pn->t_bit);
      PRINT_ADDR(&pn->t_key);
      DEBUG("[TABLE](ln) new leaf bit %d new leaf key:\n", ln->t_bit);
      /*, (dir? "right":"left"));*/
      PRINT_ADDR(&ln->t_key);
      DEBUG("[TABLE](fn) moved leaf bit %d moved leaf key:\n", fn->t_bit); 
      /*, (addr_bit(&fn->t_key, plen)?"right":"left"));*/
      PRINT_ADDR(&fn->t_key);
      return (0);
   }
   DEBUG("[ROUTE]table_add_1: OUT with failure\n");
   return (-1);			                /* failed to add a rt_node */
}
/*--END FUNCTION--(table_add_1)----------------------------------------------*/

/*--START FUNCTION--(table_lookup_1)----------------------------------------- 
 |
 | Lookup a route to destination *addr.
 |
 | Input: an ipv6 address
 |
 | Returns a table leaf.
 |
 --------------------------------------------------------------------------*/
struct table_leaf *table_lookup_1(struct in6_addr *addr, 
				  struct gen_table *table, 
				  __u16 nobacktrack) {
   struct table_node *fn, *next;
   int dir;

   DEBUG("[TABLE]rt_lookup_1: IN table = %p nbck = %d addr =\n", 
	 table, nobacktrack);
   PRINT_ADDR(addr);

   fn = table->gt_root;
   for (;;) {			/* walk down the tree as far as possible */
      dir = addr_bit(addr, fn->t_bit);
      next = dir ? fn->t_right : fn->t_left;
      if (next) {
	 fn = next;
	 continue;
      }
      break;
   }
   
   /* 
    * ND, flow lookups
    */
   if (nobacktrack) {
      if (fn->t_flags & RTN_BACKTRACK) {
	 if ((fn->t_bit == 128) && SAME_ADDR6(*addr, fn->t_key)) {
	    DEBUG("[TABLE]table_lookup_1: OUT match addr:\n");
	    PRINT_ADDR(&fn->t_key);
	    atomic_inc(&((struct rt_dest *)fn->t_leaf->tl_data)->rt_ref);
	    return fn->t_leaf;
	 } 
      }
      goto out1;
   }
   
   while ((fn->t_flags & RTN_ROOT) == 0) {	/* while not at root... */
      if (fn->t_flags & RTN_BACKTRACK)	/* we can backtrack... */
	 if (pfx_match(&fn->t_key, addr, fn->t_bit)) {
	    DEBUG("[TABLE]table_lookup_1: OUT match bit = %d addr:\n", 
		  fn->t_bit);
	    PRINT_ADDR(&fn->t_key);
	    atomic_inc(&((struct rt_dest *)fn->t_leaf->tl_data)->rt_ref);
	    return fn->t_leaf;
	 }      
      DEBUG("[TABLE]table_lookup_1: |up|\n");	
      fn = fn->t_parent;
   }

   if (table->gt_default) {
      DEBUG("[TABLE]table_lookup_1: OUT default addr: OUT\n");
      PRINT_ADDR(&((struct rt_dest *)table->gt_default->tl_data)->rt_addr);
      atomic_inc(&((struct rt_dest *)table->gt_default->tl_data)->rt_ref);
      return (void *)table->gt_default->tl_data;
      /* Assume only one default */
   }
out1:
   DEBUG("[TABLE]table_lookup_1: OUT no match\n");
   return NULL;
}
/*--END FUNCTION--(table_lookup_1)-----------------------------------------*/

/*--START FUNCTION--(table_del_3)------------------------------------------
 |
 | Called to trim the tree of intermediate nodes when possible
 |
 | INPUT: table_node to be deleted.
 | CALLED BY: table_del_2.
 -----------------------------------------------------------------------*/
static void table_del_3(struct table_node *fn, struct gen_table *table) {
   register struct table_node *tmp;
   register int dir      = 0;
   register int children = 0;
   
   DEBUG("[TABLE]table_del_3: IN bit = %d key:\n", fn->t_bit);
   PRINT_ADDR(&fn->t_key);
   /*
    *      0 or one children:
    *              delete the node
    *      2 children:
    *              move the bit down
    */
   if (fn->t_left) {
      children++;
   }
   if (fn->t_right) {
      children++;
      dir = 1;
   }
   if (children < 2) {
      tmp = fn->t_parent;
      if (tmp->t_left == fn) {
	 tmp->t_left = dir ? fn->t_right : fn->t_left;
	 if (children)		/* tmp->left != NULL */
	    tmp->t_left->t_parent = tmp;
      } else {
	 tmp->t_right = dir ? fn->t_right : fn->t_left;
	 if (children)		/* tmp->right != NULL */
	    tmp->t_right->t_parent = tmp;
      }
      /* 
       *      try to colapse on top
       */
      if (fn->t_parent && ((fn->t_parent->t_flags & RTN_ROOT)==0)) { 
	 if ((fn->t_parent->t_flags & (RTN_BACKTRACK)) == 0)
	    table_del_3(fn->t_parent, table);
      }
      kfree(fn);
      atomic_dec(&table->gt_count);	
      DEBUG("[TABLE]table_del_3: OUT, node deleted. table count: %d\n", 
	    table->gt_count); 
      return;
   }
   /* Exactly two children ... */
   /* bit = addr_diff(&fn->left->leaf->rd, 
      &fn->right->leaf->rd, fn->rn_bit);
      
      fn->rn_bit = bit; */
   fn->t_flags &= ~RTN_BACKTRACK;	/* turn off backtrack bit */
   fn->t_leaf = NULL;
   COPY_ADDR6(fn->t_key, fn->t_left->t_key);
   DEBUG("[TABLE]table_del_3: OUT, fn updated, key:\n");
   PRINT_ADDR(&fn->t_key);
   return;
}
/*--END FUNCTION--(table_del_3)---------------------------------------------*/
/*--START FUNCTION--(table_del_2)---------------------------------------------
 |
 | Removes a route destination entry.
 | INPUT: an ipv6 address *addr, 
 |        its prefix length, 
 |        the route gateway, 
 |        the route device.
 | ASSUMPTION: 
 | Returns: the radix node where the entry was deleted or NULL if not found.
 | CALLED BY: rt_del_1
 |       
--------------------------------------------------------------------------*/
static struct table_node *table_del_2(struct in6_addr *addr, 
				      __u32 prefixlen,
				      struct in6_addr *gw, 
				      struct device *dev,
				      struct gen_table *table) {
   struct table_node *fn;
   DEBUG("[TABLE]table_del_2: IN \n");
   /*
    * Run the tree
    */
   for (fn = table->gt_root; fn;) {
      int dir;

      if ((fn->t_flags & RTN_BACKTRACK) &&
	  prefixlen == fn->t_bit &&
	  pfx_match(&fn->t_key, addr, fn->t_bit)) {
	 /* We've found a matching rt_node */
	 break;
      }
      /* Go down the tree */
      DEBUG("BCKTCK: %d, PREF: %d, BIT: %d MATCH %d \n",
	     ((fn->t_flags & RTN_BACKTRACK) ? 1 : 0),
	     prefixlen, fn->t_bit,
	     ((pfx_match(&fn->t_key, addr, fn->t_bit) ? 1 : 0)));
      dir = addr_bit(addr, fn->t_bit);
      fn = dir ? fn->t_right : fn->t_left;
   }
   /* 
    *  search matching rt_node leaves...
    */
   if (fn) {
      struct table_leaf *back = NULL;
      struct table_leaf *lf;

      for (lf = fn->t_leaf; lf; lf = lf->tl_next) {
	 if (SAME_ADDR6(*addr, ((struct rt_dest *)lf->tl_data)->rt_addr)) {
	    /* Detach this entry */
	    if (back == NULL) fn->t_leaf = lf->tl_next;
	    else back->tl_next = lf->tl_next;
	    /* Schedule it for deletion */
	    leaf_release(table, lf);	    
	    DEBUG("[TABLE]rt_del_2: OUT found\n");
	    return fn;
	 }
	 back = lf;
      }
   }
   DEBUG("[TABLE]table_del_2: OUT not found\n");
   return NULL;
}
/*--END FUNCTION--(table_del_2)----------------------------------------------*/

/*--START FUNCTION--(table_del_1)---------------------------------------------
 |
 | Input: an ipv6 address,
 |        its prefix length
 |        a gateway address for that destination.
 |        the device to that destination.
 |
 |--------------------------------------------------------------------------*/
int table_del_1(struct in6_addr *addr,
		       __u32 prefixlen,
		       struct in6_addr *gw,
		       struct device *dev,
		       struct gen_table *table) {

   struct table_node *fn;
   DEBUG("[TABLE]table_del_1: IN table: %p addr:\n", table);
   PRINT_ADDR(addr);      
   
   if (!prefixlen) {		/* default route */
      struct table_leaf *tmp;	/* FIXME: Assume only one default */
      tmp = table->gt_default;
      if (tmp) {
	 /* dev ? */
	 table->gt_default = tmp->tl_next;
	 leaf_release(table, tmp);
	 DEBUG("[TABLE]table_del_1: OUT default\n");
	 return (0);
      } else {
	 DEBUG("[TABLE]table_del_1: OUT default missing\n");
	 return -ENXIO;
      }
   }
  
   fn = table_del_2(addr, prefixlen, gw, dev, table);
   if (fn == NULL) {
      DEBUG("[TABLE]table_del_1: OUT (NOENT)\n");
      return -ENOENT;		/* No entry found by rt_del_2 */
   }
   if (fn->t_leaf == NULL) {	
      /* No more rt_dest struct linked 
       * Delete rt_node
       */
      table_del_3(fn, &routing_table);
   }
   DEBUG("[TABLE]table_del_1: OUT \n");
   return (0);
}
/*--END FUNCTION--(table_del_1)----------------------------------------------*/

/*--START FUNCTION--(table_flush_3)------------------------------------------
 |
 | Flush a route node with leaves...
 |
 -------------------------------------------------------------------------*/
static void table_flush_3(struct table_node *fn, struct gen_table *table)
{
   struct table_leaf *leaf;
   DEBUG("[TABLE]table_flush_3 : IN\n");
   /* Flush all leaves */
   if (fn->t_flags & RTN_BACKTRACK) {	/* Flush all leaves */
      for (leaf = fn->t_leaf; leaf;) {
	 fn->t_leaf = leaf->tl_next;
	 leaf_release(table, leaf);
	 leaf = fn->t_leaf;
      }
   }
   /* Flush route node */
   if (fn->t_parent->t_left == fn)
      fn->t_parent->t_left = NULL;
   else
      fn->t_parent->t_right = NULL;
   kfree(fn);
   atomic_dec(&table->gt_count);
   DEBUG("[TABLE]table_flush_3 : OUT\n");
   return;
}

/*--START FUNCTION--(table_flush_2)------------------------------------------
 |
 | Flush the route table...
 |
 -------------------------------------------------------------------------*/
static void table_flush_2(struct table_node *fn, struct gen_table *table) {
   DEBUG("[TABLE]table_flush_2 : IN\n");
   if (fn->t_left)
      table_flush_2(fn->t_left, table);

   if (fn->t_right)
      table_flush_2(fn->t_right, table);

   if (fn != table->gt_root)
      table_flush_3(fn, table);
   DEBUG("[TABLE]table_flush_2 : OUT\n");
   return;
}

/*--START FUNCTION--(table_flush_1)------------------------------------------
 |
 | Flush a table...
 | Could be replaced by table_flush_2(&table_root)
 -------------------------------------------------------------------------*/
void table_flush_1(struct gen_table *table)
{
   struct table_node *fn;
   DEBUG("[TABLE]table_flush_1 : IN table: %p\n", table);
   while (table->gt_lock)
      sleep_on(&table->gt_wait);
   table_fast_lock(table);
   fn = table->gt_root;
   table_flush_2(fn, table);
   table_unlock(table);
   wake_up(&table->gt_wait);
   DEBUG("[TABLE]table_flush_1 : OUT\n");
   return;
}

