/*
 * DRET-IPv6  An implementation of the TCP/IP protocol suite for the LINUX
 *              operating system.  INET6 is implemented using the BSD Socket
 *              interface as the means of communication with the user level.
 *
 * Contacts:    
 *              INRIA      <Christophe.Diot@sophia.inria.fr>
 *              MASI       <Eric.Horlait@masi.ibp.fr>
 *              This software has been developped with the financial support
 *              of DRET (French Military Research Agency).
 *
 * Version:    $Id$ 
 *	
 * Authors:	
 *	        Benoit Brodard	<Brodard@sophia.inria.fr>
 *
 *
 * Fixes:
 *
 *
 * Description:
 *
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *              This notice must be copied with the distributed package.
 *
 *              Portions of the software are derived from  various 
 *              networking code publicly available, mainly:
 *                      NRL IPv6 code 
 *                      NetBSD code from INRIA (Francis.Dupont@inria.fr)
 *                      Pedro Roque's Linux version (Roque@di.fc.up.pt)
 *
 */

/** Contents *************************************************************

   ipv6_setsockopt
   ipv6_getsockopt
   
 ************************************************************************/

#include <asm/uaccess.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/sched.h>
#include <linux/net.h>
#include <linux/in.h>
#include <linux/netdevice.h>

#include <net/sock.h>
#include <net/snmp.h>
#include <net/tcp.h>
#include <net/udp.h>

#include <linux/dret/in6.h>
#include <net/dret/ipv6.h>
#include <net/dret/nd.h>
#include <net/protocol.h>
#include <net/dret/route6.h>
#include <net/dret/addrconf.h>
#include <net/dret/sit.h>


extern struct proto_ops		inet_stream_ops;
extern struct proto_ops		inet_dgram_ops;

/*
 *	address family specific functions
 */
extern struct tcp_func	ipv4_specific;



int ipv6_setsockopt(struct sock *sk, int level, int optname, char *optval,
		    int optlen) {
   struct ipv6_pinfo *np = &sk->net_pinfo.af_inet6;
   int val, err;
   
   if(level != IPPROTO_IPV6)
      return -EOPNOTSUPP;
   
   if (optval == NULL) {
      val = 0;
   } else {
      err = get_user(val, (int *) optval);
      if (err)
	 return (err);
   }
	
   switch (optname) {
   case IPV6_HDRINCL:
      if ((val < 0) || (val > 2)) /* FIXME semantics 0, 1, 2 */
	 return -EINVAL;
      sk->ip_hdrincl = val;
      break;
      
   case IPV6_ADDRFORM:
      /* convert from ipv4 to ipv6 or vice-versa */
      
      if (val == PF_INET){
	 if (sk->protocol != IPPROTO_UDP &&
	     sk->protocol != IPPROTO_TCP){			
	    return -EOPNOTSUPP;
	 }
	 
	 if (sk->state != TCP_ESTABLISHED){
	    return -ENOTCONN;   
	 }
			
	 if (!IS_MAPPEDADDR6(np->pi6_fl.fll_da)){
	    return -EADDRNOTAVAIL;
	 }
	 
	 if (sk->protocol == IPPROTO_TCP){
	    struct tcp_opt *tp = &(sk->tp_pinfo.af_tcp);
	    
	    sk->prot = &tcp_prot;
	    tp->af_specific = &ipv4_specific;
	    sk->socket->ops = &inet_stream_ops;
	 }else{
	    sk->prot = &udp_prot;
	    sk->socket->ops = &inet_dgram_ops;
	 }	 
      } else{
	 return -EINVAL;
      }
      break;
      
      /*
       * Ancillary data reception
       */
   case IPV6_PKTINFO:
      if ((val) && (val != 1))
	 return -EINVAL;
      np->pktinfo = val;
      break;          
 
   case IPV6_HOPLIMIT:
      if ((val) && (val != 1))
	 return -EINVAL;
      np->rxhlim = val;
      break;
   
   case IPV6_HOPOPTS:
      if ((val) && (val != 1))
	 return -EINVAL;
      np->hopopts = val;
      break;
      
   case IPV6_DSTOPTS:
      if ((val) && (val != 1))
	 return -EINVAL;
      np->dstopts = val;
      break;
      
   case IPV6_SRCRT:
      if ((val) && (val != 1))
	 return -EINVAL;
      np->recvsrcrt = val;
      break;
    
   case IPV6_UNICAST_HOPS:
      if ((val < 1) || (val > 255))
	 return -EINVAL;
      np->mcast_hops = val;
      break;
      
   case IPV6_MULTICAST_HOPS:
      if ((val < 1) || (val > 255))
	 return -EINVAL;
      np->mcast_hops = val;
      break;
      
   case IPV6_MULTICAST_LOOP:
      if ((val) && (val != 1))
	 return -EINVAL;
      np->mc_loop = val;
      break;
		
   case IPV6_MULTICAST_IF:
      {
	 unsigned int index;
	 struct device *dev;
	 struct ipv6_pinfo *info = &sk->net_pinfo.af_inet6;
	 
	 err = copy_from_user(&index, optval, sizeof(index));
	 if (err)
	    return -EFAULT;
	 if ((dev = dev_get_by_index(index)) && 
	     (dev->flags && IFF_MULTICAST)){
	    printk(KERN_DEBUG "[SETSOCKOPT]Multicast set on %s \n",
		   dev->name); 
	    info->mc_if = dev;
	 } else
	    return -EINVAL;
      }
   break;
   
   case IPV6_ADD_MEMBERSHIP:
      {
	 struct ipv6_mreq mreq;
	 struct device *dev;
	 
	 err = copy_from_user(&mreq, optval, sizeof(struct ipv6_mreq));
	 if (err)
	    return -EFAULT;
	 if ((dev = dev_get_by_index(mreq.ipv6mr_interface)))
	    return(ipv6_dev_mc_inc(dev, &mreq.ipv6mr_multiaddr));
	 else
	    return -EINVAL;
      }
   
   case IPV6_DROP_MEMBERSHIP:
      {
	 struct ipv6_mreq mreq;
	 struct device *dev;
	 
	 err = copy_from_user(&mreq, optval, sizeof(struct ipv6_mreq));
	 if (err)
	    return -EFAULT;
	 if ((dev = dev_get_by_index(mreq.ipv6mr_interface)))
	    return(ipv6_dev_mc_dec(dev, &mreq.ipv6mr_multiaddr));
	 else
	    return -EINVAL;
	 break;
      }

   case IPV6_CHECKSUM: {
      struct raw6_tpinfo *raw6_opt = &sk->tp_pinfo.raw6;
      
      if ((raw6_opt->ck_offset = val) >= 0)
	 raw6_opt->checksum = 1;	/* Checksum enabled */
      else
	 raw6_opt->checksum = 0; /* Checksum disabled */
      break;
   }

   default:
      return(-ENOPROTOOPT);   
   }
   return 0;
}

int ipv6_getsockopt(struct sock *sk, int level, int optname, char *optval,
		    int *optlen)
{
   struct ipv6_pinfo *np = &sk->net_pinfo.af_inet6;
   int val, err, len;

   if (level != IPPROTO_IPV6)
      return -EOPNOTSUPP;

   switch (optname) {
   case IPV6_HDRINCL:
      val = sk->ip_hdrincl;
      break;

   case IPV6_SRCRT:
      val = np->recvsrcrt;
      break;

   case IPV6_PKTINFO:
      val= np->pktinfo;
      break;

   case IPV6_UNICAST_HOPS:
      val = np->hop_limit;
      break;

   case IPV6_MULTICAST_HOPS:
      val = np->mcast_hops;
      break;

   case IPV6_MULTICAST_LOOP:
      val = np->mc_loop;
      break;

   case IPV6_MULTICAST_IF:
      len = strlen(np->mc_if->name);
      err = put_user(len, (int *) optlen);
      if (err)
	 return (err);
      err = copy_to_user((void *) optval, np->mc_if->name, len);
      if (err)
	 return (err);
      return 0;

   case IPV6_CHECKSUM:{
	 struct raw6_tpinfo *raw6_opt = &sk->tp_pinfo.raw6;

	 val = raw6_opt->ck_offset;
	 break;
      }
   default:
      return (-ENOPROTOOPT);
   }
   err = put_user(sizeof(int), (int *) optlen);
   if (err)
      return (err);
   err = put_user(val, (int *) optval);
   return (err);
}
